/*
 * Copyright (c) 2005-2006 Carnegie Mellon University and Intel Corporation.
 * All rights reserved.
 * See the file "LICENSE" for licensing terms.
 */

#include "gtcd.h"
#include "chunker/chunkerPlugin_all.h"
#include "xfer/xferPlugin_all.h"
#include "storage/storagePlugin_all.h"
#include "configfile.h"

static str gtcd_listen_sock("/tmp/gtcd.sock");
gtcd *gtcdp;
qhash<str, sPluginNew_cb> sPluginTab;
qhash<str, xPluginNew_cb> xPluginTab;

static void
accept_connection(int fd)
{
    struct sockaddr_un sun;
    socklen_t sunlen = sizeof(sun);
    bzero(&sun, sizeof(sun));

    int cs = accept(fd, (struct sockaddr *) &sun, &sunlen);
    if (fd < 0) {
        if (errno != EAGAIN)
            warn << "accept; errno = " << errno << "\n";
        return;
    }

    vNew client(cs, sun, wrap(gtcdp, &gtcd::dispatch));
}

static void
gtcd_start(str gtcd_listen_sock)
{
    mode_t m = umask(0);

    int fd = unixsocket(gtcd_listen_sock);
    if (fd < 0 && errno == EADDRINUSE) {
        /* XXX - This is a slightly race-prone way of cleaning up after a
         * server bails without unlinking the socket.  If we can't connect
         * to the socket, it's dead and should be unlinked and rebound.
         * Two daemons could do this simultaneously, however. */
        int xfd = unixsocket_connect(gtcd_listen_sock);
        if (xfd < 0) {
            unlink(gtcd_listen_sock);
            fd = unixsocket(gtcd_listen_sock);
        }
        else {
            warn << "closing the socket\n";
            close (xfd);
            errno = EADDRINUSE;
        }
    }
    if (fd < 0)
        fatal ("%s: %m\n", gtcd_listen_sock.cstr ());

    close_on_exec(fd);
    make_async(fd);

    umask(m);

    listen(fd, 150);

    /* XXX: Daemonize */

    warn << progname << " (DOT Generic Transfer Client) version "
        << VERSION << ", pid " << getpid() << "\n";

    fdcb(fd, selread, wrap(accept_connection, fd));
}

template <class T>
static void
print_plugin(const str &s, T *cb)
{
    warnx << s << " ";
}

static void
print_plugins()
{
    warnx << "Available Storage Plugins: ";
    sPluginTab.traverse(wrap(print_plugin<sPluginNew_cb>));
    warnx << "\n";

    warnx << "Available Transfer Plugins: ";
    xPluginTab.traverse(wrap(print_plugin<xPluginNew_cb>));
    warnx << "\n";
}

static void
usage()
{
    fprintf(stderr, "usage:  gtcd [-h] [-f configfile] [-p gtcd_listen_sock] [-v paramfile]\n");
}

static void
help()
{
    usage();
    fprintf(stderr,
	    "    -h .......... help (this message)\n"
	    "    -f file ..... configuration file\n"
	    "    -v file ..... parameter file\n"
	    "    -p listen ... socket on which to listen (def /tmp/gtcd.soc)\n"
	    "    -D <level> .. debug / do not daemonize\n"
	    "\n");
    print_plugins();
}

template <class T, class U>
T *plumb(vec<str> *p_list, U *p_tab)
{
    T *p = NULL;
    vec<T *> pvec;
    qhash<str, T *> p_ptr;
    
    while (p_list->size()) {
        str conf = p_list->pop_back();
	str plg_list = p_list->pop_back();
	str plugin = p_list->pop_back();

	dwarn(DEBUG_INIT) << "Adding Plugin: " << plugin
			  << " chained to "
			  << plg_list << " with params " << conf << "\n";
	
        if (!(*p_tab)[plugin]) {
            warn("plugin (%s) does not exist\n", plugin.cstr());
            print_plugins();
            exit(-1);
        }

	if (plg_list == "" ||
	    plg_list == "null") {
	    dwarn(DEBUG_INIT) << "Passing NULL\n";
	    p = NULL;
	}
	else {
	    if (strchr(plg_list.cstr(), ',')) {
		char *name = strdup(plg_list.cstr());
		char *temp = strtok(name,",");
		
		while (temp != NULL) {
		    str plg(temp);
		    		    
		    p = (*p_ptr[plg]);
		    if (!p) 
			fatal << "Plugin " << plg << " not instantiated\n";
		    else
			dwarn(DEBUG_INIT) << "passing " << plg << "\n";

		    pvec.push_back(p);
		    temp = strtok(NULL,",");
		}
	    }
	    else {
		//just one plugin, not a vec
		p = (*p_ptr[plg_list]);
		if (!p) 
		    fatal << "Transfer plugin " << plg_list << " not instantiated\n";
		else
		    dwarn(DEBUG_INIT) << "passing " << plg_list << "\n";
	    }
	}
	if (pvec.size()) {
	    p = NULL;
	    p = (*(*p_tab)[plugin])(gtcdp, p);
	    p->set_more_plugins(pvec);
	}
	else {
	    p = (*(*p_tab)[plugin])(gtcdp, p);
	}

	p_ptr.insert(plugin, p);
	
        if (!p->configure(conf))
            fatal("Plugin (%s) configuration failed\n", plugin.cstr());
    }
    return p;
}

static void
instantiate_plugins(str configfile)
{
    storagePlugin *sp = NULL;
    xferPlugin *xp = NULL;
        
    vec<str> sp_list;
    vec<str> xp_list;

    if (parse_config(configfile, &sp_list, &xp_list))
        fatal << "Cannot parse config file\n";

    sp = plumb<storagePlugin, typeof(sPluginTab)>(&sp_list, &sPluginTab);
    gtcdp->set_storagePlugin(sp);

    xp = plumb<xferPlugin, typeof(xPluginTab)>(&xp_list, &xPluginTab);
    gtcdp->set_xferPlugin(xp);
    
    gtcdp->set_chunkerPlugin(New chunkerPlugin_default(gtcdp->sp));
}

int
main(int argc, char * argv[])
{
    storagePlugin_maker _spm; // populate sPluginTab
    xferPlugin_maker _xpm;    // populate xPluginTab

    char ch;
    bool daemonize = true;
    str configfile;
    str paramfile;

    setprogname(argv[0]);

    while ((ch = getopt(argc, argv, "D:hp:f:v:")) != -1)
        switch(ch) {
	case 'D':
	    daemonize = false;
	    if (set_debug(optarg)) {
		exit(-1);
	    }
	    break;
        case 'p':
            gtcd_listen_sock = optarg;
            break;
        case 'f':
            configfile = optarg;
            break;
	case 'v':
	    paramfile = optarg;
	    break;
	case 'h':
	    help();
	    exit(0);
	default:
	    usage();
	    exit(-1);
	}

    gtcd_start(gtcd_listen_sock);

    /* Configure the graph of plugins */
    gtcdp = New gtcd();

    parse_paramfile(paramfile);
    instantiate_plugins(configfile);

    amain();
}
