/*
 * Copyright (c) 2005-2006 Carnegie Mellon University and Intel Corporation.
 * All rights reserved.
 * See the file "LICENSE" for licensing terms.
 */

#include "configfile.h"

enum config_state_t { CONFIG_NONE, CONFIG_STORAGE, CONFIG_XFER };

bool
parse_config(str file, vec<str> *sp_list, vec<str> *xp_list)
{
    assert(sp_list);
    assert(xp_list);

    if (!file) {
        warn << "No config file specified!\n";
        return true;
    }

    parseargs pa(file);
    int line;
    vec<str> av;

    enum config_state_t state = CONFIG_NONE;
    bool saw_storage = false;
    bool saw_xfer = false;
    bool errors = false;

    while (pa.getline(&av, &line)) {
        if (!strcasecmp(av[0], "[storage]")) {
            if (saw_storage) {
                errors = true;
                warn << file << ":" << line << ": Duplicate [storage] section\n";
            }
            else {
                state = CONFIG_STORAGE;
                saw_storage = true;
            }
        }
        else if (!strcasecmp(av[0], "[transfer]")) {
            if (saw_xfer) {
                errors = true;
                warn << file << ":" << line << ": Duplicate [transfer] section\n";
            }
            else {
                state = CONFIG_XFER;
                saw_xfer = true;
            }
        }
        else {
            if (state == CONFIG_NONE) {
                errors = true;
                warn << file << ":" << line << ": Not inside a plugin section\n";
            }
            else if (state == CONFIG_STORAGE) {
                sp_list->push_back(av.pop_front());
                if (av.size()) {
		    //for plg_list
		    sp_list->push_back(av.pop_front());
		    //for conf
                    sp_list->push_back(join(" ", av));
		}
                else {
		    //for plg_list and conf
                    sp_list->push_back("");
		    sp_list->push_back("");
		}
		    
            }
            else if (state == CONFIG_XFER) {
                xp_list->push_back(av.pop_front());
                if (av.size()) {
		    //for plg_list
		    xp_list->push_back(av.pop_front());
		    //for conf
                    xp_list->push_back(join(" ", av));
		}
                else {
		    //for plg_list and conf
		    xp_list->push_back("");
		    xp_list->push_back("");
		}
            }
        }
    }

    if (!saw_storage) {
        warn << file << ": Missing [storage] section\n";
        errors = true;
    }
    if (!saw_xfer) {
        warn << file << ": Missing [transfer] section\n";
        errors = true;
    }

    if (errors)
        warn << "Errors processing file: " << file << "\n";

    return errors;
}

//unsigned int MAX_OPEN_FDS = 800;
unsigned int CHUNK_SIZE = 32768;
int NUM_SHINGLES = 30;
unsigned int DEFAULT_DHT_PORT = 5852;
unsigned int NUM_ALLOWED_REQS = 10;
unsigned int NUM_ALLOWED_SRCS = 10;

int REFRESH_OID_LOOKUP = 10;
int REFRESH_BITMAP = 1;
unsigned int MAX_PEERS_ALBT = 50;
int REFRESH_BITMAP_ALBT = 2;

int LARGE_COST; //huge cost
int THRESH_CHANGE = 10;
int MIN_BLOCKS = 2;

//operations
str AIOD_PATH = "./sfslite/async/aiod";
str NEW_ROOT = "/";
double ALPHA = 0.2;
int XDISK_CACHE_SIZE = 10485760;
double DIR_STAT_COST = 0.05;
double CHUNK_READ_HASH_COST = 0.001;
double ADJ_WT = 4.08;
double CACHE_CHECK_COST = 0;
unsigned int LOAD_HASH_FACTOR = 1;
unsigned int LOAD_STAT_FACTOR = 1;

int REOPT_TIME_SEC = 1;
int REOPT_TIME_NSEC = 000;
int CLIENT_OUTSTANDING_THRESHOLD = 0;

unsigned int MAX_CHUNKS_MEM_FOOTPRINT = 104857600;
unsigned int CHUNKS_HIGH_WATER = ((size_t)(((double)MAX_CHUNKS_MEM_FOOTPRINT * CHUNKS_HIGH_WATER)/100));
unsigned int CHUNKS_LOW_WATER = ((size_t)(((double)MAX_CHUNKS_MEM_FOOTPRINT * CHUNKS_LOW_WATER)/100));

int CHUNK_PURGER_SEC = 1200;
int CHUNK_PURGER_NSEC = 000;

int mode = OPTIMAL;
unsigned int AIOD_READ_SIZE = 0x8000;

bool
parse_paramfile(str file)
{
    if (!file) {
        warn << "No parameter file specified!\n";
        return false;
    }

    LARGE_COST = 999999; //huge cost
    
    parseargs pa(file);
    int line;
    vec<str> av;

    warnx << "----------------------------------------------------\n";
    warnx << "----------------------------------------------------\n";
    
    while (pa.getline(&av, &line)) {

// 	if (!strcasecmp(av[0], "MAX_OPEN_FDS")) {
// 	    MAX_OPEN_FDS = atoi(av[2]);
// 	    warnx << "Open fds are " << MAX_OPEN_FDS << "\n";
// 	    continue;
// 	}
	
	if (!strcasecmp(av[0], "CHUNK_SIZE")) {
	    CHUNK_SIZE = atoi(av[2]);
	    warnx << "Chunk is " << CHUNK_SIZE << "\n";
	    continue;
	}

	if (!strcasecmp(av[0], "NUM_SHINGLES")) {
	    NUM_SHINGLES = atoi(av[2]);
	    warnx << "Shingles are " << NUM_SHINGLES << "\n";
	    continue;
	}
	if (!strcasecmp(av[0], "DEFAULT_DHT_PORT")) {
	    DEFAULT_DHT_PORT = atoi(av[2]);
	    warnx << "Dht port is " << DEFAULT_DHT_PORT << "\n";
	    continue;
	}
	
	if (!strcasecmp(av[0], "NUM_ALLOWED_REQS")) {
	    NUM_ALLOWED_REQS = atoi(av[2]);
	    warnx << "Reqs are " << NUM_ALLOWED_REQS << "\n";
	    continue;
	}
	if (!strcasecmp(av[0], "NUM_ALLOWED_SRCS")) {
	    NUM_ALLOWED_SRCS = atoi(av[2]);
	    warnx << "Srcs are " << NUM_ALLOWED_SRCS <<"\n";
	    continue;
	}
	
	if (!strcasecmp(av[0], "REFRESH_OID_LOOKUP")) {
	    REFRESH_OID_LOOKUP = atoi(av[2]);
	    warnx << "Lookup oid " << REFRESH_OID_LOOKUP << " seconds\n";
	    continue;
	}
	if (!strcasecmp(av[0], "REFRESH_BITMAP")) {
	    REFRESH_BITMAP = atoi(av[2]);
	    warnx << "Bitmap " << REFRESH_BITMAP << " seconds\n";
	    continue;
	}
	if (!strcasecmp(av[0], "MAX_PEERS_ALBT")) {
	    MAX_PEERS_ALBT = atoi(av[2]);
	    warnx << "Max sources are " << MAX_PEERS_ALBT << "\n";
	    continue;
	}
	if (!strcasecmp(av[0], "REFRESH_BITMAP_ALBT")) {
	    REFRESH_BITMAP_ALBT = atoi(av[2]);
	    warnx << "Refresh bitmap " << REFRESH_BITMAP_ALBT << " seconds\n";
	    continue;
	}
	
	if (!strcasecmp(av[0], "THRESH_CHANGE")) {
	    THRESH_CHANGE = atoi(av[2]);
	    warnx << "Threshold change " << THRESH_CHANGE << " %\n";
	    continue;
	}
	if (!strcasecmp(av[0], "MIN_BLOCKS")) {
	    MIN_BLOCKS = atoi(av[2]);
	    warnx << "Min blocks to net are " << MIN_BLOCKS <<"\n";
	    continue;
	}

	if (!strcasecmp(av[0], "AIOD_PATH")) {
	    AIOD_PATH = av[2];
	    warnx << "Aiod path is " << AIOD_PATH << "\n";
	    continue;
	}
	if (!strcasecmp(av[0], "NEW_ROOT")) {
	    NEW_ROOT = av[2];
	    warnx << "New root is " << NEW_ROOT << "\n";
	    continue;
	}

	if (!strcasecmp(av[0], "ALPHA")) {
	    ALPHA = atof(av[2]);
	    fprintf(stderr, "Alpha is set to %f seconds\n", ALPHA);
	    continue;
	}
	if (!strcasecmp(av[0], "XDISK_CACHE_SIZE")) {
	    XDISK_CACHE_SIZE = atoi(av[2]);
	    warnx << "Xdisk cache size " << XDISK_CACHE_SIZE << " bytes\n";
	    continue;
	}
	if (!strcasecmp(av[0], "DIR_STAT_COST")) {
	    DIR_STAT_COST = atof(av[2]);
	    fprintf(stderr, "Statting cost %f seconds\n", DIR_STAT_COST);
	    continue;
	}
	if (!strcasecmp(av[0], "CHUNK_READ_HASH_COST")) {
	    CHUNK_READ_HASH_COST = atof(av[2]);
	    fprintf(stderr, "Per chunk hash cost %f seconds\n", CHUNK_READ_HASH_COST);
	    continue;
	}
	if (!strcasecmp(av[0], "ADJ_WT")) {
	    ADJ_WT = atof(av[2]);
	    fprintf(stderr, "Weight adjustment for stat cost %f\n", ADJ_WT);
	    continue;
	}
	if (!strcasecmp(av[0], "CACHE_CHECK_COST")) {
	    CACHE_CHECK_COST = atof(av[2]);
	    fprintf(stderr, "Cache check cost %f seconds\n", CACHE_CHECK_COST);
	    continue;
	}
	if (!strcasecmp(av[0], "LOAD_HASH_FACTOR")) {
	    LOAD_HASH_FACTOR = atoi(av[2]);
	    warnx << "Load hash factor " << LOAD_HASH_FACTOR << "\n";
	    continue;
	}
	if (!strcasecmp(av[0], "LOAD_STAT_FACTOR")) {
	    LOAD_STAT_FACTOR = atoi(av[2]);
	    warnx << "Load stat factor " << LOAD_STAT_FACTOR << "\n";
	    continue;
	}
	
	
	if (!strcasecmp(av[0], "MAX_CHUNKS_MEM_FOOTPRINT")) {
	    MAX_CHUNKS_MEM_FOOTPRINT = atoi(av[2]);
	    warnx << "DOT cache size " << MAX_CHUNKS_MEM_FOOTPRINT << " bytes\n";
	    continue;
	}
	if (!strcasecmp(av[0], "CHUNKS_HIGH_WATER")) {
	    CHUNKS_HIGH_WATER = atoi(av[2]);
	    CHUNKS_HIGH_WATER = ((size_t)(((double)MAX_CHUNKS_MEM_FOOTPRINT * CHUNKS_HIGH_WATER)/100));
	    warnx << "DOT cache high " << CHUNKS_HIGH_WATER << " bytes\n";
	    continue;
	}
	if (!strcasecmp(av[0], "CHUNKS_LOW_WATER")) {
	    CHUNKS_LOW_WATER = atoi(av[2]);
	    CHUNKS_LOW_WATER = ((size_t)(((double)MAX_CHUNKS_MEM_FOOTPRINT * CHUNKS_LOW_WATER)/100));
	    warnx << "DOT cache low " << CHUNKS_LOW_WATER << " bytes\n";
	    continue;
	}

	if (!strcasecmp(av[0], "CHUNK_PURGER_SEC")) {
	    CHUNK_PURGER_SEC = atoi(av[2]);
	    warnx << "Chunk purge at " << CHUNK_PURGER_SEC << " seconds\n";
	    continue;
	}
	if (!strcasecmp(av[0], "CHUNK_PURGER_NSEC")) {
	    CHUNK_PURGER_NSEC = atoi(av[2]);
	    warnx << "Chunk purge at " << CHUNK_PURGER_NSEC << " nanoseconds\n";
	    continue;
	}

	if (!strcasecmp(av[0], "OPERATION_MODE")) {
	    mode = atoi(av[2]);
	    warnx << "Mode is " << mode << "\n";
	    continue;
	}
    }

    warnx << "----------------------------------------------------\n";
    warnx << "----------------------------------------------------\n";
    
    return true;
}
