#include "gcp.h"
struct timeval start;
static struct timeval end;
static int td_pending;
static bool err_flag = false;

static void
finish(svccb *sbp, int outfd, str name, ptr<gcp_sput_arg> td, str destpath,
       ref<aclnt> gtc_c, str err)
{
    td_pending--;

    if (err) {
	warnx << "Error " << err << "\n"; 
	err_flag = true;
    }

    assert(outfd == 0);
    str n = strbuf() << name << "." << getpid();
    if (return_fd(n)) {
	if (rename(n.cstr(), name.cstr()) < 0) 
	    warn << "Rename failed\n";
    }

    warn << "Just finished " << name << "\n";

    if (td)
	get_oid(td, destpath, sbp, gtc_c);
    
    if (td_pending <= 0) {

	gettimeofday(&end, NULL);
	fprintf(stderr, "time for GCP_PROC_PUT start-finish == %.2f\n",
		timeval_diff(&start, &end));

	gcp_put_res res(true);
	
	if (err_flag) {
	    res.set_ok(false);
	    warnx << "get failed\n";
	    *res.errmsg = err;
	}

	if (sbp)
	    sbp->replyref(res);
	else
	    exit(err_flag ? 1 : 0);
    }
}

void
generate_disk_hint(str compname, unsigned int size,
		   int modtime, oid_hint *h, str ig_path)
{
    str name;
    str target_path;
    char *line, *brkt;
    
    //clean the path
    str newname = "";
    char *ptr = strdup(compname);
    if (*ptr == '/')
	newname = strbuf() << "/";

    //warnx << "path is " << newname << "\n";
     
    //1. remove two consec slashes
    //2. if there is a ./ remove it
    for (line = strtok_r(ptr, "/", &brkt);  
         line; line = strtok_r(NULL, "/", &brkt)) {
        str a(line, strlen(line));
        if (a == "." ) continue;
	if (newname == "/" || newname == "")
	    newname = strbuf() << newname << a;
	else
	    newname = strbuf() << newname << "/" << a ;
    }

    dwarn(DEBUG_CLIENT) << "Clean path is " << newname << "\n";
    
    //separate comp name into name and path
    char *dup = strdup(newname);
    char *lastslash = strrchr(dup, '/');
    if (lastslash) {
        name = strbuf() << (lastslash + 1);
	lastslash++;
	*lastslash = '\0';
	target_path = strbuf() << dup;
    }
    else {
        name = compname;
	target_path = strbuf() << "./";
    }
    
    //name of the file, target directory, size, mod time, pid
    name = strbuf() << "xdisk://" << name << ":"
		    << target_path << ":" << size
		    << ":" << modtime << ":" << getpid()
		    << ":" <<ig_path;
    h->name = name;
    dwarn(DEBUG_CLIENT) << "Xdisk hint is " << h->name << "\n";
}

static void
setup_path(file_struct fs, const char *path)
{
    struct stat sb;
    if (stat(path, &sb) == 0) {
	if (!S_ISDIR(sb.st_mode))
	    fatal("Not a dir: %s\n", path);
	return;
    }
    
    dwarn(DEBUG_CLIENT) << "setup_path: " << path << "\n";

    if (mkdir(path, S_IRWXU) < 0)
        warn("mkdir failed: %m: %s\n", path);
    else if (chmod(path, fs.mode) < 0)
        warn("chmod failed: %m: %s\n", path);
}

static void
setup_symlink(file_struct fs, const char *path, const char *target)
{
    dwarn(DEBUG_CLIENT) << "setup_symlink: " << path
        << " -> " << target << "\n";

    if (strlen(target) == 0)
        warn("empty symlink target: %s\n", path);
    else if (symlink(target, path) < 0)
        warn("symlink failed: %m: %s\n", path);
}

static int
setup_file(file_struct fs, ptr<str> name)
{
    struct stat sb;
    strbuf outf;
    str s;
    int outfd = 0;
       
    dwarn(DEBUG_CLIENT) << "setup_file " << fs.destpath << " and " << fs.filename << "\n";
    
    if (fs.destpath && 0 == stat(fs.destpath, &sb) && (sb.st_mode & S_IFDIR)) {
	str dp = fs.destpath;
	str fn = fs.filename;
	s = outf.fmt("%s/%s", dp.cstr(), fn.cstr());
	*name = s;
	s = strbuf() << name->cstr() << "." << getpid();
	dwarn(DEBUG_CLIENT) << "Name is " << s.cstr() << "\n";
    } 
    else {
	str dp = fs.destpath;
	*name = dp;
	s = outf.fmt("%s.%d", dp.cstr(), getpid());
    }

    dwarn(DEBUG_CLIENT) << "Opening 2 " << s << "\n";

    outfd = get_new_fd(s);
    if (outfd < 0) {
        warn("open failed: %m: %s\n", s.cstr());
        return -1;
    }

    if (fchmod(outfd, fs.mode) < 0) {
        warn("fchmod failed: %m: %s\n", s.cstr());
	close(outfd);
        return -1;
    }

    dwarn(DEBUG_CLIENT) << "Returning " << outfd << "\n";
    return 0;
}

void
get_oid(ptr<gcp_sput_arg> td, str destpath, svccb *sbp,
	ref<aclnt> gtc_c)
{
    if (td->last_index >= td->list.size())
	return;
    
    gcp_put_arg arg = td->list[td->last_index];
    td->last_index++;

    switch (arg.type) {
    case OID_FILE:
    {
        int outfd;
        ptr<str> name = New refcounted<str>;
        
        //adjust the relative path
        arg.file.destpath = strbuf() << destpath << "/" << arg.file.destpath;
                    
        if ((outfd = setup_file(arg.file, name)) >= 0) {
            ref<vec<oid_hint> > hints = New refcounted<vec<oid_hint> >;
            for(unsigned int i = 0;  i < arg.hints.size(); i++) {
                hints->push_back(arg.hints[i]);
            }
            dwarn(DEBUG_CLIENT) << "Final name is " << *name << "\n";
            oid_hint h;
            generate_disk_hint(*name, arg.file.size, arg.file.modtime, &h, destpath);
            hints->push_back(h);
            
            td_pending++;
            
            str fn = strbuf() << *name << "." << getpid();
            vNew get_client(arg.oid, hints, outfd, fn, gtc_c, wrap(finish, sbp, 
                                                                   outfd, *name, td,
                                                                   destpath, gtc_c));
        }
        else {
            gcp_put_res res (false);
            *res.errmsg = "could not open remote file...giving up";
            if (sbp)
                sbp->replyref(res);
            return;
        }
        break;
    }
    case OID_SYMLINK:
    {
        str path = strbuf() << destpath << "/" << arg.file.destpath;
        setup_symlink(arg.file, path, arg.file.filename);
        break;
    }
    case OID_DIR:
    {
        str path = strbuf() << destpath << "/" << arg.file.destpath;
        setup_path(arg.file, path);
        break;
    }
    default:
    {
        gcp_put_res res (false);
        *res.errmsg = "unknown oid_type";
        if (sbp)
            sbp->replyref(res);
        return;
    }
    }
}


void
get_td_done(ptr<suio> buf, svccb *sbp, ref<aclnt> gtc_c,
	    str destpath, str err)
{
    if (err) {
	warnx << err << "\n";
	gcp_put_res res (false);
        *res.errmsg = err;
	if (sbp)
	    sbp->replyref(res);
        else
            exit(1);
	return;
    }
    
    rpc_bytes<> value;
    ptr<gcp_sput_arg> td = New refcounted<gcp_sput_arg>;
    
    value.setsize(buf->resid());
    buf->copyout(value.base(), value.size());
    if (!bytes2xdr(*td, value)) {
        err = "Could not unmarshal tree descriptor";
	warnx << err << "\n";
	gcp_put_res res (false);
        *res.errmsg = err;
	if (sbp)
	    sbp->replyref(res);
        else
            exit(1);
	return;
    }

    err_flag = false;
    td_pending = 0;

    int td_size = td->list.size();
    td_size = min(td_size, MAX_ALLOWED_FILES);

    td->last_index = 0;
    for (int i = 0; i < td_size; i++) {
	get_oid(td, destpath, sbp, gtc_c);
    }
}

void
get_td(gcp_put_arg arg, svccb *sbp, ref<aclnt> gtc_c)
{
    ref<vec<oid_hint> > hints = New refcounted<vec<oid_hint> >;
    for(unsigned int i = 0;  i < arg.hints.size(); i++) 
	hints->push_back(arg.hints[i]);

    str destpath = arg.file.destpath;
    struct stat sb;
    //check if destpath exists
    if (stat(destpath.cstr(), &sb) < 0) {
	warnx << "Path does not exist " << destpath << "\n";
	if (mkdir(destpath.cstr(), S_IRWXU) < 0)
	    fatal << "Cannot create " << destpath << "\n";
    }
    else if (!S_ISDIR(sb.st_mode))
	fatal << "Cannot create " << destpath << "\n";
    
    ptr<suio> buf = New refcounted<suio>;
    vNew get_client(arg.oid, hints, buf, gtc_c,
		    wrap(get_td_done, buf, sbp, gtc_c, destpath));
}

void
get_dispatch(ref<aclnt> gtc_c, svccb *sbp)
{
    if (!sbp) {
	dwarn(DEBUG_CLIENT)("gc::dispatch(): client closed connection\n");
	/* XXX - need something to keep track of our state.  */
	exit(0);
    }

    switch(sbp->proc()) {
    case GCP_PROC_PUT: {
	gettimeofday(&start, NULL);
        int outfd;
        gcp_put_arg *arg = sbp->Xtmpl getarg<gcp_put_arg>();
	ptr<str> name = New refcounted<str>;

	if (arg->type == OID_TREE) {
	    get_td(*arg, sbp, gtc_c);
	}
	else if ((outfd = setup_file(arg->file, name)) >= 0) {
	    assert(arg->type == OID_FILE);
	    ref<vec<oid_hint> > hints = New refcounted<vec<oid_hint> >;
	    for(unsigned int i = 0;  i < arg->hints.size(); i++) {
		hints->push_back(arg->hints[i]);
            }
	    oid_hint h;
	    generate_disk_hint(*name, arg->file.size, arg->file.modtime, &h, "NONE");
	    hints->push_back(h);
	    
	    warn << "Final name is " << *name << "\n";
	    //warn << "Fd is " << outfd << "\n";
	    str fn = strbuf() << *name << "." << getpid();
	    ptr<gcp_sput_arg> td = NULL;
            vNew get_client(arg->oid, hints, outfd, fn, gtc_c, wrap(finish, sbp, 
								    outfd, *name, td, fn, gtc_c));
	}
	else {
	    gcp_put_res res (false);
	    *res.errmsg = "could not open remote file";
	    sbp->replyref(res);
	}
	break;
    }
    default:
	sbp->reject(PROC_UNAVAIL);
	break;
    }
}
