#include "gcp.h"
#include "rxx.h"
#include "parseopt.h"

typedef enum { CLIENT_PUT, CLIENT_PUT_ONLY, CLIENT_GET, CLIENT_GET_OID } client_mode_t;

static ptr<aclnt> gtc_c;
static ptr<aclnt> gcp_c;
static ptr<asrv> gcp_s;
static char **files;
static int nfiles;
static client_mode_t mode;

struct timeval gcp_start;

extern struct timeval start;
bool passdp;

static void
usage()
{
    fprintf(stderr,
	    "usage: gcp [-hrnfPdD] [-g gchost] file [file2, ... fileN] <dest>\n");
}

static void
help()
{
    usage();
    fprintf(stderr,
	    "            -h ............ help (this message)\n"
	    "            -n ............ use the 'none' cipher for ssh\n"
	    "            -f ............ pass fds to the GTCD\n"
	    "            -g <gtcd host>  use specified GTCD host\n"
	    "            -p <gtcd port>  use specified GTCD port\n"
	    "            -r ............ Operate in receive mode\n"
	    "            -P ............ Put only, do not ssh\n"
	    "            -d ............ Pass data path to help gtc caching\n"
            "            -D <debug> .... debug level (-D list for help)\n"
	    "\n"
	    "           <dest> is [user@]host:/path/on/remote/end\n\n"
	    "  Alternate use:  gcp dot://<OID:hints:type> <localfile for xdisk hints> <localfile>\n\n");
}


void
rsh_reap(int kid)
{
    // XXX:
    fatal << "rsh_reap kid exited: " << kid << "\n";
}

ref<aclnt>
rsh_run(char *dest, bool cnone)
{
    vec<str> av;
    str gcp_rsh = getenv("GCP_RSH");
    if (!gcp_rsh) {
	gcp_rsh = DEFAULT_GCP_RSH;
    }

    str gcp_rsh_path = find_program(gcp_rsh);
    if (!gcp_rsh_path) {
	fatal << "Could not locate ssh program " << gcp_rsh << "\n";
    }
	
    av.push_back(gcp_rsh_path);
    av.push_back("-x");
    av.push_back("-a");
    if (cnone) {
        av.push_back("-c");
        av.push_back("none");
    }
    av.push_back(dest);
    dwarn(DEBUG_CLIENT) << "Running rsh to " << dest << "\n";
    av.push_back(strbuf() << "gcp -r -D " << debug);

    ptr<axprt_unix> gcp_x (axprt_unix_aspawnv(av[0], av));
    if (!gcp_x) {
	fatal << "Could not start remote\n";
    }
    gcp_x->allow_recvfd = false;
    pid_t pid = axprt_unix_spawn_pid;
    dwarn(DEBUG_CLIENT) << "Started rsh pid " << pid << "\n";
    chldcb(pid, wrap(rsh_reap));
    return aclnt::alloc(gcp_x, gcp_program_1);
}

void acceptconn() {    
    ref<axprt> gcp_x =
	axprt_pipe::alloc(STDIN_FILENO, STDOUT_FILENO, MAX_PKTSIZE);
    gcp_s = asrv::alloc(gcp_x, gcp_program_1, wrap(get_dispatch, gtc_c));
    
}

static void
local_finish(int outfd, char *name, str err)
{
    str n = strbuf() << name << "." << getpid();
    if (return_fd(n)) {
	//rename temp file back to the right thing
	if (rename(n.cstr(), name) < 0) 
	    warn("Rename failed: %m\n");
    }

    if (err)
	fatal << "Transfer failed: " << err << "\n";

    warn << "Transfer complete\n";
    exit(0);
}

static void
get_only()
{
    static rxx getrx ("^dot://([0-9a-fA-F]{40}):(.+):(\\d+):(\\d)$");
    if (!getrx.match(files[0])) {
        warn("Could not parse DOT descriptor: %s\n", files[0]);
        usage();
	exit(-1);
    }

    int portnum, t;
    str oid = hex2bytes(getrx[1]);
    str hostname = getrx[2];
    convertint(getrx[3], &portnum);
    convertint(getrx[4], &t);

    assert(oid);
    if (portnum == 0)
        fatal("Port number cannot be zero: %d\n", portnum);

    dot_oid_md oidmd;
    oidmd.id.set((char *)oid.cstr(), oid.len());
            
    oid_hint h;
    ref<vec<oid_hint> > hints = New refcounted<vec<oid_hint> >;

    /* add default gtc plugin hint */
    h.name = strbuf() << "gtc://" << hostname << ":" << portnum;
    hints->push_back(h);

    oid_type type = (oid_type) (t);
    if (type == OID_TREE) {
        gettimeofday(&start, NULL);
        gcp_put_arg arg;
        arg.oid = oidmd;
        arg.hints.set(hints->base(), hints->size());
        arg.type = type;
        arg.file.destpath = strbuf() << files[nfiles];
        get_td(arg, NULL, gtc_c);
    }
    else if (type == OID_FILE) {
        /* add a disk hint */
        struct stat sb;
        if (stat(files[1], &sb) < 0) {
            warn("Could not stat hint file (ignoring): %m: %s\n", files[1]);
        }
        else {
            str compname(files[nfiles]);
            generate_disk_hint(compname, sb.st_size, sb.st_mtime, &h, "NONE");
            hints->push_back(h);
        }
        
        /* Ugly - shouldn't override files like this! */
        str name = strbuf() << files[nfiles] << "." << getpid();
        dwarn(DEBUG_CLIENT) << "Opening " << name << "\n";
        int outfd = get_new_fd(name);
        if (outfd < 0) {
            warn("Could not open output file: %m: %s\n", files[nfiles]);
            exit(1);
        }
        vNew get_client(oidmd, hints, 0, name, gtc_c,
                        wrap(local_finish, outfd, files[nfiles]));
    }
    else {
        warn("Unknown type in 'get only' mode: %d\n", t);
        exit(1);
    }
}

static void
ctrlconnect(char *dest, char *destpath, bool cnone, bool passfd)
{
    /* Setup control connection */
    switch (mode) {
    case CLIENT_PUT:
        gcp_c = rsh_run(dest, cnone);
        do_put(files, nfiles, gtc_c, gcp_c, destpath, passfd);
        break;
    case CLIENT_PUT_ONLY:
	do_put(files, nfiles, gtc_c, NULL, NULL, passfd);
        break;
    case CLIENT_GET:
	acceptconn();
        break;
    case CLIENT_GET_OID:
        get_only();
        break;
    default:
        usage();
	exit(-1);
    }
}

static void
gtc_connected(char **argv, int argc, char *dest, char *destpath, int fd, 
              bool cnone, bool passfd)
{
    if (fd == -1)
	fatal("Could not open put_client-gtc socket: %m\n");

    // warn("Connected via FD: %d\n", fd);

    /* Setup GTC connection */
    ref<axprt_unix> gtc_x = axprt_unix::alloc(fd, MAX_PKTSIZE);
    gtc_c = aclnt::alloc(gtc_x, gtc_program_1);

    files = argv;
    nfiles = argc;

    ctrlconnect(dest, destpath, cnone, passfd);
}

int
main(int argc, char * argv[])
{
    extern char *optarg;
    extern int optind;
    char ch;
    char *gtchost = "127.0.0.1";
    //    int gtcport = 12000;
    char *dest = NULL;
    char *destpath = NULL;
    str gcp_connect_host("/tmp/gtcd.sock");
    bool cnone = false;
    bool passfd = false;
    passdp = false;
    /* XXX - probably replace that IP with localhost again when we've
     * debugged the libasync DNS dependency */

    gettimeofday(&gcp_start, NULL);    
    setprogname(argv[0]);

    mode = CLIENT_PUT;
    
    while ((ch = getopt(argc, argv, "hrndfg:p:PD:")) != -1)
	switch(ch) {
        case 'D':
            if (set_debug(optarg)) {
                exit(-1);
            }
            break;
 	case 'p':
 	    gcp_connect_host = optarg;
 	    break;
	case 'r':
	    mode = CLIENT_GET;
	    break;
	case 'P':
	    mode = CLIENT_PUT_ONLY;
	    break;
	case 'g':
	    gtchost = optarg;
	    break;
	case 'n':
            cnone = true;
            break;
	case 'f':
            passfd = true;
            break;
	case 'd':
            passdp = true;
            break; 
	case 'h':
	    help();
	    exit(0);
	default:
	    usage();
	    exit(-1);
	}

    argc -= optind;
    argv += optind;

    if (mode == CLIENT_PUT && argv[0] && !strncmp(argv[0], "dot://", 6))
	mode = CLIENT_GET_OID;

    if (mode == CLIENT_PUT) {
	if (argc < 2) {
	    usage();
	    exit(-1);
	}
	dest = strdup(argv[argc-1]);
	assert(dest != NULL);
	destpath = strchr(dest, ':');
	if (!destpath) {
	    warnx << "Invalid destination: " << dest << "\n";
	    usage();
	    exit(-1);
	}
	*destpath++ = '\0';
    }

    int fd = unixsocket_connect(gcp_connect_host);
    if (fd < 0)
        fatal("%s: %m\n", gcp_connect_host.cstr());

    int numfiles;
    if (mode == CLIENT_PUT_ONLY)
	numfiles = argc;
    else
	numfiles = argc - 1;

    gtc_connected(&argv[0], numfiles, dest, destpath, fd, cnone, passfd);

    amain();
    /* XXX - notreached... */
    if (dest) {
	free(dest);
    }
}
