/* Gcov.c: prepend line execution counts and branch probabilities to a
   source file.
   Copyright (C) 1990, 1991, 1992, 1993, 1994, 1996, 1997, 1998,
   1999, 2000, 2001, 2002 Free Software Foundation, Inc.
   Contributed by James E. Wilson of Cygnus Support.
   Mangled by Bob Manson of Cygnus Support.
   Mangled further by Nathan Sidwell <nathan@codesourcery.com>

Gcov is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Gcov is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Gcov; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/* ??? Print a list of the ten blocks with the highest execution counts,
   and list the line numbers corresponding to those blocks.  Also, perhaps
   list the line numbers with the highest execution counts, only printing
   the first if there are several which are all listed in the same block.  */

/* ??? Should have an option to print the number of basic blocks, and the
   percent of them that are covered.  */

/* ??? Does not correctly handle the case where two .bb files refer to
   the same included source file.  For example, if one has a short
   file containing only inline functions, which is then included in
   two other files, then there will be two .bb files which refer to
   the include file, but there is no way to get the total execution
   counts for the included file, can only get execution counts for one
   or the other of the including files. this can be fixed by --ratios
   --long-file-names --preserve-paths and perl.  */

/* Need an option to show individual block counts, and show
   probabilities of fall through arcs.  */

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tm.h"
#include "intl.h"
#include "version.h"
#undef abort

#include <getopt.h>

typedef HOST_WIDEST_INT gcov_type;
#include "gcov-io.h"

/* The bbg file is generated by -ftest-coverage option. The da file is
   generated by a program compiled with -fprofile-arcs. Their formats
   are documented in gcov-io.h.  */

/* The functions in this file for creating and solution program flow graphs
   are very similar to functions in the gcc source file profile.c.  In
   some places we make use of the knowledge of how profile.c works to
   select particular algorithms here.  */

/* This is the size of the buffer used to read in source file lines.  */

#define STRING_SIZE 200

struct function_info;
struct block_info;

/* Describes an arc between two basic blocks.  */

typedef struct arc_info
{
  /* source and destination blocks.  */
  struct block_info *src;
  struct block_info *dst;

  /* transition counts.  */
  gcov_type count;

  unsigned int count_valid : 1;
  unsigned int on_tree : 1;
  unsigned int fake : 1;
  unsigned int fall_through : 1;

  /* Arc to a call.  */
  unsigned int is_call : 1;
  
  /* Next branch on line.  */
  struct arc_info *line_next;
  
  /* Links to next arc on src and dst lists.  */
  struct arc_info *succ_next;
  struct arc_info *pred_next;
} arc_t;

/* Describes a basic block. Contains lists of arcs to successor and
   predecessor blocks.  */

typedef struct block_info
{
  /* Chain of exit and entry arcs.  */
  arc_t *succ;
  arc_t *pred;

  /* Number of unprocessed exit and entry arcs.  */
  gcov_type num_succ;
  gcov_type num_pred;

  /* Block execution count.  */
  gcov_type count;
  unsigned count_valid : 1;
  unsigned valid_chain : 1;
  unsigned invalid_chain : 1;

  /* Array of line numbers and source files. source files are
     introduced by a linenumber of zero, the next 'line number' is the
     number of the source file.  Always starts with a source file.  */
  unsigned *encoding;
  unsigned num_encodings;

  /* Temporary chain for solving graph.  */
  struct block_info *chain;
  
} block_t;

/* Describes a single function. Contains an array of basic blocks.  */

typedef struct function_info
{
  /* Name of function.  */
  char *name;
  unsigned checksum;

  /* Array of basic blocks.  */
  block_t *blocks;
  unsigned num_blocks;

  /* Raw arc coverage counts.  */
  gcov_type *counts;
  unsigned num_counts;
  
  /* Next function.  */
  struct function_info *next;
} function_t;

/* Describes coverage of a file or function.  */

typedef struct coverage_info
{
  int lines;
  int lines_executed;
  
  int branches;
  int branches_executed;
  int branches_taken;
  
  int calls;
  int calls_executed;
  
  char *name;
} coverage_t;

/* Describes a single line of source. Contains a chain of basic blocks
   with code on it.  */

typedef struct line_info
{
  gcov_type count;	   /* execution count */
  arc_t *branches; 	   /* branches from blocks that end on this
			      line.  */
  unsigned exists : 1;
} line_t;

/* Describes a file mentioned in the block graph.  Contains an array
   of line info.  */

typedef struct source_info
{
  /* Name of source file.  */
  char *name;
  unsigned index;

  /* Array of line information.  */
  line_t *lines;
  unsigned num_lines;

  coverage_t coverage;
  
  /* Next source file.  */
  struct source_info *next;
} source_t;

/* Holds a list of function basic block graphs.  */

static function_t *functions;

/* This points to the head of the sourcefile structure list.  */

static source_t *sources;

/* Modification time of graph file.  */

static time_t bbg_file_time;

/* Name and file pointer of the input file for the basic block graph.  */

static char *bbg_file_name;

/* Name and file pointer of the input file for the arc count data.  */

static char *da_file_name;

/* Output branch probabilities.  */

static int flag_branches = 0;

/* Output a gcov file if this is true.  This is on by default, and can
   be turned off by the -n option.  */

static int flag_gcov_file = 1;

/* For included files, make the gcov output file name include the name
   of the input source file.  For example, if x.h is included in a.c,
   then the output file name is a.c##x.h.gcov instead of x.h.gcov.  */

static int flag_long_names = 0;

/* Output summary info for each function.  */

static int flag_function_summary = 0;

/* Object directory file prefix.  This is the directory/file where the
   graph and data files are looked for, if nonzero.  */

static char *object_directory = 0;

/* Preserve all pathname components. Needed when object files and
   source files are in subdirectories. '/' is mangled as '#', '.' is
   elided and '..' mangled to '^'.  */

static int flag_preserve_paths = 0;

/* Output the number of times a branch was taken as opposed to the percentage
   of times it was taken.  */

static int flag_counts = 0;

/* Forward declarations.  */
static void fnotice PARAMS ((FILE *, const char *, ...)) ATTRIBUTE_PRINTF_2;
static int process_args PARAMS ((int, char **));
static void print_usage PARAMS ((int)) ATTRIBUTE_NORETURN;
static void print_version PARAMS ((void)) ATTRIBUTE_NORETURN;
static void process_file PARAMS ((const char *));
static void create_file_names PARAMS ((const char *));
static int read_graph_file PARAMS ((void));
static int read_count_file PARAMS ((void));
static void solve_flow_graph PARAMS ((function_t *));
static void add_branch_counts PARAMS ((coverage_t *, const arc_t *));
static void add_line_counts PARAMS ((coverage_t *, const function_t *));
static void function_summary PARAMS ((const coverage_t *, const char *));
static const char *format_gcov PARAMS ((gcov_type, gcov_type, int));
static void accumulate_line_counts PARAMS ((source_t *));
static void output_lines PARAMS ((FILE *, const source_t *));
static char *make_gcov_file_name PARAMS ((const char *, const char *));
static void release_structures PARAMS ((void));
extern int main PARAMS ((int, char **));

int
main (argc, argv)
     int argc;
     char **argv;
{
  int argno;
  
  gcc_init_libintl ();

  argno = process_args (argc, argv);
  if (optind == argc)
    print_usage (true);

  for (; argno != argc; argno++)
    {
      release_structures ();
      
      process_file (argv[argno]);
    }
  
  return 0;
}

static void
fnotice VPARAMS ((FILE *file, const char *msgid, ...))
{
  VA_OPEN (ap, msgid);
  VA_FIXEDARG (ap, FILE *, file);
  VA_FIXEDARG (ap, const char *, msgid);

  vfprintf (file, _(msgid), ap);
  VA_CLOSE (ap);
}

/* More 'friendly' abort that prints the line and file.
   config.h can #define abort fancy_abort if you like that sort of thing.  */
extern void fancy_abort PARAMS ((void)) ATTRIBUTE_NORETURN;

void
fancy_abort ()
{
  fnotice (stderr, "Internal gcov abort.\n");
  exit (FATAL_EXIT_CODE);
}

/* Print a usage message and exit.  If ERROR_P is nonzero, this is an error,
   otherwise the output of --help.  */

static void
print_usage (error_p)
     int error_p;
{
  FILE *file = error_p ? stderr : stdout;
  int status = error_p ? FATAL_EXIT_CODE : SUCCESS_EXIT_CODE;
  
  fnotice (file, "Usage: gcov [OPTION]... SOURCEFILE\n\n");
  fnotice (file, "Print code coverage information.\n\n");
  fnotice (file, "  -h, --help                      Print this help, then exit\n");
  fnotice (file, "  -v, --version                   Print version number, then exit\n");
  fnotice (file, "  -b, --branch-probabilities      Include branch probabilities in output\n");
  fnotice (file, "  -c, --branch-counts             Given counts of branches taken\n\
                                    rather than percentages\n");
  fnotice (file, "  -n, --no-output                 Do not create an output file\n");
  fnotice (file, "  -l, --long-file-names           Use long output file names for included\n\
                                    source files\n");
  fnotice (file, "  -f, --function-summaries        Output summaries for each function\n");
  fnotice (file, "  -o, --object-directory DIR|FILE Search for object files in DIR or called FILE\n");
  fnotice (file, "  -p, --preserve-paths            Preserve all pathname components\n");
  fnotice (file, "\nFor bug reporting instructions, please see:\n%s.\n",
	   bug_report_url);
  exit (status);
}

/* Print version information and exit.  */

static void
print_version ()
{
  char v[4];
  unsigned version = GCOV_VERSION;
  unsigned ix;

  for (ix = 4; ix--; version >>= 8)
    v[ix] = version;
  fnotice (stdout, "gcov %.4s (GCC %s)\n", v, version_string);
  fnotice (stdout, "Copyright (C) 2002 Free Software Foundation, Inc.\n");
  fnotice (stdout,
	   "This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\n");
  exit (SUCCESS_EXIT_CODE);
}

static const struct option options[] =
{
  { "help",                 no_argument,       NULL, 'h' },
  { "version",              no_argument,       NULL, 'v' },
  { "branch-probabilities", no_argument,       NULL, 'b' },
  { "branch-counts",        no_argument,       NULL, 'c' },
  { "no-output",            no_argument,       NULL, 'n' },
  { "long-file-names",      no_argument,       NULL, 'l' },
  { "function-summaries",   no_argument,       NULL, 'f' },
  { "preserve-paths",       no_argument,       NULL, 'p' },
  { "object-directory",     required_argument, NULL, 'o' },
  { "object-file",          required_argument, NULL, 'o' },
};

/* Process args, return index to first non-arg.  */

static int
process_args (argc, argv)
     int argc;
     char **argv;
{
  int opt;

  while ((opt = getopt_long (argc, argv, "hvbclnfo:p", options, NULL)) != -1)
    {
      switch (opt)
	{
	case 'h':
	  print_usage (false);
	  /* print_usage will exit.  */
	case 'v':
	  print_version ();
	  /* print_version will exit.  */
	case 'b':
	  flag_branches = 1;
	  break;
	case 'c':
	  flag_counts = 1;
	  break;
	case 'n':
	  flag_gcov_file = 0;
	  break;
	case 'l':
	  flag_long_names = 1;
	  break;
	case 'f':
	  flag_function_summary = 1;
	  break;
	case 'o':
	  object_directory = optarg;
	  break;
	case 'p':
	  flag_preserve_paths = 1;
	  break;
	default:
	  print_usage (true);
	  /* print_usage will exit.  */
	}
    }

  return optind;
}

/* Process a single source file.  */

static void
process_file (file_name)
     const char *file_name;
{
  source_t *src;
  function_t *fn;
  
  create_file_names (file_name);
  if (read_graph_file ())
    return;
  
  if (!functions)
    {
      fnotice (stderr, "%s:no functions found\n", bbg_file_name);
      return;
    }
  
  if (read_count_file ())
    return;
  
  for (fn = functions; fn; fn = fn->next)
    solve_flow_graph (fn);
  for (src = sources; src; src = src->next)
    src->lines = (line_t *) xcalloc (src->num_lines, sizeof (line_t));
  for (fn = functions; fn; fn = fn->next)
    {
      coverage_t coverage;
      
      memset (&coverage, 0, sizeof (coverage));
      coverage.name = fn->name;
      add_line_counts (flag_function_summary ? &coverage : NULL, fn);
      if (flag_function_summary)
	{
	  function_summary (&coverage, "Function");
	  fnotice (stdout, "\n");
	}
    }
  
  for (src = sources; src; src = src->next)
    {
      accumulate_line_counts (src);
      function_summary (&src->coverage, "File");
      if (flag_gcov_file)
	{
	  char *gcov_file_name = make_gcov_file_name (file_name, src->name);
	  FILE *gcov_file = fopen (gcov_file_name, "w");
	  
	  if (gcov_file)
	    {
	      fnotice (stdout, "%s:creating `%s'\n",
		       src->name, gcov_file_name);
	      output_lines (gcov_file, src);
	      if (ferror (gcov_file))
		    fnotice (stderr, "%s:error writing output file `%s'\n",
			     src->name, gcov_file_name);
	      fclose (gcov_file);
	    }
	  else
	    fnotice (stderr, "%s:could not open output file `%s'\n",
		     src->name, gcov_file_name);
	  free (gcov_file_name);
	}
      fnotice (stdout, "\n");
    }
}

/* Release all memory used.  */

static void
release_structures ()
{
  function_t *fn;
  source_t *src;
  
  free (bbg_file_name);
  free (da_file_name);
  da_file_name = bbg_file_name = NULL;
  bbg_file_time = 0;
  
  while ((src = sources))
    {
      sources = src->next;

      free (src->name);
      free (src->lines);
    }
  
  while ((fn = functions))
    {
      unsigned ix;
      block_t *block;
      
      functions = fn->next;
      for (ix = fn->num_blocks, block = fn->blocks; ix--; block++)
	{
	  arc_t *arc, *arc_n;

	  for (arc = block->succ; arc; arc = arc_n)
	    {
	      arc_n = arc->succ_next;
	      free (arc);
	    }
	  free (block->encoding);
	}
      free (fn->blocks);
      free (fn->counts);
    }
}

/* Generate the names of the graph and data files. If OBJECT_DIRECTORY
   is not specified, these are looked for in the current directory,
   and named from the basename of the FILE_NAME sans extension. If
   OBJECT_DIRECTORY is specified and is a directory, the files are in
   that directory, but named from the basename of the FILE_NAME, sans
   extension. Otherwise OBJECT_DIRECTORY is taken to be the name of
   the object *file*, and the data files are named from that.  */

static void
create_file_names (file_name)
     const char *file_name;
{
  char *cptr;
  char *name;
  int length = strlen (file_name);
  int base;
  
  if (object_directory && object_directory[0])
    {
      struct stat status;

      length += strlen (object_directory) + 2;
      name = xmalloc (length);
      name[0] = 0;
      
      base = !stat (object_directory, &status) && S_ISDIR (status.st_mode);
      strcat (name, object_directory);
      if (base && name[strlen (name) - 1] != '/')
	strcat (name, "/");
    }
  else
    {
      name = xmalloc (length + 1);
      name[0] = 0;
      base = 1;
    }
  
  if (base)
    {
      /* Append source file name */
      cptr = strrchr (file_name, '/');
      strcat (name, cptr ? cptr + 1 : file_name);
    }
  
  /* Remove the extension.  */
  cptr = strrchr (name, '.');
  if (cptr)
    *cptr = 0;
  
  length = strlen (name);
  
  bbg_file_name = xmalloc (length + strlen (GCOV_GRAPH_SUFFIX) + 1);
  strcpy (bbg_file_name, name);
  strcpy (bbg_file_name + length, GCOV_GRAPH_SUFFIX);

  da_file_name = xmalloc (length + strlen (GCOV_DATA_SUFFIX) + 1);
  strcpy (da_file_name, name);
  strcpy (da_file_name + length, GCOV_DATA_SUFFIX);
  
  return;
}

/* Read the graph file. Return nonzero on fatal error.  */

static int
read_graph_file ()
{
  FILE *file;
  struct stat status;
  unsigned magic, version;
  unsigned current_tag = 0;
  unsigned tag;
  struct function_info *fn = NULL;
  source_t *src = NULL;
  unsigned ix;

  file = fopen (bbg_file_name, "rb");
  if (!file)
    {
      fnotice (stderr, "%s:cannot open graph file\n", bbg_file_name);
      return 1;
    }
  if (!fstat (fileno (file), &status))
    bbg_file_time = status.st_mtime;
  if (gcov_read_unsigned (file, &magic) || magic != GCOV_GRAPH_MAGIC)
    {
      fnotice (stderr, "%s:not a gcov graph file\n", bbg_file_name);
      fclose (file);
      return 1;
    }

  if (gcov_read_unsigned (file, &version) || version != GCOV_VERSION)
    {
      char v[4], e[4];

      magic = GCOV_VERSION;
      
      for (ix = 4; ix--; magic >>= 8, version >>= 8)
	{
	  v[ix] = version;
	  e[ix] = magic;
	}
      fnotice (stderr, "%s:version `%.4s', prefer `%.4s'\n",
	       bbg_file_name, v, e);
    }
  
  while (!gcov_read_unsigned (file, &tag))
    {
      unsigned length;
      long base;

      if (gcov_read_unsigned (file, &length))
	goto corrupt;

      base = gcov_save_position (file);
      
      if (tag == GCOV_TAG_FUNCTION)
	{
	  char *function_name = NULL;
	  unsigned checksum;

	  if (gcov_read_string (file, &function_name, NULL)
	      || gcov_read_unsigned (file, &checksum))
	    goto corrupt;
	  fn = (function_t *)xcalloc (1, sizeof (function_t));
	  fn->name = function_name;
	  fn->checksum = checksum;

	  fn->next = functions;
	  functions = fn;
	  current_tag = tag;
	}
      else if (fn && tag == GCOV_TAG_BLOCKS)
	{
	  if (fn->blocks)
	    fnotice (stderr, "%s:already seen blocks for `%s'\n",
		     bbg_file_name, fn->name);
	  else
	    {
	      fn->num_blocks = length / 4;
	      fn->blocks
		= (block_t *)xcalloc (fn->num_blocks, sizeof (block_t));
	    }
	}
      else if (fn && tag == GCOV_TAG_ARCS)
	{
	  unsigned src;
	  unsigned num_dests = (length - 4) / 8;
	  unsigned dest, flags;

	  if (gcov_read_unsigned (file, &src)
	      || src >= fn->num_blocks
	      || fn->blocks[src].succ)
	    goto corrupt;
	  
	  while (num_dests--)
	    {
	      struct arc_info *arc;
	      
	      if (gcov_read_unsigned (file, &dest)
		  || gcov_read_unsigned (file, &flags)
		  || dest >= fn->num_blocks)
		goto corrupt;
	      arc = (arc_t *) xcalloc (1, sizeof (arc_t));
	      
	      arc->dst = &fn->blocks[dest];
	      arc->src = &fn->blocks[src];
	      
	      arc->count = 0;
	      arc->count_valid = 0;
	      arc->on_tree = !!(flags & GCOV_ARC_ON_TREE);
	      arc->fake = !!(flags & GCOV_ARC_FAKE);
	      arc->fall_through = !!(flags & GCOV_ARC_FALLTHROUGH);
	      
	      arc->succ_next = fn->blocks[src].succ;
	      fn->blocks[src].succ = arc;
	      fn->blocks[src].num_succ++;
	      
	      arc->pred_next = fn->blocks[dest].pred;
	      fn->blocks[dest].pred = arc;
	      fn->blocks[dest].num_pred++;

	      arc->is_call = arc->fake;
	      
	      if (!arc->on_tree)
		fn->num_counts++;
	    }
	}
      else if (fn && tag == GCOV_TAG_LINES)
	{
	  unsigned blockno;
	  unsigned *line_nos
	    = (unsigned *)xcalloc ((length - 4) / 4, sizeof (unsigned));

	  if (gcov_read_unsigned (file, &blockno)
	      || blockno >= fn->num_blocks
	      || fn->blocks[blockno].encoding)
	    goto corrupt;
	  
	  for (ix = 0; ;  )
	    {
	      unsigned lineno;
	      
	      if (gcov_read_unsigned (file, &lineno))
		goto corrupt;
	      if (lineno)
		{
		  if (!ix)
		    {
		      line_nos[ix++] = 0;
		      line_nos[ix++] = src->index;
		    }
		  line_nos[ix++] = lineno;
		  if (lineno >= src->num_lines)
		    src->num_lines = lineno + 1;
		}
	      else
		{
		  char *file_name = NULL;
		  
		  if (gcov_read_string (file, &file_name, NULL))
		    goto corrupt;
		  if (!file_name)
		    break;
		  for (src = sources; src; src = src->next)
		    if (!strcmp (file_name, src->name))
		      {
			free (file_name);
			break;
		      }
		  if (!src)
		    {
		      src = (source_t *)xcalloc (1, sizeof (source_t));
		      src->name = file_name;
		      src->coverage.name = file_name;
		      src->index = sources ? sources->index + 1 : 1;
		      src->next = sources;
		      sources = src;
		    }
		  line_nos[ix++] = 0;
		  line_nos[ix++] = src->index;
		}
	    }
	  
	  fn->blocks[blockno].encoding = line_nos;
	  fn->blocks[blockno].num_encodings = ix;
	}
      else if (current_tag && !GCOV_TAG_IS_SUBTAG (current_tag, tag))
	{
	  fn = NULL;
	  current_tag = 0;
	}
      if (gcov_resync (file, base, length))
	{
	corrupt:;
	  fnotice (stderr, "%s:corrupted\n", bbg_file_name);
	  fclose (file);
	  return 1;
	}
    }
  fclose (file);
  
  /* We built everything backwards, so nreverse them all */
  
  /* Reverse sources. Not strictly necessary, but we'll then process
     them in the 'expected' order.  */
  {
    source_t *src, *src_p, *src_n;

    for (src_p = NULL, src = sources; src; src_p = src, src = src_n)
      {
	src_n = src->next;
	src->next = src_p;
      }
    sources =  src_p;
  }

  /* Reverse functions.  */
  {
    function_t *fn, *fn_p, *fn_n;

    for (fn_p = NULL, fn = functions; fn; fn_p = fn, fn = fn_n)
      {
	unsigned ix;
	
	fn_n = fn->next;
	fn->next = fn_p;

	/* Reverse the arcs */
	for (ix = fn->num_blocks; ix--;)
	  {
	    arc_t *arc, *arc_p, *arc_n;
	    
	    for (arc_p = NULL, arc = fn->blocks[ix].succ; arc;
		 arc_p = arc, arc = arc_n)
	      {
		arc_n = arc->succ_next;
		arc->succ_next = arc_p;
	      }
	    fn->blocks[ix].succ = arc_p;

	    for (arc_p = NULL, arc = fn->blocks[ix].pred; arc;
		 arc_p = arc, arc = arc_n)
	      {
		arc_n = arc->pred_next;
		arc->pred_next = arc_p;
	      }
	    fn->blocks[ix].pred = arc_p;
	  }
      }
    functions = fn_p;
  }
  return 0;
}

/* Reads profiles from the count file and attach to each
   function. Return nonzero if fatal error.  */

static int
read_count_file ()
{
  FILE *file;
  unsigned ix;
  char *function_name_buffer = NULL;
  unsigned magic, version;
  function_t *fn = NULL;

  file = fopen (da_file_name, "rb");
  if (!file)
    {
      fnotice (stderr, "%s:cannot open data file\n", da_file_name);
      return 1;
    }
  if (gcov_read_unsigned (file, &magic) || magic != GCOV_DATA_MAGIC)
    {
      fnotice (stderr, "%s:not a gcov data file\n", da_file_name);
    cleanup:;
      free (function_name_buffer);
      fclose (file);
      return 1;
    }
  if (gcov_read_unsigned (file, &version) || version != GCOV_VERSION)
    {
      char v[4], e[4];
      
      magic = GCOV_VERSION;
      for (ix = 4; ix--; magic >>= 8, version >>= 8)
	{
	  v[ix] = version;
	  e[ix] = magic;
	}
      fnotice (stderr, "%s:version `%.4s', prefer version `%.4s'\n",
	       da_file_name, v, e);
    }
  
  while (1)
    {
      unsigned tag, length;
      long base;
      
      if (gcov_read_unsigned (file, &tag)
	  || gcov_read_unsigned (file, &length))
	{
	  if (feof (file))
	    break;
	  
	corrupt:;
	  fnotice (stderr, "%s:corrupted\n", da_file_name);
	  goto cleanup;
	}
      base = gcov_save_position (file);
      if (tag == GCOV_TAG_FUNCTION)
	{
	  unsigned checksum;
	  struct function_info *fn_n = functions;
	  
	  if (gcov_read_string (file, &function_name_buffer, NULL)
	      || gcov_read_unsigned (file, &checksum))
	    goto corrupt;

	  for (fn = fn ? fn->next : NULL; ; fn = fn->next)
	    {
	      if (fn)
		;
	      else if ((fn = fn_n))
		fn_n = NULL;
	      else
		{
		  fnotice (stderr, "%s:unknown function `%s'\n",
			   da_file_name, function_name_buffer);
		  break;
		}
	      if (!strcmp (fn->name, function_name_buffer))
		break;
	    }

	  if (!fn)
	    ;
	  else if (checksum != fn->checksum)
	    {
	    mismatch:;
	      fnotice (stderr, "%s:profile mismatch for `%s'\n",
		       da_file_name, function_name_buffer);
	      goto cleanup;
	    }
	}
      else if (tag == GCOV_TAG_ARC_COUNTS && fn)
	{
	  if (length != 8 * fn->num_counts)
	    goto mismatch;
	  
	  if (!fn->counts)
	    fn->counts
	      = (gcov_type *)xcalloc (fn->num_counts, sizeof (gcov_type));
	  
	  for (ix = 0; ix != fn->num_counts; ix++)
	    {
	      gcov_type count;
	      
	      if (gcov_read_counter (file, &count))
		goto corrupt;
	      fn->counts[ix] += count;
	    }
	}
      gcov_resync (file, base, length);
    }

  fclose (file);
  free (function_name_buffer);
  return 0;
}

/* Solve the flow graph. Propagate counts from the instrumented arcs
   to the blocks and the uninstrumented arcs.  */

static void
solve_flow_graph (fn)
     function_t *fn;
{
  unsigned ix;
  arc_t *arc;
  gcov_type *count_ptr = fn->counts;
  block_t *valid_blocks = NULL;    /* valid, but unpropagated blocks.  */
  block_t *invalid_blocks = NULL;  /* invalid, but inferable blocks.  */
  
  if (fn->num_blocks < 2)
    fnotice (stderr, "%s:`%s' lacks entry and/or exit blocks\n",
	     bbg_file_name, fn->name);
  else
    {
      if (fn->blocks[0].num_pred)
	fnotice (stderr, "%s:`%s' has arcs to entry block\n",
		 bbg_file_name, fn->name);
      else
	/* We can't deduce the entry block counts from the lack of
	   predecessors.  */
	fn->blocks[0].num_pred = ~(unsigned)0;
      
      if (fn->blocks[fn->num_blocks - 1].num_succ)
	fnotice (stderr, "%s:`%s' has arcs from exit block\n",
		 bbg_file_name, fn->name);
      else
	/* Likewise, we can't deduce exit block counts from the lack
	   of its successors.  */
	fn->blocks[fn->num_blocks - 1].num_succ = ~(unsigned)0;
    }

  /* Propagate the measured counts, this must be done in the same
     order as the code in profile.c  */
  for (ix = 0; ix != fn->num_blocks; ix++)
    {
      block_t const *prev_dst = NULL;
      int out_of_order = 0;
      
      for (arc = fn->blocks[ix].succ; arc; arc = arc->succ_next)
	{
	  if (!arc->on_tree)
	    {
	      if (count_ptr)
		arc->count = *count_ptr++;
	      arc->count_valid = 1;
	      fn->blocks[ix].num_succ--;
	      arc->dst->num_pred--;
	    }
	  if (prev_dst && prev_dst > arc->dst)
	    out_of_order = 1;
	  prev_dst = arc->dst;
	}
      
      /* Sort the successor arcs into ascending dst order. profile.c
	 normally produces arcs in the right order, but sometimes with
	 one or two out of order.  We're not using a particularly
	 smart sort.  */
      if (out_of_order)
	{
	  arc_t *start = fn->blocks[ix].succ;
	  unsigned changes = 1;
	  
	  while (changes)
	    {
	      arc_t *arc, *arc_p, *arc_n;
	      
	      changes = 0;
	      for (arc_p = NULL, arc = start; (arc_n = arc->succ_next);)
		{
		  if (arc->dst > arc_n->dst)
		    {
		      changes = 1;
		      if (arc_p)
			arc_p->succ_next = arc_n;
		      else
			start = arc_n;
		      arc->succ_next = arc_n->succ_next;
		      arc_n->succ_next = arc;
		      arc_p = arc_n;
		    }
		  else
		    {
		      arc_p = arc;
		      arc = arc_n;
		    }
		}
	    }
	  fn->blocks[ix].succ = start;
	}
      
      /* Place it on the invalid chain, it will be ignored if that's
	 wrong.  */
      fn->blocks[ix].invalid_chain = 1;
      fn->blocks[ix].chain = invalid_blocks;
      invalid_blocks = &fn->blocks[ix];
    }

  while (invalid_blocks || valid_blocks)
    {
      block_t *blk;
      
      while ((blk = invalid_blocks))
	{
	  gcov_type total = 0;
	  const arc_t *arc;
	  
	  invalid_blocks = blk->chain;
	  blk->invalid_chain = 0;
	  if (!blk->num_succ)
	    for (arc = blk->succ; arc; arc = arc->succ_next)
	      total += arc->count;
	  else if (!blk->num_pred)
	    for (arc = blk->pred; arc; arc = arc->pred_next)
	      total += arc->count;
	  else
	    continue;
	  
	  blk->count = total;
	  blk->count_valid = 1;
	  blk->chain = valid_blocks;
	  blk->valid_chain = 1;
	  valid_blocks = blk;
	}
      while ((blk = valid_blocks))
	{
	  gcov_type total;
	  arc_t *arc, *inv_arc;

	  valid_blocks = blk->chain;
	  blk->valid_chain = 0;
	  if (blk->num_succ == 1)
	    {
	      block_t *dst;
	      
	      total = blk->count;
	      inv_arc = NULL;
	      for (arc = blk->succ; arc; arc = arc->succ_next)
		{
		  total -= arc->count;
		  if (!arc->count_valid)
		    inv_arc = arc;
		}
	      dst = inv_arc->dst;
	      inv_arc->count_valid = 1;
	      inv_arc->count = total;
	      blk->num_succ--;
	      dst->num_pred--;
	      if (dst->count_valid)
		{
		  if (dst->num_pred == 1 && !dst->valid_chain)
		    {
		      dst->chain = valid_blocks;
		      dst->valid_chain = 1;
		      valid_blocks = dst;
		    }
		}
	      else
		{
		  if (!dst->num_pred && !dst->invalid_chain)
		    {
		      dst->chain = invalid_blocks;
		      dst->invalid_chain = 1;
		      invalid_blocks = dst;
		    }
		}
	    }
	  if (blk->num_pred == 1)
	    {
	      block_t *src;
	      
	      total = blk->count;
	      inv_arc = NULL;
	      for (arc = blk->pred; arc; arc = arc->pred_next)
		{
		  total -= arc->count;
		  if (!arc->count_valid)
		    inv_arc = arc;
		}
	      src = inv_arc->src;
	      inv_arc->count_valid = 1;
	      inv_arc->count = total;
	      blk->num_pred--;
	      src->num_succ--;
	      if (src->count_valid)
		{
		  if (src->num_succ == 1 && !src->valid_chain)
		    {
		      src->chain = valid_blocks;
		      src->valid_chain = 1;
		      valid_blocks = src;
		    }
		}
	      else
		{
		  if (!src->num_succ && !src->invalid_chain)
		    {
		      src->chain = invalid_blocks;
		      src->invalid_chain = 1;
		      invalid_blocks = src;
		    }
		}
	    }
	}
    }
  
  /* If the graph has been correctly solved, every block will have a
     valid count.  */
  for (ix = 0; ix < fn->num_blocks; ix++)
    if (!fn->blocks[ix].count_valid)
      {
	fnotice (stderr, "%s:graph is unsolvable for `%s'\n",
		 bbg_file_name, fn->name);
	break;
      }
}



/* Increment totals in COVERAGE according to arc ARC.  */

static void
add_branch_counts (coverage, arc)
     coverage_t *coverage;
     const arc_t *arc;
{
  if (arc->is_call)
    {
      coverage->calls++;
      if (arc->src->count)
	coverage->calls_executed++;
    }
  else
    {
      coverage->branches++;
      if (arc->src->count)
	coverage->branches_executed++;
      if (arc->count)
	coverage->branches_taken++;
    }
}

/* Format a HOST_WIDE_INT as either a percent ratio, or absolute
   count.  If dp >= 0, format TOP/BOTTOM * 100 to DP decimal places.
   If DP is zero, no decimal point is printed. Only print 100% when
   TOP==BOTTOM and only print 0% when TOP=0.  If dp < 0, then simply
   format TOP.  Return pointer to a static string.  */

static char const *
format_gcov (top, bottom, dp)
     gcov_type top, bottom;
     int dp;
{
  static char buffer[20];
  
  if (dp >= 0)
    {
      float ratio = bottom ? (float)top / bottom : 0;
      int ix;
      unsigned limit = 100;
      unsigned percent;
  
      for (ix = dp; ix--; )
	limit *= 10;
      
      percent = (unsigned) (ratio * limit + (float)0.5);
      if (percent <= 0 && top)
	percent = 1;
      else if (percent >= limit && top != bottom)
	percent = limit - 1;
      ix = sprintf (buffer, "%.*u%%", dp + 1, percent);
      if (dp)
	{
	  dp++;
	  do
	    {
	      buffer[ix+1] = buffer[ix];
	      ix--;
	    }
	  while (dp--);
	  buffer[ix + 1] = '.';
	}
    }
  else
    sprintf (buffer, HOST_WIDEST_INT_PRINT_DEC, (HOST_WIDEST_INT)top);
  
  return buffer;
}


/* Output summary info for a function.  */

static void
function_summary (coverage, title)
     const coverage_t *coverage;
     const char *title;
{
  fnotice (stdout, "%s `%s'\n", title, coverage->name);

  if (coverage->lines)
    fnotice (stdout, "Lines executed:%s of %d\n",
	     format_gcov (coverage->lines_executed, coverage->lines, 2),
	     coverage->lines);
  else
    fnotice (stdout, "No executable lines");

  if (flag_branches)
    {
      if (coverage->branches)
	{
	  fnotice (stdout, "Branches executed:%s of %d\n",
		   format_gcov (coverage->branches_executed,
				coverage->branches, 2),
		   coverage->branches);
	  fnotice (stdout, "Taken at least once:%s of %d\n",
		   format_gcov (coverage->branches_taken,
				coverage->branches, 2),
		   coverage->branches);
	}
      else
	fnotice (stdout, "No branches\n");
      if (coverage->calls)
	fnotice (stdout, "Calls executed:%s of %d\n",
		 format_gcov (coverage->calls_executed, coverage->calls, 2),
		 coverage->calls);
      else
	fnotice (stdout, "No calls\n");
    }
}

/* Generate an output file name. LONG_OUTPUT_NAMES and PRESERVE_PATHS
   affect name generation. With preserve_paths we create a filename
   from all path components of the source file, replacing '/' with
   '#', without it we simply take the basename component. With
   long_output_names we prepend the processed name of the input file
   to each output name (except when the current source file is the
   input file, so you don't get a double concatenation). The two
   components are separated by '##'. Also '.' filename components are
   removed and '..'  components are renamed to '^'.  */

static char *
make_gcov_file_name (input_name, src_name)
     const char *input_name;
     const char *src_name;
{
  char *cptr;
  char *name = xmalloc (strlen (src_name) + strlen (input_name) + 10);
  
  name[0] = 0;
  if (flag_long_names && strcmp (src_name, input_name))
    {
      /* Generate the input filename part.  */
      cptr = flag_preserve_paths ? NULL : strrchr (input_name, '/');
      strcat (name, cptr ? cptr + 1 : input_name);
      strcat (name, "##");
    }
   
  /* Generate the source filename part.  */
  cptr = flag_preserve_paths ? NULL : strrchr (src_name, '/');
  strcat (name, cptr ? cptr + 1 : src_name);
  
  if (flag_preserve_paths)
    {
      /* Convert '/' to '#', remove '/./', convert '/../' to '/^/' */
      char *prev;
      
      for (cptr = name; (cptr = strchr ((prev = cptr), '/'));)
 	{
 	  unsigned shift = 0;
 	  
 	  if (prev + 1 == cptr && prev[0] == '.')
 	    {
 	      /* Remove '.' */
 	      shift = 2;
 	    }
 	  else if (prev + 2 == cptr && prev[0] == '.' && prev[1] == '.')
 	    {
 	      /* Convert '..' */
 	      shift = 1;
 	      prev[1] = '^';
 	    }
 	  else
 	    *cptr++ = '#';
 	  if (shift)
 	    {
 	      cptr = prev;
 	      do
 		prev[0] = prev[shift];
	      while (*prev++);
 	    }
 	}
    }
  
  strcat (name, ".gcov");
  return name;
}

/* Scan through the bb_data for each line in the block, increment
   the line number execution count indicated by the execution count of
   the appropriate basic block.  */

static void
add_line_counts (coverage, fn)
     coverage_t *coverage;
     const function_t *fn;
{
  unsigned ix;
  line_t *line = NULL; /* this is propagated from one iteration to the
			  next.  */

  /* Scan each basic block.  */
  for (ix = 0; ix != fn->num_blocks; ix++)
    {
      const block_t *block = &fn->blocks[ix];
      unsigned *encoding;
      const source_t *src = NULL;
      unsigned jx;

      for (jx = 0, encoding = block->encoding;
	   jx != block->num_encodings; jx++, encoding++)
	if (!*encoding)
	  {
	    unsigned src_n = *++encoding;

	    for (src = sources; src->index != src_n; src = src->next)
	      continue;
	    jx++;
	  }
	else
	  {
	    line = &src->lines[*encoding];

	    if (coverage)
	      {
		if (!line->exists)
		  coverage->lines++;
		if  (!line->count && block->count)
		  coverage->lines_executed++;
	      }
	    line->exists = 1;
	    line->count += block->count;
	  }
      
      if (line && flag_branches)
	{
	  arc_t *arc;
	  
	  for (arc = block->succ; arc; arc = arc->succ_next)
	    {
	      /* Ignore fall through arcs as they aren't really branches.  */
	      if (arc->fall_through)
		continue;
	      
	      arc->line_next = line->branches;
	      line->branches = arc;
	      if (coverage)
		add_branch_counts (coverage, arc);
	    }
	}
    }
  if (!line)
    fnotice (stderr, "%s:no lines for `%s'\n", bbg_file_name, fn->name);
}

/* Accumulate the line counts of a file.  */

static void
accumulate_line_counts (src)
     source_t *src;
{
  line_t *line;
  unsigned ix;
  
  for (ix = src->num_lines, line = src->lines; ix--; line++)
    {
      arc_t *arc, *arc_p, *arc_n;
      
      /* Total and reverse the branch information.  */
      for (arc = line->branches, arc_p = NULL; arc; arc_p = arc, arc = arc_n)
	{
	  arc_n = arc->line_next;
	  arc->line_next = arc_p;

	  add_branch_counts (&src->coverage, arc);
	}
      line->branches = arc_p;

      if (line->exists)
	{
	  src->coverage.lines++;
	  if (line->count)
	    src->coverage.lines_executed++;
	}
    }
}

/* Read in the source file one line at a time, and output that line to
   the gcov file preceded by its execution count and other
   information.  */

static void
output_lines (gcov_file, src)
     FILE *gcov_file;
     const source_t *src;
{
  FILE *source_file;
  unsigned line_num;       	/* current line number.  */
  const line_t *line;           /* current line info ptr.  */
  char string[STRING_SIZE];     /* line buffer.  */
  char const *retval = "";	/* status of source file reading.  */

  fprintf (gcov_file, "%9s:%5d:Source:%s\n", "-", 0, src->name);
  fprintf (gcov_file, "%9s:%5d:Graph:%s\n", "-", 0, bbg_file_name);
  fprintf (gcov_file, "%9s:%5d:Data:%s\n", "-", 0, da_file_name);
  
  source_file = fopen (src->name, "r");
  if (!source_file)
    {
      fnotice (stderr, "%s:cannot open source file\n", src->name);
      retval = NULL;
    }
  else
    {
      struct stat status;
      
      if (!fstat (fileno (source_file), &status)
	  && status.st_mtime > bbg_file_time)
	{
	  fnotice (stderr, "%s:source file is newer than graph file `%s'\n",
		   src->name, bbg_file_name);
	  fprintf (gcov_file, "%9s:%5d:Source is newer than graph\n",
		   "-", 0);
	}
    }

  for (line_num = 1, line = &src->lines[line_num];
       line_num < src->num_lines; line_num++, line++)
    {
      /* For lines which don't exist in the .bb file, print '-' before
 	 the source line.  For lines which exist but were never
 	 executed, print '#####' before the source line.  Otherwise,
 	 print the execution count before the source line.  There are
 	 16 spaces of indentation added before the source line so that
 	 tabs won't be messed up.  */
      fprintf (gcov_file, "%9s:%5u:",
	       !line->exists ? "-" : !line->count ? "#####"
	       : format_gcov (line->count, 0, -1), line_num);
      
      if (retval)
	{
	  /* Copy source line.  */
	  do
	    {
	      retval = fgets (string, STRING_SIZE, source_file);
	      if (!retval)
		{
		  fnotice (stderr, "%s:unexpected EOF\n", src->name);
		  break;
		}
	      fputs (retval, gcov_file);
	    }
	  while (!retval[0] || retval[strlen (retval) - 1] != '\n');
	}
      if (!retval)
	fputs ("??\n", gcov_file);
      
      if (flag_branches)
	{
	  int ix;
	  arc_t *arc;
	  
	  for (ix = 0, arc = line->branches; arc; arc = arc->line_next, ix++)
	    {
	      if (arc->is_call)
		{
		  if (arc->src->count)
		    fnotice
		      (gcov_file, "call   %2d returns %s\n", ix,
		       format_gcov (arc->src->count - arc->count,
				    arc->src->count,
				    -flag_counts));
		  else
		    fnotice (gcov_file, "call   %2d never executed\n", ix);
		}
	      else
		{
		  if (arc->src->count)
		    fnotice
		      (gcov_file, "branch %2d taken %s\n", ix,
		       format_gcov (arc->count, arc->src->count,
				    -flag_counts));
		  else
		    fnotice (gcov_file, "branch %2d never executed\n", ix);
		}
	    }
	}
    }
  
  /* Handle all remaining source lines.  There may be lines after the
     last line of code.  */
  if (retval)
    {
      for (; (retval = fgets (string, STRING_SIZE, source_file)); line_num++)
	{
	  fprintf (gcov_file, "%9s:%5u:%s", "-", line_num, retval);
	  
	  while (!retval[0] || retval[strlen (retval) - 1] != '\n')
	    {
	      retval = fgets (string, STRING_SIZE, source_file);
	      if (!retval)
		break;
	      fputs (retval, gcov_file);
	    }
	}
    }
  
  if (source_file)
    fclose (source_file);
}
