#!/usr/bin/perl
#
# gtconvert.pl
#
# Initial Author: Jeff Pang <jeffpang+441@cs.cmu.edu>
# Class: 15-441 (Spring 2004)
#
# This script converts a graph generated by GraphThing into topology 
# configuration files for the project. GraphThing is a simple GUI graph tool:
#
# http://graph.seul.org/
#
# This script can only handle GraphThing graphs (*.gt format) that have
# undirected edges and no edge weights. The vertex labels in the graph
# are converted directly into nodeids so you can not use non-integer labels.
#
# usage: ./gtconvert.pl <group_num> <hostname> <graphthing.gt file>
#
# Where group_num is your group number (used for port assignment),
# hostname is the host you will run your servers on (e.g., localhost),
# and graphthing.gt file is the graphthing file in *.gt format.
#
# The genconfig.pl script must be in the same directory for this script
# to work.
#
# Note: please check your output to make sure it looks correct, especially if
# you make a very esoteric graph. I'm not 100% sure this will work properly
# for all GraphThing output!
#
###############################################################################

use strict;

our $basedir   = `dirname $0`;
chomp $basedir;
our $genconfig = "$basedir/genconfig.pl";
our $tmpfile   = "/tmp/gtconvert.$$";

if (@ARGV != 3) {
    print STDERR "usage: ./gtconvert.pl <group_num> <hostname> <graphthing.gt file>\n";
    exit(1);
}

our $grpnum = $ARGV[0];
our $host   = $ARGV[1];
our $gtfile = $ARGV[2];

# vertex_name => { vertex_edge... }
our %vertices;

# parse the graphthing file
open(GTFILE, "<$gtfile") or die "ERROR: can't open $gtfile for reading\n";
my $line;
my $lineno;
while ($line = <GTFILE>) {
    chomp $line;
    $lineno++;

    if ($line =~ /^vertex/) {

	if ($line =~ /^vertex \"(\d+)\" at \(\d+,\d+\)$/) {
	    my $label = $1;
	    if (! exists $vertices{$label}) {
		$vertices{$label} = {};
	    }
	} else {
	    warn "WARNING: skipping bad vertex line $lineno: $line\n";
	}

    } elsif ($line =~ /^edge/) {

	if ($line =~ /^edge \"(\d+)\" -- \"(\d+)\"$/) {
	    my($label1, $label2) = ($1, $2);
	    if (! exists $vertices{$label1}) {
		$vertices{$label1} = {};
	    }
	    if (! exists $vertices{$label2}) {
		$vertices{$label2} = {};
	    }

	    $vertices{$label1}->{$label2} = 1;
	    $vertices{$label2}->{$label1} = 1;
	} else {
	    warn "WARNING: skipping bad edge line $lineno: $line\n";
	}

    } else {
	# skip
    }
}
close(GTFILE);

# generate our simple adjacency list format
open(TMP, ">$tmpfile") or die "ERROR: can't open temporary file $tmpfile\n";
foreach my $k (sort { $a <=> $b } keys %vertices) {
    print TMP "$k";
    foreach my $e (sort { $a cmp $b } keys %{$vertices{$k}}) {
	print TMP " $e";
    }
    print TMP "\n";
}
close(TMP);

# call genconfig.pl to generate the config files
system("$genconfig $grpnum $host $tmpfile");

# cleanup
unlink $tmpfile;

