function [normalMaps, wasPredicted] = getSparseResultsFromDetections(I, detections, data, options)
    %David Fouhey, Abhinav Gupta, Martial Hebert
    %Data-Driven 3D Primitives For Single Image Understanding 
    %ICCV 2013
    %
    %Inference-only code


    h = size(I,1); w = size(I,2);
       
    threshes = options.sparseThresholds;

    %Figure out the cutoff point for the detectors for each threshold
    cutoffs = zeros(numel(data.precisionVSVMScore),numel(threshes));
    for i=1:numel(data.precisionVSVMScore)
        for j=1:numel(threshes)
            cutoffs(i,j) = computePrecisionCutoff(data.precisionVSVMScore{i},threshes(j));
        end 
    end

    %allocate the maps we're going to be writing to
    %we'll just sum up the normal contributions then
    %divide
    normalSumMaps = cell(numel(threshes),1);
    normalDivMaps = cell(numel(threshes),1);
    for threshI=1:numel(threshes)
        normalSumMaps{threshI} = zeros([h,w,3]);
        normalDivMaps{threshI} = zeros([h,w]);
    end



    for i=1:size(detections,1)
        clusterId = detections(i,1); score = detections(i,2);
        minX = detections(i,3); maxX = detections(i,5);
        minY = detections(i,4); maxY = detections(i,6);

        xferPatchWidth = maxX - minX;
        xferPatchHeight = maxY - minY;

        %reject too big patches irrespective of threshold
        if( (xferPatchWidth > w*options.maxPatchSizeFrac) || ...
            (xferPatchHeight > h*options.maxPatchSizeFrac))
            continue;
        end
        
        priorRankMap = [];

        for threshI=1:numel(threshes)
            if score < cutoffs(clusterId,threshI)
                continue;
            end
            
            %compute the prior rank map if we haven't already
            if numel(priorRankMap) == 0
                priorRankMap = computePriorRankImage(data.clusterData.priors{clusterId}, ...
                                                     options.priorStrength, h, w);
            end

            meanX = fix(mean([minX,maxX])); meanY = fix(mean([minY,maxY]));

            if priorRankMap(meanY,meanX) < options.priorMinRank
                continue;
            end

            %the median patch and our transfer strength
            medianPatch = data.clusterData.medianPatches{clusterId}.medianPatch;
            patches = {medianPatch.patchNx, medianPatch.patchNy, medianPatch.patchNz};
            toTransfer = data.clusterData.patchStats{clusterId}.patchStats.thetaMax < options.patchSelfConsistencyThresh; 

            wparams = data.weibullCalibrations.weibullParams{clusterId};
            calibratedScore = wblcdf(score+wparams.bias,wparams.a,wparams.b);

            %transfer the normals, if known, per channel
            for c=1:3
                xferPatch = normalSumMaps{threshI}(minY:maxY-1,minX:maxX-1,c);
                xferPatchNorm = ones(size(xferPatch)) * calibratedScore;
                scaleUp = imresize_old(patches{c}, size(xferPatch));
                if options.doMaskedTransfer
                    try
                        %in very, very rare cases we might not want to 
                        %transfer anything. It's easier and probably faster 
                        %just to catch rather than check explicitly
                        xferPatch(~toTransfer) = 0;
                        xferPatchNorm(~toTransfer) = 0;
                    catch
                        continue;
                    end 
                end
                xferPatch(~isnan(scaleUp)) = scaleUp(~isnan(scaleUp));
                %xferPatchNorm(~isnan(scaleUp)) = 1;
                normalSumMaps{threshI}(minY:maxY-1,minX:maxX-1,c) = normalSumMaps{threshI}(minY:maxY-1,minX:maxX-1,c) + xferPatch; 
                normalDivMaps{threshI}(minY:maxY-1,minX:maxX-1) = normalDivMaps{threshI}(minY:maxY-1,minX:maxX-1) + xferPatchNorm; 
            end
        end
    end
    
    normalMaps = cell(numel(threshes),1);
    wasPredicted = cell(numel(threshes),1);

    for threshI=1:numel(threshes)
        normalMap = normalSumMaps{threshI} ./ repmat(normalDivMaps{threshI} + eps, [1, 1, 3]);
        %normalize
        normalDiv = sum(normalMap.^2,3).^0.5;
        normalMap = normalMap ./ repmat(normalDiv, [1, 1, 3]);
        normalMaps{threshI} = normalMap;
        wasPredicted{threshI} =  (normalDivMaps{threshI} > 0) .* (sum(abs(normalMaps{threshI}),3) > 0);
    end

end
