(* type.sml
   Implements type conversion, displaying pieces of dcds,
   deciding if a dcds is infinite, type environment manipulation. *)


signature TYPE =
    sig
    exception TypeError of string
    val convertDcds : CDSBasic.dcds_tree -> string * CDSInternal.idcds
    val show : int * string -> CDSBasic.cell list
    val showMore : int * string -> 
	(CDSBasic.cell * CDSBasic.value list * CDSBasic.access list) list
    val infinite : CDSInternal.idcds -> bool
    val isSubtype : CDSInternal.idcds * CDSInternal.idcds -> (bool * CDSEnv.subkind)
    val insert : string * CDSInternal.idcds -> unit
    val insertReplace : string * CDSInternal.idcds -> unit
    val cellLength : CDSBasic.cell -> int
    val listDcds : int -> CDSInternal.idcds -> 
	(CDSBasic.cell * CDSBasic.value list * CDSBasic.access list) list
    val typeLookup : ''a * (''a * 'b * 'c) list -> ('b * 'c)
    val includedCell : CDSBasic.cell * CDSBasic.cell -> bool
    val includedVal : CDSBasic.value * CDSBasic.value -> bool
    val nonVar : CDSBasic.arexpr -> bool
    val includedValues : CDSBasic.value list * CDSBasic.value list -> bool
    val countDepth : CDSInternal.idcds -> int
    val stratify :(CDSBasic.cell * CDSBasic.value list * CDSBasic.access list) list ->
	(CDSBasic.cell * CDSBasic.value list * CDSBasic.access list) list list
    val extends : CDSInternal.idcds * CDSInternal.idcds -> bool
    val partOf : CDSInternal.idcds * CDSInternal.idcds -> bool
    end;
    

functor TypeFUN (structure Internal : INTERNAL
		 structure Printer : PRINTER) : TYPE =
    struct
    local open CDSBasic
	  open CDSInternal
	  open CDSEnv
    in

    exception TypeError of string

    fun append a b = a @ b

    fun unravelGraft (Cell_graft(c,t)) =
	  (case c of
	      (Cell_name s) => let fun f x = Cell_graft(x,t)
			       in (s, f)
			       end
	    | (Cell_graft(c',t')) => 
		  let val (name,f) = unravelGraft(Cell_graft(c',t'))
		      fun f' x = Cell_graft(f x,t)
		  in (name,f')
		  end
	    | _ => raise TypeError "unravelGraft: incorrect graft")
      | unravelGraft _ = raise TypeError "unravelGraft: incorrect graft"

	(* Used for grafting.  Applies f to all cells in an event list. *)
    fun tagAccess f [] = []
      | tagAccess f ((c,v)::l) = (f c,v)::(tagAccess f l)

	(* Puts together 2 access lists after a graft. *)
    fun zipAccess ([], a2List) = a2List
      | zipAccess (a1List, []) = a1List
      | zipAccess ([a1], a2List) = map (append a1) a2List
      | zipAccess (a1::a1List, a2List) = 
	  (map (append a1) a2List) @ zipAccess(a1List,a2List)

	(* Given 2 access lists, first one being the access list   *)
	(* from a recursive graft, readies the second for tagging. *)
    fun matchReplace _ _ graftedAccess [] = graftedAccess
      | matchReplace _ _ [] oldAccess = oldAccess
      | matchReplace first f (a::graftedAccess) oldAccess =
	  let fun matchAccess _ _ ([], old) = old
		| matchAccess first f ((c,v)::graft, old) =
		    let fun match (c1, c2) = 
			if (c1 = c2) then true
			else case c2 of
			    Cell_graft(c2',t) => match(c1,c2')
			  | _ => false
			fun occursIn (_, _, (c,v), [], access) = (c,v)::access
			  | occursIn (first, f, (c,v), (c1,v1)::l, access) =
		     (* If recursive access condition is the same as base *)
		     (* case and we are part of f1 def (and not fi), then *)
                     (* we begin recurring on it right away *)
			    if (c=c1) andalso first 
				then access @ ((f c1,v1)::l)
			    else if (match(c,c1)) then access @ ((c1,v1)::l)
			    else occursIn (first,f,(c,v),l,[(c1,v1)] @ access)
			fun occursInList (_, _, (c,v), []) = []
			  | occursInList (first, f, (c,v), a::alist) =
			      occursIn(first,f,(c,v),a,[]) :: 
			      occursInList(first,f,(c,v),alist)
		    in
			matchAccess first f (graft, 
					     occursInList(first,f,(c,v),old))
		    end
	  in
	      matchReplace first f graftedAccess 
	                   (matchAccess first f (a,oldAccess))
	  end

        (* When grafting an already defined cds into a new one,  *)
        (* takes its cva list and tags the cells and inserts the *)
        (* extra access conditions. *)
    fun graftCva ([], fc, alist) = []
      | graftCva (Plain(c,v,a)::l, fc, alist) =
	  let val newCell = fc c
	      val newAccess = zipAccess(alist,map (tagAccess fc) a)
	  in
	      Plain(newCell,v,newAccess)::graftCva(l,fc,alist)
	  end
      | graftCva (Delay(dcds,f)::l, fc, alist) =
	  let val g = fn (c,v,a) => 
	        (fc c,v,zipAccess(alist,map (tagAccess fc) a))
	  in Delay(dcds,(g o f))::graftCva(l,fc,alist)
	  end

        (* Given a component list, constructs a list of cells, values,    *)
        (* access conditions (cva list).  Need to be careful with grafts. *)
    fun extractCells (_, [], cvaList) = Nonrec(rev cvaList)
      | extractCells (localList, comp::l, cvaList) = 
	  case comp of
	      (Comp_cell(c,vlist,alist)) => 
		  extractCells(localList, l, Plain(c,vlist,alist)::cvaList)
	    | (Comp_graft(c,alist)) => 
		  let val (s, fc) = unravelGraft c
		      val dcds = find (s,localList) handle Find =>
			  lookup(s,typeList)
		  in case dcds of
		      (Nonrec cvas) => 
			  extractCells(localList, l, 
				       graftCva(cvas,fc,alist) @ cvaList)
		    | (Rec _) => 
			  let val f = fn (c0,v0,a0) =>
			      (fc c0,v0,zipAccess(alist,map (tagAccess fc) a0))
			  in extractCells(localList, l, Delay(dcds,f)::cvaList)
			  end
		  end

        (* Same as extractCells, except it handles recursive dcds  *)
	(* definitions.  When it encounters the recursive graft,   *)
        (* it constructs a Rec idcds (if there are other cell defs *)
        (* after the rec graft, they get processed before it).     *)
    fun extractCellsRec (_, [], name, cvaList) = 
	 raise TypeError "extractCellsRec: incorrect recursive dcds definition"
      | extractCellsRec (localList, comp::l, name, cvaList) = 
	  case comp of
	      (Comp_cell(c,vlist,alist)) => 
		  extractCellsRec(localList,
				  l,name,Plain(c,vlist,alist)::cvaList)
	    | (Comp_graft(c,alist)) =>
		  let val (s, fc) = unravelGraft c
		  in if (s = name)   (* this the recursive graft *)
			 then if (l <> nil)
				  then extractCellsRec(localList,
						       l@[comp],name,cvaList)
			      else 
				  let fun f1 (c0,v0,a0) =
				     (fc c0, v0, matchReplace true fc alist a0)
				      fun fi (c0,v0,a0) =
					  (fc c0, v0, map (tagAccess fc) 
					   (matchReplace false fc alist a0))
				  in Rec(rev cvaList, f1, fi)
				  end
		     else   (* grafting some other thing in here *)
			 let val graftDcds = find (s,localList) handle Find =>
			     lookup(s,typeList)
			 in (case graftDcds of
				 (Nonrec graftCvaList) =>
				     let val newCva = 
					   graftCva(graftCvaList,fc,alist)
				     in extractCellsRec(localList,
							l,name,newCva@cvaList)
				     end
			       | (Rec _) => 
				     let val f = fn (c0,v0,a0) => 
					 (fc c0, v0,zipAccess
					  (alist,map (tagAccess fc) a0))
					 val recCva = Delay(graftDcds,f)
				     in extractCellsRec(localList,
							l,name,recCva::cvaList)
				     end)
			 end
		  end

	(* Converts a dcds from parse tree form to internal form. *)
    fun convertDcds d = convertDcdsRec [] d

    and convertDcdsRec localList (Normal(s,clist)) = 
	(s,extractCells(localList,clist,[]))
      | convertDcdsRec localList (Recursive(s,clist)) = 
	(s,extractCellsRec(localList,clist,s,[]))
      | convertDcdsRec localList (Local(d1,d2)) = 
	let val (s1,d1cnv) = convertDcdsRec localList d1
	in convertDcdsRec ((s1,d1cnv)::localList) d2
	end

	(* Returns first i (c,v,a) from a recursive dcds declaration. *)
    fun iterate (0,_,_,_) _ = []
      | iterate (i, first, f1, fi) (c,v,a) =
	  if (first)     (* first iteration: apply f1 *)
	      then if (i = 1) then [(c,v,a)]
		   else let val newCva = f1(c,v,a)
			in (c,v,a)::(newCva::(iterate(i-2,false,f1,fi) newCva))
			end
	  else let val newCva = fi(c,v,a)
	       in newCva::(iterate (i-1,false,f1,fi) newCva)
	       end

	(* Returns first i cva's from a cva list.  If a recursive dcds *)
	(* is grafted into another one, will list more than i items.   *)
    fun listCva (0, _) = []
      | listCva (i, []) = []
      | listCva (i, cva::l) = 
	  (case cva of
	       (Plain(c,v,a)) => [(c,v,a)] @ (listCva(i-1,l))
	     | (Delay(dcds,f)) => 
		   (case dcds of
			(Nonrec _) => 
			    raise TypeError "listCva: incorrect representation"
		      | (Rec(cvaList,f1,fi)) => 
			    let val recPart = map (iterate(i,true,f1,fi))
				                  (listCva(i,cvaList))
				val ordered = flatten recPart
			    in (map f ordered) @ (listCva(i-1,l))
			    end))

	(* Lists first i cva's from any dcds. *)
    fun listDcds i (Nonrec cvaList) = listCva(i, cvaList)
      | listDcds i (Rec(cvaList,f1,fi)) = 
	  let val recPart = map (iterate(i,true,f1,fi)) (listCva(i,cvaList))
	  in 
	      flatten recPart
	  end

        (* Next 2 functions return piecemeal info on a dcds. *)
    fun show (i, name) =
	  let val dcds = lookup(name, typeList)
	      val cvaList = listDcds i dcds
	      val cellList = map (#1) cvaList
	  in
	      cellList
	  end

    fun showMore (i, name) =
	  let val dcds = lookup(name, typeList)
	      val cvaList = listDcds i dcds
	  in
	      cvaList
	  end

        (* Decide if a dcds has an infinite number of cells arising  *)
        (* from a recursive declaration, and not stuff like R.[0..]. *)
    fun infinite (Nonrec cvaList) = 
	  let fun infiniteCva [] = false
		| infiniteCva ((Plain(c,v,a))::l) = infiniteCva l
		| infiniteCva ((Delay(d,f))::l) = 
		    let val dInfinite = infinite d
		    in if (dInfinite) then true
		       else infiniteCva l
		    end
	  in
	      infiniteCva cvaList
	  end
      | infinite (Rec(cvaList,f1,fi)) = true


	(* Is an arithmetic expression free of variables? *)
    fun nonVar (Arexpr_var _) = false
      | nonVar (Arexpr_minus a) = nonVar a
      | nonVar (Arexpr_plus(a1,a2)) = (nonVar a1) andalso (nonVar a2)
      | nonVar (Arexpr_sub(a1,a2)) = (nonVar a1) andalso (nonVar a2)
      | nonVar (Arexpr_mult(a1,a2)) = (nonVar a1) andalso (nonVar a2)
      | nonVar (Arexpr_div(a1,a2)) = (nonVar a1) andalso (nonVar a2)
      | nonVar _ = true

	(* Same for values --> this not done yet **** *)
    fun nonVarVal (Val_arexpr a) = nonVar a
      | nonVarVal (Val_with(_,_)) = false
      | nonVarVal _ = true

	(* c1 has variables -- extracts them and their respective *)
	(* values in corresponding position in c2. *)
	(* Not complete -- should return a list *** *)
    fun extractVars (Cell_graft(c1,t1),Cell_graft(c2,t2)) =
	(case t1 of
	     (Tag_arexpr(Arexpr_var s)) => 
		 (case t2 of
		      (Tag_str s') => (s,Val_string s')::extractVars(c1,c2)
		    | (Tag_arexpr a) => (s,Val_arexpr a)::extractVars(c1,c2)
		    | (Tag_interval v) => (s,v)::extractVars(c1,c2))
	   | _ => extractVars(c1,c2))
      | extractVars (Cell_fun(e1,c1),Cell_fun(e2,c2)) =
	 extractVars(c1,c2)
      | extractVars (_,_) = []

	(* Substitutes an int for a var and simplifies. *)
    fun expandArexpr (Arexpr_int i,_,_) = Arexpr_int i
      | expandArexpr (Arexpr_var s1,s,i) = if s = s1 then Arexpr_int i
					   else Arexpr_var s1
      | expandArexpr (Arexpr_minus a,s,i) = 
	  Internal.simplify(Arexpr_minus(expandArexpr(a,s,i)))
      | expandArexpr (Arexpr_plus(a1,a2),s,i) = 
	  Internal.simplify(Arexpr_plus(expandArexpr(a1,s,i),
					expandArexpr(a2,s,i)))
      | expandArexpr (Arexpr_sub(a1,a2),s,i) = 
	  Internal.simplify(Arexpr_sub(expandArexpr(a1,s,i),
				       expandArexpr(a2,s,i)))
      | expandArexpr (Arexpr_mult(a1,a2),s,i) = 
	  Internal.simplify(Arexpr_mult(expandArexpr(a1,s,i),
					expandArexpr(a2,s,i)))
      | expandArexpr (Arexpr_div(a1,a2),s,i) = 
	  Internal.simplify(Arexpr_div(expandArexpr(a1,s,i),
				       expandArexpr(a2,s,i)))

	(* Is at least one branch of a boolexp free of variables? *)
    fun usableConstraint (Boolexp_gt(a1,a2)) = (nonVar a1) orelse (nonVar a2)
      | usableConstraint (Boolexp_gteq(a1,a2)) = (nonVar a1) orelse (nonVar a2)
      | usableConstraint (Boolexp_lt(a1,a2)) = (nonVar a1) orelse (nonVar a2)
      | usableConstraint (Boolexp_lteq(a1,a2)) = (nonVar a1) orelse (nonVar a2)
      | usableConstraint (Boolexp_eq(v1,v2)) = 
	  (nonVarVal v1) orelse (nonVarVal v2)
      | usableConstraint (Boolexp_noteq(v1,v2)) = 
	  (nonVarVal v1) orelse (nonVarVal v2)
      | usableConstraint (Boolexp_or(b1,b2)) = 
	  (usableConstraint b1) orelse (usableConstraint b2)
      | usableConstraint (Boolexp_and(b1,b2)) = 
	  (usableConstraint b1) orelse (usableConstraint b2)


	(* If variable s is replaced with v in boolexp b, does *)
	(* it hold?  We can have intervals here. *)
    fun satConstraint (Boolexp_gt(a1,a2),s,v) =
	(case v of
	    (Val_arexpr(Arexpr_int i)) => 
		let val a1' = expandArexpr(a1,s,i)
		    val a2' = expandArexpr(a2,s,i)
		in case a1' of
		    (Arexpr_int i1) => (case a2' of 
					    (Arexpr_int i2) => i1 > i2
					  | _ => false)
		  | _ => false
		end
	  | (Val_interval_inf) => true
	  | (Val_interval_from i) => true
	  | (Val_interval_to i) => 
		let val a1' = expandArexpr(a1,s,i)
		    val a2' = expandArexpr(a2,s,i)
		in case a1' of
		    (Arexpr_int i1) => (case a2' of 
					    (Arexpr_int i2) => i1 > i2
					  | _ => false)
		  | _ => false
		end
	  | (Val_interval_fromto(low,high)) => 
		let val a1' = expandArexpr(a1,s,high)
		    val a2' = expandArexpr(a2,s,high)
		in case a1' of
		    (Arexpr_int i1) => (case a2' of 
					    (Arexpr_int i2) => i1 > i2
					  | _ => false)
		  | _ => false
		end
	  | _ => false)
      | satConstraint (Boolexp_gteq(a1,a2),s,v) =
	(case v of
	    (Val_arexpr(Arexpr_int i)) => 
		let val a1' = expandArexpr(a1,s,i)
		    val a2' = expandArexpr(a2,s,i)
		in case a1' of
		    (Arexpr_int i1) => (case a2' of 
					    (Arexpr_int i2) => i1 >= i2
					  | _ => false)
		  | _ => false
		end
	  | (Val_interval_inf) => true
	  | (Val_interval_from i) => true
	  | (Val_interval_to i) => 
		let val a1' = expandArexpr(a1,s,i)
		    val a2' = expandArexpr(a2,s,i)
		in case a1' of
		    (Arexpr_int i1) => (case a2' of 
					    (Arexpr_int i2) => i1 >= i2
					  | _ => false)
		  | _ => false
		end
	  | (Val_interval_fromto(low,high)) => 
		let val a1' = expandArexpr(a1,s,high)
		    val a2' = expandArexpr(a2,s,high)
		in case a1' of
		    (Arexpr_int i1) => (case a2' of 
					    (Arexpr_int i2) => i1 >= i2
					  | _ => false)
		  | _ => false
		end
	  | _ => false)
      | satConstraint (Boolexp_lt(a1,a2),s,v) = true
      | satConstraint (Boolexp_lteq(a1,a2),s,v) = true
      | satConstraint (Boolexp_eq(v1,v2),s,v) = true
      | satConstraint (Boolexp_noteq(v1,v2),s,v) = true
      | satConstraint (Boolexp_or(b1,b2),s,v) =
	  satConstraint(b1,s,v) orelse satConstraint(b2,s,v)
      | satConstraint (Boolexp_and(b1,b2),s,v) =
	  satConstraint(b1,s,v) andalso satConstraint(b2,s,v)


	(* Check for value inclusion.  Can have intervals in here. *)
	(* Assume that only v1 can contain variables. *)
    fun includedVal (v1, v2) =
	if (v1 = v2) then true
	else case v1 of
	    (Val_output v1') =>
		(case v2 of
		     (Val_output v2') => includedVal(v1',v2')
		   | _ => false)
	  | (Val_arexpr(Arexpr_int n)) => 
		(case v2 of
		     (Val_interval_inf) => true
		   | (Val_interval_from nl) => n >= nl
		   | (Val_interval_to nh) => n <= nh
		   | (Val_interval_fromto(nl,nh)) => n >= nl andalso n <= nh
		   | _ => false)
	  | (Val_arexpr(Arexpr_var _)) => true
	  | (Val_arexpr a) => 
		if nonVar a then false
		else (case v2 of
			  (Val_arexpr _) => true
			| (Val_interval_inf) => true
			| (Val_interval_from _) => true
			| (Val_interval_to _) => true
			| (Val_interval_fromto(_,_)) => true
			| _ => false)
	  | (Val_with(s,b)) => 
		if (usableConstraint b) 
		    then satConstraint(b,s,v2)
		(* Could do more here *** *)
		else true
	  | (Val_pair(v11,v12)) => 
		(case v2 of
		     (Val_pair(v21,v22)) => includedVal(v11,v21) andalso
			                    includedVal(v12,v22)
		   | _ => false)
	  | (Val_interval_inf) => 
		(case v2 of
		     (Val_interval_inf) => true
		   | _ => false)
	  | (Val_interval_from n1) =>
		(case v2 of
		     (Val_interval_inf) => true
		   | (Val_interval_from n2) => n2 <= n1
		   | _ => false)
	  | (Val_interval_to n1) =>
		(case v2 of
		     (Val_interval_inf) => true
		   | (Val_interval_to n2) => n2 >= n1
		   | _ => false)
	  | (Val_interval_fromto(n1l,n1h)) =>
		(case v2 of
		     (Val_interval_inf) => true
		   | (Val_interval_from n2l) => n1l >= n2l
		   | (Val_interval_to n2h) => n1h <= n2h
		   | (Val_interval_fromto(n2l,n2h)) => ((n1l >= n2l) andalso
							(n1h <= n2h))
		   | _ => false)
	  | _ => false

        (* Check if tag t1 is same as or included in tag t2. *)
    fun includedTag (t1, t2) =
	  if (t1 = t2) then true
	  else case t1 of
	      (Tag_interval i1) => 
		  (case t2 of
		       (Tag_interval i2) => includedVal(i1,i2)
		     | _ => false)
	    | (Tag_arexpr a1) => 
		  (case t2 of
		       (Tag_interval i2) => includedVal(Val_arexpr a1,i2)
		     | _ => false)
	    | _ => false

    fun printBindings [] = ""
      | printBindings ((s,v)::rest) = "("^s^", "^(Printer.unparseVal v)^"), "^(printBindings rest)


	(* Is cell c1 covered by cell c2?  This is not trivial equality *)
	(* because there can be grafts with intervals (R.[0..] etc.).   *)
        (* Only cell c1 can be a variable or have variables. *)
    fun includedCell (c1, c2) =
	  if (c1 = c2) then true
	  else case c1 of
	      (Cell_graft(c1',t1)) =>
		  (case t1 of
		       (Tag_arexpr(Arexpr_var v)) => stripTags(c1',c2)
		     | _ => (case c2 of
				 (Cell_graft(c2',t2)) => 
				     includedCell(c1',c2') andalso 
				     includedTag(t1,t2)
			       | _ => false))
	    | (Cell_var _) => true
	    | (Cell_with(c1',b)) => 
		  if includedCell(c1',c2)
		    then if (usableConstraint b)
			   then let val bindings = extractVars(c1',c2)
				in if null bindings then false
				   else if (length bindings) = 1
					    then let val (s,v) = hd bindings
						 in satConstraint(b,s,v)
						 end
					else raise TypeError 
			("includedCell: cannot handle this yet, c1 =  "^(Printer.unparseCell c1)^
			 ", c2 = "^(Printer.unparseCell c2)^",  bindings = "^(printBindings bindings))
				end
			 else true
		  else false
	    | _ => false

        (* c1 had a variable tag, so now we want to see if we can *)
        (* strip tags from c2 and make it match c1. *)
    and stripTags (c1,Cell_graft(c2',t2)) = if includedCell(c1,c2') then true
					    else stripTags(c1,c2')
      | stripTags (c1, c2) = if includedCell(c1,c2) then true
			    else false

        (* Check if a value list is a subset of another value list. *)
    fun includedValues ([], _) = true
      | includedValues (_, []) = false
      | includedValues (v1::v1List, v2List) =
	  let fun includedValList (v1, []) = false
		| includedValList (v1, v2::v2List) =
		    if (includedVal(v1,v2)) then true
		    else includedValList(v1,v2List)
	  in
	      if (includedValList(v1,v2List)) 
		  then includedValues(v1List,v2List)
	      else false
	  end

        (* Is x1 a subset of x2? *)
    fun subset ([], _) = true
      | subset (_, []) = false
      | subset ((c1,v1)::x1, x2) =
	  let fun includedEvent (c1,v1,[]) = false
                | includedEvent (c1,v1,(c2,v2)::x) =
		    if (includedCell(c1,c2) andalso includedVal(v1,v2))
			then true
		    else includedEvent(c1,v1,x)
	  in 
	      if (includedEvent(c1,v1,x2)) then subset(x1,x2)
	      else false
	  end

        (* Verify a1 "weaker than" a2 relation on accessibility lists. *)
    fun weakerAccess ([], _) = true
      | weakerAccess (_, []) = false
      | weakerAccess (a1List, a2::a2List) =
	    (* Does a2 occur in or is it a superset of a set in a1List. *)
	let fun containedOrSuperset (a2, []) = false
	      | containedOrSuperset (a2, a1::a1List) =
		if (a1 = a2) then true
		     (* Now check containment -- remember we have intervals. *)
		else if (length(a1) = length(a2) andalso
			 subset(a2,a1)) then true
		     (* Check for superset now. *)
		     else if (length(a2) > length(a1) andalso
			      subset(a1,a2)) then true
			  else containedOrSuperset(a2,a1List)
	in if (containedOrSuperset(a2,a1List)) then 
	    if (a2List = nil) then true
	    else weakerAccess(a1List, a2List)
	   else false
	end

        (* Check if a cva element occurs in a cva list. *)
    fun checkCva (c1,v1,a1,[]) = false
      | checkCva (c1,v1,a1,(Plain(c2,v2,a2))::cva2) =
	  if includedCell(c1,c2)
	      then let val match = includedValues(v1,v2) andalso
		                   weakerAccess(a2,a1)
		   in if (match) then true
		      else checkCva(c1,v1,a1,cva2)
		   end
	  else checkCva(c1,v1,a1,cva2)

        (* Check if a cva element occurs in a cva list ignoring access conditions *)
    fun checkOnlyCv (c1,v1,a1,[]) = false
      | checkOnlyCv (c1,v1,a1,(Plain(c2,v2,a2))::cva2) =
	  if (includedCell(c1,c2)) andalso (includedValues(v1,v2)) then true
	  else checkOnlyCv(c1,v1,a1,cva2)

        (* Check if a cva element occurs in a cva list with same *)
	(* access condition and fewer values. *)
    fun checkSameAccCva (c1,v1,a1,[]) = false
      | checkSameAccCva (c1,v1,a1,(Plain(c2,v2,a2))::cva2) =
	  if includedCell(c1,c2)
	      then let val match = includedValues(v1,v2) andalso (a1 = a2)
		   in if (match) then true
		      else checkSameAccCva(c1,v1,a1,cva2)
		   end
	  else checkSameAccCva(c1,v1,a1,cva2)

        (* Check if a cva element occurs in a cva list with same *)
	(* access condition and more values *)
    fun checkMoreValCva (c1,v1,a1,[]) = false
      | checkMoreValCva (c1,v1,a1,(Plain(c2,v2,a2))::cva2) =
	  if includedCell(c1,c2)
	      then let val match = includedValues(v2,v1) andalso (a1 = a2)
		   in if (match) then true
		      else checkMoreValCva(c1,v1,a1,cva2)
		   end
	  else checkMoreValCva(c1,v1,a1,cva2)

	(* Decide if one finite dcds is a subtype of another finite one. *)
    fun included ([], _) = true
      | included (_, []) = false
      | included ((Plain(c1,v1,a1))::cva1, cva2) =
	  checkCva(c1,v1,a1,cva2) andalso included(cva1,cva2)

	(* Are all cells and values of cva1 included in cva2 *)
    fun includedOnlyCv ([], _) = true
      | includedOnlyCv (_, []) = false
      | includedOnlyCv ((Plain(c1,v1,a1))::cva1, cva2) =
	  checkOnlyCv(c1,v1,a1,cva2) andalso includedOnlyCv(cva1,cva2)

        (* Decide if cva1's cells all occur in cva2 with same *)
        (* access condition and fewer values *)
    fun partofCva ([], _) = true
      | partofCva (_, []) = false
      | partofCva ((Plain(c1,v1,a1))::cva1, cva2) =
	  checkSameAccCva(c1,v1,a1,cva2) andalso partofCva(cva1,cva2)

        (* Decide if cva1's cells all occur in cva2 with same *)
        (* access condition but potentially more values. *)
    fun extendsCva ([], _) = true
      | extendsCva (_, []) = false
      | extendsCva ((Plain(c1,v1,a1))::cva1, cva2) =
	  checkMoreValCva(c1,v1,a1,cva2) andalso extendsCva(cva1,cva2)

	(* Count length of cell name in sense of number of tags. *)
    fun cellLength (Cell_name _) = 1
      | cellLength (Cell_var _) = 1
      | cellLength (Cell_fun(e,c)) = cellLength c
      | cellLength (Cell_graft(c,t)) = 1 + cellLength c
      | cellLength (Cell_with(c,b)) = cellLength c

	(* Make a Plain(c,v,a) into a (c,v,a) and the other way. *)
    fun stripPlain (Plain(c,v,a)) = (c,v,a)
      | stripPlain _ = raise TypeError "stripPlain: not a Plain"

    fun addPlain (c,v,a) = Plain(c,v,a)

	(* Next 2 functions are used to figure out how much of a *)
	(* dcds to expose so that even the most deeply embedded  *)
        (* recursive graft has been unrolled at least once.      *)
    fun findDelay [] = 0
      | findDelay ((Delay(d,f))::l) = 1 + (countDepth d)
      | findDelay ((Plain(c,v,a))::l) = findDelay l

    and countDepth (Nonrec cvas) = (length cvas) + (findDelay cvas)
      | countDepth (Rec(cvas,f1,fi)) = 1 + (length cvas) + (findDelay cvas)

    fun cvaIfy [] = []
      | cvaIfy ((c,v)::rest) = Plain(c,[v],[])::(cvaIfy rest)

	(* Is a cva element enabled by a list of cvas? *)
    fun enabledBy (c,vlist,alist) enab = 
	if null alist then true
	else let fun checkOne [] _ = false
		   | checkOne (a::others) enab = if included(cvaIfy a, enab) then true
						 else checkOne others enab
	     in checkOne alist enab
	     end
	 
	(* A version of enabled by for use with stratify.  It does not *)
        (* require any weaker than conditions on accessibility relation *)
    fun enabledStrat (c,vlist,alist) enab = 
	if null alist then true
	else let fun checkOne [] _ = false
		   | checkOne (a::others) enab = if includedOnlyCv(cvaIfy a, enab) then true
						 else checkOne others enab
	     in checkOne alist enab
	     end

        (* Separate cva list into cvas enabled by some list of cvas and *)
	(* those not enabled. *)
    fun findEnabled (enab, [], yes, no) = (yes, no)
      | findEnabled (enab, ((c,vlist,alist)::cva), yes, no) =
	if enabledBy (c,vlist,alist) enab then
	    findEnabled(enab, cva, (c,vlist,alist)::yes, no)
	else findEnabled(enab, cva, yes, (c,vlist,alist)::no)

        (* Given a (cell, value list, access list) stratify it according *)
        (* to the access conditions: initial, cvas enabled by initial, etc. *)
    fun stratify cvaList =
	let fun findEnabledStrat (enab, [], yes, no) = (yes, no)
	      | findEnabledStrat (enab, ((c,vlist,alist)::cva), yes, no) =
		if enabledStrat (c,vlist,alist) enab then
		    findEnabledStrat(enab, cva, (c,vlist,alist)::yes, no)
		else findEnabledStrat(enab, cva, yes, (c,vlist,alist)::no)
	    fun stratifyRec [] done = []
	      | stratifyRec cvaList done =
		let val (yes,no) = findEnabledStrat(done,cvaList,[],[])
		in yes :: (stratifyRec no (done @ (map Plain yes)))
		end
	in stratifyRec cvaList []
	end

        (* Given two cvas, construct list of common parts (including *)
	(* intersecting the value lists) and access conditions. *)
    fun intersectCva (_, []) = []
      | intersectCva ([], _) = []
      | intersectCva (cva1,(c,vlist,alist)::cva2) =
	let fun onePass (c,vlist,alist) [] = []
	      | onePass (c,vlist,alist) ((c1,v1list,a1list)::rest) =
		if includedCell (c,c1) 
		    then [(c,intersect(vlist,v1list),intersect(alist,a1list))]
		else onePass (c,vlist,alist) rest
	in (onePass (c,vlist,alist) cva1) @ (intersectCva(cva1,cva2))
	end

	(* Is every maximal state in cva1 maximal in cva2?  Both cvas *)
	(* are finite.  First check that all initial cells from cva2 *)
	(* are in cva1, then all cells enabled by common values in cva2 *)
	(* are also in cva1, and so on. *)
    fun maximal ([], _, _) = true
      | maximal (cva1, cva2, enab) =
	let val (enab1,not1) = findEnabled(map Plain enab,cva1,[],[])
	    val (enab2,not2) = findEnabled(map Plain enab,cva2,[],[])
	in if extendsCva(map Plain enab2, map Plain enab1) then
	    let val newEnab = intersectCva(enab2, enab1)
	    in if null newEnab then true else maximal (not1, not2, newEnab)
	    end
	   else false
	end

	(* Does d1 belong to a partition of d2?   That is, are the sets *)
	(* of states of d1 included in d2, same accesibility conditions, *)
	(* and any maximal state of d1 is maximal in d2? *)
    fun partOf (d1, d2) = 
	case (infinite d1, infinite d2) of
	    (true, false) => false
	  | (false, false) => 
		(case (d1,d2) of
		     (Nonrec cva1, Nonrec cva2) => 
			 partofCva(cva1,cva2) andalso 
			 maximal(map stripPlain cva1,map stripPlain cva2,[])
		   | _ => raise TypeError "dcds actually infinite")
	  | (false, true) => 
		(case d1 of
		     (Nonrec cva1) =>
			 let val cva1List = map stripPlain cva1
			     val lengthList = map cellLength 
				 (map (#1) cva1List)
			     val n = fold max lengthList 0
			     val cva2List = listDcds n d2
			     val cva2 = map addPlain cva2List
			 in
			     partofCva(cva1, cva2) andalso
			     maximal(cva1List, cva2List, [])
			 end
		   | _ => raise TypeError "dcds actually infinite")
	  | (true, true) => 
		(* Unroll all recursions enough times in d1 to diff from d2. *)
		let val d2Depth = countDepth d2
		    val d2Len = fold max (map cellLength (map (#1) (listDcds d2Depth d2))) 0
		    val d1Depth = countDepth d1
		    val cva1List = listDcds (d1Depth + d2Len) d1
		    val d1Len = fold max (map cellLength (map (#1) cva1List)) 0
		    val cva2List = listDcds (d2Depth + d1Len) d2
		    val cva1 = map addPlain cva1List
		    val cva2 = map addPlain cva2List
		in
		    partofCva(cva1, cva2) andalso 
		    maximal(cva1List, cva2List, [])
		end

        (* Is d1 an extension of d2?  That is, does d1 have d2's initial cells, *)
        (* fewer values per those cell, and the same accessibility conditions? *)
    fun extends (d1, d2) = 
	case (infinite d1, infinite d2) of
	    (false, true) => false
	  | (false, false) => 
		(case (d1,d2) of
		     (Nonrec cva1, Nonrec cva2) => 
			 if extendsCva(cva2,cva1) then
			     if (length cva1 < length cva2) then false
			     else true
			 else false
		   | _ => raise TypeError "dcds actually infinite")
	  | (true, false) => 
		(case d2 of
		     (Nonrec cva2) =>
			 let val cva2List = map stripPlain cva2
			     val lengthList = map cellLength 
				 (map (#1) cva2List)
			     val n = fold max lengthList 0
			     val cva1List = listDcds n d1
			 in
			     extendsCva(cva2,map addPlain cva1List)
			 end
		   | _ => raise TypeError "dcds actually infinite")
	  | (true, true) => 
		(* Unroll all recursions enough times in d2 to diff from d1. *)
		let val d1Depth = countDepth d1
		    val d1Len = fold max (map cellLength (map (#1) (listDcds d1Depth d1))) 0
		    val d2Depth = countDepth d2
		    val cva2List = listDcds (d2Depth + d1Len) d2
		    val d2Len = fold max (map cellLength (map (#1) cva2List)) 0
		    val cva1List = listDcds (d1Depth + d2Len) d1
		in
		    if extendsCva(map addPlain cva2List,map addPlain cva1List)
			(* Check if d1 and d2 actually have same cells, so d1 is no extension *)
			then if partOf (d1, d2) then false
			     else true
		    else false
		end

    fun isSubtype (d1, d2) = if partOf(d1, d2) then (true, partof)
			     else if extends(d1, d2) then (true, ext)
				  else (false, notsubtype)

      
        (* Partition a set into three: subtypes, supertypes and *)
        (* incomparables with respect to dcds d. *)
    fun partition (d,[],subs,sups,incomps) = (subs,sups,incomps)
      | partition (d,(M,k)::l,subs,sups,incomps) =
	let val d' = lookup(M,typeList)
	    val (success, kind) = isSubtype(d,d')
	in if success then partition(d,l,subs,sups@[(M,kind)],incomps)
	   else let val (success, kind) = isSubtype(d',d)
		in if success then partition(d,l,subs@[(M,kind)],sups,incomps)
		   else partition(d,l,subs,sups,incomps@[(M,k)])
		end
	end

	(* Utility functions for handling the type hierarchy list. *)
    fun typeInsert (name,(parents,children)) =
	hierarchy := (!hierarchy) @ [(name,parents,children)]

    fun typeReplace (M,Mup,Mdown) =
	  let val newHierarchy = replace3Ord((M,Mup,Mdown),!hierarchy,[])
	  in hierarchy := newHierarchy
	  end

    fun typeMerge ((Mup,Mdown),Mup',Mdown') =
	  (Mup @ Mup', Mdown @ Mdown')

    fun typeLookup (t, []) = raise Lookup ""
      | typeLookup (t, (t', tUp, tDown)::l) = if t = t' then (tUp,tDown)
					      else typeLookup(t,l)

	(* Inserts a new type in the subtype hierarchy.  Modifies  *)
	(* hierarchy accordingly.  Only nearest parent or child is *)
	(* stored (and not the transitive closure.) *)
    fun insertRec (name, d, [], (parents,children)) = 
	  typeInsert(name,(duplicates parents, duplicates children))
      | insertRec (name, d, (Mname, Mup, Mdown)::l, entry) =
	let val d' = lookup(Mname,typeList)
	    val (subtype, kind) = isSubtype(d,d')
	in if subtype then 
	    let val (subs,sups,incomps) = partition(d,Mdown,[],[],[])
	    in case (subs,sups,incomps) of
		([],[],_) => (typeReplace(Mname,Mup,Mdown@[(name,kind)]);
			      insertRec(name,d,l,typeMerge(entry,[(Mname,kind)],[])))
	      | (_,[],[]) => (typeReplace(Mname,Mup,[(name,kind)]);
			      insertRec(name,d,l,
					typeMerge(entry,[(Mname,kind)],Mdown)))
	      | ([],_,[]) => insertRec(name,d,l,entry)
	      | (_,[],_) => (typeReplace(Mname,Mup,incomps@[(name,kind)]);
			     insertRec(name,d,l,typeMerge(entry,[(Mname,kind)],subs)))
	      | ([],_,_) => insertRec(name,d,l,entry)
	      | _ => raise TypeError "insert: inconsistent subtypes"
	    end
	   else let val (supertype, kind) = isSubtype(d',d)
		in if supertype then
		    let val (subs,sups,incomps) = partition(d,Mup,[],[],[])
		    in case (subs,sups,incomps) of
			([],[],_) => (typeReplace(Mname,Mup@[(name,kind)],Mdown);
				      insertRec(name,d,l,
						typeMerge(entry,[],[(Mname,kind)])))
		      | ([],_,[]) => (typeReplace(Mname,[(name,kind)],Mdown);
				      insertRec(name,d,l,
						typeMerge(entry,Mup,[(Mname,kind)])))
		      | (_,[],[]) => insertRec(name,d,l,entry)
		      | ([],_,_) => (typeReplace(Mname,incomps@[(name,kind)],Mdown);
				     insertRec(name,d,l,
					       typeMerge(entry,sups,[(Mname,kind)])))
		      | (_,[],_) => insertRec(name,d,l,entry)
		      | _ => raise TypeError "insert: inconsistent subtypes"
		    end
		   else (* name and Mname are incomparable *)
		       insertRec(name,d,l,entry)
		end
	end

        (* Wrapper for insertRec.  Calls it with starting conditions. *)
    fun insert (name, d) = insertRec(name,d,!hierarchy,([],[]))

	(* Fixes up hierarchy after removal of an element. Highs *)
	(* and lows are the parents/children of defunct element. *)
	(* Each element in highs gets all in lows as extra  *)
	(* children _unless_ there is a path between them. *)
	(* Similarly the other way around. *)
	(* *** This should be different with new subtyping rules: *)
	(* Now it's possible for highs not to have lows as subtypes *)
    fun splice ([], _) = ()
      | splice ((M,k)::highs, lows) =
	(* M1>=M2.  Is M2 reachable from M1? *)
	let fun reachable(M1,M2) =
	    let val (parents,children) = typeLookup(M1,!hierarchy)
	    in
		if member(M2,map #1 children) then true
		else let fun reachableList ([], M) = false
			   | reachableList (M'::l, M) =
			     if reachable(M',M) then true
			     else reachableList(l, M)
		     in
			 reachableList(map #1 children,M2)
		     
		     end
	    end
	(* Splice not reachable children of M into its descendant list *)
        (* and put M into their parent lists. *)
	    fun spliceList (M, []) = ()
	      | spliceList (M, (M',kind)::lows) =
		let val (parentsM,childrenM) = typeLookup(M,!hierarchy)
		    val (parentsM',childrenM') = typeLookup(M',!hierarchy)
		in
		    if reachable(M,M') then spliceList(M,lows)
		    else (typeReplace(M, parentsM, (M',kind)::childrenM);
			  typeReplace(M', (M,kind)::parentsM', childrenM');
			  spliceList(M,lows))
		end
	in
	    (spliceList(M, lows);
	     splice(highs, lows))
	end

        (* Gets rid of all occurences of 'name' from hierarchy *)
        (* (including parent/children lists in hierarchy). *)
    fun excise (name, []) = []
      | excise (name, (M,Mup,Mdown)::rest) =
	(* Remove first occurence of an element in a list *)
	let fun dispose (e,[]) = []
	      | dispose (e,(e1,k)::l) = if e = e1 then l else (e1,k)::(dispose(e,l))
	in
	    if (name = M) then excise(name,rest)
	    else
		let val newMup = dispose(name,Mup)
		    val newMdown = dispose(name,Mdown)
		in
		    (M,newMup,newMdown)::(excise(name,rest))
		end
	end

	(* We first remove the old name from the hierarchy, before *)
	(* inserting new guy.  Must make sure to "sew" the hierarchy *)
	(* back together after removal. *)
    fun insertReplace (name, d) = 
	let val (Mup,Mdown) = typeLookup(name,!hierarchy) 
	in 
	    (hierarchy := excise(name,!hierarchy);
	     splice(Mup,Mdown);
	     insert(name,d))
	end

    end
    end;
