(* toplevel.sml
   Top level loop of the CDS interpreter and related functions *)

(* Signature for the toplevel functions of the CDS0 interpreter *)

signature TOPLEVEL =
    sig
    val ProcessInput : CDSBasic.ParseTree -> unit
    val ProcessEXP : CDSBasic.expr -> unit
    datatype typeStyle = regular | refined
    val cds0 : typeStyle -> unit
    end;


functor ToplevelFUN (structure CDSParser : PARSER
		     structure CellParser : PARSER
		     structure CDSLex : LEXER
                     structure Interface : INTERFACE
		     structure Parser : PARSER_INTERFACE
		     structure Internal : INTERNAL
		     structure Eval : EVALUATOR
		     structure Printer : PRINTER
		     structure Match : MATCH
		     structure Type : TYPE
		     structure TypeChecker : TYPE_CHECKER
		     structure QandA : QANDA
		     structure Refine : REFINE
		     structure PcfCode : PCF_CODE) : TOPLEVEL =
    struct

    local open CDSBasic
	  open CDSEnv
	  open CDSInternal
    in
    val currentExp : forest ref = ref (forest_basic(0,[]))
    val loopFlag : bool ref = ref true
    val systemTimer : System.Timer.timer ref = ref (System.Timer.start_timer())

    fun ProcessDCDS d = 
	  let val (dcdsName, dcds) = Type.convertDcds d
	  in
	      (if search(dcdsName, !typeList)
		   then Type.insertReplace(dcdsName, dcds)
	       else 
		   Type.insert(dcdsName, dcds);
	       store(dcdsName,dcds,typeList);
	       incTimeStamp ();
	       output(std_out, "Type "^dcdsName^" defined.\n"))
	  end

    fun ProcessEXP e = 
	(let val f = Internal.convert e
	     val _ = if (!typing) then
		 (let val (rt, gT) = Refine.mgType e
		      val rtstring = Printer.printType rt
		      val gtstring = Printer.printGType gT
		  in output(std_out, "r: "^rtstring^"\n : "^gtstring^"\n")
		  end) handle 
		      TypeChecker.InferError s =>
			  output(std_out, "Error: Type inference: "^s^"\n")
		    | Refine.RefineError s => 
			  output(std_out, 
				 "Error: Refinement type inference: "^s^"\n")
		     else ()
	     val _ = (currentExp := f; loopFlag := true)
	 in
	 while (!loopFlag = true) do
	     (output(std_out, "request? ");
	      let val new_input = Parser.cell_kbd()
	      in case new_input of
		  Empty => loopFlag := false
		| Name c => 
		      let val ic = Internal.convertCell c
			  val timeStart = System.Timer.check_timer 
			      (!systemTimer)
			  val v = if !trace 
				      then Eval.evalDebug(1,f,ic,emptyenv)
				  else Eval.eval(f,ic,emptyenv)
			  val timeEnd = System.Timer.check_timer (!systemTimer)
		      in
			  (output(std_out, "--> "^(Printer.printval v)^"\n");
			   if !timer
			       then output(std_out, "time : "^
  (System.Timer.makestring(System.Timer.sub_time(timeEnd,timeStart)))^" sec\n")
			   else ())
		      end
	      end)
	 end) handle Internal.InternalError s => output(std_out, "Error: Internal: "^s^"\n")
                   | Eval.EvalError s => output(std_out, "Error: Eval: "^s^"\n")
		   | Eval.NoAccess => output(std_out, "Error: No access.\n")
		   | CDSParser.ParseError => output(std_out, "Error: Parsing error.\n")
		   | CellParser.ParseError => output(std_out, "Error: Cell parsing error.\n")
		   | Lookup s => output(std_out, "Error: Identifier "^s^" not defined.\n")
		   | BogusForest => output(std_out, "Error: Query does not have proper state part.\n")
		   | CDSLex.LexError => output(std_out, "Error: Lexer: illegal symbol used.\n")

    fun ProcessCOM c = case c of
	(Com_abbreviate(name, exp)) => 
	    (((store(name,Internal.convert exp,exprList);
	      if (!typing) then 
		  (let val (rt, gT) = Refine.mgType exp
		       val rtstring = Printer.printType rt
		       val gtstring = Printer.printGType gT
		       val _ = storeExpType nameExpRTypeList
			       (name,!currentTimeStamp,exp,TYPE rt)
		       val _ = storeExpType nameExpTypeList
			       (name,!currentTimeStamp,exp,gT)
		   in output(std_out, "r: "^rtstring^"\n : "^gtstring^"\n")
		   end) handle 
		       TypeChecker.InferError s =>
			   output(std_out, "Error: Type inference: "^s^"\n")
		     | Refine.RefineError s =>
			   output(std_out, 
				  "Error: Refinement type inference: "^s^"\n")
	      else storeExpType nameExpTypeList 
		               (name,!currentTimeStamp,exp,UNTYPED);
		   storeExpType nameExpRTypeList
		               (name,!currentTimeStamp,exp,UNTYPED));
	    output(std_out, "Abbreviation \""^name^"\" defined.\n"))
	    handle Internal.InternalError s => 
		output(std_out, "Error: Internal: "^s^"\n"))
      | (Com_print name) => Printer.print name
      | (Com_load file) => (Parser.load (file, ProcessInput)
	    handle Io s => output(std_out, "Error: "^s^".\n"))
      | (Com_loadecho file) => (Parser.loadecho (file, ProcessInput)
	    handle Io s => output(std_out, "Error: "^s^".\n"))
      | (Com_traceon) => trace := true
      | (Com_traceoff) => trace := false
      | (Com_timeron) => timer := true
      | (Com_timeroff) => timer := false
      | (Com_typingon) => typing := true
      | (Com_typingoff) => typing := false
      | (Com_show(i,dcds)) => 
	    let val cellList = Type.show(i,dcds)
		val strings = map Printer.unparseCell cellList
	    in Printer.printList strings
	    end
      | (Com_show_more(i,dcds)) => 
	    let val cvaList = Type.showMore(i,dcds)
		val cl = map Printer.unparseCell (map (#1) cvaList)
		val vl = map Printer.unparseValList (map (#2) cvaList)
		val al = map Printer.unparseAccess (map (#3) cvaList)
		fun glue ([],[],[]) = []
		  | glue (c::cl,v::vl,a::al) = 
		      ("\n"^c^" values "^v^
		       (if (a="") then "" else " access "^a))::glue(cl,vl,al)
	    in Printer.printList (glue(cl,vl,al))
	    end
      | (Com_hierarchy s) => let val file = open_out(s)
			     in (output(file,".GS 7 10 fill\n");
				 output(file,"draw nodes as Circle;\n\n");
				 Printer.printHierarchy(file,!hierarchy);
				 output(file,"\n\n.GE\n");
				 close_out(file))
			     end
      | (Com_env) => output(std_out, Printer.printenv(!exprList) ^"\n")
      | (Com_refine nlist) => refineOnlyList := (!refineOnlyList) @ nlist
      | (Com_PCF) => (tempExprList := !exprList;
		      tempTypeList := !typeList;
		      tempHierarchy := !hierarchy;
		      tempNameExpTypeList := !nameExpTypeList;
		      tempNameExpRTypeList := !nameExpRTypeList;
		      tempRefineOnlyList := !refineOnlyList;
		      exprList := !pcfExprList;
		      typeList := !pcfTypeList;
		      hierarchy := !pcfHierarchy;
		      nameExpTypeList := !pcfNameExpTypeList;
		      nameExpRTypeList := !pcfNameExpRTypeList;
		      refineOnlyList := !pcfRefineOnlyList;
		      tempTyping := !typing;
		      typing := true;
		      PcfCode.pcf(ProcessEXP,ProcessCOM) handle PcfCode.ExitPCFLoop => 
			  (exprList := !tempExprList;
			   typeList := !tempTypeList;
			   hierarchy := !tempHierarchy;
			   nameExpTypeList := !tempNameExpTypeList;
			   nameExpRTypeList := !tempNameExpRTypeList;
			   refineOnlyList := !tempRefineOnlyList;
			   typing := !tempTyping))

    and ProcessInput (EMPTY) = ()
      | ProcessInput (COM c) = ProcessCOM c
      | ProcessInput (EXP e) = ProcessEXP e
      | ProcessInput (DCDS d) = ProcessDCDS d

	
    fun TopLevelLoop () = 
	while true do
	    (output(std_out, "# ");
	     (let val new_input = Parser.kbd()
	      in ProcessInput new_input
	      end) handle CDSParser.ParseError => 
		          output(std_out, "Error: Parsing error.\n")
	            | Lookup s => output(std_out, 
				    "Error: Identifier "^s^" not defined\n")
		    | Match.MatchError s => 
			  output(std_out, "Error: MatchError: "^s^"\n")
		    | CDSLex.LexError => 
			output(std_out, "Error: Lexer: illegal symbol used.\n")
		    | Type.TypeError s => 
			  output(std_out, "Error: TypeError: "^s^"\n")
                    | Interface.CommentError s =>
			  output(std_out, "Error: "^s^"\n")
	    )

    datatype typeStyle = regular | refined
	    
    fun cds0 (tStyle) = (exprList := [];
		   typeList := [];
		   hierarchy := [];
		   nameExpTypeList := [];
		   nameExpRTypeList := [];
		   refineOnlyList := [];
		   trace := false;
		   timer := false;

		   (* load the PCF constants *)
		   output(std_out, "-- Loading PCF constants.\n");
		   typing := true;
		   if tStyle = regular then
		       ProcessCOM(Com_load "pcfenv.cds")
		   else ProcessCOM(Com_load "pcfrefenv.cds");
		   pcfExprList := !exprList;
		   pcfTypeList := !typeList;
		   pcfHierarchy := !hierarchy;
		   pcfNameExpTypeList := !nameExpTypeList;
		   pcfNameExpRTypeList := !nameExpRTypeList;
		   pcfRefineOnlyList := !refineOnlyList;

		   exprList := [];
		   typeList := [];
		   hierarchy := [];
		   nameExpTypeList := [];
		   nameExpRTypeList := [];
		   refineOnlyList := [];
		   typing := false;
		   currentTimeStamp := 0;
		   systemTimer := System.Timer.start_timer();
		   output(std_out, "\nCDS0 version 1.1 --- June 11, 1997\n");
		   TopLevelLoop())

    end
    end;
