(* subtype.sml
   Implements decision procedure for subtyping, both for ground dcds's,
   and for type expressions.  Includes a constraint solver.
   Also functions for taking the lub and glb of a set of types. *)


signature SUBTYPE =
    sig
    exception SubtypingFailure
    val subtype : string * string -> bool
    val subtypeV : CDSInternal.typeExp * CDSInternal.typeExp -> 
	CDSInternal.subst
    exception SupInfFailure
    val supremum : CDSInternal.typeExp list -> CDSInternal.typeExp list
    val infimum : CDSInternal.typeExp list -> CDSInternal.typeExp list
    end;
    

functor SubtypeFUN (structure Internal : INTERNAL
		    structure Type : TYPE
		    structure Printer : PRINTER) : SUBTYPE =
    struct
    local open CDSBasic
	  open CDSInternal
	  open CDSEnv
    in
  exception SubtypingFailure

      (* *** SUBTYPING FOR GROUND DCDS *** *)

      (* Is Dcds s1 <= Dcds s2? *)
  fun subtype (s1, s2) = 
      if s1 = s2 then true 
      else let val (s1Parents, s1Children) = Type.typeLookup(s1,!hierarchy)
	   in if s1Parents = nil then false
	      else subtypeRecList s1Parents s2
	   end

      (* Check if any member of s1list is a descendant of s2. *)
  and subtypeRecList [] s2 = false
    | subtypeRecList ((s1,_)::s1list) s2 =
      if (subtype(s1,s2)) then true else subtypeRecList s1list s2


      (* *** CONSTRAINT SATISFACTION *** *)
  
  exception ConstraintFailure

  fun makeArrow t1list [] = []
    | makeArrow [] t2list = []
    | makeArrow (t1::t1list) t2list = 
      let fun makeArrowOne t [] = []
	    | makeArrowOne t (t'::rest) = (Arrow(t,t'))::(makeArrowOne t rest)
      in duplicates ((makeArrowOne t1 t2list) @ (makeArrow t1list t2list))
      end

      (* Given a list of lists of types, it returns a list of product types. *)
      (* consisting of all possible combinations of the input lists. *)
  fun makeProd [] = []
    | makeProd [t1list] = []
    | makeProd (types as (t1list::rest)) = 
      let fun insert t [] = []
	    | insert t ((Prod tlist)::rest) = 
	      (Prod (t::tlist)) :: (insert t rest)
	  fun combine [] _ = []
	    | combine (t1::t1list) prods =
	      (insert t1 prods) @ combine t1list prods
	  fun makeProd' [] = []
	    | makeProd' [t1list] = map (fn x => Prod [x]) t1list
	    | makeProd' (t1list::rest) = combine t1list (makeProd' rest)
      in duplicates (makeProd' types)
      end

  fun dropOne typeCon [] = raise ConstraintFailure
    | dropOne typeCon [t] = raise ConstraintFailure
    | dropOne typeCon [t1,t2] = [t1, t2]
    | dropOne typeCon tlist = 
      let fun removeOne _ [] = []
	    | removeOne [] (t::rest) = rest :: (removeOne [t] rest)
	    | removeOne done (t::rest) =
	      (done @ rest) :: (removeOne (done @ [t]) rest)
      in map typeCon (removeOne [] tlist)
      end

  fun getParents (Dcds s) = 
      let val (sParents, _) = Type.typeLookup(s,!hierarchy)
      in map Dcds (map #1 sParents)
      end
    | getParents (Alpha i) = raise ConstraintFailure
    | getParents (Arrow(t1,t2)) = 
      let val t1children = getChildren t1
	  val t2parents = getParents t2
      in makeArrow t1children t2parents
      end
    | getParents (Prod tlist) = 
      let val tparents = map getParents tlist
      in makeProd tparents
      end
    | getParents (And tlist) = 
      ((dropOne And tlist) handle
       ConstraintFailure => if null tlist then [] else getParents (hd tlist))
    | getParents (Meet tlist) = 
      ((dropOne Meet tlist) handle
       ConstraintFailure => if null tlist then [] else getParents (hd tlist))

  and getChildren (Dcds s) = 
      let val (_, sChildren) = Type.typeLookup(s,!hierarchy)
      in map Dcds (map #1 sChildren)
      end
    | getChildren (Alpha i) = raise ConstraintFailure
    | getChildren (Arrow(t1,t2)) = 
      let val t1parents = getParents t1
	  val t2children = getChildren t2
      in makeArrow t1parents t2children
      end
    | getChildren (Prod tlist) = 
      let val tchildren = map getChildren tlist
      in makeProd tchildren
      end
    | getChildren (And tlist) = raise ConstraintFailure
    | getChildren (Meet tlist) = raise ConstraintFailure


  fun isThereOne t [] = false 
    | isThereOne t (t'::rest) = let val result = subNoVar t' t 
				in if null result then (isThereOne t rest) else true
				end

  and belowAll t [] = true
    | belowAll t (t'::rest) = let val result = subNoVar t t'
			      in if null result then (belowAll t rest) else false
			      end

  and forEachOne [] _ = true
    | forEachOne (t2::t2list) t1list = if (isThereOne t2 t1list) then (forEachOne t2list t1list)
				       else false

  and subNoVar (Dcds s1) (Dcds s2) = if subtype(s1,s2) then [Dcds s1] else []
    | subNoVar (Meet t1list) (Meet t2list) = if (forEachOne t2list t1list) then [Meet t1list] else []
    | subNoVar (Meet tlist) t2 = if (isThereOne t2 tlist) then [Meet tlist] else []
    | subNoVar t1 (Meet tlist) = if (belowAll t1 tlist) then [t1] else []
    | subNoVar (Arrow(t11,t12)) (Arrow(t21,t22)) = 
      if (not (null (subNoVar t21 t11))) andalso (not (null (subNoVar t12 t22)))
	  then [Arrow(t11,t12)] else []
    | subNoVar (Prod t1list) (Prod t2list) = 
      ((flatten (map (fn (x,y) => subNoVar x y) 
		 (zip t1list t2list))) handle Zip => [])
    | subNoVar _ _ = []

  fun intersectType t1 t2 = let val try = subNoVar t1 t2 
			    in if null try then subNoVar t2 t1 else try
			    end 

  fun separateVars [] var nonVar = (var, nonVar)
    | separateVars (t::tlist) var nonVar = 
      if variable t then separateVars tlist (t::var) nonVar
      else separateVars tlist var (t::nonVar)
      
  fun intersectNonVar [] = []
    | intersectNonVar [t] = [t]
    | intersectNonVar tlist =
      let fun intersectRec [] = []
	    | intersectRec (t::rest) = 
	      (flatten (map (intersectType t) rest)) @ (intersectRec rest)
      in intersectRec tlist
      end

  fun intersect tlist =
      let val (var, nonVar) = separateVars tlist [] []
	  val nonVarInt = intersectNonVar nonVar
      in var @ nonVarInt
      end

      (* Given a list of types, first it intersects them.  If that  *)
      (* is empty, it calls itself on the parents on the types.      *)
      (* If the lub involves other type variables, we generate a new *)
      (* constraint.  We also return the final constraint for a, to  *)
      (* be used when backtracking. *)
  fun lub a [] = raise ConstraintFailure
    | lub a tlist = 
      let val result = intersect tlist
      in if null result 
	     then lub a (duplicates (flatten (map getParents tlist)))
	 else if variableInList result
		  then (result, Lub result, [(a,Lub result)])
	      else (result, Lub result, [])
      end

      (* Find a such that a >= glb tlist.  Similar to lub. *)
  fun glb a [] = raise ConstraintFailure
    | glb a tlist = 
      let val result = intersect tlist
      in if null result 
	     then glb a (duplicates (flatten (map getChildren tlist)))
	 else if variableInList result
		  then (result, Glb result, [(a,Glb result)])
	      else (result, Glb result, [])
      end

      (* We have the requirement that above <= 'a <= below for some 'a *)
      (* Make sure that all elements of above are <= those of below. *)
      (* There are no variables in the inputs here *)
  fun verifySeparation above below = 
      let (* val _ = let val abovestr = implode (map Printer.printType above)
	              val belowstr = implode (map Printer.printType below) 
		  in output(std_out, "above = "^abovestr^", below = "^belowstr^"\n")
		  end *)
	  val ourSet = intersect (above @ below)
	  (* val _ = output(std_out, "outSet = "^(implode (map Printer.printType ourSet))^"\n") *)
      in (null ourSet, ourSet)
      end

  exception RecursiveConstraints

  fun getSandwichConstr [t1] [t2] = getConstraints(t1,t2)
    | getSandwichConstr t1list t2list = getSandwichList t1list t2list

  and getSandwichList [] _ = []
    | getSandwichList _ [] = []
    | getSandwichList (t1::t1list) t2list =
      let fun doManyRhs _ [] = []
	    | doManyRhs t1 (t2::t2list) = (getConstraints(t1,t2)) @ (doManyRhs t1 t2list)
      in (doManyRhs t1 t2list) @ (getSandwichList t1list t2list)
      end

  and narrow a [] [] = raise ConstraintFailure
    | narrow a below [] = lub a below
    | narrow a [] above = glb a above
    | narrow a below above = (* experimental *)
      let val resultA = intersect above       (* resultA <= a <= resultB *)
	  val resultB = intersect below
	  (* val _ = let val abovestr = implode (map Printer.printType above)
	              val belowstr = implode (map Printer.printType below) 
		  in output(std_out, "narrow:  above = "^abovestr^", below = "^belowstr^"\n")
		  end *)
      in (case (resultA, resultB) of
	      ([],[]) => raise ConstraintFailure
	    | (_,_) => (case (variableInList resultA, variableInList resultB) of
			    (true,true) => if resultA = resultB then (resultB, Lub resultB, [])
					   else (case (resultA, resultB) of
						     ([Alpha i], _) => 
							 ([], Lub [], [(Alpha i, Lub (a::resultB))])
						   | (_, [Alpha j]) => 
							 ([], Lub [], [(Alpha j, Glb (a::resultA))])
						   | _ => let val newConstr = getSandwichConstr resultA resultB
							  in ([], Lub [], 
							      newConstr @ [(a, Lub resultB), (a, Glb resultA)])
							  end)
			  | (false,false) => let val (empty,ourSet) = verifySeparation resultA resultB
					     in if not empty then (ourSet, Lub ourSet, [])  (* NOT GENERAL *)
						else raise ConstraintFailure
					     end
			  | (true,false) => (case resultA of
						 ([Alpha i]) => (resultB, Lub resultB, 
								 [(a, Lub resultB),(Alpha i, Lub resultB)])
					       | _ => let val newConstr = getSandwichConstr resultA resultB
						      in ([], Lub [], 
							  newConstr @ [(a, Lub resultB), (a, Glb resultA)])
						      end)
			  | (false,true) => (case resultB of
						 ([Alpha i]) => (resultA, Glb resultA, 
								 [(a, Glb resultA),(Alpha i, Glb resultA)])
					       | _ => let val newConstr = getSandwichConstr resultA resultB
						      in ([], Lub [], 
							  newConstr @ [(a, Lub resultB), (a, Glb resultA)])
						      end)))
      end

      (* Same as above, but we have all vars now *)
  and narrowVar (a as (Alpha i)) below [] = 
      (case below of
	   [t] => newSub i t
	 | _ => let val (target, newTlist) = identifyTarget below []
		in shootSubst target (a::newTlist)
		end)
    | narrowVar (a as (Alpha i)) [] above =
      (case above of
	   [t] => newSub i t 
	 | _ => let val (target, newTlist) = identifyTarget above []
		in shootSubst target (a::newTlist)
		end)
    | narrowVar (a as (Alpha i)) below above = 
      let val resultA = intersect above       (* resultA <= a <= resultB *)
	  val resultB = intersect below
      in if resultA = resultB then narrowVar a resultB []
	 else (case (resultA, resultB) of
		   ([ta],[tb]) => let val s = subtypeV(ta,tb)
				      val newTb = apply s tb
				  in compose (newSub i newTb) s
				  end
		 | _ => raise ConstraintFailure)
      end

  and findBelow a [] done = ([], done)
    | findBelow a (c::clist) done =
      case c of
	  (a', Lub tlist) => 
	      if a=a' then let val (below, other) = findBelow a clist done
			   in (tlist @ below, other)
			   end
	      else findBelow a clist (c::done)
	| _ => findBelow a clist (c::done)

  and findAbove a [] done = ([], done)
    | findAbove a (c::clist) done =
      case c of
	  (a', Glb tlist) => 
	      if a=a' then let val (above, other) = findAbove a clist done
			   in (tlist @ above, other)
			   end
	      else findAbove a clist (c::done)
	| _ => findAbove a clist (c::done)

      (* Given a list of dependencies of the form (alpha, typeConstraint) *)
      (* divide into those that depend on some Alpha i and others.  For   *)
      (* the dependents, change Alpha i into whatever s specifies. *)
  and cullDependents s (Alpha i) constr = 
      let fun cullRec s (Alpha i) [] depend others = (depend, others)
	    | cullRec s (Alpha i) ((a,tcon)::rest) depend others =
	      case tcon of
		  (Glb tlist) => 
		      if member(Alpha i,tlist)
			  then cullRec s (Alpha i) rest 
			      ((a,Glb (map (apply s) tlist))::depend) others
		      else cullRec s (Alpha i) rest depend ((a,tcon)::others)
		| (Lub tlist) => 
		      if member(Alpha i,tlist)
			  then cullRec s (Alpha i) rest 
			      ((a,Lub (map (apply s) tlist))::depend) others
		      else cullRec s (Alpha i) rest depend ((a,tcon)::others)
      in cullRec s (Alpha i) constr [] []
      end

  and onlyVars tlist = fold (fn (x,y) => x andalso y) (map variable tlist) true

  and separateConstraints [] (workable, hopeless) = (workable, hopeless)
    | separateConstraints ((con as (a,tcon))::rest) (workable, hopeless) =
      case tcon of
	  (Lub tlist) => 
	      if onlyVars tlist
		  then separateConstraints rest (workable, con::hopeless)
	      else separateConstraints rest (con::workable, hopeless)
	| (Glb tlist) => 
	      if onlyVars tlist
		  then separateConstraints rest (workable, con::hopeless)
	      else separateConstraints rest (con::workable, hopeless)

      (* Apply a substitution to the rhs of a constraint *)
  and applyToConstr s (a,Lub tlist) = (a,Lub (map (apply s) tlist))
    | applyToConstr s (a,Glb tlist) = (a,Glb (map (apply s) tlist))

  and occursInList i [] = false
    | occursInList i (t::rest) = if occursIn i t then true else occursInList i rest

      (* Given a list of constraints, choose one such that the variables *)
      (* on its rhs do not appear on the lhs of any other constraint. If *)
      (* this is not possible, raise an exception. *)
  and pickRightNoLeft [] _ = raise RecursiveConstraints
    | pickRightNoLeft ((a,tcon)::constr) done =
      let fun findAllAlphas _ [] notAlpha = ([],notAlpha)
	    | findAllAlphas a ((a',tcon)::rest) notAlpha = 
	      let val (aCons,others) = findAllAlphas a rest notAlpha
	      in if a=a' then (tcon::aCons,others) else (aCons,(a',tcon)::others)
	      end
	  fun stripCon (Lub tlist) = tlist
	    | stripCon (Glb tlist) = tlist
	  val (aCons,others) = findAllAlphas a constr []
	  val relevantTlist = flatten (map stripCon aCons)
	  fun checkTcon _ [] = true
	    | checkTcon tlist ((Alpha j, _)::rest) = 
	      if occursInList j tlist then false else checkTcon tlist rest
      in if checkTcon relevantTlist others then ((a,tcon),done@constr)
	 else pickRightNoLeft others (done @ [(a,tcon)] @ (map (fn l => (a,l)) aCons))
      end

      (* Given a list of types, separate the Alphas from the _one_ *)
      (* different type.  Doesn't check if there is more than one  *)
      (* that is different.  If all are alphas just pick one.  Can be improved. *)
  and identifyTarget [] done = (hd done, tl done)
    | identifyTarget (t::tlist) done =
      (case t of
	   Alpha _ => identifyTarget tlist (done @ [t])
	 | _ => (t, done @ tlist))

  and shootSubst t [] = emptySub
    | shootSubst t ((Alpha i)::rest) = let val s = newSub i t
				       in compose s (shootSubst t rest)
				       end
    | shootSubst _ _ = raise ConstraintFailure

      (* Handles constraints with variables on both sides *)
  and lastDitch [] = emptySub
    | lastDitch constr =
      let val ((a as (Alpha i), tcon), rest) = (pickRightNoLeft constr [])
	  handle RecursiveConstraints => raise ConstraintFailure
	  val (belowAlpha, constr') = findBelow a constr []
	  val (aboveAlpha, constr'') = findAbove a constr' []
	  val S = narrowVar a (duplicates belowAlpha) (duplicates aboveAlpha)
      in compose (lastDitch (map (applyToConstr S) constr'')) S
      end

      (* like unification: returns pairs of type variables and type exps *)
  and matchUp (Dcds s1) (Dcds s2) =
        if s1=s2 then [] else raise ConstraintFailure
    | matchUp (Alpha i1) (Alpha i2) = 
        if i1 = i2 then [] else [(Alpha i1, Alpha i2)]
    | matchUp (Alpha i) t2 = [(Alpha i, t2)]
    | matchUp t1 (Alpha i) =
        matchUp (Alpha i) t1
    | matchUp (Arrow(t1,t2)) (Arrow(u1,u2)) =
        let val match1 = matchUp t1 u1
	    val match2 = matchUp t2 u2
	in 
	    match1 @ match2
	end
    | matchUp (And tlist1) (And tlist2) =
        matchUpList tlist1 tlist2
    | matchUp (Prod tlist1) (Prod tlist2) =
        matchUpList tlist1 tlist2
    | matchUp _ _ = raise ConstraintFailure

  and matchUpList [] [] = []
    | matchUpList (t1::tlist1) (t2::tlist2) =
        let val match1 = matchUp t1 t2
	    val match2 = matchUpList tlist1 tlist2
	in
	    match1 @ match2
	end
    | matchUpList _ _ = raise ConstraintFailure

  and noVars tlist = not (fold (fn (x,y) => x orelse y) (map variable tlist) false)

      (* Pick out constraints of the kind alpha <= (>=) t, where t is variable-free *)
      (* but make sure not to get any constraints involving a on the lhs into easy *)
  and separateEasy _ [] result = result
    | separateEasy a ((a',tcon)::rest) (easy,hard) = 
      if a=a' then separateEasy a rest (easy, (a',tcon)::hard)
      else (case tcon of
		Lub tlist => if noVars tlist then separateEasy a rest ((a',Lub tlist)::easy, hard)
			     else separateEasy a rest (easy, (a',Lub tlist)::hard)
	      | Glb tlist => if noVars tlist then separateEasy a rest ((a',Glb tlist)::easy, hard)
			     else separateEasy a rest (easy, (a',Glb tlist)::hard))


      (* We are down to a list of constraints involving other vars.  *)
      (* Attempt to solve this: if in a list of constraints we have  *)
      (* some non vars, we can equate the vars with the non-vars and *)
      (* see what happens. *)
  and resolveVarConstr [] = emptySub
    | resolveVarConstr constr =
      let val (workable, hopeless) = separateConstraints constr ([],[])
      in if null workable then lastDitch hopeless
	 else doOneVar workable hopeless
      end

  and doOneVar [] hopeless = resolveVarConstr hopeless
    | doOneVar ((Alpha i, tcon)::constr) hopeless =
      let val s = 
	  case tcon of
	      (Lub tlist) => 
		  let val (var, nonVar) = separateVars tlist [] []
		      val newa = case nonVar of [t] => t
		                               | _ => raise ConstraintFailure
		      val matchPairs = flatten(map (matchUp newa) var)
		      val newconstr = map (fn (x,y) => (x, Glb [y])) matchPairs
		      val newconstr' = (Alpha i, Lub [newa]) :: newconstr
		  in resolveRec (newconstr' @ constr @ hopeless) []
		  end
	    | (Glb tlist) => 
		  let val (var, nonVar) = separateVars tlist [] []
		      val newa = case nonVar of [t] => t
			                       | _ => raise ConstraintFailure
		      val matchPairs = flatten(map (matchUp newa) var)
		      val newconstr = map (fn (x,y) => (x, Lub [y])) matchPairs
		      val newconstr' = (Alpha i, Glb [newa]) :: newconstr
		  in resolveRec (newconstr' @ constr @ hopeless) []
		  end
      in s
      end

      (* Second argument keeps list of constraints involving another *)
      (* variable.  When we get an answer for one variable, we must  *)
      (* check the constraints it affects with other vars, and we    *)
      (* might have to backtrack. *)
  and resolveRec [] freshConstr = resolveVarConstr freshConstr
    | resolveRec constr freshConstr = 
      let val _ = if !trace 
		      then let val (s1,s2) = Printer.printConstraint constr 
			                     freshConstr
			   in output(std_out, "resolve: constraints = "^
				  s1^"\nfreshConstr = "^s2^"\n")
			   end
		  else ()
	  val (workable, hopeless) = separateConstraints constr ([],[])
      in if null workable then lastDitch (hopeless @ freshConstr)
	 (* Find constraints on alpha "a" in constr *)
	 else let val (a as (Alpha i),tcon) = hd workable
		  val (belowAlpha, constr') = findBelow a constr []
		  val (belowAlphaFresh, freshConstr') = findBelow a freshConstr []
		  val (aboveAlpha, constr'') = findAbove a constr' []
		  val (aboveAlphaFresh, freshConstr'') = findAbove a freshConstr' []
		  val belowList = belowAlpha @ belowAlphaFresh
		  val aboveList = aboveAlpha @ aboveAlphaFresh
		  val (newaList, tcon, newcons) = narrow a (duplicates belowList) (duplicates aboveList)
	      in if null newcons 
		     (* then we have solved this one for now -- for all  *)
		     (* possible a's in newaList, try to solve remainder *)
		     then keepTrying a newaList tcon constr'' freshConstr''
		 (* else it must depend on other vars -- add it to freshConstr *)
		 else let val (easy, hard) = separateEasy a newcons ([],[])
		      in resolveRec (easy @ constr'') (hard @ freshConstr'')
		      end
	      end
      end
    
      (* Tries to resolve all constraints.  If one fails, we backtrack *)
      (* to the previous type constraint and try something else.       *)
  and keepTrying (Alpha i) [] tcon constr freshConstr = 
      (* we've exhausted all choices for a at this level *)
      (* examine tcon and generate a new batch of choices *)
      let val (newaList, tcon', newconstr) = 
	  (case tcon of
	       (Lub tlist) => 
		  lub (Alpha i) (duplicates (flatten (map getParents tlist)))
	     | (Glb tlist) => 
		  glb (Alpha i) (duplicates (flatten (map getChildren tlist))))
      in if null newconstr
	     (* this should be null now, since it was before *)
	     (* and we are just going up or down a level     *)
	     then keepTrying (Alpha i) newaList tcon constr freshConstr
	 else raise ConstraintFailure
      end
    | keepTrying (Alpha i) (newa::newaList) tcon constr freshConstr =
      (* for all a's in newalist see how far we can get; if we fail *)
      (* we backtrack and try another a from newaList. First let's  *)
      (* see if we can solve any of the fresh constraints generated *)
      let val s = newSub i newa
	  val modConstr = map (applyToConstr s) constr
	  val modFreshConstr = map (applyToConstr s) freshConstr
	  val (dependOnA,others) = cullDependents s (Alpha i) modFreshConstr
      in if null dependOnA
	     then compose s (resolveRec modConstr modFreshConstr)
	 else   (* for each of the guys that we can solve do a keepTrying *)
	     compose s (resolveRec (modConstr @ dependOnA) others)
      end handle ConstraintFailure =>
	  keepTrying (Alpha i) newaList tcon constr freshConstr


      (* Resolve constraints.  Returns a substitution if successful *)
      (* and raises ConstraintFailure otherwise.  Constraints have  *)
      (* the form (alpha, typeConstraint).                          *)
  and resolve [] = emptySub
    | resolve constraints = resolveRec constraints []


      (* *** SUBTYPING FOR TYPE EXPRESSIONS *** *)

      (* Is every member of l2 in l1? *)
  and searchList l1 [] = emptySub
    | searchList l1 (t::l2) = compose (findOne l1 t) (searchList l1 l2)

      (* Is every elem of list a subtype of t? *)
  and searchAll t [] = emptySub
    | searchAll t (t'::l) = compose (subtypeV(t,t')) (searchAll t l)

      (* Is one elem of list a subtype of t? *)
  and findOne [] t = raise SubtypingFailure
    | findOne (t'::l) t = (subtypeV(t',t) handle SubtypingFailure =>
			   findOne l t)

      (* Same as subtype but works on type expressions (variables). *)
      (* Returns a substitution if successful. *)
  and subtypeV (Dcds s1, Dcds s2) = 
      if (subtype(s1,s2)) then emptySub else raise SubtypingFailure
    | subtypeV (Alpha i1, Alpha i2) = 
      if i1 = i2 then emptySub else newSub i1 (Alpha i2)
    | subtypeV (t1, Alpha i2) = newSub i2 t1
    | subtypeV (Alpha i1, t2) = newSub i1 t2
    | subtypeV (Arrow(s1,t1), Arrow(s2,t2)) = 
      (let val cons1 = getConstraints(s2,s1)
	   val cons2 = getConstraints(t1,t2)
       in resolve (cons1 @ cons2)
       end handle ConstraintFailure => raise SubtypingFailure)
    | subtypeV (And t1list, And t2list) = 
      (* Every member of t2list must be a supertype of some elem in t1list *)
      searchList t1list t2list
    | subtypeV (t1, And t2list) = (* t1 must be <= every element of t2 *)
      searchAll t1 t2list
    | subtypeV (And t1list, t2) = 
      (* There must be some t1 in t1list such that t1 <= t2 *)
      findOne t1list t2
    | subtypeV (Meet t1list, Meet t2list) = searchList t1list t2list
    | subtypeV (t1, Meet t2list) = searchAll t1 t2list
    | subtypeV (Meet t1list, t2) = findOne t1list t2
    | subtypeV (Prod t1list, Prod t2list) = 
      ((resolve (getConstraintsList t1list t2list)) 
       handle ConstraintFailure => raise SubtypingFailure)
    | subtypeV (_, _) = raise SubtypingFailure

  and getConstraintsList [] [] = []
	    | getConstraintsList (t1::tlist1) (t2::tlist2) =
	      (getConstraints(t1,t2)) @ (getConstraintsList tlist1 tlist2)
	    | getConstraintsList _ _ = raise SubtypingFailure

      (* What is needed to make t1 <= t2 ? *)
  and getConstraints (Dcds s1, Dcds s2) =
      if (subtype(s1,s2)) then [] else raise SubtypingFailure
    | getConstraints (Alpha i1, Alpha i2) = 
      if i1 = i2 then [] else [(Alpha i1, Lub [Alpha i2])]
    | getConstraints (t1, Alpha i2) = [(Alpha i2, Glb [t1])]
    | getConstraints (Alpha i1, t2) = [(Alpha i1, Lub [t2])]
    | getConstraints (Arrow(s1,t1), Arrow(s2,t2)) = 
      (getConstraints(s2,s1) @ getConstraints(t1,t2))
    | getConstraints (Prod t1list, Prod t2list) = 
      getConstraintsList t1list t2list
    | getConstraints (t1, t2) = let val s = subtypeV(t1, t2) in [] end


  exception SupInfFailure

  fun getParentsOrSelf x = let val plist = getParents x 
			   in if null plist then [x]
			      else plist
			   end

  fun getChildrenOrSelf x = let val plist = getChildren x 
			   in if null plist then [x]
			      else plist
			   end
      
  fun recursiveLub [] = []
    | recursiveLub tlist =
      let val parents = (duplicates (flatten (map getParentsOrSelf tlist)))
	  handle ConstraintFailure => 
	      (output(std_out, "recursiveLub: getParents dies on: "^
		      (implode (map (fn x => (Printer.printType x)^", ") 
				tlist))^"\n");
	       raise SupInfFailure)
      in if (tlist = parents) then raise SupInfFailure
	 else let val result = intersect parents
	      in if null result then recursiveLub parents
		 else result
	      end
      end

  fun recursiveGlb [] = []
    | recursiveGlb tlist =
      let val children = (duplicates (flatten (map getChildrenOrSelf tlist)))
	  handle ConstraintFailure => 
	      (output(std_out, "recursiveGlb: getChildren dies on: "^
		      (implode (map (fn x => (Printer.printType x)^", ") 
				tlist))^"\n");
	       raise SupInfFailure)
      in if (tlist = children) then raise SupInfFailure
	 else let val result = intersect children
	      in if null result then recursiveGlb children
		 else result
	      end
      end

  fun subCheck (t1, t2) = 
      (let val s = subtypeV(t1, t2)
       in true
       end) handle SubtypingFailure => false


      (* NOTE!  might have to do some kind of constraint solving here  *)
      (* as well, for things like 'a -> 'a and true -> false.  I don't *)
      (* think that is necessary at this point. *)
  fun unifyLub (Dcds s1) (Dcds s2) =
        if s1=s2 then (emptySub,[Dcds s1])
	else if subtype(s1,s2) then (emptySub,[Dcds s2])
	     else if subtype(s2,s1) then (emptySub,[Dcds s1])
		  else (emptySub, recursiveLub [Dcds s1, Dcds s2])
    | unifyLub (Alpha i1) (Alpha i2) = 
        if i1 = i2 then (emptySub, [Alpha i1]) 
	else (newSub i1 (Alpha i2), [Alpha i2])
    | unifyLub (Alpha i) t2 = (newSub i t2, [t2])
    | unifyLub t1 (Alpha i) = unifyLub (Alpha i) t1
    | unifyLub (Arrow(t1,t2)) (Arrow(u1,u2)) =
        let val (sub1, domainList) = unifyGlb t1 u1
	    val (t2',u2') = (apply sub1 t2, apply sub1 u2)
	    val (sub2, rangeList) = unifyLub t2' u2'
	in 
	    (compose sub2 sub1, makeArrow domainList rangeList)
	end
    | unifyLub (And tlist1) (And tlist2) =
      if subCheck(And tlist1, And tlist2) then (emptySub, [And tlist2])
      else if subCheck(And tlist2, And tlist1) then (emptySub, [And tlist1])
	   else (emptySub, recursiveLub (tlist1 @ tlist2))
    | unifyLub t1 (And tlist2) = 
      if subCheck(And tlist2, t1) then (emptySub, [t1])
      else if subCheck(t1, And tlist2) then (emptySub, [And tlist2])
	   else (emptySub, recursiveLub(t1::tlist2))
    | unifyLub (And tlist1) t2 = unifyLub t2 (And tlist1)
    | unifyLub (Meet tlist1) (Meet tlist2) =
      if subCheck(Meet tlist1, Meet tlist2) then (emptySub, [Meet tlist2])
      else if subCheck(Meet tlist2, Meet tlist1) 
	       then (emptySub, [Meet tlist1])
	   else (emptySub, recursiveLub (tlist1 @ tlist2))
    | unifyLub t1 (Meet tlist2) = 
      if subCheck(Meet tlist2, t1) then (emptySub, [t1])
      else if subCheck(t1, Meet tlist2) then (emptySub, [Meet tlist2])
	   else (emptySub, recursiveLub(t1::tlist2))
    | unifyLub (Meet tlist1) t2 = unifyLub t2 (Meet tlist1)
    | unifyLub (Prod tlist1) (Prod tlist2) =
      let val (s, l) = unifyLubList tlist1 tlist2
      in (s, makeProd l)
      end
    | unifyLub _ _ = raise SupInfFailure

  and unifyLubList [] [] = (emptySub, [])
    | unifyLubList (t1::tlist1) (t2::tlist2) =
        let val (sub1, tlist) = unifyLub t1 t2
	    val newlist1 = map (apply sub1) tlist1
	    val newlist2 = map (apply sub1) tlist2
	    val (sub2, rest) = unifyLubList newlist1 newlist2
	in
	    (compose sub2 sub1, tlist::rest)
	end
    | unifyLubList _ _ = raise SupInfFailure

      (* we do not try to unify alphas here, because we might have    *)
      (* situations like the conditional: bool * 'a * 'a -> 'a versus *)
      (* something like true * 'a * 'b -> 'a.  Refinement type is the *)
      (* left argument. *)
  and unifyGlb (Dcds s1) (Dcds s2) =
        if s1=s2 then (emptySub,[Dcds s1])
	else if subtype(s1,s2) then (emptySub,[Dcds s1])
	     else if subtype(s2,s1) then (emptySub,[Dcds s2])
		  else (emptySub, recursiveGlb [Dcds s1, Dcds s2])
    | unifyGlb (Alpha i1) (Alpha i2) = (emptySub, [Alpha i1])
    | unifyGlb (Alpha i) t2 = (newSub i t2, [t2])
    | unifyGlb t1 (Alpha i) = unifyGlb (Alpha i) t1
    | unifyGlb (Arrow(t1,t2)) (Arrow(u1,u2)) =
        let val (sub1, domainList) = unifyLub t1 u1
	    val (t2',u2') = (apply sub1 t2, apply sub1 u2)
	    val (sub2, rangeList) = unifyGlb t2' u2'
	in 
	    (compose sub2 sub1, makeArrow domainList rangeList)
	end
    | unifyGlb (And tlist1) (And tlist2) =
      if subCheck(And tlist1, And tlist2) then (emptySub, [And tlist1])
      else if subCheck(And tlist2, And tlist1) then (emptySub, [And tlist2])
	   else (emptySub, recursiveGlb (tlist1 @ tlist2))
    | unifyGlb t1 (And tlist2) = 
      if subCheck(And tlist2, t1) then (emptySub, [And tlist2])
      else (let val s = subtypeV(t1, And tlist2) 
	    in (s, [apply s t1])
	    end handle SubtypingFailure => 
		(emptySub, recursiveGlb (t1::tlist2)))
    | unifyGlb (And tlist1) t2 = unifyGlb t2 (And tlist1)
    | unifyGlb (Meet tlist1) (Meet tlist2) =
      (let val s = subtypeV(Meet tlist1, Meet tlist2)
       in (s, [apply s (Meet tlist1)])
       end handle SubtypingFailure =>
	   let val s1 = subtypeV(Meet tlist2, Meet tlist1)
	   in (s1, [apply s1 (Meet tlist2)])
	   end handle SubtypingFailure =>
	       (emptySub, recursiveGlb (tlist1 @ tlist2)))
    | unifyGlb t1 (Meet tlist2) = 
      (let val s = subtypeV(t1, Meet tlist2) 
       in (s, [apply s t1])
       end handle SubtypingFailure =>
	   let val s1 = subtypeV(Meet tlist2, t1)
	   in (s1, [apply s1 (Meet tlist2)])
	   end handle SubtypingFailure =>
	       (emptySub, recursiveGlb (t1::tlist2)))
    | unifyGlb (Meet tlist1) t2 = unifyGlb t2 (Meet tlist1)
    | unifyGlb (Prod tlist1) (Prod tlist2) =
      let val (s, l) = unifyGlbList tlist1 tlist2
      in (s, makeProd l)
      end
    | unifyGlb _ _ = raise SupInfFailure

  and unifyGlbList [] [] = (emptySub, [])
    | unifyGlbList (t1::tlist1) (t2::tlist2) =
        let val (sub1, tlist) = unifyGlb t1 t2
	    val newlist1 = map (apply sub1) tlist1
	    val newlist2 = map (apply sub1) tlist2
	    val (sub2, rest) = unifyGlbList newlist1 newlist2
	in
	    (compose sub2 sub1, tlist::rest)
	end
    | unifyGlbList _ _ = raise SupInfFailure


      (* find lub of a list of types, also doing unification *)
  fun supremum [] = []
    | supremum [t] = [t]
    | supremum (t1::t2::ts) = 
      let val (s, tlist) = unifyLub t1 t2
      in tryAllSups tlist (map (apply s) ts)
      end

  and tryAllSups [] rest = raise SupInfFailure
    | tryAllSups (t::others) rest =
      (supremum (t::rest)) handle SupInfFailure => tryAllSups others rest

  fun infimum [] = []
    | infimum [t] = [t]
    | infimum (t1::t2::ts) = 
      let val (s, tlist) = unifyGlb t1 t2
      in tryAllInfs tlist (map (apply s) ts)
      end

  and tryAllInfs [] rest = raise SupInfFailure
    | tryAllInfs (t::others) rest =
      (infimum (t::rest)) handle SupInfFailure => tryAllInfs others rest

    end
    end;
