(* refine.sml
   Implements the refinement (i.e., abstract interpretation) of a state. *)


signature REFINE =
    sig
    exception RefineError of string
    val mgType : CDSBasic.expr -> (CDSInternal.typeExp * CDSInternal.TYPE)
    end;
    

functor RefineFUN (structure Printer : PRINTER
		   structure Internal : INTERNAL
		   structure Type : TYPE
		   structure Subtype : SUBTYPE
		   structure TypeChecker : TYPE_CHECKER
		   structure QandA : QANDA) : REFINE =
    struct
    local open CDSBasic
	       CDSInternal
	       CDSEnv
    in
    exception RefineError of string


    (* ******************************************************************** *)
    (* Construct a dependency list out of internal algorithm structure      *)
    (* ******************************************************************** *)

        (* Get rid of one output from internal representation of *)
        (* the value in tree_Result. *)
    fun doOutput (0, v) = v
      | doOutput (_, v) = (case v of
			       (Ival_output iv) => iv
			     | _ => raise RefineError "doOutput: no output")

        (* Depth First Search of the instruction tree.  Output is    *)
        (* a list of pairs of the form:                              *)
	(*  (output cell, value), [(input cell, which input, value)] *)
        (* The values are actually lists;  they are either empty     *)
        (* (valof but no values listed), or contain 1 element.       *)
    fun dfs (n, oc, tree_Result(i,iv), deps) = 
	[((oc,[doOutput(n,iv)]), rev deps)]
      | dfs (n, oc, tree_Valof(ic, i, tqlist), deps) =
	if null tqlist 
	    then [((oc,[]), rev ((ic,i,[])::deps))]
	else dfsTqlist(n, oc, ic, i, tqlist, deps)
      | dfs (n, oc, tree_From(ic, i, tqlist), deps) =
	if null tqlist 
	    then [((oc,[]), rev ((ic,i,[])::deps))]
	else dfsTqlist(n, oc, ic, i, tqlist, deps)

	
    and dfsTqlist (n, oc, ic, i, [], deps) = []
      | dfsTqlist (n, oc, ic, i, (iv,ins)::rest, deps) =
	let val branch = dfs(n, oc, ins, (ic,i,[iv])::deps)
	in branch @ (dfsTqlist(n, oc, ic, i, rest, deps))
	end


        (* Generate dependency information for a tree from a forest *)
        (* This is going to be a pair containing:                   *)
        (* (output cell, value), list of input cells and their      *)
        (*                       values encountered along the path  *)
	(*                       to that particular output value    *)
    fun depInfo n (oc, ins) = dfs (n, oc, ins, [])



    (* ******************************************************************** *)
    (* Transform a deplist from internal to parse tree form                 *)
    (* ******************************************************************** *)

    fun convCell (Icell_name s) = Cell_name s
      | convCell (Icell_var s) = Cell_var s
      | convCell (Icell_fun(f,ic)) = Cell_fun(convForest f, convCell ic)
      | convCell (Icell_graft(ic,t)) = Cell_graft(convCell ic, t)
      | convCell (Icell_with(ic,ib)) = Cell_with(convCell ic, convBool ib)

    and convVal (Ival_string s) = Val_string s
      | convVal (Ival_output iv) = Val_output (convVal iv)
      | convVal (Ival_valof ic) = Val_valof (convCell ic)
      | convVal (Ival_arexpr a) = Val_arexpr a
      | convVal (Ival_omega) = Val_omega
      | convVal (Ival_with(s,ib)) = Val_with(s, convBool ib)
      | convVal (Ival_pair(iv1,iv2)) = Val_pair(convVal iv1, convVal iv2)

    and convBool (Iboolexp_gt(a1,a2)) = Boolexp_gt(a1,a2)
      | convBool (Iboolexp_gteq(a1,a2)) = Boolexp_gteq(a1,a2)
      | convBool (Iboolexp_lt(a1,a2)) = Boolexp_lt(a1,a2)
      | convBool (Iboolexp_lteq(a1,a2)) = Boolexp_lteq(a1,a2)
      | convBool (Iboolexp_or(b1,b2)) = Boolexp_or(convBool b1, convBool b2)
      | convBool (Iboolexp_and(b1,b2)) = Boolexp_and(convBool b1, convBool b2)
      | convBool (Iboolexp_eq(iv1,iv2)) = Boolexp_eq(convVal iv1, convVal iv2)
      | convBool (Iboolexp_noteq(iv1,iv2)) = 
	  Boolexp_noteq(convVal iv1, convVal iv2)

    and convForest (forest_basic(i, tlist)) = Expr_state(convTlist i tlist)
      | convForest _ = raise RefineError "convForest: non-basic forest"

    and convTlist _ [] = []
      | convTlist i ((oc,ins)::rest) = 
	let val rootCell = convCell oc
	in (flattenIns(rootCell,ins)) @ (convTlist i rest)
	end

    and flattenIns (root, tree_Result(i,iv)) = 
	  [(root, convVal iv)]       (* does not work at higher-order *)
      | flattenIns _ = raise RefineError "flattenIns: non tree_Result"
	
        (* Convert a list of events from internal form (icell, ivalue) *)
        (* to parse tree form (cell, value). *)
    fun toParseTree [] = []
      | toParseTree (((oc,ovList),deps)::rest) =
	let fun mapFn (ic, i, ivList) = (convCell ic, i, map convVal ivList)
	in ((convCell oc, map convVal ovList), 
	    map mapFn deps)::(toParseTree rest)
	end

    fun isOmega (Val_string s) = false
      | isOmega (Val_output v) = isOmega v
      | isOmega (Val_valof c) = false
      | isOmega (Val_arexpr a) = false
      | isOmega (Val_omega) = true
      | isOmega (Val_with(s,ib)) = false
      | isOmega (Val_pair(iv1,iv2)) = false

	(* We want to get rid of OMEGA values.  Instead, there should *)
        (* be nothing. *)
    fun removeOmegas ((oc,ovList), deps) =
	let fun removeOne [v] = if (isOmega v) then [] else [v]
	      | removeOne vlist = vlist
	    val mapfn = fn (c,i,vlist) => (c,i,removeOne vlist)
	in ((oc, removeOne ovList), map mapfn deps)
	end


    (* ******************************************************************** *)
    (* Given a deplist, determine its type                                  *)
    (* ******************************************************************** *)

    fun makeI 0 result = result
      | makeI i result = makeI (i-1) ([] :: result)

    fun putInIth 1 (c,v) (x::xs) = (x @ [(c,v)]) :: xs
      | putInIth i (c,v) (x::xs) = x :: (putInIth (i-1) (c,v) xs)

    fun collate [] result = result
      | collate ((c,i,v)::rest) result = collate rest (putInIth i (c,v) result)

    fun makeOut (0,v) = v
      | makeOut (i,v) = Val_output(makeOut(i-1,v))

	(* Takes a cvList generated from a dependence list (hence *)
	(* guaranteed to have only one value per cell) and makes  *)
        (* a state out of it;  careful to preserve order *)
    fun cvToState [] = []
      | cvToState ((c,vlist)::rest) = 
	(case vlist of
	     ([v]) => (c,v)::(cvToState rest)
	   | ([]) => (c,makeOut((cellOrder c) - 1,Val_omega))::(cvToState rest)
	   | _ => raise RefineError "cvToState: more than one value per cell")


    (* ******************************************************************** *)
    (* Sift through deplists, determining cell and value dependencies       *)
    (* ******************************************************************** *)

    fun twoMeets (Arrow(t1,t2)) =
	let fun findMeet (Meet _) = true
	      | findMeet (Prod tlist) = fold (fn (x,y) => x orelse y)
		                        (map findMeet tlist) false
	      | findMeet (Arrow(t1,t2)) = (findMeet t1) orelse (findMeet t2)
	      | findMeet _ = false
	in (findMeet t1) andalso (findMeet t2)
	end
      | twoMeets _ = false

    fun meetRemove (Meet tlist) = tlist
      | meetRemove t = [t]

    (* Given a list of types put /\ around it if it contains >1 element *)
    fun meetWrap t = 
	let val t' = duplicates t
	in if (length t') > 1 then Meet t'
	   else if t' = nil then raise RefineError "term does not have a type"
		else hd t'
	end

    fun makeArrowMeet [] _ = []
      | makeArrowMeet (t1::t1list) t2list =
	let fun makeOneList t1 [] = []
	      | makeOneList t1 (t2::t2list) = 
		(Arrow(t1,t2))::(makeOneList t1 t2list)
	in (makeOneList t1 t2list)@(makeArrowMeet t1list t2list)
	end

    fun makeArrowCurryMeet [t1list, t2list] = makeArrowMeet t1list t2list
      | makeArrowCurryMeet (tlist::rest) = makeArrowMeet tlist (makeArrowCurryMeet rest)

    fun makeOneProd [] tlist = []
      | makeOneProd (t1::t1list) [] = [t1] :: (makeOneProd t1list [])
      | makeOneProd (t1::t1list) tlist = 
	(map (fn x => t1::x) tlist) @ (makeOneProd t1list tlist)

    fun makeProdMeet [] = []
      | makeProdMeet (t1list::rest) = 
	let val restlist = makeProdMeet rest
	in makeOneProd t1list restlist
	end

    fun distributeMeets (Arrow(t1,t2)) =
	let val t1m = distributeMeets t1
	    val t2m = distributeMeets t2
	in meetWrap(makeArrowMeet (meetRemove t1m) (meetRemove t2m))
	end
      | distributeMeets (Prod tlist) = 
	let val tmlist = map distributeMeets tlist
	    val tmlistlist = map meetRemove tmlist
	in meetWrap(map Prod (makeProdMeet tmlistlist))
	end
      | distributeMeets (Meet tlist) =
	let val tmlist = map distributeMeets tlist
	    val tmlistlist = map meetRemove tmlist
	in meetWrap (flatten tmlistlist)
	end
      | distributeMeets t = t

    fun distTailMeet (Arrow(t1, t2)) = 
	(case t2 of 
	     (Meet tlist) => makeArrowMeet [t1] tlist
	   | _ => [Arrow(t1,t2)])
      | distTailMeet t = [t]

    fun distHeadMeet (Arrow(t1, t2)) = 
	(case t1 of
	     (Meet tlist) => makeArrowMeet tlist [t2]
	   | _ => [Arrow(t1,t2)])
      | distHeadMeet t = [t]

    fun distMeet (Arrow(t1, t2)) =
	(case t1 of
	     (Meet tlist1) => (case t2 of
				  (Meet tlist2) => makeArrowMeet tlist1 tlist2
				| _ => makeArrowMeet tlist1 [t2])
	   | _ => (case t2 of
		       (Meet tlist2) => makeArrowMeet [t1] tlist2
		     | _ => [Arrow(t1,t2)]))
      | distMeet t = [t]

    fun distGroundMeet (Arrow(t1,t2)) = 
	let val t1list = distGroundMeet t1
	    val t2list = distGroundMeet t2
	in makeArrowMeet t1list t2list
	end
      | distGroundMeet (Meet tlist) = 
	(case (hd tlist) of
	     (Dcds _) => tlist
	   | _ => [Meet tlist])
      | distGroundMeet (Prod tlist) =
	let val tlistlist = map distGroundMeet tlist
	in map Prod (makeProdMeet tlistlist)
	end 
      | distGroundMeet t = [t]

	(* Like distGroundMeet, but leaves any Meet in the output alone *)
    fun distGroundMeetNoTail (Arrow(t1,t2)) =
	let val (_,out,inputs) = stripArrows (Arrow(t1,t2)) 0
	    val inputsList = map distGroundMeet inputs
	in makeArrowCurryMeet (inputsList @ [[out]])
	end
      | distGroundMeetNoTail t = distGroundMeet t
	   

	(* Given a list of (outType name, inType) where inType can be a Meet *)
	(* we construct a type out of it *)
    fun constructType [] = []
      | constructType ((oname, it)::rest) = (Arrow(it, Dcds oname)) :: (constructType rest)


        (* For debugging. *)
    fun printCVlist [] = []
      | printCVlist ((c,vlist)::CV) = ("("^(Printer.unparseCell c)^
				       ", ["^(Printer.unparseValList vlist)^"])")::(printCVlist CV)

    fun printCVTlist [] = "}\n"
      | printCVTlist ((CVlist, t)::CVT) =
	let val s = "{["^(implode(printCVlist CVlist))^"] : "
	    ^(Printer.printType t)^", "^(printCVTlist CVT)
	in s
	end

    fun printCVTPlist [] = "}\n"
      | printCVTPlist ((CVlist, t, prodIndex)::CVT) =
	let val s = "{["^(implode(printCVlist CVlist))^"] : "
	    ^(Printer.printType t)^"."^(implode (map (makestring : int -> string) prodIndex))
	    ^", "^(printCVTPlist CVT)
	in s
	end

    fun printCVInst [] = ""
      | printCVInst ((t,cvlist)::rest) = (Printer.printType t)^
	">> "^(implode(printCVlist cvlist))^", "^(printCVInst rest)

    fun breakUpProd (Prod tlist, cvList) = 
	let val dots = TypeChecker.separate 1 cvList []
	    val cvtLists = zip tlist dots
	in flatten (map breakUpProd cvtLists)
	end
      | breakUpProd (t, cvList) = [(cvList, t)]

    fun genIndexList n =
	let fun genIndexList' i n = if i = n then [n] else i :: (genIndexList' (i+1) n)
	in genIndexList' 1 n
	end 

    fun zip3 [] [] [] = []
      | zip3 (x::xs) (y::ys) (z::zs) = (x,y,z)::(zip3 xs ys zs)
      | zip3 _ _ _ = raise Zip

    fun breakUpInProd (Prod tlist, cvList, prodIndex) =
	let val dots = TypeChecker.separate 1 cvList []
	    val newIndex = genIndexList (length tlist)
	    val newProdIndex = map (fn x => prodIndex @ [x]) newIndex
	    val cvtLists = zip3 tlist dots newProdIndex
	in flatten (map breakUpInProd cvtLists)
	end
      | breakUpInProd (t, cvList, prodIndex) = [(cvList, t, prodIndex)]

    fun higherOrder (Arrow _) = true
      | higherOrder (Meet tlist) = if higherOrder (hd tlist) then true else false
      | higherOrder _ = false

    fun getNames (Dcds s) = [s]
      | getNames (Meet tlist) = flatten (map getNames tlist)
      | getNames (Alpha _) = []
      | getNames _ = raise RefineError "getNames: not ground"

    fun id (c : cell) = c

    datatype varMatch = CellMatch of (string * cell) 
                      | ValMatch of (string * value) 
                      | TagMatch of string * (cell -> cell)

    exception FindMatch

    fun instArexpr (a1, a2, env) =
	if a1 = a2 then (true, env)
	else (false, env)  (* Should more be checked here? *)

	(* Only left input can have variables in it.  Collect all variables and *)
	(* their respective values in case of a match. *)
    fun instCell (c1, c2, env) = 
	if c1 = c2 then (true, env)
	else case c1 of 
	    (Cell_var s1) => (true, (CellMatch(s1, c2))::env)
	  | (Cell_graft (c1',t1)) => 
		(case c2 of 
		     (Cell_name s2) => 
			 (case instCell(c1',c2,env) of
			      (true, newEnv) => (case t1 of 
						     (Tag_arexpr(Arexpr_var v)) => (true, (TagMatch(v,id)) :: newEnv)
						   | _ => (false, env))
			    | _ => (false, env))
		   | (Cell_graft(c2',t2)) => 
			 (case t1 of 
			      (Tag_arexpr(Arexpr_var v)) => instManyTags(Cell_graft(c2',t2),c1',v,id,env)
			    | _ => (case (instTag(t1,t2,env)) of
					(true, newEnv) => instCell(c1',c2',newEnv)
				      | (false, _) => (false, env))))
	  | (Cell_with (c1',b)) => if Type.includedCell(c1,c2) then instCell(c1',c2,env)
				   else (false, env)
	  | _ => (false, env)
				       
    and instTag (t1, t2, env) =
	if t1 = t2 then (true, env)
	else case t1 of 
	    (Tag_str s1) => 
		(case t2 of 
		     (Tag_arexpr(Arexpr_int i2)) => if s1 = makestring i2 then (true,env) else (false,env)
		   | _ => (false, env))
	  | (Tag_arexpr a1) => 
		(case t2 of 
		     (Tag_arexpr a2) => instArexpr(a1,a2,env)
		   | (Tag_str s2) => (case a1 of
					  (Arexpr_int i1) => if s2 = makestring i1 then (true,env) else (false,env)
					| _ => (false, env)))

	(* We have (c1.$v) matched against (c2.t2).  We want to strip off *)
	(* as many tags from c2 to make a match.  Example: C.1.$T should  *)
	(* match C.1.2.2.3.4 and $T should be a function that tags 2.2.3.4 *)
    and instManyTags (Cell_graft(c2,t2),c1,v,f,env) =
	let val g = fn x => Cell_graft(x,t2)
	in case instCell(c1,c2,env) of
	    (true, newEnv) => (true, (TagMatch(v,g o f)) :: newEnv)
	  | (false, _) => case c2 of
		(Cell_graft(c22,t22)) => instManyTags(c2,c1,v,g o f,env)
	      | _ => (false, env)
	end
      | instManyTags (_,_,_,_,env) = (false, env)

    fun instVal (v1, v2, env) =
	if v1 = v2 then (true, env)
	else case v1 of 
	    (Val_arexpr(Arexpr_var s)) => (true, (ValMatch(s,v2)) :: env)
	  | (Val_arexpr _) => (Type.includedVal(v1,v2), env)
	  | (Val_with(s,b)) => if Type.includedVal(v1,v2) 
				   then (true, (ValMatch(s,v2)) :: env)
			       else (false, env)
	  | (Val_pair(v11,v12)) => 
		(case v2 of
		     (Val_pair(v21,v22)) => (case instVal(v11,v21,env) of
						 (true, newEnv) => instVal(v12,v22,newEnv)
					       | (false,_) => (false,env))
		   | _ => (false,env))
	  | (Val_string _) => (Type.includedVal(v1,v2), env)   (* NEW !! *)

    fun instUnit (c,v,[]) = []
      | instUnit (c,v,(c1,vlist,_)::rest) = 
	let val (success, cellMatchList) = instCell(c,c1,[])
	in if success 
	       then let val valMatchAttempts = map (fn x => instVal(v,x,[])) vlist
			val yes = fold (fn (x,y)=>x orelse y) (map #1 valMatchAttempts) false
			val valMatchList = flatten (map #2 valMatchAttempts)
		    in if yes then (cellMatchList @ valMatchList) :: instUnit(c,v,rest)
		       else instUnit(c,v,rest)
		    end
	   else instUnit(c,v,rest)
	end

    fun joinMatches [] [] = []
      | joinMatches (vmlist::oneMatch) (oneList::otherMatches) =
	if null oneList then [vmlist] @ (joinMatches oneMatch otherMatches)
	else (map (fn l => vmlist @ l) oneList) @ (joinMatches oneMatch otherMatches)
      | joinMatches _ _ = raise RefineError "joinMatches:  unequal lenghts"

        (* Given an cvlist and a list of pairs of a dcds name and dcds internal *)
        (* representation (i.e., cva), match each event from the cvlist against *)
        (* the cva (thus obtaining a varMatch list list), apply the match to the *)
        (* rest of the cvlist, and continue getting matches, appending the new ones *)
        (* to the instantiated ones. *)
        (* Returns a list of lists of (dcds name, varMatch list list) pairs *)
    fun instStateVars cvlist (names,cvas) =
	let val scvas = zip names cvas
	    fun applyMatchesToInCV _ [] = []
	      | applyMatchesToInCV inCV (vmlist::rest) = (plugOneCV inCV vmlist) :: (applyMatchesToInCV inCV rest)
	        (* val instInVars : (cell * value list) list
		                     -> 'a * (cell * value list * 'b) list -> varMatch list list  *)
	    fun instInVars _ [] = []
	      | instInVars (s,cva) ((c,[v])::inCV) = 
		let val oneMatch = instUnit(c,v,cva)
		    (* val _ = output(std_out, "oneMatch = "^(glueStrings (map printMatchList oneMatch))^"\n")  *)
		in if null oneMatch
		       then instInVars (s,cva) inCV
		    (* for each varMatch list in oneMatch, apply it to inCV and do the rest *)
		    (* of the instantiations, concatenating new matches to that varMatch list *)
		   else let val newInCVs = applyMatchesToInCV inCV oneMatch
			    val otherMatches = map (instInVars (s,cva)) newInCVs
			in joinMatches oneMatch otherMatches
			end
		end
	    (* while instatiating values we could have created a cvlist with more than one value per cell *)
	      | instInVars (s,cva) ((c,vlist)::inCV) = 
		let fun tryAllVals _ _ (c,[]) = []
		      | tryAllVals (s,cva) inCV (c,(v::vlist)) =
			(instInVars (s,cva) ((c,[v])::inCV)) @ (tryAllVals (s,cva) inCV (c,vlist))
		in tryAllVals (s,cva) inCV (c,vlist)
		end
	    fun instInCvaVars cvlist [] = []
	      | instInCvaVars cvlist ((s,cva)::rest) = 
		(instInVars (s,cva) cvlist) :: (instInCvaVars cvlist rest)
	    val thisBatch = instInCvaVars cvlist scvas
	in zip names thisBatch
	end

    and instInputVars [] [] = []
      | instInputVars (cvlist::cvlists) ((names,cvas)::types) = 
	(instStateVars cvlist (names,cvas)) :: (instInputVars cvlists types)

    and printMatchList [] = ""
      | printMatchList ((CellMatch(s,c))::rest) = s^"->"^(Printer.unparseCell c)^", "^
	(printMatchList rest)
      | printMatchList ((ValMatch(s,v))::rest) = s^"->"^(Printer.unparseVal v)^", "^
	(printMatchList rest)
      | printMatchList ((TagMatch(s,f))::rest) = "(c."^s^")->"^(Printer.unparseCell (f (Cell_name "c")))^", "^
	(printMatchList rest)

    and printProdInst [] = ""
      | printProdInst ((t,mlist)::rest) = let val mstr = glueStrings (map printMatchList mlist)
					  in (Printer.printType t)^" == ["^mstr^"], "^
					      (printProdInst rest)
					  end
    and printInst [] = ""
      | printInst ((s,mlist)::rest) = let val mstr = glueStrings (map printMatchList mlist)
				      in s^" == ["^mstr^"], "^(printInst rest)
				      end
   
    (* Assuming each variable has a different name. *)
    and findMatch s [] = raise FindMatch
      | findMatch s ((CellMatch(s1, c))::rest) = if s = s1 then CellMatch(s1, c) 
						 else findMatch s rest
      | findMatch s ((ValMatch(s1, v))::rest) = if s = s1 then ValMatch(s1, v)
						 else findMatch s rest
      | findMatch s ((TagMatch(s1, f))::rest) = if s = s1 then TagMatch(s1, f)
						 else findMatch s rest

    and restrictToVal [] = []
      | restrictToVal ((CellMatch(_, _))::rest) = restrictToVal rest
      | restrictToVal ((TagMatch(_, _))::rest) = restrictToVal rest
      | restrictToVal ((ValMatch(s1, v))::rest) = [ValMatch(s1, v)] :: (restrictToVal rest)

	(* Took it apart, now have to put it back together.  Given varMatches *)
	(* in varList, expand cell name.  *)
    and plugCell (Cell_name s) _= Cell_name s
      | plugCell (Cell_var s) varList = 
	(let val (CellMatch(s1,c1)) = findMatch s varList
	 in c1
	 end handle FindMatch => raise RefineError "plugCell: var not defined")
      | plugCell (Cell_graft(c,t)) varList =
	(let val c' = plugCell c varList
	 in case t of 
	     (Tag_arexpr(Arexpr_var s)) => 
		 let val (TagMatch(s1,f)) = findMatch s varList
		 in f c'
		 end
	   | _ => Cell_graft(c',t)
	 end handle FindMatch => raise RefineError "plugCell: var not defined")
      | plugCell (Cell_with(c,b)) varList = plugCell c varList
      | plugCell c _ = c

    and plugVal (Val_string s) _ = Val_string s
      | plugVal (Val_pair(v1,v2)) varList = Val_pair(plugVal v1 varList, plugVal v2 varList)
      | plugVal (Val_arexpr a) varList =
	(case a of 
	     (Arexpr_var s) => (let val (ValMatch(s1,v)) = findMatch s varList
				in v
				end handle FindMatch => Val_arexpr a)
	   | _ => Val_arexpr a)
      | plugVal (Val_with(s,b)) varList = 
	(let val (ValMatch(s1,v)) = findMatch s varList
	 in v
	 end handle FindMatch => Val_with(s,b))
      | plugVal v _ = v

    and plugValWrapper v varList = [plugVal v varList]

	(* List of variables is a varMatch list *)
    and plugOneCV [] _ = []
      | plugOneCV ((c,[v])::rest) varList =
	let val c' = plugCell c varList
	    val valVarList = restrictToVal varList
	    val vlist = if null valVarList then [v] else duplicates (flatten (map (plugValWrapper v) valVarList))
	in (c', vlist) :: (plugOneCV rest varList)
	end

	(* Given a cvlist and the info generated by instVars, plug in those vars *)
	(* in the cvList.  Return a list of pairs of cvlists and the output type *)
	(* the info came from. *)
    fun plugCV inCV [] = []
      | plugCV inCV ((d,varList)::rest) =
	let val newCV = flatten (map (plugOneCV inCV) varList)
	in (d,newCV)::(plugCV inCV rest)
	end

        (* Given a list of (product tag, type, matches), reconstructs product *)
    fun rebuildProd ttlist = 
	let fun separable ([] : (int list * typeExp * varMatch list list) list list) = 
	                                 raise RefineError "rebuildProd: empty ttlist"
	      | separable (l::ls) = if null (#1 (hd l)) then false else true
	    fun separateProd ttlist =
		let fun findRemove n [] (nTags, others) = (nTags, others)
		      | findRemove n (matchList::rest) (nTags, others) =
			let val firstTag = #1 (hd matchList)
			in if n = (hd firstTag)
			       then let val moreNTags = map (fn (x,y,z) => (tl x, y, z)) matchList
				    in findRemove n rest (moreNTags::nTags, others)
				    end 
			   else findRemove n rest (nTags, matchList::others)
			end 
		    fun keepFinding n ttlist result =
			let val (nTags, others) = findRemove n ttlist ([],[])
			in if null nTags then rev result
			   else keepFinding (n+1) others (nTags::result)
			end 
		in keepFinding 1 ttlist []
		end 
	    fun addOne [] _ = []
	      | addOne ((t',mlist')::rest) (t,mlist) =
		case t' of
		    (Prod tlist) => (Prod (tlist@[t]), mlist' @ mlist) :: (addOne rest (t,mlist))
		  | _ => raise RefineError "rebuildProd: not product type"
	        (* Given a list of lists of lists of (type, varMatch) where order *)
	        (* denoted place in product, make all possible combinations of prods *)
	    fun makeAllCombos [] _ = raise RefineError "rebuildProd: empty input to makeAllCombos"
	      | makeAllCombos [l] soFar = flatten (map (addOne soFar) l)
	      | makeAllCombos (typeMatch::rest) soFar = 
		makeAllCombos rest (flatten (map (addOne soFar) typeMatch))
	in if separable ttlist 
	       then let val separatedList = separateProd ttlist
			val typeMatchList = map rebuildProd separatedList
		    in makeAllCombos typeMatchList [(Prod [], [])]
		    end
	   else (flatten (map (map (fn (x,y,z) => (y,z))) ttlist))
	end


        (* Given a (cvlist, type, product tag) list and a corresponding *)
        (* (dcds name, varMatch list list) list list, it assembles them *)
        (* back into a product type and a varMatch list.  Wherever there *)
        (* was a meet in the product branch initially, we now have several *)
        (* (dcds name, varMatch list list) pairs.  We want to create a *)
        (* product type with each of them in turn. *)
    fun createProdTypes CVTPlist matchList =
	    (* makeTagType : ('a * typeExp * 'b) list
	                       -> (string * 'c list) list list -> ('b * typeExp * 'c list) list list  *)
	let fun makeTagType [] _ = []
	      | makeTagType _ [] = []
	      | makeTagType ((cvList, t, prodIndex)::CVTPlist) (matches::rest) =
		case t of
		    (Alpha _) => [(prodIndex, t, [])] :: (makeTagType CVTPlist rest)
		  | _ => let fun f (s, mlist) = (prodIndex, Dcds s, mlist)
			 in (map f matches) :: (makeTagType CVTPlist rest)
			 end
	    val ttlist = makeTagType CVTPlist matchList
	in rebuildProd ttlist
	end


    exception RefUnificationFailure

        (* see if a refinement type refines a regular type *)
    fun unifyRef (Dcds s1) (Dcds s2) =
        if Subtype.subtype(s1,s2) then [] else raise RefUnificationFailure
    | unifyRef (Alpha i1) (Alpha i2) = []
    | unifyRef (Alpha i) t2 = [(i, t2)]
    | unifyRef (Arrow(t1,t2)) (Arrow(u1,u2)) =
        let val sub1 = unifyRef t1 u1
	    val sub2 = unifyRef t2 u2
	in sub1 @ sub2
	end
    | unifyRef (And tlist1) (And tlist2) =
        unifyRefList tlist1 tlist2
    | unifyRef (Prod tlist1) (Prod tlist2) =
        unifyRefList tlist1 tlist2
    | unifyRef (Meet tlist1) (Meet tlist2) = 
      (* each member of tlist1 must be a refinement of some member of tlist2 *)
      unifyRefSome tlist1 tlist2
    | unifyRef (Meet tlist1) t = (* each member of tlist1 must be a refinement of t *)
      unifyRefEach tlist1 t
    | unifyRef t (Meet tlist2) = unifyRefEach tlist2 t
    | unifyRef _ _ = raise RefUnificationFailure

  and unifyRefList [] [] = []
    | unifyRefList (t1::tlist1) (t2::tlist2) =
        let val sub1 = unifyRef t1 t2
	    val sub2 = unifyRefList tlist1 tlist2
	in sub1 @ sub2
	end
    | unifyRefList _ _ = raise RefUnificationFailure

    and unifyRefEach [] _ = []
      | unifyRefEach (t1::tlist1) t = (unifyRef t1 t) @ (unifyRefEach tlist1 t)

    and unifyRefSome [] _ = []
      | unifyRefSome (t1::tlist1) tlist2 =
	let fun unifyOne t [] = raise RefUnificationFailure
	      | unifyOne t (t2::tlist2) = 
		(unifyRef t t2 
		 handle RefUnificationFailure => unifyOne t tlist2)
	in (unifyOne t1 tlist2) @ (unifyRefSome tlist1 tlist2)
	end

    fun allRefs (Dcds s) =
	let val (_, sChildren) = Type.typeLookup(s,!hierarchy)
	    fun pickRefs [] = []
	      | pickRefs ((s1, ext)::rest) = pickRefs rest
	      | pickRefs ((s1, partof)::rest) = (Dcds s1)::(pickRefs rest)
	in (Dcds s)::(pickRefs sChildren)
	end
      | allRefs (Arrow(t1,t2)) =
	let val t1refs = allRefs t1
	    val t2refs = allRefs t2
	in makeArrowMeet t1refs t2refs
	end
      | allRefs (Prod tlist) =
	map Prod (makeProdMeet (map allRefs tlist))
      | allRefs (Meet tlist) = map Meet (map allRefs tlist)
      | allRefs t = [t]
	

        (* Given a list of type variable indices paired with a regular type, we list *)
        (* all possible refinements of the regular type and pair each with the index *)
    fun generateAllRefs [] = []
      | generateAllRefs ((i,t)::rest) =
	let val tlist = (* If ground type, just take regular type --- Correct? *)
	    case t of
		(Arrow(_,_)) => allRefs t
	      | _ => [t]
	in (map (fn x => (i,x)) tlist) :: (generateAllRefs rest)
	end

	(* Given the result of generateAllRefs, generate a sub for each pair and *)
        (* apply one from each list.  Make all combinations. *)
    fun genApplySubs t subListList = 
	let fun genOne [] = []
	      | genOne ((i,t2)::others) = (newSub i t2)::(genOne others)
	    fun composeAll [] = []
	      | composeAll [subs] = subs
	      | composeAll (subs::others) = 
		composeOne subs (composeAll others)
	    and composeOne [] _ = []
	      | composeOne (s::others) done =
		(map (compose s) done) @ (composeOne others done)
	    val subsRaw = map genOne subListList
	    val subs = composeAll subsRaw
	in map (fn s => apply s t) subs
	end

        (* Figure out where an alpha in the ref type is really some other type *)
        (* from the regular type.  Also eliminate non refinements of regType. *)
    fun plugInAlphas regType [] = []
      | plugInAlphas regType (t::tlist) = 
	(let val subList = unifyRef t regType
	 in if null subList then t :: (plugInAlphas regType tlist)
	    else let val newSubList = generateAllRefs subList
		 in (genApplySubs t newSubList) @ (plugInAlphas regType tlist)
		 end
	 end
              handle RefUnificationFailure => plugInAlphas regType tlist)

    fun isMeetResult (Meet tlist) = fold (fn (x,y) => x orelse y) (map isMeetResult tlist) false
      | isMeetResult t =
	let val (n,out,inputs) = stripArrows t 0
	in isMeet out
	end

    fun isMeetInHead (t as (Arrow(_,_))) = 
	let val (n,out,inputs) = stripArrows t 0
	in fold (fn (x,y) => x orelse y) (map isMeetInHead inputs) false
	end
      | isMeetInHead (Prod tlist) = fold (fn (x,y) => x orelse y) (map isMeetInHead tlist) false
      | isMeetInHead (Meet _) = true
      | isMeetInHead _ = false

        (* Separate ref type lists into those in which we figured out dependence info *)
        (* and those which still have meets in the output. *)
    fun separateByInfo regType tlist =
	let fun sepInfo [] (info, useless) = (info, useless) 
	      | sepInfo (t::tlist) (info, useless) = if (isMeetResult t)
					 then sepInfo tlist (info, t::useless)
				     else sepInfo tlist (t::info, useless)
	in if isMeetResult regType then (tlist, [])
	   else sepInfo tlist ([],[])
	end

    fun matchingInputs [] [] = true
      | matchingInputs (t1::t1list) (t2::t2list) = 
	(let val _ = TypeChecker.unify t1 t2
	 in matchingInputs t1list t2list
	 end handle TypeChecker.UnificationFailure => false)

        (* t is an arrow.  Look for types in the second argument that have the same input. *)
        (* If there are any, and they have a different output, discard t.  Else keep it. *)
    fun superseded t tlist = 
	let fun findFrom _ [] = []
	      | findFrom inputs ((t' as (Arrow(_,_)))::rest) = 
		let val (_,out',inputs') = stripArrows t' 0
		in if (matchingInputs inputs inputs') then out'::(findFrom inputs rest)
		   else findFrom inputs rest
		end
	      | findFrom inputs (t2::rest) = raise RefineError "superseded: non arrow type"
	in case t of
	    (Arrow(_, _)) => let val (n,out,inputs) = stripArrows t 0
				 val conflicts = intersect(meetRemove out, findFrom inputs tlist)
			     in if null conflicts then [t] else []
			     end
	  | _ => [t]
	end
		 
	(* The first argument consists of types that still have meets in the *)
	(* output or input, that could not be eliminated by the dependency phase.  The *)
	(* second argument are types for which we figured out dependencies. *)
	(* Now we're going to distribute the meets from each of the first types *)
	(* but we're only going to keep those that do not conflict with the *)
        (* second argument. *)
    fun trimUseless [] _ = []
      | trimUseless (t::useless) info = (superseded t info) @ (trimUseless useless info)

        (* Eliminate types in the useless list which are less specific than *)
        (* those deduced by the dependence finder. *)
    fun mostSpecific [] _ = []
      | mostSpecific (t::useless) info = 
	let fun isRefinedBy t [] = false
	      | isRefinedBy t (t1::tlist) = 
		let val tryFirst = (let val _ = unifyRef t1 t in true end
					handle RefUnificationFailure => false)
		in if tryFirst then true else isRefinedBy t tlist
		end
	in if isRefinedBy t info then mostSpecific useless info
	   else t::(mostSpecific useless info)
	end	

    fun findStart t [] (startT, others) = (startT, others)
      | findStart t (t1::tlist) (startT, others) = 
	(case t1 of 
	     (Arrow(t1',t2)) => if t = t1' then findStart t tlist (t2::startT, others)
				else findStart t tlist (startT, t1::others)
	   | _ => findStart t tlist (startT, t1::others))

        (* If functions start with same argument, then combine them into one with *)
        (* the result type a union of all the result types *)
    fun monotonicize [] = []
      | monotonicize (t::tlist) = 
	(case t of
	     (Arrow(_,Arrow(_,_))) => (t::tlist)   (* Do nothing on curried types *)
	   | (Arrow(t1,t2)) => (let val (startT, others) = findStart t1 tlist ([], [])
				    val newTail = Subtype.supremum (t2::startT)
				in (Arrow(t1, meetWrap newTail))::(monotonicize others)
				end handle Subtype.SupInfFailure => raise RefineError "monotonicize failed")
	   | _ => (t::tlist))

        (* We want types like a'->sig /\ b'->sig to be identified.  Also, if there is *)
        (* no difference but in alphas bet reg type and ref type, make alphas same. *)
    fun identifyAlphas tlist regType =
	let fun doTlist [] = []
	      | doTlist [t] = [t]
	      | doTlist (t::tlist) =
		let fun doOneType t [] = emptySub
		      | doOneType t (t'::tlist) =
			(let val s = TypeChecker.unify t t'
			 in compose s (doOneType t tlist)
			 end handle TypeChecker.UnificationFailure => doOneType t tlist)
		    val s = doOneType t tlist
		in (apply s t) :: (doTlist tlist)
		end
	    val newTlist = duplicates (doTlist tlist)
	    fun compareWithRegType regType [] = []
	      | compareWithRegType regType (t::tlist) =
		(let val s = TypeChecker.unify t regType
		 in (apply s t) :: (compareWithRegType regType tlist)
		 end handle TypeChecker.UnificationFailure => t :: (compareWithRegType regType tlist))
	in duplicates (compareWithRegType regType newTlist)
	end

    (* ******************************************************************** *)
    (* Various utility functions for printing, combining types.             *)
    (* ******************************************************************** *)

    fun printDepInfo [] = ""
      | printDepInfo (((oc,ovList),deps)::rest) =
	let val ocname = Printer.unparseCell oc
	    val ovname = implode (map Printer.unparseVal ovList)
	    fun prfun (c, i:int, vList) = (Printer.unparseCell c)^" ("^
	    (makestring i)^") = "^(implode (map Printer.unparseVal vList))^", "
	    val onestr = "(("^ocname^" = "^ovname^", ["^
		(implode (map prfun deps))^"])\n"
	in onestr ^ (printDepInfo rest)
	end

        (* If we don't have a refinement type, use the regular type *)
    fun makeRefType [] regType = regType
      | makeRefType [t] _ = t
      | makeRefType l _ = meetWrap l    (* distributeMeets (Meet l) *)
	;

    (* ******************************************************************** *)
    (* This is the main sequence of functions:  find most general type      *)
    (* (a pair of the refinement type and "global" type), find refinement   *)
    (* type for a state, find refinement type for a deplist.                *)
    (* ******************************************************************** *)

    fun figureDep _ (t, []) = t
      | figureDep 1 (t, cvtList) = 
	(* First figure out simple matches of the kind $C <---> $C.1 *)
	(* These are the same that we had to deal with in overloaded types *)
	(* so we will reuse the technology. *)
	let val (outCV, outType) = hd cvtList
	    val (inCV, inType) = hd (tl cvtList)
	in if (TypeChecker.isOverloaded inCV outCV) then
	    let	val (dummyIn, matchIn) = TypeChecker.genDummy inType []
		val (dummyOut, matchOut) = TypeChecker.genDummy outType []
		val inCVT = breakUpProd (inType, inCV)
		val outCVT = breakUpProd (outType, outCV)
		val S = TypeChecker.match(matchIn, inCVT, matchOut, outCVT)
		val matchList = matchIn @ matchOut
		val inType' = TypeChecker.genMeet (apply S dummyIn) matchList
		val outType' = TypeChecker.genMeet (apply S dummyOut) matchList
	    in 
		(* Now the exciting part, finding correspondences between types *)
		if ((isMeet outType') orelse (isMeet inType')) andalso 
		    (not (higherOrder inType')) then 
		    let (* First find out how much to unroll each dcds that's involved *)
			val allCells = (map #1 (flatten(map #1 outCVT))) @ (map #1 (flatten (map #1 inCVT)))
			val lengthList = map Type.cellLength allCells
			val outNames = getNames outType'
			val inNoMeetType = TypeChecker.genEmbeddedMeet (apply S dummyIn) matchList
			val inCVTP = breakUpInProd (inNoMeetType, inCV, [])
			val inNames = map getNames (map #2 inCVTP)
			val inInternalTypes = map (map (fn x => lookup(x,typeList))) inNames
			val inTypesDepth = map Type.countDepth (flatten inInternalTypes)
			val outInternalTypes = map (fn x => lookup(x,typeList)) outNames
			val outTypesDepth = map Type.countDepth outInternalTypes
			val n = fold max (lengthList @ inTypesDepth @ outTypesDepth) 0
			(* Get a portion of depth n out of all input types *)
			val inCvaLists = map (map (Type.listDcds n))  inInternalTypes
			(* Figure out variable instantiations for the input for each type *)
			val inInst = instInputVars (map #1 inCVTP) (zip inNames inCvaLists)
			val inProdInst = createProdTypes inCVTP inInst
	                (* For each of those types, plug in variables into output *)
			val outCVInst = plugCV outCV inProdInst
			(* Debugging info *)
			(* val _ = let val cvtpstr = printCVTPlist inCVTP
				    val intstr = Printer.printType inType'
				    val outstr = printCVInst outCVInst
				    val instr = printProdInst inProdInst
				    val instr' = implode (map printInst inInst)
				    val _ = output(std_out, "inInst = "^instr'^"\n")
				    val _ = output(std_out, "inProdInst = "^instr^"\n")
				    val _ = output(std_out, "outCVInst = "^outstr^"\n")
				in output(std_out, "inCVT = "^cvtpstr^"inType' = "^intstr^"\n\n")
				end   *)
	                (* Retype the output, seeing if a mapping is established *)
			val newOutTypes = map (fn (inT, cvlist) => 
					       (Arrow(inT, TypeChecker.typeCVlist true cvlist))) outCVInst
		    in meetWrap newOutTypes
		    end
		(* We are not in a position to figure out dependencies *)
		else Arrow(inType', outType')
	    end
	   else t
	end 
      | figureDep _ (t, cvtList) = 
	(* Same as above, except for curried *)
	let val (outCV, outType) = hd cvtList
	    val inputs = tl cvtList
	    val inCVlist = map (#1) inputs
	    val inTypes = map (#2) inputs
	in if (TypeChecker.isOverloadedC inCVlist outCV) then
	    let val outCVT = breakUpProd (outType, outCV)
		val inCVTlist = map breakUpProd (zip inTypes inCVlist)
		val (dummyOut, matchOut) = TypeChecker.genDummy outType []
		val inDummyMatchList = map (fn x => TypeChecker.genDummy x []) inTypes
		fun genSubst [] _ _ = emptySub
		  | genSubst ((matchIn, inCVType)::rest) matchOut outCVType =
		    let val S = TypeChecker.match(matchIn, inCVType, matchOut, outCVType)
		    in compose S (genSubst rest matchOut outCVType)
		    end
		val matchInList = map (#2) inDummyMatchList
		val dummyInList = map (#1) inDummyMatchList
		val S = genSubst (zip matchInList inCVTlist) matchOut outCVT
		val matchList = fold op@ matchInList matchOut
		val outType' = TypeChecker.genMeet (apply S dummyOut) matchList
		val inTypes' = map (fn x => TypeChecker.genMeet x matchList) (map (apply S) dummyInList)
	    in 
		TypeChecker.genCurryType (inTypes' @ [outType'])
	    end
	   else t
	end

        (* Given the dependency info generated by depInfo (but in *)
        (* parse tree form), generate a type for that particular  *)
        (* "slice" of the algorithm.  Uses the routines of the    *)
        (* TypeChecker module.  Packages the dependency info with *)
        (* type for output---we will need to check it later.      *)
	(*   n -- 0=basic state, 1=algorithm, >1=curried algo     *)
    and refType n ((oc,ovList), deps) =
	let val outputCV = [(oc,ovList)]
	in case n of
	    1 => let val inputCV = map (fn (x,y,z) => (x,z)) deps
		 in if TypeChecker.isPoly inputCV outputCV
		     then let val t = TypeChecker.genPolyType inputCV outputCV
			  in (t, [])
			  end
		    else let val outputState = cvToState outputCV
			     val inputState = cvToState inputCV
			     val outGType = TypeChecker.typeState true outputState
			     val inGType = TypeChecker.typeState true inputState
			     val (inType, _) = refine inGType inputState
			     val (outType, _) = refine outGType outputState
			 in (Arrow(inType, outType), 
			     [(outputCV, outType), (inputCV, inType)])
			 end
		 end
	  | _ => let val inputCVlist = collate deps (makeI n [])
		 in if TypeChecker.isPolyC inputCVlist outputCV
			then let val t = 
			    TypeChecker.genPolyCType inputCVlist outputCV
			     in (t, [])
			     end
		    else let val outputState = cvToState outputCV
			     val inputStateList = map cvToState inputCVlist
			     val outGType = TypeChecker.typeState true outputState
			     val inGTypeList = map (TypeChecker.typeState true) inputStateList
			     val inTypeList = map (fn (x,y) => refine x y)
				 (zip inGTypeList inputStateList)
			     val (outType, _) = refine outGType outputState
			     val inRTypeList = map (#1) inTypeList
			     val t = TypeChecker.genCurryType 
				 (inRTypeList @ [outType])
			 in (t, (outputCV, outType)::
			        (zip inputCVlist inRTypeList))
			 end
		 end
	end

    and refineForest regType (n,tlist) =
	let val depInternalLists = map (depInfo n) tlist
	    val depListsOmega = flatten (map toParseTree depInternalLists)
	    val depLists = map removeOmegas depListsOmega
	    val _ = if !trace then 
		output(std_out, "dep list:\n"^(printDepInfo depLists))
		    else ()
	in  (* If we have a ground state, treat it as one piece, rather *)
	    (* than a separate one for each c=v *)
	    if n = 0 
	       then let val outputCV = map #1 depLists
			val outputState = cvToState outputCV
		    in (TypeChecker.typeState true outputState, regType)
		    end
	    else let val depTypeLists = map (refType n) depLists
		     val typeLists = if null depLists then [Alpha(newVar())]
				     else map (figureDep n) depTypeLists
		     val tlist = duplicates typeLists
		     val tlistNoAlphas = flatten (map meetRemove (plugInAlphas regType tlist))
		     val (infoNA, uselessNA) = separateByInfo regType 
			 (flatten (map distGroundMeetNoTail tlistNoAlphas))
		     (* val _ = let val infostr = implode (map Printer.printType infoNA)
				 val uselessstr = implode (map Printer.printType uselessNA)
			     in output(std_out, "info: "^infostr^"\nuseless: "^uselessstr^"\n")
			     end *)
		     val useful = flatten (map distGroundMeet (trimUseless uselessNA infoNA))
		     val newUseful = mostSpecific useful infoNA
		     val finalList = (monotonicize infoNA) @ newUseful
		     val finalList' = identifyAlphas finalList regType
		 in (makeRefType finalList' regType, regType)
		 end
	end

        (* Find refinement type given regular type for a state *)
    and refine regType x = 
	let val f = Internal.convert (Expr_state x)
	in case f of
	    (forest_basic(n,tlist)) => refineForest regType (n,tlist)
	  | _ => raise RefineError "refine: non-state"
	end
    
    and mgType e = 
	let val gT = (TypeChecker.inferType e) 
	    handle Lookup s => raise TypeChecker.InferError ("ident "^s^" not found")
		 | TypeChecker.OccursCheck => raise TypeChecker.InferError "occurs check"
		 | TypeChecker.UnificationFailure => 
		       raise TypeChecker.InferError "unification failure"
		 | Subtype.SubtypingFailure => raise TypeChecker.InferError "subtype failure"
	    val gt = case gT of
		TYPE t => t
	      | _ => raise RefineError "no global type"
	    val rt = findRefType gt e
		handle QandA.QandAError "nothing to do" => gt
		     | QandA.QandAError s => raise RefineError ("Q & A: "^s)
		     | QandA.Loop => (output(std_out,"This expression loops.\n"); gt)
		     | QandA.OutOfRange => (output(std_out,
				   "This expression may loop.  Refinement type inference gives up.\n"); gt)
	in (rt, gT)
	end

    and findRefType gt (Expr_id s) = 
	let val (time,e,T) = lookupExpType(s,!nameExpRTypeList)
	    handle Lookup str => raise RefineError ("nonexistent identifier: "^s)
	in case T of
	    UNTYPED => let val rt = findRefType gt e
			   val _ = storeExpType nameExpRTypeList
			       (s, !currentTimeStamp, e, TYPE rt)
		       in rt
		       end
	  | TYPE t' => if time = !currentTimeStamp
			   then let val (freshrt', _) = freshInst t' emptyInst
				in freshrt'
				end
		       else let val newrt = findRefType gt e
				val _ = storeExpType nameExpRTypeList
				  (s, !currentTimeStamp, e, TYPE newrt)
			    in newrt
			    end
	end
      | findRefType gt (Expr_state x) =
	let val (rt, _) = refine gt x
	in rt
	end
      | findRefType gt (Expr_algo a) =
	findRefType gt (Expr_state (Internal.algoToState(a,[])))
      | findRefType gt e =
	let val relevantState = QandA.refine gt e
	    val (rt, _) =  refine gt relevantState
	in rt
	end 

    end
    end;

