(* qanda.sml
   Abstract interpretation.  Figure out cell dependencies across
   an expression through a series of questions and answers, without looping. *)

signature QANDA =
    sig
    exception AiError of string
    exception Loop
    exception OutOfRange
    val eval :  CDSInternal.forest * CDSInternal.icell * CDSInternal.env -> 
	CDSInternal.ivalue
    exception QandAError of string
    val refine : CDSInternal.typeExp -> CDSBasic.expr -> (CDSBasic.cell * CDSBasic.value) list
    end;


functor QandAFUN (structure Internal : INTERNAL
		  structure Printer : PRINTER
		  structure Match : MATCH
		  structure Eval : EVALUATOR
		  structure Type : TYPE) : QANDA =
    struct

    local open CDSBasic
	  open CDSInternal
	  open CDSEnv
	  open Internal
	  open Printer
	  open Match
    in
    
    exception AiError of string
    exception Loop
    exception OutOfRange
    exception NoAccess

fun printCVlist [] = []
      | printCVlist ((c,vlist)::CV) = ("("^(Printer.unparseCell c)^
				       ", ["^(Printer.unparseValList vlist)^"])")::(printCVlist CV)
exception QandAError of string

(* ************************************************************** *)
(* First we have some functions which construct an annotated type *)
(* given a regular type.  The annotated type contains information *)
(* about product tags, number of inputs in a curried type, etc.   *)
(* ************************************************************** *)

datatype refineUnit = Ground of typeExp * typeExp list
                    | Ho of annotated 

withtype outputRefinement = (int list * refineUnit) list

and inputRefinement = (int * int list * refineUnit) list

and annotated = int * outputRefinement * inputRefinement

   (* Creates a list 1..n to be used to tag curried inputs and products *)
fun genIndexList n =
    let fun genIndexList' i n = if i = n then [n] else i :: (genIndexList' (i+1) n)
    in genIndexList' 1 n
    end
    
    (* Flattens a product into constituent pieces and decorates them with an int list *)
    (* which specifies the tag.   $C.2.1.1 |--> [2,1,1], $C *)
fun breakUpProd (Prod tlist, prodIndex) =
    let val newIndex = genIndexList (length tlist)
	val newProdIndex = map (fn x => prodIndex @ [x]) newIndex
	val typeIndexList = zip tlist newProdIndex
    in flatten (map breakUpProd typeIndexList)
    end
  | breakUpProd (t, prodIndex) = [(t, rev prodIndex)]

    (* Looks up refinement types for a type *)
fun getRefs (Dcds s) = 
    let val (_, sChildren) = Type.typeLookup(s,!hierarchy)
	fun pickRefs [] = []
	      | pickRefs ((s1, ext)::rest) = pickRefs rest
	      | pickRefs ((s1, partof)::rest) = (Dcds s1)::(pickRefs rest)
	in (Dcds s, pickRefs sChildren)
    end
  | getRefs t = (t, [])

    (* Appends all the refinement lists from an annotated type *)
fun collectAnnotatedRefs (_,refOut,refIn) =
    let val outRefs = getOutRefs refOut
	val inRefs = getInRefs refIn
    in outRefs @ inRefs
    end

and getOutRefs [] = []
  | getOutRefs ((pI,Ground(t,refs))::rest) = refs @ (getOutRefs rest)
  | getOutRefs ((pI,Ho annType)::rest) = (collectAnnotatedRefs annType) @ (getOutRefs rest)

and getInRefs [] = []
  | getInRefs ((_,_,Ground(t,refs))::rest) = refs @ (getInRefs rest)
  | getInRefs ((_,_,Ho annType)::rest) = (collectAnnotatedRefs annType) @ (getInRefs rest)

fun makeRefOut (t, pI) = 
    (case t of
	 (Arrow(_,_)) => let val (_,refT) = refinable t
			in (pI, Ho refT)
			end
      | _ => let val (_,refs) = getRefs t
	     in (pI, Ground(t,refs))
	     end)

and makeRefIn [] = []
  | makeRefIn ((i,prodIndex,t)::rest) = 
    (case t of
	 (Arrow(_,_)) => let val (_,refT) = refinable t
			 in (i,prodIndex,Ho refT) :: (makeRefIn rest)
			 end
       | _ => let val (_,refs) = getRefs t
	      in (i,prodIndex,Ground (t,refs)) :: (makeRefIn rest)
	      end)

    (* Given a type, it constructs an annotated type *)
and refinable t =
    (case t of
	 (Arrow(_,_)) => let val (n,out,inputs) = stripArrows t 0
			     val brokenOut = breakUpProd (out, [])
			     val prelimInputs = map (fn t => (t,[])) inputs
			     val brokenInList = map breakUpProd prelimInputs
			     val curryIndex = genIndexList (length inputs)
			     fun insertIndex [] _ = []
			       | insertIndex (i::indices) (refIn::inList) =
				 let fun insertIndexList i [] = []
				       | insertIndexList i ((t,prodIndex)::rest) = 
					 (i,prodIndex,t)::(insertIndexList i rest)
				 in (insertIndexList i refIn) @ (insertIndex indices inList)
				 end
			     val taggedInList = insertIndex curryIndex brokenInList
			     val refOut = map makeRefOut brokenOut
			     val refIn = makeRefIn taggedInList
			     val annType = (n,refOut,refIn)
			     val allRefs = collectAnnotatedRefs annType
			 in (null allRefs, annType)
			 end
       | (Meet tlist) => raise QandAError "cannot refine conjunct types for now"
       | t => let val brokenOut = breakUpProd (t, [])
		  val refOut = map makeRefOut brokenOut
		  val annType = (0,refOut,[])
		  val allRefs = collectAnnotatedRefs annType
	      in (null allRefs, annType)
	      end)


(* ************************************************************ *)
(* Following are functions to figure out what are the names of  *)
(* cells to be queried are in the output of some annotated type *)
(* ************************************************************ *)

fun createTag [] c = c
  | createTag (1::prodIndex) c = createTag prodIndex (Cell_graft(c, Tag_arexpr(Arexpr_int 1)))
  | createTag (2::prodIndex) c = createTag prodIndex (Cell_graft(c, Tag_arexpr(Arexpr_int 2)))

fun addState 0 c = c
  | addState n c = Cell_fun(Expr_state [], addState (n-1) c)

fun makeCellNames _ [] = []
  | makeCellNames n (c::rest) = (addState n c) :: (makeCellNames n rest)

fun applyTag prodIndex cvaList =
    let val tag = createTag prodIndex
	fun tagAccess f [] = []
	  | tagAccess f (x::xs) = (map (fn (c,v) => (f c, v)) x) :: (tagAccess f xs)
	fun tagCva f [] = []
	  | tagCva f ((c,vlist,alist)::cvaList) = 
	    (f c, vlist, tagAccess f alist) :: (tagCva f cvaList)
    in tagCva tag cvaList
    end

fun getNames (Dcds s) = [s]
  | getNames (Meet tlist) = flatten (map getNames tlist)
  | getNames (Alpha _) = []
  | getNames _ = raise QandAError "getNames: not ground"

fun different [] _ = false
  | different (v::vlist) voutlist = if member(v,voutlist) then different vlist voutlist else true

fun findC c [] = raise Find
  | findC c ((c1,vlist,alist)::rest) = if c=c1 then (c,vlist,alist) else findC c rest
	
fun infiniteVal (Val_interval_inf) = true
  | infiniteVal (Val_interval_from _) = true
  | infiniteVal (Val_interval_to _) = true
  | infiniteVal (Val_interval_fromto(_,_)) = true
  | infiniteVal _ = false

fun infiniteValList vlist = fold (fn (x,y) => x orelse y) (map infiniteVal vlist) false

fun distinguish _ [] = []
  | distinguish outCva (refCva::rest) =
    let fun intersect _ [] result = result
	  | intersect outCva ((c,vlist,alist)::refCva) result =
	    (let val (c',vlist',alist') = findC c outCva
	     in if different vlist' vlist then intersect outCva refCva ((c',vlist',alist')::result)
		else intersect outCva refCva result
	     end handle Find => intersect outCva refCva result)
	val inter1 = intersect outCva refCva []
    in duplicates (inter1 @ (distinguish outCva rest))
    end

fun pickFiniteVals [] = []
  | pickFiniteVals ((c,vlist,alist)::rest) = if infiniteValList vlist then pickFiniteVals rest
					     else [(c,vlist,alist)]

    (* Takes stratified cva lists of the refinement types and attempts to *)
    (* come up with cva elements that distinguish among them. *)
fun distinguishRefs [] = []
  | distinguishRefs [refCva] = []
  | distinguishRefs (refCva::rest) =
    let fun makeOnePass _ [] = []
	  | makeOnePass refCva (refCva'::rest) =
	    let fun cvaList2comp [] [] = []
		  | cvaList2comp [] (cvaList'::rest) = pickFiniteVals cvaList'
		  | cvaList2comp (cvaList::rest) [] = pickFiniteVals cvaList
		  | cvaList2comp (cvaList::rest) (cvaList'::rest') = 
		    let fun cvaListComp _ [] result = result
			  | cvaListComp [] _ result = result
			  | cvaListComp cvaList ((c,vlist,alist)::cvaList') result =
			    (let val _ = findC c cvaList 
			     in cvaListComp cvaList cvaList' result
			     end handle Find => cvaListComp cvaList cvaList' ((c,vlist,alist)::result))
		    in (cvaListComp cvaList cvaList' []) @ (cvaList2comp rest rest')
		    end
	    in (cvaList2comp refCva refCva') @ (makeOnePass refCva rest)
	    end
    in (makeOnePass refCva rest) @ (distinguishRefs rest)
    end

fun removeRedundant outCompare [] result = outCompare @ result
  | removeRedundant outCompare ((c,vlist,alist)::refCompare) result =
    (let val _ = findC c outCompare
     in removeRedundant outCompare refCompare result
     end handle Find => removeRedundant outCompare refCompare ((c,vlist,alist)::result))

fun collectByI n icvaList =
    let fun findI i [] (yesi,noi) = (yesi,noi)
	  | findI i ((i',cvaList)::rest) (yesi,noi)= 
	    if i=i' then findI i rest (cvaList @ yesi,noi)
	    else findI i rest (yesi,(i',cvaList)::noi)
	fun findN i n icvaList = 
	    if i=n then [(flatten (map #2 icvaList))]
	    else let val (yes,no) = findI i icvaList ([],[])
		 in yes :: (findN (i+1) n no)
		 end
    in findN 1 n icvaList
    end

fun addOutput 0 v = v
  | addOutput n v = Val_output (addOutput (n-1) v)

    (* Creates initial cva list for a higher-order curried type with n inputs. *)
    (* The input cva lists are stratified. *)
fun cvaGenerator (n, outStratCva, inStratCvaList) = 
    let val outInitial = hd outStratCva
	fun makeSimpleOutputs _ [] = []
	  | makeSimpleOutputs n ((c,vlist,_)::rest) =
	    let val newVals = map (addOutput n) vlist
	    in (addState n c, newVals, []) :: (makeSimpleOutputs n rest)
	    end
    in makeSimpleOutputs n outInitial
    end    

fun listGround (Ground(t,refs)) =
    let val outName = case t of Dcds s => s | _ => raise QandAError "listGround"
	val outInternal = lookup(outName,typeList)
	val outDepth = Type.countDepth outInternal
	val outCvaList = Type.listDcds outDepth outInternal
    in if null refs
	   (* No refinements for this guy, just unroll output dcds *)
	   then outCvaList
       (* Else figure out minimum cva list we need to look at to differentiate *)
       else let val refNames = flatten (map getNames refs)
		val refInternal = map (fn s => lookup(s,typeList)) refNames
		val refDepth = map Type.countDepth refInternal
		val d = fold max refDepth 0
		val refCvaLists = map (Type.listDcds d) refInternal
		val outCmp = distinguish outCvaList refCvaLists
		val refCmp = distinguishRefs (map Type.stratify refCvaLists)
	    in removeRedundant outCmp refCmp []
	    end 
    end

and listRefOut (pI, Ground(t,refs)) = applyTag pI (listGround (Ground(t,refs)))
  | listRefOut (pI, Ho annType) = applyTag pI (listHo annType)

and listRefIn (i, pI, Ground(t,refs)) = (i, applyTag pI (listGround (Ground(t,refs))))
  | listRefIn (i, pI, Ho annType) = (i, applyTag pI (listHo annType))

and listHo (n,refOut,refIn) = 
    (case n of
	 0 => flatten (map listRefOut refOut)
       | _ => let val outCvas = flatten (map listRefOut refOut)
		  val inCvaList = map listRefIn refIn
		  val inCvaList2 = collectByI n inCvaList
		  val outCvaStrat = Type.stratify outCvas
		  val inCvasStrat = map Type.stratify inCvaList2
	      in cvaGenerator (n, outCvaStrat, inCvasStrat)
	      end)

fun getIs i [] = []
  | getIs i ((i',pI,r)::rest) = if i=i' then (i',pI,r)::(getIs i rest)
				else getIs i rest

    (* Given an outputRefinement construct a list of all initial cells *)
    (* with which we will start our querying. *)
fun getRelevant n [] = []
  | getRelevant n ((prodIndex,Ground(t,refs))::rest) = 
    let val outCva = listGround(Ground(t,refs))
	val outCvaInit = hd (Type.stratify outCva)
	val outCells = map #1 (applyTag prodIndex outCvaInit) (* changed *) 
    in (makeCellNames n outCells) @ (getRelevant n rest)
    end
  | getRelevant n ((prodIndex,Ho annT)::rest) = 
    let val hoCva = listHo annT
	val outCells = map #1 (applyTag prodIndex hoCva)
    in (makeCellNames n outCells) @ (getRelevant n rest)
    end


(* ********************************************************* *)
(* Functions which pose, interpret, and generate new queries *)
(* ********************************************************* *)

exception ExpressionAnswer

    (* Convert icell -> cell *)
fun revertCell (Icell_name s) = Cell_name s
  | revertCell (Icell_graft(ic,t)) = Cell_graft(revertCell ic,t)
  | revertCell (Icell_with(ic,ib)) = Cell_with(revertCell ic,revertBoolexp ib)
  | revertCell (Icell_fun(f,ic)) = Cell_fun(revertForest f,revertCell ic)

and revertForest (forest_basic(i,ctreeInsList)) = Expr_state(revertTreeIns i ctreeInsList)
  | revertForest f = raise ExpressionAnswer

and revertTreeIns _ [] = []
  | revertTreeIns i ((ic,tree_Result(i',iv))::rest) = 
    (revertCell ic, revertVal iv) :: (revertTreeIns i rest)
  | revertTreeIns _ _ = raise QandAError "revertTreeIns: non result tree"

and revertVal (Ival_string s) = Val_string s
  | revertVal (Ival_arexpr a) = Val_arexpr a
  | revertVal (Ival_omega) = Val_omega
  | revertVal (Ival_with(s,ib)) = Val_with(s,revertBoolexp ib)
  | revertVal (Ival_pair(iv1,iv2)) = Val_pair(revertVal iv1,revertVal iv2)
  | revertVal (Ival_output iv) = Val_output (revertVal iv)
  | revertVal (Ival_valof ic) = Val_valof (revertCell ic)

and revertBoolexp (Iboolexp_gt(a1,a2)) = Boolexp_gt(a1,a2)
  | revertBoolexp (Iboolexp_gteq(a1,a2)) = Boolexp_gteq(a1,a2)
  | revertBoolexp (Iboolexp_lt(a1,a2)) = Boolexp_lt(a1,a2)
  | revertBoolexp (Iboolexp_lteq(a1,a2)) = Boolexp_lteq(a1,a2)
  | revertBoolexp (Iboolexp_eq(iv1,iv2)) = Boolexp_eq(revertVal iv1,revertVal iv2)
  | revertBoolexp (Iboolexp_noteq(iv1,iv2)) = Boolexp_noteq(revertVal iv1,revertVal iv2)
  | revertBoolexp (Iboolexp_or(ib1,ib2)) = Boolexp_or(revertBoolexp ib1,revertBoolexp ib2)
  | revertBoolexp (Iboolexp_and(ib1,ib2)) = Boolexp_and(revertBoolexp ib1,revertBoolexp ib2)

fun whichInput (Ival_output iv) = let val (i,iv') = whichInput iv in (1 + i, iv') end
  | whichInput iv = (1,iv)

fun whichInputV (Val_output v) = let val (i,v') = whichInputV v in (1 + i, v') end
  | whichInputV v = (1,v)

fun ripTag (Cell_graft(c,Tag_arexpr(Arexpr_int 1))) soFar = ripTag c (1::soFar)
  | ripTag (Cell_graft(c,Tag_arexpr(Arexpr_int 2))) soFar = ripTag c (2::soFar)
  | ripTag c soFar = (c,soFar)

fun ripStateTag [] = []
  | ripStateTag ((c,v)::x) = let val (c',theTag) = ripTag c []
			     in (theTag, (c',v)) :: ripStateTag x
			     end

exception FindByTag
exception FindInCva

fun findByTag _ [] = raise FindByTag
  | findByTag theTag ((i,prodIndex,r)::rest) = 
    if theTag = prodIndex then (i,prodIndex,r) else findByTag theTag rest

fun findInCva c [] = raise FindInCva
  | findInCva c ((c',vlist,alist)::rest) = if c=c' then vlist else findInCva c rest

fun peelNStates 0 c = ([],c)
  | peelNStates 1 (Cell_fun(e,c)) = ([e], c)
  | peelNStates n (Cell_fun(e,c)) = let val (elist, theC) = peelNStates (n-1) c
				    in (e::elist,theC)
				    end

fun leepStates [] c = c
  | leepStates (e::elist) c = Cell_fun(e,leepStates elist c)

fun peelNOutputs 0 v = v
  | peelNOutputs 1 (Val_output v) = v
  | peelNOutputs n (Val_output v) = peelNOutputs (n-1) v

fun findInOutByTag _ [] = raise FindByTag
  | findInOutByTag theTag ((prodIndex,r)::rest) = 
    if theTag = prodIndex then (prodIndex,r) else findInOutByTag theTag rest

fun findInStateByTag _ [] = []
  | findInStateByTag pI ((pI',evt)::rest) = 
    if pI=pI' then evt :: (findInStateByTag pI rest) else findInStateByTag pI rest

fun findByAlist _ [] = []
  | findByAlist alist ((c,vlist,alistList)::rest) = 
    if member(alist,alistList) then c :: (findByAlist alist rest) else findByAlist alist rest

fun bunchUpIs n refIn =
    let fun findI i [] (yesi,noi) = (yesi,noi)
	  | findI i ((i',pI,r)::rest) (yesi,noi)= 
	    if i=i' then findI i rest ((pI,r)::yesi,noi)
	    else findI i rest (yesi,(i',pI,r)::noi)
	fun findN i n refIn = 
	    if i=n then [map (fn (x,y,z) => (y,z)) refIn]
	    else let val (yes,no) = findI i refIn ([],[])
		 in yes :: (findN (i+1) n no)
		 end
    in findN 1 n refIn
    end

    (* Is an access list fulfilled by a state? *)
fun fulfilled [] x = false
  | fulfilled (a1list::alist) x =
    let fun checkOneAccess [] x = true
	  | checkOneAccess (evt::alist) x =
	    if member (evt, x) then checkOneAccess alist x
	    else false
    in if checkOneAccess a1list x then true 
       else fulfilled alist x
    end 

    (* Given a state and a cva, find those cells in cva that *)
    (* are enabled but not filled in the state *)
fun enabledInCva x [] = []
  | enabledInCva x ((c,_,alist)::cvaList) =
    if search(c,x) then enabledInCva x cvaList
    else if null alist then c :: (enabledInCva x cvaList)
	 else if (fulfilled alist x) then c :: (enabledInCva x cvaList)
	      else enabledInCva x cvaList

    (* Given a state, separate it into events of the form c=ouput v *)
    (* and c=valof c'.  State can be curried with n inputs *)
fun splitByOut [] (outX, valofX) = (rev outX, rev valofX)
  | splitByOut ((c,v)::x) (outX, valofX) =
    let val (i,v') = whichInputV v
    in case v' of 
	(Val_valof c') => splitByOut x (outX, (c,i,c')::valofX)
      | _ => splitByOut x ((c,v)::outX, valofX)
    end 

    (* Given a state and a cell list, remove cells already filled in the state *)
fun removeFilled x [] = []
  | removeFilled x (c::clist) = if search(c,x) then removeFilled x clist
				else c :: (removeFilled x clist)

fun splitAtI 1 [] = raise Ith
  | splitAtI 1 (e::l) = ([], e, l)
  | splitAtI i [] = raise Ith
  | splitAtI i (e::l) = let val (left,elem,right) = splitAtI (i-1) l
			in (e::left,elem,right)
			end

    (* When querying cell c we were asked for valof c' from the i'th state part *)
    (* of c.  Put in each value in vlist for c' in that state. *)
fun packageQuery c i c' vlist =
    let fun unwrapCell (Cell_fun(e,c)) = let val (elist,cellName) = unwrapCell c
					 in (e::elist,cellName)
					 end
	  | unwrapCell c = ([], c)
	val (stateList, cName) = unwrapCell c
	val (bef,stateI,after) = splitAtI i stateList
	val eventListI = case stateI of (Expr_state l) => l | _ => raise QandAError "packageQuery"
	fun makeStates _ _ [] = []
	  | makeStates eventList c' (v::vlist) =
	    (Expr_state(eventList @ [(c',v)])) :: (makeStates eventList c' vlist)
	val newStatesI = makeStates eventListI c' vlist
	fun insertEach _ _ [] = []
	  | insertEach bef after (e::elist) = (bef @ [e] @ after) :: (insertEach bef after elist)
	val newStateLists = insertEach bef after newStatesI
	fun wrapCell c [] = c
	  | wrapCell c (e::elist) = Cell_fun(e,wrapCell c elist)
    in map (wrapCell cName) newStateLists
    end;

    (* List possible relevant (from the point of view of refinements) *)
    (* values for cell c from the i'th input given an annotated type *)
    (* and a state of that type. *)
fun possibleValues i c (n,refOut,refIn) history =
    let val inList = getIs i refIn
	val (cNoTag,theTag) = ripTag c []
	val (inputIndex,prodIndex,r) = (findByTag theTag inList) 
	    handle FindByTag => raise QandAError "possibleValues: no matching input"
    in case r of
	(Ground(_,_)) => ((findInCva cNoTag (listGround r)) handle FindInCva => [])
      | (Ho (n',refOut',refIn')) => possibleHOValues cNoTag (n',refOut',refIn')
    end

and possibleOutValues c refOut =
    let val (cNoTag,theTag) = ripTag c []
	val (prodIndex,r) = (findInOutByTag theTag refOut) 
	    handle FindByTag => raise QandAError "possibleValues: no matching input"
    in case r of
	(Ground(_,_)) => ((findInCva cNoTag (listGround r)) handle FindInCva => [])
      | (Ho (n',refOut',refIn')) => possibleHOValues cNoTag (n',refOut',refIn')
    end

and possibleHOValues c (n, refOut, refIn) =
    let val (elist,outC) = peelNStates n c
	val inputList = bunchUpIs n refIn
	val clistList = map enabledByState (zip elist inputList)
	val vlist = possibleOutValues outC refOut
	val outvlist = map (addOutput n) vlist
	val clistIndexed = zip (genIndexList n) clistList
	val valoflist = flatten (map (fn (i,cells) => map (addOutput (i-1)) (map Val_valof cells)) clistIndexed)
    in outvlist @ valoflist
    end
    
and enabledByState (_, []) = []
  | enabledByState (Expr_state x, (pI,Ground(t,refs))::inList) = 
    let val inCva = applyTag pI (listGround (Ground(t,refs)))
    in (enabledInCva x inCva) @ (enabledByState (Expr_state x, inList))
    end
  | enabledByState (Expr_state x, (pI, Ho (n,refOut,refIn))::inList) =
    let val relevantX = findInStateByTag pI (ripStateTag x)
	val (outvalX, valofX) = splitByOut relevantX ([],[])
	val initCells = map #1 (listHo (n,refOut,refIn))
	val outEnabled = findOutEnabled outvalX (n,refOut,refIn)
	val valofEnabled = findValofEnabled valofX (n,refOut,refIn)
	val clist = removeFilled relevantX (initCells @ outEnabled @ valofEnabled)
    in (map (createTag pI) clist) @ (enabledByState (Expr_state x, inList))
    end

and findOutEnabled outvalX (n,refOut,refIn) =
    flatten (map (findNewlyEnabled (n,refOut,refIn) []) outvalX)

and findValofEnabled [] _ = []
  | findValofEnabled ((c,i,c')::valofX) (n,refOut,refIn) =
    let val vlist = possibleValues i c' (n,refOut,refIn) []
    in (packageQuery c i c' vlist) @ (findValofEnabled valofX (n,refOut,refIn))
    end    

    (* We have just figured out a new c=v event (appended at the end of history) *)
    (* so now we want to see if this enables any new cells in the output so we *)
    (* can query them.  *** Only works on filiform dcds currently. *** *)
and findNewlyEnabled (n,refOut,refIn) history (c,v) =
    let val (cNoTag,theTag) = ripTag c []
	val (prodIndex, r) = (findInOutByTag theTag refOut)
	    handle FindByTag => raise QandAError "findNewlyEnabled: no matching output"
    in case r of
	(Ground(_,_)) => let val (elist,outCell) = peelNStates n cNoTag
			     val outVal = peelNOutputs n v
			     val outCva = listGround r
			     val clist = findByAlist [(outCell,outVal)] outCva
			     fun putStates [] _ = []
			       | putStates (oneCell::rest) elist =
				 (leepStates elist oneCell) :: (putStates rest elist)
			 in map (createTag theTag) (putStates clist elist)
			 end
      | (Ho (n',refOut',refIn')) => (* we have a higher-order piece of a product output, i.e., n=0 *)
	    let val clist = findNewlyEnabled (n',refOut',refIn') [] (cNoTag,v)
	    in map (createTag theTag) clist
	    end
    end

fun poseQuery _ _ [] history = history
  | poseQuery (n,refOut,refIn) f (c::toAsk) history =
    (let val ic = Internal.convertCell c
	val _ = (fixList := [];  fixCounter := 0)
	val iv = eval(f,ic,emptyenv)
	(*val _ = output(std_out, "poseQuery: c = "^(Printer.unparseCell c)^"\n")*)
    in case iv of
	(Ival_output _) => let val (i,iv') = whichInput iv
			       val v = revertVal iv
			       val newHistory = history @ [(c,v)]
			   in case iv' of
			       (Ival_valof ic') =>
				   (case n of
					0 => let val (cNoTag,theTag) = ripTag c []
						 val (prodIndex, r) = (findInOutByTag theTag refOut)
					     in case r of
						 (Ho (n',refOut',refIn')) => 
						     let val c' = revertCell ic'
							 val vlist = possibleValues i c' (n',refOut',refIn') newHistory
						     in if infiniteValList vlist orelse null vlist
							    then poseQuery (n,refOut,refIn) f toAsk history (*newHistory*)
							else let val newAsk = packageQuery cNoTag i c' vlist
								 val newAsk' = map (createTag theTag) newAsk
							   in poseQuery (n,refOut,refIn) f (newAsk' @ toAsk) newHistory
							     end
						     end
					       | _ => raise QandAError "poseQuery: not a product output"
					     end
				      | _ => let val c' = revertCell ic'
						 val vlist = possibleValues i c' (n,refOut,refIn) newHistory
					     in if infiniteValList vlist orelse null vlist
						    then poseQuery (n,refOut,refIn) f toAsk history (*newHistory*)
						else let val newAsk = packageQuery c i c' vlist
						     in poseQuery (n,refOut,refIn) f (newAsk @ toAsk) newHistory
						     end
					     end)
			     | _ => let val newAsk = findNewlyEnabled (n,refOut,refIn) newHistory (c,v)
				    in poseQuery (n,refOut,refIn) f (newAsk @ toAsk) newHistory
				    end
			   end
      | (Ival_valof ic') => 
	    (case n of
		   (* We have something like int * (bool -> bool) on our hands *)
		 0 => let val (cNoTag,theTag) = ripTag c []
			  val (prodIndex, r) = (findInOutByTag theTag refOut)
			  val v = revertVal iv
			  val newHistory = history @ [(c,v)]
		      in case r of
			  (Ho (n',refOut',refIn')) => 
			      let val c' = revertCell ic'
				  val vlist = possibleValues 1 c' (n',refOut',refIn') newHistory
			      in if infiniteValList vlist orelse null vlist
				     then poseQuery (n,refOut,refIn) f toAsk history (*newHistory*)
				 else let val newAsk = packageQuery cNoTag 1 c' vlist
					  val newAskTagged = map (createTag theTag) newAsk
				      in poseQuery (n,refOut,refIn) f (newAskTagged @ toAsk) newHistory
				      end
			      end
			| _ => raise QandAError "poseQuery: not a product output"
		      end
	           (* The output is just a ground dcds *)
	       | _ => let val c' = revertCell ic'
			  val newHistory = history @ [(c,Val_valof c')]
			  val vlist = possibleValues 1 c' (n,refOut,refIn) newHistory
		      in if infiniteValList vlist orelse null vlist
			     then poseQuery (n,refOut,refIn) f toAsk history (*newHistory*)
			 else let val newAsk = packageQuery c 1 c' vlist
			      in poseQuery (n,refOut,refIn) f (newAsk @ toAsk) newHistory
			      end
		      end)
      | (Ival_omega) => (* Why did this happen?  It was either an access problem or loop *)
	    poseQuery (n,refOut,refIn) f toAsk history
      | _ => let val v = revertVal iv
		 val newHistory = history @ [(c,v)]
		 val newAsk = findNewlyEnabled (n,refOut,refIn) newHistory (c,v)
	     in poseQuery (n,refOut,refIn) f (newAsk @ toAsk) newHistory
	     end
     end
          handle ExpressionAnswer => poseQuery (n,refOut,refIn) f toAsk history)


(* *********************************************************** *)
(* Main functions in this module: figure out what questions to *)
(* ask, ask them, interpret results, potentially generate new  *)
(* questions, find dependencies, and generate a type from them *)
(* *********************************************************** *)

and last [] result = (NONE, result)
  | last [cv] result = (SOME cv, result)
  | last (cv::cvList) result = last cvList (result @ [cv])

and elimLastValofs cvList =
    let val (lastElem, rest) = last cvList []
    in case lastElem of 
	SOME (c,v) => let val (_, v') = whichInputV v
		      in case v' of
			  Val_valof _ => elimLastValofs rest
			| _ => cvList
		      end
      | NONE => cvList
    end

and endsInAlpha t = let val (n,out,inputs) = stripArrows t 0
		    in variable out 
		    end

and QandA (n,refOut,refIn) f =
    let (* Assume cells are ordered to account for access conditions *)
	val relevantCells = getRelevant n refOut
    in poseQuery (n, refOut, refIn) f relevantCells []
    end
	
and refine regType e =
    let val (noRefs, annotatedType) = refinable regType
	handle QandAError "cannot refine conjunct types for now" => 
	    raise QandAError "nothing to do"
    in  (* If type contains no types which have refinements, don't bother *)
	if noRefs orelse (endsInAlpha regType) then raise QandAError "nothing to do"
	(* Otherwise do a questions and answers session to find dependencies *)
	else let val cvList = QandA annotatedType (Internal.convert e)
		 val newCvList = elimLastValofs cvList
	     in if null newCvList then raise QandAError "nothing to do"
		else newCvList
	     end 
    end


(* ******************************* *)
(* Non-looping evaluator functions *)
(* ******************************* *)

	(* Projection functions. *)
    and remove label (Icell_graft(c,Tag_str l),ins) = 
	  if label=l then [(c, ins)] else []
      | remove label (Icell_graft(c,Tag_arexpr(Arexpr_int i)),ins) =
	  if label = makestring(i) then [(c, ins)] else []
      | remove label _ = []

    and proj (s, forest_basic(i, tlist)) = 
	                forest_basic(i, (flatten (map (remove s) tlist)))
      | proj (s, forest_prod flist) =
	  let val i = ord s - ord "0"
	  in ith(i, flist) handle Ith 
		          => raise AiError "project: index out of bounds"
	  end
      | proj (s, forest_apply(forest_pair flist, f2)) =
	  let val i = ord s - ord "0"
	  in forest_apply(ith(i, flist),f2) handle Ith 
		          => raise AiError "project: index out of bounds"
	  end
      | proj _ = raise AiError "attempting to project wrong type"


	(* Executes one tree instruction when querying cell c.     *)
	(* There are 3 kinds of tree instructions:  Result, Valof, *)
	(* and From.  The semantics is the same as in Devin.       *)
    and exec (tree_Result(i,v),c,env) = makeOutput(i,expandVal(v,env))
      | exec (tree_Valof(c1,i,tqlist),c,env) = 
	  let val fi = getIthForest(i,c) handle IthForest => 
	                                    raise AiError "exec: IthForest"
	      val c1' = expandCell(c1,env)
	      val v1 = eval(fi,c1',env)
	  in if v1 = Ival_omega 
		 then makeOutput(i-1,Ival_valof c1')
	     else (let val (ins,newEnv) = findValMatch(v1,tqlist,env)
		   in exec(ins,c,newEnv)
		   end handle Find => Ival_omega)
	  end
      | exec (tree_From(c1,i,tqlist),c,env) =
	  let val fi = getIthForest(i,c) handle IthForest => 
	                                    raise AiError "exec: IthForest"
	      val c1' = expandCell(c1,env)
	      val v1 = eval(fi,c1',env)
	  in if v1 = Ival_omega then Ival_omega
	     else (let val (ins,newEnv) = findValMatch(v1,tqlist,env)
		   in exec(ins,c,newEnv)
		   end handle Find => raise NoAccess)
	  end

    
        (* Main function--processes queries of the form forest ? cell. *)
        (* If we have a basic forest then just look up appropiate tree *)
        (* instruction (if any) and exec it.  Otherwise we have an     *)
        (* expression--apply the CDS02 rewrite rules.                  *)
    and eval (forest_basic(i, tlist), c, env) =
	  (let val (ins,newEnv) = findCellMatch(getIthName(i,c),tlist,env)
	   in exec(ins,c,newEnv)
	   end handle Find => Ival_omega)
      | eval (forest_apply(f1,f2), c, env) =
	  let val v = eval(f1, Icell_fun(f2,c),env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof _ => Ival_omega
	    | Ival_output v1 => expandVal(v1,env)
	    | _ => raise AiError ("application: value "^(printval v)^
		  " not in list")
	  end
      | eval (forest_comp(f1,f2), c, env) =
	  let val x = getForest c
	      val cname = getName c
	      val v = eval(f1,Icell_fun(forest_apply(f2,x),cname),env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof c1 => eval(f2,Icell_fun(x,c1),env)
	    | Ival_output v1 => Ival_output(expandVal(v1,env))
	    | _ => raise AiError "composition: value not in list"
	  end
      | eval (forest_fix f, c, env) =
	if checkInVerify (f,c) then raise Loop
	else if (!fixCounter >= 30) then raise OutOfRange
	     else let val v = eval(f, Icell_fun(forest_fix f, c), env)
		  in case v of
		      Ival_omega => let val _ = checkOut (f,c) in Ival_omega end
		    | Ival_valof _ => let val _ = checkOut (f,c) in Ival_omega end
		    | Ival_output v1 => let val _ = checkOut (f,c) in expandVal(v1,env) end
		    | _ => raise AiError "fix: value not in list"
		  end
      | eval (forest_curry f, c, env) =
	let val x = getForest c
	    val y = getForest(getName c)
	    val cname = getName(getName c)
	    val v = eval(f, Icell_fun(forest_prod [x, y], cname), env)
	in case v of
	    Ival_omega => Ival_omega
	  | Ival_valof(Icell_graft(c1,Tag_str "1")) => 
		Ival_valof(expandCell(c1,env))
	  | Ival_valof(Icell_graft(c1,Tag_arexpr(Arexpr_int 1))) => 
		Ival_valof(expandCell(c1,env))
	  | Ival_valof(Icell_graft(c2,Tag_str "2")) => 
		Ival_output(Ival_valof(expandCell(c2,env)))
	  | Ival_valof(Icell_graft(c2,Tag_arexpr(Arexpr_int 2))) =>
		Ival_output(Ival_valof(expandCell(c2,env)))
	  | Ival_output v1 => Ival_output(Ival_output(expandVal(v1,env)))
	  | _ => raise AiError "curry: value not in list"
	end
      | eval (forest_uncurry f, c, env) =
	let val x = getForest c
	    val cname = getName c
	    val v = eval(f, Icell_fun(proj("1",x),
				      (Icell_fun(proj("2",x),cname))), env)
	in case v of
	    Ival_omega => Ival_omega
	  | Ival_valof c1 => 
		Ival_valof(Icell_graft(expandCell(c1,env),
				       Tag_arexpr(Arexpr_int 1)))
	  | Ival_output(Ival_valof c2) => 
		Ival_valof(Icell_graft(expandCell(c2,env),
				       Tag_arexpr(Arexpr_int 2)))
	  | Ival_output(Ival_output v1) => Ival_output(expandVal(v1,env))
	  | _ => raise AiError "uncurry: value not in list"
	end
      | eval (forest_pair flist, c, env) =
	let val x = getForest c
	    val cname = getName c
	    val (ci,i) = case cname of
		Icell_graft(ci, Tag_str s) => (ci, ord s - ord "0")
	      | Icell_graft(ci, Tag_arexpr(Arexpr_int m)) => (ci, m)
	      | _ => raise AiError "pair: name not correct graft"
	    val fi = ith(i, flist) handle Ith 
		=> raise AiError "pair: index out of bounds"
	in eval(fi, Icell_fun(x,ci), env)
	end
      | eval (forest_prod flist, c, env) =
	let val (ci,i) = case c of
	    Icell_graft(ci, Tag_str s) => (ci, ord s - ord "0")
	  | Icell_graft(ci, Tag_arexpr(Arexpr_int m)) => (ci, m)
	  | _ => raise AiError "product: name not correct graft"
	    val fi = ith(i, flist) handle Ith 
		=> raise AiError "product: index out of bounds"
	in eval(fi, ci, env)
	end


    and checkInVerify (f,c) =
	let val sc = simplifyCell c
	in if member ((f, sc), !fixList) then true
	   else let val _ = (fixList := (f,sc)::(!fixList);
			     fixCounter := (!fixCounter) + 1)
		in false
		end
	end

    and checkOut (f,c) = (fixList := tl (!fixList);
			  fixCounter := (!fixCounter) - 1)

    and simplifyCell (Icell_fun(f,ic)) = Icell_fun(simplifyForest f, simplifyCell ic)
      | simplifyCell ic = ic

    and simplifyForest (f as (forest_apply(f1, f2))) =
	let val pi2 = find("snd",!exprList)
	in if f1 = pi2
	       then (case f2 of
			 (forest_prod [f21, f22]) => simplifyForest f22
		       | _ => f)
	   else f
	end
      | simplifyForest f = f

    end
    end;
