(* print.sml
   Makes strings out of cell names, values, and states *)


signature PRINTER =
    sig
    val printval : CDSInternal.ivalue -> string
    val printcell : CDSInternal.icell -> string
    val printforest : CDSInternal.forest -> string
    val printenv : (string * 'a) list -> string
    val printenviro : CDSInternal.env -> string
    val print : string -> unit
    val unparseVal : CDSBasic.value -> string
    val unparseValList : CDSBasic.value list -> string
    val unparseCell : CDSBasic.cell -> string
    val unparseAccess : CDSBasic.access list -> string
    val unparseExpr : CDSBasic.expr -> string
    val printList : string list -> unit
    val printHierarchy : outstream * (string * (string * CDSEnv.subkind) list * 
				      (string * CDSEnv.subkind) list) list
	-> unit
    val printType : CDSInternal.typeExp -> string
    val printGType : CDSInternal.TYPE -> string
    val printConstraint : 
	(CDSInternal.typeExp * CDSInternal.typeConstraint) list ->
	(CDSInternal.typeExp * CDSInternal.typeConstraint) list -> 
	(string * string)
    end;
    
functor PrinterFUN () : PRINTER =
    struct
    local open CDSBasic
	  open CDSInternal
	  open CDSEnv
    in
    val cell = ref false

    fun space 1 = " "
      | space n = " "^space(n-1)

    fun findForest f [] = raise Find
      | findForest f ((s,f')::rest) = if f=f' then s else findForest f rest

    fun printArexpr (Arexpr_int i) = makestring i
      | printArexpr (Arexpr_var s) = s
      | printArexpr _ = "arexpr"

    fun printval (Ival_string s) = s
      | printval (Ival_output v) = "output "^(printval v)
      | printval (Ival_valof c) = "valof "^(printcell c)
      | printval (Ival_arexpr a) = printArexpr a
      | printval (Ival_omega) = ""
      | printval (Ival_with(s,b)) = s^" with boolexp"
      | printval (Ival_pair(v1,v2)) = "("^(printval v1)^"."^(printval v2)^")"

    and printcell (Icell_name s) = s
      | printcell (Icell_var s) = s
      | printcell (Icell_fun(f,c)) = 
	  let val _ = (cell := true)
	      val result = (printforest f)^(printcell c)
	      val _ = (cell := false)
	  in result
	  end
      | printcell (Icell_graft(c,t)) = 
	  (case t of
	       (Tag_str s) => "("^(printcell c)^"."^s^")"
	     | (Tag_arexpr a) => "("^(printcell c)^"."^(printArexpr a)^")"
	     | (Tag_interval i) => "interval")
      | printcell (Icell_with(c,b)) = (printcell c)^" with boolexp"

    and printforest f = 
	(findForest f (!exprList)) handle Find => printforest' f

    and printforest' (forest_basic(_,tlist)) = "{"^(printf tlist)^"}"
      | printforest' (forest_apply(f1,f2)) = 
	                       (printforest f1)^" . "^(printforest f2)
      | printforest' (forest_comp(f1,f2)) = 
	                       "("^(printforest f1)^" | "^(printforest f2)^")"
      | printforest' (forest_fix f) = "fix ("^(printforest f)^")"
      | printforest' (forest_curry f) = "curry ("^(printforest f)^")"
      | printforest' (forest_uncurry f) = "uncurry ("^(printforest f)^")"
      | printforest' (forest_pair flist) = "<"^(printflist flist)^">"
      | printforest' (forest_prod flist) = 
	  "("^(printflist flist)^")"

    and printflist [] = ""
      | printflist [f] = printforest f
      | printflist (f::l) = (printforest f)^", "^(printflist l)

    and printf [] = ""
      | printf [(c,t)] = (printcell c)^"="^(printtree(4,t))
      | printf [(c1,t1),(c2,t2)] = (printcell c1)^"="^(printtree(4,t1))^
	     ", "^(printcell c2)^"="^(printtree(4,t2))
      | printf ((c,t)::l) = (printcell c)^"="^(printtree(4,t))^", "^
	     (printf l)

    and printtree (n,tree_Valof(c,i,tqlist)) = 
	     "valof ("^(printcell c)^", "^(makestring i)^") is\n"^
	     implode(map (printpair (n+2)) tqlist)
      | printtree (n,tree_From(c,i,tqlist)) = "from ("^(printcell c)^", "^
	     (makestring i)^") is\n"^implode(map (printpair (n+2)) tqlist)
      | printtree (n,tree_Result(i,v)) = 
	     printval(makeOutput(i,v))^(if !cell then "" else "\n")

    and printpair n (v, ins) = (space n)^(printval v)^" : "^printtree(n+2,ins)

    fun printenv [] = ""
      | printenv [(a,_)] = a
      | printenv ((a,_)::l) = a^", "^(printenv l)

    fun printGenList ([], f) = ""
      | printGenList ([(s,v)], f) = s^" |--> "^(f v)
      | printGenList (((s,v)::l), f) = s^" |--> "^(f v)^", "^
	  (printGenList(l,f))

    fun printenviro env = "cellEnv = {"^
	(printGenList(getCellList env,printcell))^"}, valEnv = {"^
	(printGenList(getValList env, printval))^"}"

    fun print s = output(std_out, s^" =\n"^(printforest'(lookup(s,exprList)))^"\n")

	(* Next few functions print out parts of the parse tree. *)
    fun unparseVal (Val_string s) = s
      | unparseVal (Val_output v) = "output "^(unparseVal v)
      | unparseVal (Val_valof c) = "valof "^(unparseCell c)
      | unparseVal (Val_arexpr a) = printArexpr a
      | unparseVal (Val_omega) = ""
      | unparseVal (Val_with(s,b)) = s^" with boolexp"
      | unparseVal (Val_pair(v1,v2)) = 
	  "("^(unparseVal v1)^"."^(unparseVal v2)^")"
      | unparseVal (Val_interval_inf) = "[..]"
      | unparseVal (Val_interval_from i) = "["^(makestring i)^"..]"
      | unparseVal (Val_interval_to i) = "[.."^(makestring i)^"]"
      | unparseVal (Val_interval_fromto(i1,i2)) =
	  "["^(makestring i1)^".."^(makestring i2)^"]"

    and unparseCell (Cell_name s) = s
      | unparseCell (Cell_var s) = s
      | unparseCell (Cell_fun(e,c)) = 
	  "{"^(unparseExpr e)^"}"^(unparseCell c)
      | unparseCell (Cell_graft(c,t)) = 
	  (case t of
	       (Tag_str s) => "("^(unparseCell c)^"."^s^")"
	     | (Tag_arexpr a) => "("^(unparseCell c)^"."^(printArexpr a)^")"
	     | (Tag_interval v) => "("^(unparseCell c)^"."^(unparseVal v)^")")
      | unparseCell (Cell_with(c,b)) = (unparseCell c)^" with boolexp"

    and unparseExpr (Expr_state x) = unparseState x
      | unparseExpr (Expr_algo a) = "algo"
      | unparseExpr (Expr_curry e) = "curry ("^(unparseExpr e)^")"
      | unparseExpr (Expr_uncurry e) = "uncurry ("^(unparseExpr e)^")"
      | unparseExpr (Expr_compose(e1,e2)) = (unparseExpr e1)^" | "^(unparseExpr e2)
      | unparseExpr (Expr_apply(e1,e2)) = (unparseExpr e1)^" . "^(unparseExpr e2)
      | unparseExpr (Expr_pair(e1,e2)) = "< "^(unparseExpr e1)^", "^(unparseExpr e2)^" >"
      | unparseExpr (Expr_prod(e1,e2)) = "( "^(unparseExpr e1)^", "^(unparseExpr e2)^" )"
      | unparseExpr (Expr_fix e) = "fix ("^(unparseExpr e)^")"
      | unparseExpr (Expr_id s) = s

    and unparseState [] = ""
      | unparseState [(c,v)] = (unparseCell c)^"="^(unparseVal v)
      | unparseState ((c,v)::x) = 
	  (unparseCell c)^"="^(unparseVal v)^", "^(unparseState x)

    fun unparseAccess [] = ""
      | unparseAccess [a] = unparseState a
      | unparseAccess (a::alist) = 
	  (unparseState a)^" or "^(unparseAccess alist)

    fun unparseValList [] = ""
      | unparseValList [v] = unparseVal v
      | unparseValList (v::vs) = (unparseVal v)^", "^(unparseValList vs)

    fun printList [] = output(std_out, "{}\n")
      | printList l = 
	  let fun printListRec [] = ""
		| printListRec [s] = s
		| printListRec (s::rest) = s^", "^(printListRec rest)
	  in
	      output(std_out, "{"^(printListRec l)^"}\n")
	  end

    fun printHierarchy (file, []) = ()
      | printHierarchy (file, (M,Mup,Mdown)::l) =
	  let fun printKind (s, ext) = s^" (e) "
		| printKind (s, partof) = s^" (p) "
		| printKind (s, notsubtype) = s^" (not) "
	      val children = map printKind Mdown
	  in (output(file,M^" "^(implode children)^";\n");
	      printHierarchy(file, l))
	  end

        (* Prints poly types as alpha23, rather then 'a *)
    fun printTypeDebug (Dcds s) = s
      | printTypeDebug (Alpha i) = "alpha"^(makestring i)
      | printTypeDebug (Arrow(t1,t2)) = 
	  let val t1string = case t1 of
	      (Arrow(_,_)) => "("^(printTypeDebug t1)^")"
	    | _ => printTypeDebug t1
	  in t1string^" -> "^(printTypeDebug t2)
	  end
      | printTypeDebug (And tlist) = "("^(printTypeDebugList " & " tlist)^")"
      | printTypeDebug (Meet tlist) = 
	  "/\\["^(printTypeDebugList ", " tlist)^"]"
      | printTypeDebug (Prod tlist) = "("^(printTypeDebugList " * " tlist)^")"

    and printTypeDebugList s [] = ""
      | printTypeDebugList s [t] = printTypeDebug t
      | printTypeDebugList s (t::tlist) = (printTypeDebug t)^s^
	                                  (printTypeDebugList s tlist)

        (* Keeps track of how many alphas are currently defined. *)
    val substCounter = ref 0

    exception undefinedAlpha of int

    fun emptyPrintSubst t = raise undefinedAlpha (!substCounter)

	(* Adds a mapping of the form i -> "'a".  The representation for *)
        (* i is "'"^(chr (value + 97)), where chr 97 = "a". *)
    fun addToSub newAlpha value f = 
	let val _ = substCounter := !substCounter + 1
	in fn t => if t=newAlpha then "'"^(chr (value + 97)) else f t
	end

        (* Keep track of enclosing type to print types clearly, e.g. *)
        (* (int -> int) * bool, rather than (int -> int * bool) *)
    datatype enclosingType = NONE
                           | ENCL of typeExp

    fun printTypeExp (Dcds s) _ f = (s,f)
      | printTypeExp (Alpha i) _ f = ((f i, f) handle undefinedAlpha num =>
	    let val f' = addToSub i num f
	    in (f' i, f')
	    end)
      | printTypeExp (Arrow(t1,t2)) et f = 
	let val (t1str, f1) = printTypeExp t1 (ENCL (Arrow(t1,t2))) f
	    val (t2str, f2) = printTypeExp t2 (ENCL (Arrow(t1,t2))) f1
	    val t1str' = case t1 of
		(Arrow(_,_)) => "("^t1str^")"
	      | _ => t1str
	in case et of
	    NONE => (t1str'^" -> "^t2str, f2)
	  | ENCL enclType => case enclType of
		Prod _ => ("("^t1str'^" -> "^t2str^")", f2)
	      | _ => (t1str'^" -> "^t2str, f2)
	end
      | printTypeExp (And tlist) et f = 
	let val (strlist, f') = printTypeExpList " & " tlist f
	in case et of
	    NONE => (strlist, f')
	  | _ => ("("^strlist^")", f')
	end
      | printTypeExp (Meet tlist) _ f = 
	let val (strlist, f') = printTypeExpList ", " tlist f
	in ("/\\["^strlist^"]", f')
	end
      | printTypeExp (Prod tlist) et f = 
	let val (strlist, f') = printTypeExpList " * " tlist f
	in case et of
	    NONE => (strlist, f')
	  | _ => ("("^strlist^")", f')
	end

    and printTypeExpList s [] f = ("", f)
      | printTypeExpList s [t] f = 
	(case s of
	     " * " => printTypeExp t (ENCL (Prod [])) f
	   | " & " => printTypeExp t (ENCL (And [])) f
	   | _ => printTypeExp t NONE f)
      | printTypeExpList s (t::tlist) f = 
	let val (tstr, f') = case s of
	    " * " => printTypeExp t (ENCL (Prod [])) f
	  | " & " => printTypeExp t (ENCL (And [])) f
	  | _ => printTypeExp t NONE f
	    val (tliststr, f'') = printTypeExpList s tlist f'
	in (tstr^s^tliststr, f'')
	end

    fun printType t = 
	if !trace then printTypeDebug t
	else let val _ = substCounter := 0
		 val (s, _) = printTypeExp t NONE emptyPrintSubst
	     in s
	     end

    fun printGType (UNTYPED) = "no type"
      | printGType (TYPE t) = printType t

    fun printConstraint constr1 constr2 =
	let fun printConstr [] f = ("", f)
	      | printConstr ((a,tcon)::rest) f =
		let val (astr, f') = printTypeExp a NONE f
		    val (tconstr,f'') = 
			(case tcon of
			     (Lub tlist) => let val (tstr, newf) = 
				                printTypeExpList ", " tlist f'
					    in (" < ["^tstr^"]", newf)
					    end
			   | (Glb tlist) => let val (tstr, newf) = 
				                printTypeExpList ", " tlist f'
					    in (" > ["^tstr^"]", newf)
					    end)
		    val (otherstr, f''') = printConstr rest f''
		in (astr^tconstr^", "^otherstr, f''')
		end
	    val _ = substCounter := 0
	    val (s1, newf) = printConstr constr1 emptyPrintSubst
	    val (s2, _) = printConstr constr2 newf
	in (s1, s2)
	end


    end
    end;
