(* Refinement type version of pcfenv.cds *)

(* Constants that are part of the PCF environment *)
(* Automatically loaded in when user switches to  *)
(* PCF interpreter. *)

(* The basic types *)

let bool = dcds cell B values tt,ff end;
let int = dcds cell N values [..] end;

(* The refinement types *)

let true = dcds cell B values tt end;
let false = dcds cell B values ff end;
refine true, false;

(* The primitive operations *)

(* cond : ((bool * 'a) * 'a) -> 'a  *)
let cond = 
  algo
    request $C do
      valof ((B.1).1) is
        tt: valof (($C.2).1) is
              $V: output $V
            end
        ff: valof ($C.2) is
              $W: output $W
            end
      end
    end
  end;

(* fst : ('a * 'b) -> 'a  *)
let fst = algo
    request $C do
    	valof ($C.1) is
	    $V: output $V
    	end
    end
end;

(* snd : ('a * 'b) -> 'a  *)
let snd = algo
    request $C do
    	valof ($C.2) is
	    $W: output $W
    	end
    end
end;

(* plus : (int * int) -> int  *)
let plus = algo
    request N do
	valof (N.1) is
	    $V1: valof (N.2) is
		  $V2: output $V1 + $V2
		 end
	end
    end
end;

let minus = algo
    request N do
	valof (N.1) is
	    $V1: valof (N.2) is
		  $V2: output $V1 - $V2
		 end
	end
    end
end;

let times = algo
    request N do
	valof (N.1) is
	    $V1: valof (N.2) is
		  $V2: output $V1 * $V2
		 end
	end
    end
end;

let div = algo
    request N do
	valof (N.1) is
	    $V1: valof (N.2) is
		  $V2: output $V1 / $V2
		 end
	end
    end
end;

(* equal : (int * int) -> bool  *)
let equal = algo
    request B do
        valof (N.1) is
            $V1: valof (N.2) is
                     $V2 with $V2 = $V1: output tt
                     $V2 with $V2 != $V1: output ff
                 end
        end
    end
end;

let less = algo
    request B do
        valof (N.1) is
            $V1: valof (N.2) is
                     $V2 with $V2 > $V1: output tt
                     $V2 with $V2 <= $V1: output ff
                 end
        end
    end
end;

let grtr = algo
    request B do
        valof (N.1) is
            $V1: valof (N.2) is
                     $V2 with $V2 < $V1: output tt
                     $V2 with $V2 >= $V1: output ff
                 end
        end
    end
end;

let leq = algo
    request B do
        valof (N.1) is
            $V1: valof (N.2) is
                     $V2 with $V2 >= $V1: output tt
                     $V2 with $V2 < $V1: output ff
                 end
        end
    end
end;

let geq = algo
    request B do
        valof (N.1) is
            $V1: valof (N.2) is
                     $V2 with $V2 <= $V1: output tt
                     $V2 with $V2 > $V1: output ff
                 end
        end
    end
end;

(* and : (bool * bool) -> bool  *)
let land =
  algo
    request B do
      valof (B.1) is
        tt: valof (B.2) is
              tt: output tt
              ff: output ff
            end
        ff: output ff
      end
    end
  end;

let lor =
  algo
    request B do
      valof (B.1) is
        tt: output tt
	ff: valof (B.2) is
              tt: output tt
              ff: output ff
            end
      end
    end
  end;


(* Now things that are not explicitly in the language but are needed *)
(* in the translation to categorical combinators. *)

(* id : 'a -> 'a  *)
let id = algo
    request $C do
   	valof $C is
	    $V : output $V
	end
    end
end;

(* The empty environment *)
let emptyenv = {};

(* the "regular" fixpoint, Y : ('a -> 'a) -> 'a  *)
(*  Y = fix (fn f => fn x => x (f x))            *)
let Y = fix((curry(curry(uncurry(id) | 
			<snd, uncurry(id) | <snd|fst, snd>>))).emptyenv);

(* the "environment" fixpoint, Yenv : (env -> 'a -> 'a) -> env -> 'a  *)
let Yenv = curry(Y | uncurry(id));


(* Integer lists *)

letrec intlist = dcds 
  cell EMPTY values true, false
  graft (int.l) access EMPTY = false
  graft (intlist.l) access EMPTY=false
end;

(* refined types *)

let empty_intlist = dcds
  cell EMPTY values true
end;

let one_intlist = dcds
  cell EMPTY values false
  cell (N.l) values [..] access EMPTY = false
  cell (EMPTY.l) values true access EMPTY = false
end;

local letrec partial_intlist = dcds
  cell (EMPTY.l) values true, false access EMPTY = false
  cell (N.l) values [..] access EMPTY = false
  graft (partial_intlist.l) access EMPTY = false
  end
in let many_intlist = dcds
  cell EMPTY values false
  cell (N.l) values [..] access EMPTY = false
  cell (EMPTY.l) values false access EMPTY = false
  cell ((N.l).l) values [..] access (EMPTY.l) = false
  graft (partial_intlist.l)
  end
end;

refine empty_intlist, one_intlist, many_intlist;


let nil = {EMPTY = true};

let null = algo
  request B do
    valof EMPTY is
      true : output tt
      false : output ff
    end
  end
end;

let cons = algo
  request EMPTY do
    output false
  end
  request (N.l) do
    valof (N.1) is 
      $V : output $V
    end
  end
  request (EMPTY.l) do
    valof (EMPTY.2) is
      $B : output $B
    end
  end
  request (((EMPTY.$T).l).l) do
    from {((EMPTY.$T).2)=false} do
      valof (((EMPTY.$T).l).2) is
        $B : output $B
      end
    end
  end
  request (((N.$T).l).l) do
    from {((EMPTY.$T).2)=false} do
      valof (((N.$T).l).2) is
	$V : output $V
      end
    end
  end
end;


let hd = algo
  request N do
    valof EMPTY is
      false : valof (N.l) is
                $V : output $V
	      end
    end
  end
end;


let tl = algo
  request (EMPTY.$T) do
    from {(EMPTY.$T)=false} do
      valof ((EMPTY.$T).l) is
        $B : output $B
      end
    end
  end
  request ((N.$T).l) do
    from {(EMPTY.$T)=false, ((EMPTY.$T).l)=false} do
      valof (((N.$T).l).l) is
        $V : output $V
      end
    end
  end
end;
