(* CDS Lexer Definition *)

structure Tokens = Tokens
structure Interface = Interface
open Interface

type pos = Interface.pos
type svalue = Tokens.svalue
type ('a,'b) token = ('a,'b) Tokens.token
type lexresult= (svalue,pos) token

(* Keep track of nested comments.  A rather ugly way of doing it. *)
val comment = ref false
val nesting = ref 0

val eof = fn () => if !comment = true 
                      then (comment := false;
                            raise CommentError "unclosed comment")
                   else Tokens.EOF(!line,!line)

%%
%header (functor PCFLexFun(structure Tokens: PCF_TOKENS
			    structure Interface : INTERFACE) : LEXER);
alpha=[a-zA-Z];
extAlpha=[A-Za-z0-9_];
fileName=[^\"\n];
digit=[0-9];
ws = [\ \t];
%%
\n       => (next_line(); lex());
{ws}+    => (lex());
{digit}+ => (if !comment = true then lex() 
             else Tokens.INTEGER
                (revfold (fn (a,r) => ord(a)-ord("0")+10*r)
                         (explode yytext) 0,
                  !line,!line));

"(*"       => ((if !comment = true then nesting := !nesting + 1
                else comment := true); lex());
"*)"       => ((if !nesting = 0 andalso !comment = false
                   then raise CommentError "unmatched close comment"
                else if !nesting = 0 then comment := false
                else nesting := !nesting - 1); lex());

"end"      => (if !comment = true then lex() else Tokens.END(!line,!line));
"or"       => (if !comment = true then lex() else Tokens.OR(!line,!line));
"and"	   => (if !comment = true then lex() else Tokens.AND(!line,!line));
"let"      => (if !comment = true then lex() else Tokens.LET(!line,!line));
"letrec"   => (if !comment = true then lex() else Tokens.LETREC(!line,!line));

"load"   => (if !comment = true then lex() else Tokens.LOAD(!line,!line));
"print"	 => (if !comment = true then lex() else Tokens.PRINT(!line,!line));
"quit"	 => (if !comment = true then lex() else Tokens.QUIT(!line,!line));

"fn"	 => (if !comment = true then lex() else Tokens.FN(!line,!line));
"val"	 => (if !comment = true then lex() else Tokens.VAL(!line,!line));
"in"	 => (if !comment = true then lex() else Tokens.IN(!line,!line));
"if"	 => (if !comment = true then lex() else Tokens.IF(!line,!line));
"then"	 => (if !comment = true then lex() else Tokens.THEN(!line,!line));
"else"	 => (if !comment = true then lex() else Tokens.ELSE(!line,!line));
"fst"	 => (if !comment = true then lex() else Tokens.FST(!line,!line));
"snd"	 => (if !comment = true then lex() else Tokens.SND(!line,!line));
"true"	 => (if !comment = true then lex() else Tokens.TRUE(!line,!line));
"false"	 => (if !comment = true then lex() else Tokens.FALSE(!line,!line));
"=>"     => (if !comment = true then lex() else Tokens.MAPSTO(!line,!line));
"<="     => (if !comment = true then lex() else Tokens.LEQ(!line,!line));
">="     => (if !comment = true then lex() else Tokens.GEQ(!line,!line));

"::"     => (if !comment = true then lex() else Tokens.CONS(!line,!line));
"hd"     => (if !comment = true then lex() else Tokens.HD(!line,!line));
"tl"     => (if !comment = true then lex() else Tokens.TL(!line,!line));
"[]"     => (if !comment = true then lex() else Tokens.NIL(!line,!line));
"null"   => (if !comment = true then lex() else Tokens.NULL(!line,!line));

","      => (if !comment = true then lex() else Tokens.COMMA(!line,!line));
"="      => (if !comment = true then lex() else Tokens.EQ(!line,!line));
";"      => (if !comment = true then lex() else Tokens.SEMI(!line,!line));
"("	 => (if !comment = true then lex() else Tokens.LB(!line,!line));
")"	 => (if !comment = true then lex() else Tokens.RB(!line,!line));
"<"	 => (if !comment = true then lex() else Tokens.LA(!line,!line));
">"	 => (if !comment = true then lex() else Tokens.RA(!line,!line));
"+"	 => (if !comment = true then lex() else Tokens.PLUS(!line,!line));
"-"	 => (if !comment = true then lex() else Tokens.SUB(!line,!line));
"*"	 => (if !comment = true then lex() else Tokens.MULT(!line,!line));
"/"	 => (if !comment = true then lex() else Tokens.DIV(!line,!line));
"~"	 => (if !comment = true then lex() else Tokens.MINUS(!line,!line));

{alpha}{extAlpha}*    => (if !comment = true then lex() 
                          else Tokens.ID(yytext,!line,!line));
\"{fileName}*\" => (if !comment = true then lex() 
                    else Tokens.FILE
		(implode(tl(CDSBasic.take((size yytext)-1, explode yytext))),
		   !line,!line));

.	=> (if !comment = true then lex() else raise LexError);
