(* match.sml
   Contains routines implementing variables/"pattern matching." *)


signature MATCH =
    sig
    exception MatchError of string
    val expandVal : CDSInternal.ivalue * CDSInternal.env -> CDSInternal.ivalue
    val expandCell : CDSInternal.icell * CDSInternal.env -> CDSInternal.icell
    val findCellMatch : CDSInternal.icell * CDSInternal.tree list * 
	CDSInternal.env -> CDSInternal.tree_instruction * CDSInternal.env
    val findValMatch : CDSInternal.ivalue * CDSInternal.tree_query list * 
	CDSInternal.env -> CDSInternal.tree_instruction * CDSInternal.env
    end;
    

functor MatchFUN () : MATCH =
    struct
    exception MatchError of string

    local open CDSBasic
          open CDSInternal
	  open CDSEnv
    in

    datatype tagChoice = TAG of tag
                       | FUN of icell -> icell

    fun id x = x

	(* Check whether 2 cell names match each other.  Doesn't do  *)
	(* unification: only one cell name can contain variables.    *)
	(* Modifies the environment if the match involves variables. *)
	(* All other "match..." functions work the same way.         *)
    fun matchCell (c1, c2, env) =
	  if c1 = c2 then (true, env)
	  else case c1 of
	      (Icell_name s1) => 
		  (case c2 of 
		       (Icell_var s2) => (true, bindCell(s2,c1,env))
		     | (Icell_with(c,b)) => satisfyCell(c1,c,b,env)
		     | (Icell_graft(c,t)) => (case t of
			   (Tag_arexpr(Arexpr_var s2)) =>
			       let val (yes,newEnv) = matchCell(c1,c,env) 
			       in if yes then (true, bindTag(s2,id,newEnv))
				  else (false,env)
			       end
			 | _ => (false,env))
		     | _ => (false, env))
	    | (Icell_var s1) => 
		  (case c2 of
		       (Icell_name s2) => (true, bindCell(s1,c2,env))
		     | (Icell_var s2) => raise MatchError
			   "matchCell: both cells are vars"
		     | (Icell_with(c,b)) => raise MatchError
			   "matchCell: both cells are vars"
		     | _ => (true, bindCell(s1,c2,env)))
	    | (Icell_fun(f1,c11)) => 
		  (case c2 of
		       (Icell_fun(f2,c22)) => 
			   (case (matchForest(f1,f2,env)) of
				(true, newEnv) => matchCell(c11,c22,newEnv)
			      | (false, _) => (false,env))
		     | (Icell_var s2) => (true, bindCell(s2,c1,env))
		     | (Icell_with(c,b)) => satisfyCell(c1,c,b,env)
		     | (Icell_graft(c,t)) => (case t of
			   (Tag_arexpr(Arexpr_var s2)) =>
			       let val (yes,newEnv) = 
				   matchCell(Icell_fun(f1,c11),c,env) 
			       in if yes then (true, bindTag(s2,id,newEnv))
				  else (false,env)
			       end
			 | _ => (false,env))
		     | _ => (false,env))
	    | (Icell_graft(c11,t1)) =>
		  (case c2 of
		       (Icell_var s2) => (true, bindCell(s2,c1,env))
		     | (Icell_graft(c22,t2)) => 
			   (case t2 of
				(Tag_arexpr(Arexpr_var v)) =>
				    longTag(Icell_graft(c11,t1),c22,v,id,env,1)
			      | _ => (case (matchTag(t1,t2,env)) of
				   (true, newEnv) => matchCell(c11,c22,newEnv)
				 | (false, _) => (false, env)))
		     | (Icell_with(c,b)) => satisfyCell(c1,c,b,env)
		     | _ => (false, env))
	    | (Icell_with(c11,b)) =>
		  (case c2 of
		       (Icell_var s2) => raise MatchError
			   "matchCell: both cells are vars"
		     | (Icell_with(c22,b2)) => raise MatchError
			   "matchCell: both cells are vars"
		     | _ => satisfyCell(c2,c11,b,env))


    and matchForest (forest_basic(_,[]), forest_basic(_,[]), env) = (true, env)
      | matchForest (forest_basic(_,(c,ins)::l), forest_basic(_,[]), env) = 
	  (false, env)
      | matchForest (forest_basic(_,[]), forest_basic(_,(c,ins)::l), env) = 
	  (false, env)
      | matchForest (forest_basic (i1, (c1,ins1)::l1), 
		     forest_basic (i2, (c2,ins2)::l2), env) = 
	  let val (result1, newEnv1) = matchCell(c1,c2,env)
	  in if result1 = false 
		 then (false, env)
	     else let val (result2, newEnv2) = matchIns(ins1,ins2,newEnv1)
		  in if result2 = false
			 then (false, env)
		     else matchForest(forest_basic(i1,l1),
				      forest_basic(i2,l2), newEnv2)
		  end
	  end
      | matchForest (_, _, _) = raise MatchError 
	                                  "matchForest: non-basic forest"

    and matchIns (ins1, ins2, env) =
	  if ins1 = ins2 then (true, env)
	  else case ins1 of 
	      (tree_Valof(c1,i1,tqlist1)) =>
		  (case ins2 of
		       tree_Valof(c2,i2,tqlist2) =>
			   if i1 <> i2 then (false, env)
			   else let val (result1,newEnv)=matchCell(c1,c2,env)
				in if result1 = false then (false, env)
				   else let val (result2,newEnv2) = 
				           matchTqlist(tqlist1,tqlist2,newEnv)
					in if result2 then (true, newEnv2)
					   else (false,env)
					end
				end
		     | _ => (false, env))
	    | (tree_From(c1,i1,tqlist1)) => 
		  (case ins2 of
		       tree_From(c2,i2,tqlist2) =>
			   if i1 <> i2 then (false, env)
			   else let val (result1,newEnv)=matchCell(c1,c2,env)
				in if result1 = false then (false, env)
				   else let val (result2,newEnv2) = 
				           matchTqlist(tqlist1,tqlist2,newEnv)
					in if result2 then (true, newEnv2)
					   else (false,env)
					end
				end
		     | _ => (false, env))
	    | (tree_Result(i1,v1)) =>
		  (case ins2 of
		       tree_Result(i2,v2) =>
			   if i1 <> i2 then (false, env)
			   else let val (result1,newEnv)=matchVal(v1,v2,env)
				in if result1 = false then (false, env)
				   else (true, newEnv)
				end
		     | _ => (false, env))


    and matchTqlist ([],[],env) = (true, env)
      | matchTqlist ([],tqlist2,env) = (false,env)
      | matchTqlist (tqlist1,[],env) = (false,env)
      | matchTqlist ((v1,ins1)::tqlist1,(v2,ins2)::tqlist2,env) =
	  let val (result1,newEnv1) = matchVal(v1,v2,env)
	  in if result1
		 then let val (result2,newEnv2) = matchIns(ins1,ins2,newEnv1)
		      in if result2
			     then matchTqlist(tqlist1,tqlist2,newEnv2)
			 else (false,env)
		      end
	     else (false,env)
	  end

		       
    and matchVal (v1,v2,env) =
	  if v1 = v2 then (true, env)
	  else case v1 of
	      (Ival_string s1) =>
		  (case v2 of
		       (Ival_arexpr(Arexpr_var v)) => (true, bindVal(v,v1,env))
		     | (Ival_with(s2,b2)) => satisfyVal(v1,s2,b2,env)
		     | _ => (false, env))
	    | (Ival_output v11) =>
		  (case v2 of
		       (Ival_arexpr(Arexpr_var v)) => (true, bindVal(v,v1,env))
		     | (Ival_with(s2,b2)) => satisfyVal(v1,s2,b2,env)
		     | (Ival_output v22) => matchVal(v11,v22,env)
		     | _ => (false, env))
	    | (Ival_valof c1) => 
		  (case v2 of
		       (Ival_arexpr(Arexpr_var v)) => (true, bindVal(v,v1,env))
		     | (Ival_with(s2,b2)) => satisfyVal(v1,s2,b2,env)
		     | (Ival_valof c2) => matchCell(c1,c2,env)
		     | _ => (false, env))
	    | (Ival_arexpr a1) => 
		  (case v2 of
		       (Ival_arexpr(Arexpr_var v)) => (true, bindVal(v,v1,env))
		     | (Ival_with(s2,b2)) => satisfyVal(v1,s2,b2,env)
		     | (Ival_arexpr a2) => matchArexpr(a1,a2,env)
		     | _ => (false, env))
	    | (Ival_omega) => 
		  (case v2 of
		       (Ival_arexpr(Arexpr_var v)) => (true, bindVal(v,v1,env))
		     | (Ival_with(s2,b2)) => satisfyVal(v1,s2,b2,env)
		     | _ => (false, env))
	    | (Ival_with(s1,b1)) => 
		  (case v2 of
		       (Ival_arexpr(Arexpr_var v)) =>
			   raise MatchError "matchVal: both vals are vars"
		     | (Ival_with(s2,b2)) => 
			   raise MatchError "matchVal: both vals are vars"
		     | _ => satisfyVal(v2,s1,b1,env))
	    | (Ival_pair(v11,v12)) =>
		  (case v2 of
		       (Ival_arexpr(Arexpr_var v)) => (true, bindVal(v,v1,env))
		     | (Ival_with(s2,b2)) => satisfyVal(v1,s2,b2,env)
		     | (Ival_pair(v21,v22)) => 
			   (case (matchVal(v11,v21,env)) of
				(true, newEnv) => matchVal(v12,v22,newEnv)
			      | _ => (false,env))
		     | _ => (false, env))


	(* Assume for now that a1 and a2 are in simplified form, that *)
	(* is, all possible simplifications have been performed.      *)
    and matchArexpr (a1, a2, env) =
	  if a1 = a2 then (true, env)
	  else case a1 of
	      (Arexpr_int i1) =>
		  (case a2 of
		       (Arexpr_var s2) => (true,bindVal(s2,Ival_arexpr a1,env))
		     | _ => (false, env))
	    | (Arexpr_var s1) => 
		  (case a2 of
		       (Arexpr_var s2) => raise MatchError 
			   "matchArexpr: both arxeprs are vars"
		     | _ => (true, bindVal(s1,Ival_arexpr a2,env)))
	    | (Arexpr_minus a1) => 
		  (case a2 of
		       (Arexpr_var s2) => (true,bindVal(s2,Ival_arexpr a1,env))
		     | _ => (false, env))
	    | (Arexpr_plus(a11,a12)) => 
		  (case a2 of
		       (Arexpr_var s2) => (true,bindVal(s2,Ival_arexpr a1,env))
		     | (Arexpr_plus(a21,a22)) =>
			   let val (result,newEnv) = matchArexpr(a11,a21,env)
			   in if result = true 
				  then matchArexpr(a12,a22,newEnv)
			      else (false,env)
			   end
		     | _ => (false, env))
	    | (Arexpr_sub(a11,a12)) => 
		  (case a2 of
		       (Arexpr_var s2) => (true,bindVal(s2,Ival_arexpr a1,env))
		     | (Arexpr_sub(a21,a22)) =>
			   let val (result,newEnv) = matchArexpr(a11,a21,env)
			   in if result = true 
				  then matchArexpr(a12,a22,newEnv)
			      else (false,env)
			   end
		     | _ => (false, env))
	    | (Arexpr_mult(a11,a12)) => 
		  (case a2 of
		       (Arexpr_var s2) => (true,bindVal(s2,Ival_arexpr a1,env))
		     | (Arexpr_mult(a21,a22)) =>
			   let val (result,newEnv) = matchArexpr(a11,a21,env)
			   in if result = true 
				  then matchArexpr(a12,a22,newEnv)
			      else (false,env)
			   end
		     | _ => (false, env))
	    | (Arexpr_div(a11,a12)) => 
		  (case a2 of
		       (Arexpr_var s2) => (true,bindVal(s2,Ival_arexpr a1,env))
		     | (Arexpr_div(a21,a22)) =>
			   let val (result,newEnv) = matchArexpr(a11,a21,env)
			   in if result = true 
				  then matchArexpr(a12,a22,newEnv)
			      else (false,env)
			   end
		     | _ => (false, env))


    and matchTag(t1,t2,env) =
	  if t1 = t2 then (true,env)
	  else case t1 of
	      (Tag_str s1) =>
		  (case t2 of
		       (Tag_arexpr(Arexpr_var s2)) => 
			   (true,bindVal(s2,Ival_string s1,env))
		     | (Tag_arexpr(Arexpr_int i2)) => if s1 = makestring(i2)
							  then (true,env)
						      else (false,env)
		     | _ => (false, env))
	    | (Tag_arexpr a1) => 
		  (case t2 of
		       (Tag_arexpr a2) => matchArexpr(a1,a2,env)
		     | (Tag_str s2) => 
			   (case a1 of
				(Arexpr_int i1) => if makestring(i1) = s2
						       then (true, env)
						   else (false,env)
			      | (Arexpr_var s1) => 
				    (true,bindVal(s1,Ival_string s2,env))
			      | _ => (false,env)))


        (* We have an Icell_graft(c22,var) to match against   *)
	(* Icell_graft(c11,t1).  We want to strip off as many *)
        (* tags from c11 to try to make a match. If we get a  *)
        (* match with only 1 tag, then store as a val and tag *)
        (* else as a tag only, with bindTag. KLUDGE! *)
    and longTag(Icell_graft(c11,t1),c22,v,f,env,level) =
	let val g = fn c => Icell_graft(c,t1)
	in case matchCell(c11,c22,env) of
	    (true,newEnv) => 
		if level = 1
		    then let val env' = bindTag(v,g,newEnv)
			 in matchTag(t1,Tag_arexpr(Arexpr_var v),env')
			 end
		else (true,bindTag(v,g o f,newEnv))
	  | (false,_) => (case c11 of
			      (Icell_graft(c111,t11)) =>
				  longTag(c11,c22,v,g o f,env,level+1)
			    | _ => (false,env))
	end
      | longTag(_,_,_,_,env,_) = (false,env)


	(* Verify that a constraint attached to a cell is satisfied. *)
    and satisfyCell(c,withCell,constraint,env) =
	  let val (result,newEnv) = matchCell(c,withCell,env)
	  in if result = true 
		 then if evalConstraint(constraint,newEnv)
			  then (true,newEnv)
		      else (false,env)
	     else (false,env)
	  end


	(* Verify that a constraint attached to a value is satisfied. *)
    and satisfyVal(value,name,constraint,env) =
	  let val newEnv = bindVal(name,value,env)
	  in if evalConstraint(constraint,newEnv)
		 then (true,newEnv)
	     else (false,env)
	  end


        (* Computes the value of the constraint in an environment. *)
    and evalConstraint(constraint,env) =
	  case constraint of
	      (Iboolexp_gt(a1,a2)) => 
		  evalArexpr(a1,env) > evalArexpr(a2,env)
	    | (Iboolexp_gteq(a1,a2)) => 
		  evalArexpr(a1,env) >= evalArexpr(a2,env)
	    | (Iboolexp_lt(a1,a2)) => 
		  evalArexpr(a1,env) < evalArexpr(a2,env)
	    | (Iboolexp_lteq(a1,a2)) => 
		  evalArexpr(a1,env) <= evalArexpr(a2,env)
	    | (Iboolexp_eq(v1,v2)) => 
		  expandVal(v1,env) = expandVal(v2,env)
	    | (Iboolexp_noteq(v1,v2)) =>
		  expandVal(v1,env) <> expandVal(v2,env)
	    | (Iboolexp_or(b1,b2)) => evalConstraint(b1,env) orelse
		                      evalConstraint(b2,env)
	    | (Iboolexp_and(b1,b2)) => evalConstraint(b1,env) andalso
		                       evalConstraint(b2,env)


	(* Evaluates an arithmetic expression in an environment. *)
    and evalArexpr(axp,env) =
	  case axp of
	      (Arexpr_int i) => i
	    | (Arexpr_var s) => 
		  let val v = (lookupVal(s,env) handle EnvLookup => 
			       raise MatchError "evalArexpr: var not in env")
		      in case v of
			  (Ival_arexpr a) => evalArexpr(a,env)
			| _ => raise MatchError "evalArexpr: var not arexpr"
		  end
	    | (Arexpr_minus a) => op~(evalArexpr(a,env))
	    | (Arexpr_plus(a1,a2)) => 
		  evalArexpr(a1,env) + evalArexpr(a2,env)
	    | (Arexpr_sub(a1,a2)) => 
		  evalArexpr(a1,env) - evalArexpr(a2,env)
	    | (Arexpr_mult(a1,a2)) => 
		  evalArexpr(a1,env) * evalArexpr(a2,env)
	    | (Arexpr_div(a1,a2)) => 
		  evalArexpr(a1,env) div evalArexpr(a2,env)


	(* "Fills out" a value by looking up any variables in it in the  *)
	(* environment.  All other "expand..." functions work similarly. *)
    and expandVal(value,env) =
	  case value of
	      (Ival_string s) => Ival_string s
	    | (Ival_output v) => Ival_output(expandVal(v,env))
	    | (Ival_valof c) => Ival_valof(expandCell(c,env))
	    | (Ival_arexpr a) => 
		  (case a of
		       (Arexpr_var s) =>
			   let val v = (lookupVal(s,env) handle EnvLookup => 
				 raise MatchError "evalArexpr: var not in env")
			   in case v of
			       (Ival_arexpr a) => 
				   Ival_arexpr(Arexpr_int(evalArexpr(a,env)))
			     | _ => expandVal(v,env)
			   end
		     | _ => Ival_arexpr(Arexpr_int(evalArexpr(a,env))))
	    | (Ival_omega) => Ival_omega
	    | (Ival_with(s,b)) => 
		  let val v = (lookupVal(s,env) handle EnvLookup => 
			  raise MatchError "expandVal: can't expand Ival_with")
		  in expandVal(v,env)
		  end
	    | (Ival_pair(v1,v2)) => 
		  Ival_pair(expandVal(v1,env),expandVal(v2,env))

    and expandCell(cell,env) =
	  case cell of
	      (Icell_name s) => Icell_name s
	    | (Icell_var s) => 
		  let val c = lookupCell(s,env) handle EnvLookup =>
		        raise MatchError "expandCell: var not defined"
		  in expandCell(c,env)
		  end
	    | (Icell_fun(f,c)) => 
		  (case f of
		       (forest_basic(i,l)) => 
			  Icell_fun(forest_basic(i,expandForest(l,env)),
				    expandCell(c,env))
		     | _ => Icell_fun(f,expandCell(c,env)))
	    | (Icell_graft(c,t)) => 
		  let val newTag = expandTag(t,env)
		  in case newTag of
		      (FUN f) => f (expandCell(c,env))
		    | (TAG t') => Icell_graft(expandCell(c,env),t')
		  end
	    | (Icell_with(c,b)) => expandCell(c,env)

    and expandForest ([],env) = []
      | expandForest ((c,ins)::l, env) =
	  (expandCell(c,env),expandIns(ins,env))::expandForest(l,env)

    and expandIns(ins,env) =
	  case ins of
	      tree_Valof(c,i,tqlist) => 
		  tree_Valof(expandCell(c,env),i,expandTqlist(tqlist,env))
	    | tree_From(c,i,tqlist) => 
		  tree_From(expandCell(c,env),i,expandTqlist(tqlist,env))
	    | tree_Result(i,v) => tree_Result(i,expandVal(v,env))

    and expandTqlist ([],env) = []
      | expandTqlist ((v,ins)::l,env) =
	  (expandVal(v,env),expandIns(ins,env))::expandTqlist(l,env)

    and expandTag(t,env) =
	  case t of
	      (Tag_str s) => TAG (Tag_str s)
	    | (Tag_arexpr a) => 
		  (case a of
		       (Arexpr_var s) =>
			   ((let val f = (lookupTag(s,env))
			    in (FUN f)
			    end) handle EnvLookup =>
				(let val v = (lookupVal(s,env))
				 in case v of
				     (Ival_arexpr a) => 
					 TAG (Tag_arexpr(Arexpr_int
							 (evalArexpr(a,env))))
				   | (Ival_string str) => TAG (Tag_str str)
				   | _ => raise MatchError 
					 "expandTag: not right var"
				 end) handle EnvLookup => 
				 raise MatchError "expandTag: var not in env")
		     | _ => TAG (Tag_arexpr(Arexpr_int(evalArexpr(a,env)))))


	(* Given a cell name and a list of (cell,_) pairs, it looks     *)
	(* for the first match.  findMatchVal works the same with vals. *)
    fun findCellMatch (c, [], env) = raise Find
      | findCellMatch (c, (cell,ins)::tlist, env) =
	  let val (result, newEnv) = matchCell(c,cell,env)
	  in if result
		 then (ins,newEnv)
	     else findCellMatch(c,tlist,env)
	  end

    fun findValMatch (v, [], env) = raise Find
      | findValMatch (v, (value,ins)::tqlist, env) =
	  let val (result, newEnv) = matchVal(v,value,env)
	  in if result
		 then (ins,newEnv)
	     else findValMatch(v,tqlist,env)
	  end

    end
    end;
