(* internal.sml
   Conversion from expression to forest *)


(* Signature for the internal representation of the interpreter *)
signature INTERNAL =
  sig
  exception InternalError of string
  val convert : CDSBasic.expr -> CDSInternal.forest
  val convertCell : CDSBasic.cell -> CDSInternal.icell
  val algoToState : CDSBasic.algo * CDSBasic.state -> CDSBasic.state
  val uncurry : int * CDSBasic.state -> CDSBasic.state
  val uncurryCVList : int * (CDSBasic.cell * CDSBasic.value list) list ->
      (CDSBasic.cell * CDSBasic.value list) list
  val simplify : CDSBasic.arexpr -> CDSBasic.arexpr
  end;


(* Implementation of the conversion from ParseTree to forest. *)
functor InternalTranslatorFUN (structure Printer : PRINTER) : INTERNAL =
  struct

  local open CDSBasic
        open CDSInternal
	open CDSEnv
  in
  exception InternalError of string


      (* Finds a cell * value pair in list wich matches c1 and        *)
      (* sticks it at the front of the list.  We know a match exists. *)
  fun shuffle (c1, [], acc) = raise InternalError "shuffle: no match found"
    | shuffle (c1, (c,v)::l, acc) = 
        if c1 = c
	    then ((c,v)::(rev acc)) @ l
	else shuffle(c1, l, (c,v)::acc)

      (* Maps cell, tree instruction pairs to cell, internal values. *)
  fun insToIval (c,tree_Valof(c1,i,l)) = 
        (case l of
	     [] => (c,Ival_valof c1)
	   | _ => raise InternalError "insToIval: valof is not 1st order")
    | insToIval (c,tree_Result(i,v)) = (c,makeOutput(i,v))
    | insToIval (c,tree_From(_,_,_)) = 
                          raise InternalError "insToIval: from present"

      (* Maps ivalues to instructions.  There is some unnecessary *)
      (* back and forth conversion, which should be fixed.        *)
  fun ivalToIns (Ival_valof c1) = tree_Valof(c1,1,[])
    | ivalToIns v = tree_Result(0,v)


      (* Generates tree of from instructions when converting a state. *)
  fun genFrom ([],result) = result
    | genFrom ((c,v)::l,result) = 
                        tree_From(c, 1, [(v, genFrom(l,result))])

      (* Breaks apart tree of instructions, inserts a new leaf, *)
      (* and puts it back together again.  A lot of copying...  *)
  fun buildIns ([],_,_) = raise InternalError "buildIns: empty state"
    | buildIns ([(c,v)],result,ins) = 
        (case ins of
	     tree_Valof(c1,i,tqlist) => 
		 if c = c1
		     then tree_Valof(c1,i,tqlist @ [(v,result)])
		 else raise InternalError ("buildIns: not same cell "^
			(Printer.printcell c)^" --- "^(Printer.printcell c1))
	   | tree_From(c1,i,tqlist) => 
		 if c = c1
		     then tree_From(c1,i,tqlist @ [(v,result)])
		 else raise InternalError ("buildIns: not same cell "^
			(Printer.printcell c)^" --- "^(Printer.printcell c1))
	   | _ => raise InternalError "buildIns: ins not a valof or from")
    | buildIns ((c,v)::l,result,ins) = 
        case ins of
	    tree_Valof(c1,i,tqlist) => 
		if c = c1
		     then let val nextIns = find(v,tqlist)
			      val newIns = buildIns(l,result,nextIns)
			      val newList = replaceOrd((v,newIns),tqlist,[])
			  in tree_Valof(c1,i,newList)
			  end
		else  (* look for a cell that matches in remainder of list *)
		    if search(c1,l)
			then buildIns(shuffle(c1,(c,v)::l,[]),result,ins)
		    else raise InternalError ("buildIns: no cell matches"^
			(Printer.printcell c1))
	  | tree_From(c1,i,tqlist) => 
		if c = c1
		     then if search(v,tqlist)   (* value done before *)
			   then let val nextIns = find(v,tqlist)
				    val newIns = buildIns(l,result,nextIns)
				    val newList = replaceOrd((v,newIns),
							     tqlist,[])
				in tree_From(c1,i,newList)
				end
			  else let val newIns = genFrom(l,result)
				   val newList = tqlist @ [(v,newIns)]
			       in tree_From(c1,i,newList)
			       end
		else (* look for a cell that matches in remainder of list *)
		    if search(c1,l)
			then buildIns(shuffle(c1,(c,v)::l,[]),result,ins)
		    else raise InternalError ("buildIns: no cell matches"^
			(Printer.printcell c1))
	  | _ => raise InternalError "buildIns: ins not a valof or from"

      (* Converts event c=v and puts it in the appropiate place in tree    *)
      (* of instructons t.  We know already that {}cname is in t.  We find *)
      (* what's in the state part of c:  for each event like c'=v', there  *)
      (* better be a valof or from c' in tree {}cname.  Insert v': v in    *)
      (* appropiate tree-query list. *)
  fun insert ((c,v),cname,t) = 
        let val ins = find(cname,t)
	    val x = let val f = (getForest c) handle BogusForest => forest_basic(1,[])
		    in case f of 
			forest_basic(i,s) => s
		      | _ => raise InternalError "insert: f not a state"
		    end
	    val newIns = buildIns(map insToIval x,ivalToIns v,ins)
	in replaceOrd((cname,newIns), t, [])
	end


      (* Converts one algo cell * instruction into a list of events. *)
  fun flattenIns (c,Ins_output v,fromState) = 
        [(Cell_fun(Expr_state fromState,c), Val_output v)]
    | flattenIns (c,Ins_valof(c1,qlist),fromState) =
	(Cell_fun(Expr_state fromState,c),Val_valof c1) :: 
	                             flattenQlist(c,c1,qlist,fromState)
    | flattenIns (c,Ins_from(flist),fromState) = 
        flattenFlist(c,flist,fromState)
    | flattenIns (c,Ins_omega,fromState) =
	[(Cell_fun(Expr_state fromState,c), Val_omega)]

      (* Converts a (value * instruction) list from a valof into events. *)
  and flattenQlist (c,c1,[],fromState) = []
    | flattenQlist (c,c1,(v,ins)::qlist,fromState) =
        let val newFromState = fromState @ [(c1,v)]
	in flattenIns(c,ins,newFromState) @ flattenQlist(c,c1,qlist,fromState)
	end

      (* Converts a (state * instruction) list from a "from" to events. *)
  and flattenFlist (c,[],fromState) = []
    | flattenFlist (c,(x,ins)::flist,fromState) =
        flattenIns(c,ins,fromState @ x) @ flattenFlist(c,flist,fromState)

      (* Converts an algorithm -- given in syntax tree from -- into a *)
      (* state -- also in syntax tree form.                           *)
  fun algoToState ([], acc) = acc
    | algoToState ((c,ins)::l, acc) = 
        let val newEvents = flattenIns(c,ins,[])
	    val newAcc = acc @ newEvents
	in algoToState(l,newAcc)
	end


      (* Makes cells in a state grafts with string label. *)
  fun graft (label, []) = []
    | graft (label, (c,v)::s) = (Cell_graft(c,label),v) :: graft(label,s)

      (* Uncurries a state -- in syntax tree form -- once. *)
  fun uncurryState [] = []
    | uncurryState ((c,v)::s) = (uncurryCell c, uncurryVal v)::uncurryState s

  and uncurryCell (Cell_fun(Expr_state x1, Cell_fun(Expr_state x2, c))) = 
        Cell_fun(Expr_state (graft(Tag_str "1", x1) @ 
			     graft(Tag_str "2", x2)), c)
    | uncurryCell (Cell_graft(Cell_fun(Expr_state x1, 
				       Cell_fun(Expr_state x2, c)), s)) =
        Cell_graft(Cell_fun(Expr_state (graft(Tag_str "1", x1) @ 
					graft(Tag_str "2", x2)), c), s)
    | uncurryCell _ = raise InternalError "uncurryCell: not appropiate cell"

  and uncurryVal (Val_valof c) = Val_valof (Cell_graft(c,Tag_str "1"))
    | uncurryVal (Val_output(Val_valof c)) = 
                                 Val_valof (Cell_graft(c,Tag_str "2"))
    | uncurryVal (Val_output(Val_output v)) = Val_output v
    | uncurryVal _ = raise InternalError "uncurryVal: not appropiate value"

      (* Uncurries a state -- in syntax tree form -- order times. *)
  fun uncurry (0, s) = s
    | uncurry (order, s) = uncurry(order-1, uncurryState s)

      (* Used for type inference.  Uncurries a (cell,value list) list. *)
  fun uncurryCVList (0, cvlist) = cvlist
    | uncurryCVList (order, cvlist) = 
      let fun uncurryList [] = []
	    | uncurryList ((c,vlist)::rest) =
	      (uncurryCell c,map uncurryVal vlist)::(uncurryList rest)
      in uncurryCVList(order-1,uncurryList cvlist)
      end


      (* Curries a (cell * index) pair. *)
  fun curryPair (Icell_graft(c,Tag_str "1"),1) = (c, 1)
    | curryPair (Icell_graft(c,Tag_str "2"),1) = (c, 2)
    | curryPair (c, i) = (c, i+1)

      (* Curries a tree of instructions. *)
  fun curryIns (tree_Valof(ic,i,tqlist)) = 
        let val (newC, newI) = curryPair(ic, i)
	in tree_Valof(newC,newI,curryTqlist tqlist)
	end
    | curryIns (tree_From(ic,i,tqlist)) = 
        let val (newC, newI) = curryPair(ic, i)
	in tree_From(newC,newI,curryTqlist tqlist)
	end
    | curryIns (tree_Result(i,v)) = tree_Result(i+1,v)

      (* Curries a (value * instruction) list. *)
  and curryTqlist [] = []
    | curryTqlist ((v,ins)::tqlist) = (v,curryIns ins) :: (curryTqlist tqlist)

      (* Curries a state -- in internal form, i.e. forest -- once. *)
  fun curryTlist [] = []
    | curryTlist ((ic, ins)::tlist) = (ic, curryIns ins) :: curryTlist tlist

      (* Curries a state -- in internal form, i.e. forest -- order times. *)
  fun curry (0, tlist) = tlist
    | curry (order, tlist) = curry(order-1, curryTlist tlist)


      (* Partially evaluates an arexpr (it might have vars embedded in it). *)
  fun simplify (Arexpr_int i) = Arexpr_int i
    | simplify (Arexpr_var s) = Arexpr_var s
    | simplify (Arexpr_minus a) = 
        let val sa = simplify a
	in case sa of
	    (Arexpr_int i) => Arexpr_int(op~(i))
	  | a1 => Arexpr_minus a1
	end
    | simplify (Arexpr_plus(a1,a2)) =
        let val sa1 = simplify a1
	    val sa2 = simplify a2
	in case sa1 of
	    (Arexpr_int a11) => (case sa2 of
				     (Arexpr_int a22) => Arexpr_int(a11+a22)
				   | _ => Arexpr_plus(sa1,sa2))
	  | _ => Arexpr_plus(sa1,sa2)
	end
  | simplify (Arexpr_sub(a1,a2)) =
        let val sa1 = simplify a1
	    val sa2 = simplify a2
	in case sa1 of
	    (Arexpr_int a11) => (case sa2 of
				     (Arexpr_int a22) => Arexpr_int(a11-a22)
				   | _ => Arexpr_sub(sa1,sa2))
	  | _ => Arexpr_sub(sa1,sa2)
	end
  | simplify (Arexpr_mult(a1,a2)) =
        let val sa1 = simplify a1
	    val sa2 = simplify a2
	in case sa1 of
	    (Arexpr_int a11) => (case sa2 of
				     (Arexpr_int a22) => Arexpr_int(a11*a22)
				   | _ => Arexpr_mult(sa1,sa2))
	  | _ => Arexpr_mult(sa1,sa2)
	end
  | simplify (Arexpr_div(a1,a2)) =
        let val sa1 = simplify a1
	    val sa2 = simplify a2
	in case sa1 of
	    (Arexpr_int a11) => 
		(case sa2 of
		     (Arexpr_int a22) => Arexpr_int(a11 div a22)
		   | _ => Arexpr_div(sa1,sa2))
	  | _ => Arexpr_div(sa1,sa2)
	end


      (* Converts a first-order state into internal form.  Must *)
      (* be careful with functional cells whose "parents" have  *)
      (* already been converted--they are in acc (= accumulate).*)
      (* If a functional cell is not initial, must construct    *)
      (* tree of from instructions.                             *)
  fun convertState ([], acc) = acc
    | convertState ((c,v)::x, acc) =
        let val ic = hybridCell c
	    val iv = hybridVal v
	    val icName = getFinalName ic
	    val f = (getForest ic) handle BogusForest => forest_basic(1,[])
	in if search(icName,acc) then         (* if ic is converted already *)
	    let val newAcc = insert((ic,iv),icName,acc)
	    in convertState(x,newAcc)
	    end
	else 
	    case f of
		(forest_basic(_,[])) =>       (* c is initial *) 
		   (case iv of
		     Ival_valof c1 => 
		         convertState(x, acc @ [(icName, tree_Valof(c1,1,[]))])
		   | _ => convertState(x, acc @ [(icName, tree_Result(0,iv))]))
	      | (forest_basic(_,state)) =>    (* c will have "from's" *)
		    convertState(x, acc @ [(icName, 
					    genFrom(map insToIval state, 
						    ivalToIns iv))])
	      | _ => raise InternalError "convertState: state not basic"
	end

      (* Does a partial conversion of a state to a forest.  It's not *)
      (* a real forest--it's flat--just an icell * ivalue list.      *)
  and hybridState ([], acc) = rev acc
    | hybridState ((c,v)::x, acc) = 
        hybridState(x,(hybridCell c, ivalToIns(hybridVal v))::acc)

      (* Next 2 value and cell conversion routines are used for       *)
      (* hybrid state conversion.  It turns out that we don't want to *)
      (* build a real forest for the state part of a functional cell  *)
      (* only to have to tear it apart and have to reconstruct the    *)
      (* actual cell name--state and all--when building instructions. *)
  and hybridVal (Val_string s) = Ival_string s
    | hybridVal (Val_output v) = Ival_output (hybridVal v)
    | hybridVal (Val_valof c) = Ival_valof (hybridCell c)
    | hybridVal (Val_arexpr a) = Ival_arexpr (simplify a)
    | hybridVal (Val_omega) = Ival_omega
    | hybridVal (Val_pair(v1, v2)) = Ival_pair(hybridVal v1,hybridVal v2)
    | hybridVal (Val_with(s, b)) = Ival_with(s, convertBoolexp b)
    | hybridVal _ = raise InternalError
                         "hybridVal: interval values not allowed in algorithms"
  
  and hybridCell (Cell_name s) = Icell_name s
    | hybridCell (Cell_var s) = Icell_var s
    | hybridCell (Cell_fun(e,c)) = 
        (case e of
	     Expr_state x => Icell_fun(forest_basic(1,hybridState(x,[])),
				       hybridCell c)
	   | _ => raise InternalError "hybridCell: non-state in fun cell")
    | hybridCell (Cell_graft(c,t)) = 
        (case t of
	     (Tag_str s) => Icell_graft(hybridCell c, Tag_str s)
	   | (Tag_arexpr a) => 
		 Icell_graft(hybridCell c, Tag_arexpr(simplify a))
	   | _ => raise InternalError "hybridCell: tag is an interval")
    | hybridCell (Cell_with(c, b)) = Icell_with(hybridCell c,convertBoolexp b)


      (* Cell conversion routine used to convert requests posed to *)
      (* the interpreter--the forest had better be real here.      *)
  and convertCell (Cell_name s) = Icell_name s
    | convertCell (Cell_var s) = Icell_var s
    | convertCell (Cell_fun(e,c)) = Icell_fun(convert e, convertCell c)
    | convertCell (Cell_graft(c,t)) = 
        (case t of
	    (Tag_str s) => Icell_graft(convertCell c, Tag_str s)
	  | (Tag_arexpr a) => 
		Icell_graft(convertCell c, Tag_arexpr(simplify a))
	  | _ => raise InternalError "hybridCell: tag is an interval")
    | convertCell (Cell_with(c, b)) = 
        Icell_with(convertCell c,convertBoolexp b)

  and convertBoolexp (Boolexp_gt (a1, a2)) = Iboolexp_gt(a1,a2)
    | convertBoolexp (Boolexp_gteq (a1, a2)) = Iboolexp_gteq(a1,a2)
    | convertBoolexp (Boolexp_lt (a1, a2)) = Iboolexp_lt(a1,a2)
    | convertBoolexp (Boolexp_lteq (a1, a2)) = Iboolexp_lteq(a1,a2)
    | convertBoolexp (Boolexp_eq (v1, v2)) = 
        Iboolexp_eq(hybridVal v1, hybridVal v2)
    | convertBoolexp (Boolexp_noteq (v1, v2)) = 
        Iboolexp_noteq(hybridVal v1, hybridVal v2)
    | convertBoolexp (Boolexp_or (b1, b2)) = 
        Iboolexp_or(convertBoolexp b1, convertBoolexp b2)
    | convertBoolexp (Boolexp_and (b1, b2)) = 
        Iboolexp_and(convertBoolexp b1, convertBoolexp b2)


      (* Converts an expression -> forest.  Looks up identifiers in   *)
      (* environment.  Algorithms are first flattened into state form *)
      (* and then if necessary uncurried to get a first order state,  *)
      (* the state is converted to a forest, and the forest curried   *)
      (* back to original type, if necessary.                         *)
  and convert (Expr_id s) = CDSEnv.lookup(s,exprList)
    | convert (Expr_state x) = 
        if x = [] then forest_basic(0,[])
	else let val order = degree(#1(hd x))
	     in if order <= 2 then forest_basic(order-1,convertState(x,[]))
		else let val firstOrderS = uncurry(order-2, x)
			 val convertedS = convertState(firstOrderS, [])
		     in forest_basic(order-1,curry(order-2, convertedS))
		     end
	     end
    | convert (Expr_algo a) = convert(Expr_state (algoToState(a,[])))
    | convert (Expr_curry e) = forest_curry(convert e)
    | convert (Expr_uncurry e) = forest_uncurry(convert e)
    | convert (Expr_compose(e1,e2)) = forest_comp(convert e1, convert e2)
    | convert (Expr_apply(e1,e2)) = forest_apply(convert e1, convert e2)
    | convert (Expr_pair(e1,e2)) = forest_pair([convert e1, convert e2])
    | convert (Expr_prod(e1,e2)) = forest_prod([convert e1, convert e2])
    | convert (Expr_fix e) = forest_fix(convert e)

  end
  end;
