(* evaluate.sml
   Evaluator---the heart of the interpreter *)

signature EVALUATOR =
    sig
    exception EvalError of string
    exception NoAccess
    val eval : CDSInternal.forest * CDSInternal.icell * CDSInternal.env -> 
	CDSInternal.ivalue
    val evalDebug : int * CDSInternal.forest * CDSInternal.icell * 
	CDSInternal.env -> CDSInternal.ivalue
    end;
    

functor EvaluatorFUN (structure Internal : INTERNAL
		      structure Printer : PRINTER
		      structure Match : MATCH) : EVALUATOR =
    struct
    exception EvalError of string
    exception NoAccess

    local open CDSBasic
	  open CDSInternal
	  open CDSEnv
	  open Internal
	  open Printer
	  open Match
    in

	(* Projection functions. *)
    fun remove label (Icell_graft(c,Tag_str l),ins) = 
	  if label=l then [(c, ins)] else []
      | remove label (Icell_graft(c,Tag_arexpr(Arexpr_int i)),ins) =
	  if label = makestring(i) then [(c, ins)] else []
      | remove label _ = []

    fun proj (s, forest_basic(i, tlist)) = 
	                forest_basic(i, (flatten (map (remove s) tlist)))
      | proj (s, forest_prod flist) =
	  let val i = ord s - ord "0"
	  in ith(i, flist) handle Ith 
		          => raise EvalError "project: index out of bounds"
	  end
      | proj (s, forest_apply(forest_pair flist, f2)) =
	  let val i = ord s - ord "0"
	  in forest_apply(ith(i, flist),f2) handle Ith 
		          => raise EvalError "project: index out of bounds"
	  end
      | proj _ = raise EvalError "attempting to project wrong type"


	(* Executes one tree instruction when querying cell c.     *)
	(* There are 3 kinds of tree instructions:  Result, Valof, *)
	(* and From.  The semantics is the same as in Devin.       *)
    and exec (tree_Result(i,v),c,env) = makeOutput(i,expandVal(v,env))
      | exec (tree_Valof(c1,i,tqlist),c,env) = 
	  let val fi = getIthForest(i,c) handle IthForest => 
	                                    raise EvalError "exec: IthForest"
	      val v1 = eval(fi,expandCell(c1,env),env)
	  in if v1 = Ival_omega 
		 then makeOutput(i-1,Ival_valof (expandCell(c1,env)))
	     else (let val (ins,newEnv) = findValMatch(v1,tqlist,env)
		   in exec(ins,c,newEnv)
		   end handle Find => Ival_omega)
	  end
      | exec (tree_From(c1,i,tqlist),c,env) =
	  let val fi = getIthForest(i,c) handle IthForest => 
	                                    raise EvalError "exec: IthForest"
	      val v1 = eval(fi,expandCell(c1,env),env)
	  in if v1 = Ival_omega then Ival_omega
	     else (let val (ins,newEnv) = findValMatch(v1,tqlist,env)
		   in exec(ins,c,newEnv)
		   end handle Find => raise NoAccess)
	  end

    
        (* Main function--processes queries of the form forest ? cell. *)
        (* If we have a basic forest then just look up appropiate tree *)
        (* instruction (if any) and exec it.  Otherwise we have an     *)
        (* expression--apply the CDS02 rewrite rules.                  *)
    and eval (forest_basic(i, tlist), c, env) =
	  (let val (ins,newEnv) = findCellMatch(getIthName(i,c),tlist,env)
	   in exec(ins,c,newEnv)
	   end handle Find => Ival_omega)
      | eval (forest_apply(f1,f2), c, env) =
	  let val v = eval(f1, Icell_fun(f2,c),env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof _ => Ival_omega
	    | Ival_output v1 => expandVal(v1,env)
	    | _ => raise EvalError ("application: value "^(printval v)^
		  " not in list")
	  end
      | eval (forest_comp(f1,f2), c, env) =
	  let val x = getForest c
	      val cname = getName c
	      val v = eval(f1,Icell_fun(forest_apply(f2,x),cname),env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof c1 => eval(f2,Icell_fun(x,c1),env)
	    | Ival_output v1 => Ival_output(expandVal(v1,env))
	    | _ => raise EvalError "composition: value not in list"
	  end
      | eval (forest_fix f, c, env) =
	  let val v = eval(f, Icell_fun(forest_fix f, c), env)
	      in case v of
		  Ival_omega => Ival_omega
		| Ival_valof _ => Ival_omega
		| Ival_output v1 => expandVal(v1,env)
		| _ => raise EvalError "fix: value not in list"
	  end
      | eval (forest_curry f, c, env) =
	  let val x = getForest c
	      val y = getForest(getName c)
	      val cname = getName(getName c)
	      val v = eval(f, Icell_fun(forest_prod [x, y], cname), env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof(Icell_graft(c1,Tag_str "1")) => 
		  Ival_valof(expandCell(c1,env))
	    | Ival_valof(Icell_graft(c1,Tag_arexpr(Arexpr_int 1))) => 
		  Ival_valof(expandCell(c1,env))
	    | Ival_valof(Icell_graft(c2,Tag_str "2")) => 
		  Ival_output(Ival_valof(expandCell(c2,env)))
	    | Ival_valof(Icell_graft(c2,Tag_arexpr(Arexpr_int 2))) =>
		  Ival_output(Ival_valof(expandCell(c2,env)))
	    | Ival_output v1 => Ival_output(Ival_output(expandVal(v1,env)))
	    | _ => raise EvalError "curry: value not in list"
	  end
      | eval (forest_uncurry f, c, env) =
	  let val x = getForest c
	      val cname = getName c
	      val v = eval(f, Icell_fun(proj("1",x),
					(Icell_fun(proj("2",x),cname))), env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof c1 => 
		  Ival_valof(Icell_graft(expandCell(c1,env),
					 Tag_arexpr(Arexpr_int 1)))
	    | Ival_output(Ival_valof c2) => 
		  Ival_valof(Icell_graft(expandCell(c2,env),
					 Tag_arexpr(Arexpr_int 2)))
	    | Ival_output(Ival_output v1) => Ival_output(expandVal(v1,env))
	    | _ => raise EvalError "uncurry: value not in list"
	  end
      | eval (forest_pair flist, c, env) =
	  let val x = getForest c
	      val cname = getName c
	      val (ci,i) = case cname of
		  Icell_graft(ci, Tag_str s) => (ci, ord s - ord "0")
		| Icell_graft(ci, Tag_arexpr(Arexpr_int m)) => (ci, m)
	        | _ => raise EvalError "pair: name not correct graft"
	      val fi = ith(i, flist) handle Ith 
		          => raise EvalError "pair: index out of bounds"
	  in eval(fi, Icell_fun(x,ci), env)
	  end
      | eval (forest_prod flist, c, env) =
	  let val (ci,i) = case c of
		  Icell_graft(ci, Tag_str s) => (ci, ord s - ord "0")
		| Icell_graft(ci, Tag_arexpr(Arexpr_int m)) => (ci, m)
	        | _ => raise EvalError "product: name not correct graft"
	      val fi = ith(i, flist) handle Ith 
		          => raise EvalError "product: index out of bounds"
	  in eval(fi, ci, env)
	  end



	(* Debug versions of the evaluator routines  *)
	(* Same as above except have to carry around *)
	(* the number of the recursive call to eval. *)   
    fun exec_dbg (n,tree_Result(i,v),c,env) = makeOutput(i,expandVal(v,env))
      | exec_dbg (n,tree_Valof(c1,i,tqlist),c,env) = 
	  let val fi = getIthForest (i, c) handle IthForest 
	              => raise EvalError "exec: forest index out of bounds"
	      val v1 = evalDebug(n+1,fi,expandCell(c1,env),env)
	  in if v1 = Ival_omega 
		 then makeOutput(i-1,Ival_valof(expandCell(c1,env)))
	     else (let val (ins,newEnv) = findValMatch(v1,tqlist,env)
                   in exec_dbg(n,ins,c,newEnv)
                   end handle Find => Ival_omega)
	  end
      | exec_dbg (n,tree_From(c1,i,tqlist),c,env) =
	  let val fi = getIthForest (i, c) handle IthForest 
	              => raise EvalError "exec: forest index out of bounds"
	      val v1 = evalDebug(n+1,fi,expandCell(c1,env),env)
	  in if v1 = Ival_omega then Ival_omega
	     else (let val (ins,newEnv) = findValMatch(v1,tqlist,env)
                   in exec_dbg(n,ins,c,newEnv)
                   end handle Find => raise NoAccess)
	  end

    and evalDebug (n, f, c, env) = 
	let val _ = output(std_out, (makestring n)^" ==> "^(printforest f)
			   ^" ? "^(printcell c)^"\nENV = {"
			   ^(printenviro env)^"}\n")
	    val v = eval_dbg(n, f, c, env)
	    val _ = output(std_out, (makestring n)^" <== "^(printval v)^"\n")
	in v
	end

    and eval_dbg (n, forest_basic(i, tlist), c, env) =
	  (let val (ins,newEnv) = findCellMatch(getIthName(i,c),tlist,env)
	   in exec_dbg(n,ins,c,newEnv)
           end handle Find => Ival_omega)
      | eval_dbg (n, forest_apply(f1,f2), c, env) =
	  let val v = evalDebug(n+1, f1, Icell_fun(f2,c),env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof _ => Ival_omega
	    | Ival_output v1 => expandVal(v1,env)
	    | _ => raise EvalError "application: value not in list"
	  end
      | eval_dbg (n, forest_comp(f1,f2), c, env) =
	  let val x = getForest c
	      val cname = getName c
	      val v = evalDebug(n+1,f1,
				Icell_fun(forest_apply(f2,x),cname),env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof c1 => evalDebug(n+1,f2,Icell_fun(x,c1),env)
	    | Ival_output v1 => Ival_output(expandVal(v1,env))
	    | _ => raise EvalError "composition: value not in list"
	  end
      | eval_dbg (n, forest_fix f, c, env) =
	  let val v = evalDebug(n+1, f, Icell_fun(forest_fix f, c),env)
	      in case v of
		  Ival_omega => Ival_omega
		| Ival_valof _ => Ival_omega
		| Ival_output v1 => expandVal(v1,env)
		| _ => raise EvalError "fix: value not in list"
	  end
      | eval_dbg (n, forest_curry f, c, env) =
	  let val x = getForest c
	      val y = getForest(getName c)
	      val cname = getName(getName c)
	      val v = evalDebug(n+1,f,Icell_fun(forest_prod [x, y],cname),env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof(Icell_graft(c1,Tag_str "1")) => 
		  Ival_valof(expandCell(c1,env))
	    | Ival_valof(Icell_graft(c1,Tag_arexpr(Arexpr_int 1))) => 
		  Ival_valof(expandCell(c1,env))
	    | Ival_valof(Icell_graft(c2,Tag_str "2")) => 
		  Ival_output(Ival_valof(expandCell(c2,env)))
	    | Ival_valof(Icell_graft(c2,Tag_arexpr(Arexpr_int 2))) =>
		  Ival_output(Ival_valof(expandCell(c2,env)))
	    | Ival_output v1 => Ival_output(Ival_output(expandVal(v1,env)))
	    | _ => raise EvalError "curry: value not in list"
	  end
      | eval_dbg (n, forest_uncurry f, c, env) =
	  let val x = getForest c
	      val cname = getName c
	      val v = evalDebug(n+1, f, Icell_fun(proj("1",x),
					(Icell_fun(proj("2",x),cname))), env)
	  in case v of
	      Ival_omega => Ival_omega
	    | Ival_valof c1 => 
		  Ival_valof(Icell_graft(expandCell(c1,env),
					 Tag_arexpr(Arexpr_int 1)))
	    | Ival_output(Ival_valof c2) => 
		  Ival_valof(Icell_graft(expandCell(c2,env),
					 Tag_arexpr(Arexpr_int 2)))
	    | Ival_output(Ival_output v1) => Ival_output(expandVal(v1,env))
	    | _ => raise EvalError "uncurry: value not in list"
	  end
      | eval_dbg (n, forest_pair flist, c, env) =
	  let val x = getForest c
	      val cname = getName c
	      val (ci,i) = case cname of
		  Icell_graft(ci, Tag_str s) => (ci, ord s - ord "0")
		| Icell_graft(ci, Tag_arexpr(Arexpr_int m)) => (ci, m)
	        | _ => raise EvalError "pair: name not correct graft"
	      val fi = ith(i, flist) handle Ith 
		          => raise EvalError "pair: index out of bounds"
	  in evalDebug(n+1, fi, Icell_fun(x,ci), env)
	  end
      | eval_dbg (n, forest_prod flist, c, env) =
	  let val (ci,i) = case c of
		  Icell_graft(ci, Tag_str s) => (ci, ord s - ord "0")
		| Icell_graft(ci, Tag_arexpr(Arexpr_int m)) => (ci, m)
	        | _ => raise EvalError "product: name not correct graft"
	      val fi = ith(i, flist) handle Ith 
		          => raise EvalError "product: index out of bounds"
	  in evalDebug(n+1, fi, ci, env)
	  end
    end
    end;
