(* defs.sml
   All-pervasive types and structures *)


(* CDS basic types *)
structure CDSBasic =
    struct

	(* Arithmetic expressions *)
    datatype arexpr = Arexpr_int of int
	            | Arexpr_var of string
		    | Arexpr_minus of arexpr
		    | Arexpr_plus of arexpr * arexpr
		    | Arexpr_sub of arexpr * arexpr
		    | Arexpr_mult of arexpr * arexpr
		    | Arexpr_div of arexpr * arexpr

    datatype tag = Tag_str of string
                 | Tag_arexpr of arexpr
		 | Tag_interval of value

	(* Values, cells *)
    and value = Val_string of string
	      | Val_output of value
	      | Val_valof of cell
	      | Val_arexpr of arexpr
	      | Val_pair of value * value
	      | Val_with of string * boolexp
	      | Val_omega
	      | Val_interval_inf
	      | Val_interval_from of int
	      | Val_interval_to of int
	      | Val_interval_fromto of int * int
    and cell = Cell_name of string
	     | Cell_var of string
    	     | Cell_fun of expr * cell
	     | Cell_graft of cell * tag
             | Cell_with of cell * boolexp
        (* Boolean expressions *)
    and boolexp = Boolexp_gt of arexpr * arexpr
                | Boolexp_gteq of arexpr * arexpr
                | Boolexp_lt of arexpr * arexpr
                | Boolexp_lteq of arexpr * arexpr
                | Boolexp_eq of value * value
		| Boolexp_noteq of value * value
		| Boolexp_or of boolexp * boolexp
		| Boolexp_and of boolexp * boolexp
        (* Algo instructions *)
    and instruction = Ins_output of value
                    | Ins_valof of cell * query list
                    | Ins_from of from_do list
                    | Ins_omega
       (* Expressions -- what's evaluated by interpreter *)
    and expr = Expr_state of state
             | Expr_algo of algo
	     | Expr_curry of expr
	     | Expr_uncurry of expr
	     | Expr_compose of expr * expr
	     | Expr_apply of expr * expr
	     | Expr_pair of expr * expr
	     | Expr_prod of expr * expr
	     | Expr_fix of expr
	     | Expr_id of string
       (* Events, states *)
    withtype event = cell * value
    and state = event list
       (* Pieces of algos *)
    and from_do = state * instruction
    and query = value * instruction
        (* Algorithms -- a cell*instruction list *)
    and algo = (cell * instruction) list

	(* Dcds and pieces of it *)
    type access = (cell * value) list
    
    datatype component = Comp_cell of cell * value list * access list
    		       | Comp_graft of cell * access list
		       
    type dcds = string * component list
			    
    datatype dcds_tree = Normal of dcds
		       | Recursive of dcds
		       | Local of dcds_tree * dcds_tree

	(* Commands -- stuff outside of the language *)
    datatype command_tree = Com_abbreviate of string * expr
	                  | Com_print of string
	                  | Com_load of string
	                  | Com_loadecho of string
			  | Com_traceon
			  | Com_traceoff
			  | Com_timeron
                          | Com_timeroff
			  | Com_typingon
			  | Com_typingoff
			  | Com_show of int * string
			  | Com_show_more of int * string
			  | Com_hierarchy of string
                          | Com_env
			  | Com_PCF
                          | Com_refine of string list

        (* The whole parse tree *)			    
    datatype ParseTree  = COM of command_tree
	                | EXP of expr
			| DCDS of dcds_tree
			| EMPTY

	(* The parse tree for cell names *)
    datatype cellParse = Empty
                       | Name of cell

    fun take (i, []) = []
      | take (i, x::xs) = if i>0 then x::take(i-1,xs) else []

    exception Ith
    fun ith (i, []) = raise Ith
      | ith (0, _) = raise Ith
      | ith (1, x::xs) = x
      | ith (i, x::xs) = ith(i-1,xs)

    fun flatten [] = []
      | flatten [l] = l
      | flatten ll = let
	  fun flat ([], l) = rev l
	    | flat (ll::r, l) = flat(r, flat2(ll, l))
	  and flat2 ([], l) = l
	    | flat2 (x::r, l) = flat2(r, x::l)
	  in
	    flat (ll, [])
	  end

    (* Intersects two lists *)
    fun intersect (_, []) = []
      | intersect ([], _) = []
      | intersect (x::xs, ys) =
	let fun onePass _ [] = []
	      | onePass x (y::ys) = if x=y then [x] else onePass x ys
	in (onePass x ys) @ (intersect(xs,ys))
	end

    exception Find

    fun search (s, []) = false
      | search (s, (s1,e1)::t) = if (s = s1) then true else search (s, t)

    fun find (s, []) = raise Find
      | find (s, (s1,e1)::t) = if (s = s1) then e1 else find (s, t)

    fun member (x,[]) = false
      | member (x,x'::l) = if x=x' then true else member(x,l)

    fun forAll p [] = true
      | forAll p (x::l) = if p x then forAll p l else false

    exception Zip

    fun zip [] [] = []
      | zip (x::xs) (y::ys) = (x,y)::(zip xs ys)
      | zip _ _ = raise Zip

	(* Remove duplicates from a list. *)
    fun duplicates [] = []
      | duplicates (e::l) = 
	let fun remove (e, []) = []
	      | remove (e, e'::l) = 
		if e=e' then remove(e,l) else e'::(remove(e,l))
	    val newl = remove(e,l)
	in e::(duplicates(remove(e,l)))
	end

    fun glueStrings [] = ""
      | glueStrings [s] = "{"^s^"}"
      | glueStrings (s::slist) = "{"^s^"}, "^(glueStrings slist)

    end;


(* Types for internal representation.  They're outside of the *)
(* InternalTranslatorFUN functor (from internal.sml) because  *)
(* the evaluator needs to be able to destruct them.           *)
structure CDSInternal = 
    struct
    local open CDSBasic
    in
	(* Internal cells.  Needed for posing intensional queries *)
    datatype icell = Icell_name of string
	     | Icell_var of string
    	     | Icell_fun of forest * icell
	     | Icell_graft of icell * tag
	     | Icell_with of icell * iboolexp
    and ivalue = Ival_string of string
               | Ival_output of ivalue
               | Ival_valof of icell
	       | Ival_arexpr of arexpr
	       | Ival_omega
	       | Ival_with of string * iboolexp
               | Ival_pair of ivalue * ivalue
    and iboolexp = Iboolexp_gt of arexpr * arexpr
	         | Iboolexp_gteq of arexpr * arexpr
	         | Iboolexp_lt of arexpr * arexpr
	         | Iboolexp_lteq of arexpr * arexpr
	         | Iboolexp_eq of ivalue * ivalue
	         | Iboolexp_noteq of ivalue * ivalue
	         | Iboolexp_or of iboolexp * iboolexp
		 | Iboolexp_and of iboolexp * iboolexp
    and tree_instruction = tree_Valof of icell * int * tree_query list
                         | tree_From of icell * int * tree_query list
			 | tree_Result of int * ivalue
    and forest = forest_basic of int * tree list
               | forest_apply of forest * forest
	       | forest_comp of forest * forest
	       | forest_fix of forest
	       | forest_curry of forest
	       | forest_uncurry of forest
	       | forest_pair of forest list
	       | forest_prod of forest list
    withtype tree_query = ivalue * tree_instruction
    and tree = icell * tree_instruction

	(* Replaces a tuple with another one, based on first element.  *)
	(* If tuple not in, it is inserted at the end.  Does not       *)
	(* maintain order of tuples in list---see replaceOrd for that. *)
    fun replace ((c,v), [], result) = (c,v)::result
      | replace ((c,v), (c1,v1)::l, result) = 
          if c=c1 then ((c,v)::l) @ result
	  else replace((c,v),l,(c1,v1)::result)

	 (* The evaluation environment. Consists of three things: *)
	 (* values, cells, and tags, represented as functions. *)
    type env = (string * ivalue) list * (string * icell) list *
	(string * (icell -> icell)) list
    val emptyenv : env = ([],[],[])
    exception EnvLookup

    fun makeEnv (l1, l2, l3) = (l1,l2,l3):env;

    fun getValList (valList, cellList, tagList) = valList

    fun getCellList (valList, cellList, tagList) = cellList

    fun getTagList (valList, cellList, tagList) = tagList

    fun bindVal (s, v, env) = makeEnv(replace((s,v),getValList env,[]), 
				      getCellList env, getTagList env)

    fun bindCell (s, c, env) = makeEnv(getValList env, 
			      replace((s,c),getCellList env,[]),getTagList env)

    fun bindTag (s, f, env) = makeEnv(getValList env, getCellList env,
				      replace((s,f),getTagList env,[]))

    fun lookupVal (s, env) = find(s,getValList env) handle Find => 
	                                                     raise EnvLookup

    fun lookupCell (s, env) = find(s,getCellList env) handle Find => 
	                                                     raise EnvLookup

    fun lookupTag (s, env) = find(s,getTagList env) handle Find => 
	                                                     raise EnvLookup

    fun joinList ([],[]) = []
      | joinList ((s,v)::l, []) = (s,v)::l
      | joinList ([], (s,v)::l) = (s,v)::l
      | joinList ((s1,v1)::l1, l2) = joinList(l1,replace((s1,v1),l2,[]))

	(* Bindings to same variable in env1 take precedence. *)
    fun joinEnv (env1, env2) = 
	  makeEnv(joinList(getValList env1,getValList env2),
		  joinList(getCellList env1,getCellList env2),
		  joinList(getTagList env1,getTagList env2))

    fun degree (Cell_fun(e,c)) = 1 + degree(c)
      | degree (Cell_graft(c,s)) = degree c
      | degree (Cell_with(c,b)) = degree c
      | degree _ = 1
	
    fun cellOrder (Cell_fun(e,c)) = 1 + cellOrder(c)
      | cellOrder (Cell_with(c,b)) = cellOrder c
      | cellOrder _ = 1

    exception BogusForest
    
    fun getForest (Icell_fun(f,c)) = f
      | getForest (Icell_with(c,b)) = getForest c
      | getForest (Icell_graft(c,s)) = getForest c
      | getForest _ = raise BogusForest

    exception IthForest
    fun getIthForest (1, Icell_fun(f,c)) = f
      | getIthForest (i, Icell_fun(f,c)) = getIthForest (i-1, c)
      | getIthForest (i, Icell_with(c,b)) = getIthForest (i, c)
      | getIthForest (i, Icell_graft(c,s)) = getIthForest (i, c)
      | getIthForest (_, _) = raise IthForest

    fun getName (Icell_name s) = Icell_name s
      | getName (Icell_var s) = Icell_var s
      | getName (Icell_fun(x,c)) = c
      | getName (Icell_graft(c,s)) = Icell_graft(getName c,s)
      | getName (Icell_with(c,b)) = Icell_with(getName c,b)

    fun getIthName (0, c) = c
      | getIthName (1, c) = getName c
      | getIthName (i, c) = getIthName(i-1, getName c)

    fun getFinalName (Icell_name s) = Icell_name s
      | getFinalName (Icell_var s) = Icell_var s
      | getFinalName (Icell_fun(x,c)) = getFinalName c
      | getFinalName (Icell_graft(c,s)) = Icell_graft(getFinalName c,s)
      | getFinalName (Icell_with(c,b)) = Icell_with(getFinalName c,b)

    fun makeOutput (0,v) = v
      | makeOutput (i,v) = Ival_output(makeOutput(i-1,v))

	(* Internal form of dcds *)
    datatype cva = Plain of cell * value list * access list
                 | Delay of idcds * ((cell * value list * access list) -> 
				     (cell * value list * access list))

    and idcds = Nonrec of cva list
              | Rec of cva list * 
		((cell * value list * access list) -> 
		 (cell * value list * access list)) *
		((cell * value list * access list) -> 
		 (cell * value list * access list))

	(* Datatype definitions for types *)
    type typeVar = int

    datatype typeExp = Dcds of string
                     | Alpha of typeVar
                     | Arrow of typeExp * typeExp
		     | And of typeExp list   (* & = intersection of cds's *)
                     | Meet of typeExp list  (* /\ = overloading *)
		     | Prod of typeExp list

        (* These are the types we store in our environment. *)
    datatype TYPE = UNTYPED | TYPE of typeExp

      (* Generate unique type variables *)
    local val counter = ref 0
    in fun newVar () = (counter := !counter + 1; !counter)
    end

    fun isMeet (Meet _) = true
      | isMeet _ = false	

    (* Does type variable i occur in type t? *)
    fun occursIn i (Dcds _) = false
      | occursIn i (Alpha j) = i = j
      | occursIn i (Arrow(t1,t2)) = (occursIn i t1) orelse (occursIn i t2)
      | occursIn i (And tlist) = exists (occursIn i) tlist
      | occursIn i (Meet tlist) = exists (occursIn i) tlist
      | occursIn i (Prod tlist) = exists (occursIn i) tlist

    (* Is a type completely polymorphic? *)
    fun isPolymorphic (Alpha _) = true
      | isPolymorphic (Dcds _) = false
      | isPolymorphic (Arrow(t1,t2)) = 
	(isPolymorphic t1) andalso (isPolymorphic t2)
      | isPolymorphic (Prod tlist) = 
	let val tvals = map isPolymorphic tlist
	in fold (fn (x,y) => x andalso y) tvals true
	end
      | isPolymorphic (And _) = false
      | isPolymorphic (Meet _) = false

    fun variable (Dcds _) = false
      | variable (Alpha _) = true
      | variable (Arrow(t1,t2)) = (variable t1) orelse (variable t2)
      | variable (Prod tlist) = variableInList tlist
      | variable (And tlist) = false
      | variable (Meet tlist) = variableInList tlist

    and variableInList tlist = let val tvalues = map variable tlist
			       in fold (fn (x,y) => x orelse y) tvalues false
			       end

    (* Substitutions are represented as functions. *)
    datatype subst = Sub of typeVar -> typeExp

        (* The identity substitution. *)
    val emptySub = Sub Alpha

    fun assoc [] oldassocs a = oldassocs a
      | assoc ((a1,b1)::rest) oldassocs a = 
          if a = a1 then b1 else assoc rest oldassocs a

        (* Creates a new substitution function. *)
    fun newSub v t = Sub(assoc [(v,t)] Alpha)

        (* Applies a substitution to all variables in a type expression. *)
    fun apply (Sub f) (Alpha i) = f i
      | apply _ (Dcds s) = Dcds s
      | apply subs (Arrow(t1,t2)) = Arrow(apply subs t1, apply subs t2)
      | apply subs (And tlist) = And(map (apply subs) tlist)
      | apply subs (Meet tlist) = Meet(map (apply subs) tlist)
      | apply subs (Prod tlist) = Prod(map (apply subs) tlist)

        (* Compose 2 substitutions. Sub2 gets applied first. *)
    fun compose sub1 (Sub f) = Sub(apply sub1 o f) 

    datatype typeConstraint = Glb of typeExp list
                            | Lub of typeExp list

      (* For handling the construction of a fresh instance of a type *)
    exception EmptyInstance
    fun emptyInst i = raise EmptyInstance
    fun extendInst f i = let val new = newVar()
			 in fn x => if x=i then new else f x
			 end

      (* Given a type, generate a fresh instance of the type with all *)
      (* alphas replaced consistently with fresh ones. *)
    fun freshInst (Dcds s) f = (Dcds s, f)
      | freshInst (Alpha i) f = ((Alpha (f i), f) handle EmptyInst =>
				 let val f' = extendInst f i
				 in (Alpha (f' i), f')
				 end)
      | freshInst (Arrow(t1,t2)) f = 
	let val (t1', f') = freshInst t1 f
	    val (t2', f'') = freshInst t2 f'
	in (Arrow(t1',t2'), f'')
	end
      | freshInst (And tlist) f = let val (tlist', f') = freshListInst tlist f
				  in (And tlist', f')
				  end
      | freshInst (Meet tlist) f = let val (tlist', f') = freshListInst tlist f
				   in (Meet tlist', f')
				   end
      | freshInst (Prod tlist) f = let val (tlist', f') = freshListInst tlist f
				   in (Prod tlist', f')
				   end
      
    and freshListInst [] f = ([], f)
      | freshListInst (t::tlist) f = 
	let val (t', f') = freshInst t f
	    val (rest, f'') = freshListInst tlist f' 
	in (t'::rest, f'')
	end

        (* Makes a list out of the inputs in a curried type *)
    fun stripArrows (Arrow(t1,t2)) n =
	let val (i,out,is) = stripArrows t2 n
	in (i+1,out,t1::is)
	end 
      | stripArrows t n = (n,t,[])

    end
    end;


(* CDS environment globals and storage functions *)
structure CDSEnv =
    struct
    local open CDSBasic
    in
    exception Lookup of string
	
    val timer = ref false
    val trace = ref false
    val typing = ref false
    val tempTyping = ref false
    val currentTimeStamp = ref 0

    fun incTimeStamp () = currentTimeStamp := !currentTimeStamp + 1

	(* Expressions: name * internal representation *)
    val exprList : (string * CDSInternal.forest) list ref = ref []
    val pcfExprList : (string * CDSInternal.forest) list ref = ref []
    val tempExprList : (string * CDSInternal.forest) list ref = ref []

	(* Types: dcds name * internal representation *)
    val typeList : (string * CDSInternal.idcds) list ref = ref []
    val pcfTypeList : (string * CDSInternal.idcds) list ref = ref []
    val tempTypeList : (string * CDSInternal.idcds) list ref = ref []

    datatype subkind = ext | partof | notsubtype

	(* Subtype hierarchy: dcds name * parents * children *)
    val hierarchy : (string * (string * subkind) list * (string * subkind) list) list ref = ref []
    val pcfHierarchy : (string * (string * subkind) list * (string * subkind) list) list ref = ref []
    val tempHierarchy : (string * (string * subkind) list * (string * subkind) list) list ref = ref []

	(* expression name * timestamp * parse tree * type *)
    val nameExpTypeList : (string * int * expr * CDSInternal.TYPE) 
	list ref = ref []
    val pcfNameExpTypeList : (string * int * expr * CDSInternal.TYPE) 
	list ref = ref []
    val tempNameExpTypeList : (string * int * expr * CDSInternal.TYPE) 
	list ref = ref []

	(* expression name * timestamp * parse tree * refinement type *)
    val nameExpRTypeList : (string * int * expr * CDSInternal.TYPE) 
	list ref = ref []
    val pcfNameExpRTypeList : (string * int * expr * CDSInternal.TYPE) 
	list ref = ref []
    val tempNameExpRTypeList : (string * int * expr * CDSInternal.TYPE) 
	list ref = ref []

	(* list of type names to be used only for refinement type inference *)
    val refineOnlyList : string list ref = ref []
    val pcfRefineOnlyList : string list ref = ref []
    val tempRefineOnlyList : string list ref = ref []

      (* Used for overwriting a tuple with another one.  If    *)
      (* tuple to be overwritten not in, new tuple is inserted *)
      (* at the end.  Maintains order of tuples in list.       *)
    fun replaceOrd ((c,v), [], result) = (rev result) @ [(c,v)]
      | replaceOrd ((c,v), (c1,v1)::l, result) = 
          if c=c1 then (rev result) @ ((c,v)::l)
	  else replaceOrd((c,v),l,(c1,v1)::result)

	(* Same as above excepts works with triples. *)
    fun replace3Ord ((M,Mup,Mdown), [], result) = (rev result)@[(M,Mup,Mdown)]
      | replace3Ord ((M,Mup,Mdown), (M',Mup',Mdown')::l, result) = 
          if M=M' then (rev result) @ ((M,Mup,Mdown)::l)
	  else replace3Ord((M,Mup,Mdown),l,(M',Mup',Mdown')::result)

	(* Same as above excepts works with 4-tuples. *)
    fun replace4Ord ((M1,M2,M3,M4), [], result) = (rev result)@[(M1,M2,M3,M4)]
      | replace4Ord ((M1,M2,M3,M4), (M1',M2',M3',M4')::l, result) = 
          if M1=M1' then (rev result) @ ((M1,M2,M3,M4)::l)
	  else replace4Ord((M1,M2,M3,M4),l,(M1',M2',M3',M4')::result)

    fun store (s, e, l) = if search (s, !l) 
			   then l := replaceOrd ((s,e), !l, [])
		       else l := (s, e)::(!l)

    fun lookup (s, l) = find (s, !l) handle Find => raise Lookup s

	(* functions for handling type environment manipulation *)
    fun lookupExpType (s, []) = raise Lookup ""
      | lookupExpType (s, (s',time,e,T)::l) = 
        if s = s' then (time,e,T) else lookupExpType(s,l)

    fun storeExpType Tlist (s, time, e, T) = 
        Tlist := replace4Ord((s,time,e,T),!Tlist,[])

	(* We keep track of pending recursions with these two vars *)
    val fixList : (CDSInternal.forest * CDSInternal.icell) list ref = ref []
    val fixCounter : int ref = ref 0
    end
    end;


(* PCF parse tree *)
structure PCFBasic =
    struct
    datatype PCF_Expr = pcf_Bool of bool
          | pcf_Int of int
          | pcf_Ident of string
	  | pcf_App of PCF_Expr * PCF_Expr
	  | pcf_Lam of string * PCF_Expr
	  | pcf_Let of string * PCF_Expr * PCF_Expr
	  | pcf_Letrec of string * PCF_Expr * PCF_Expr
	  | pcf_Cond of PCF_Expr * PCF_Expr * PCF_Expr
	  | pcf_Couple of PCF_Expr * PCF_Expr
	  | pcf_Fst of PCF_Expr
	  | pcf_Snd of PCF_Expr
	  | pcf_Cons of PCF_Expr * PCF_Expr
          | pcf_Head of PCF_Expr
          | pcf_Tail of PCF_Expr
          | pcf_Nil
          | pcf_Null of PCF_Expr
	  | pcf_Bop of PCF_Boperation

    and PCF_Boperation = pcf_Plus of PCF_Expr * PCF_Expr
          | pcf_Minus of PCF_Expr * PCF_Expr
	  | pcf_Times of PCF_Expr * PCF_Expr
	  | pcf_Div of PCF_Expr * PCF_Expr
	  | pcf_Equal of PCF_Expr * PCF_Expr
	  | pcf_Less of PCF_Expr * PCF_Expr
	  | pcf_Grtr of PCF_Expr * PCF_Expr
	  | pcf_Leq of PCF_Expr * PCF_Expr
	  | pcf_Geq of PCF_Expr * PCF_Expr
	  | pcf_And of PCF_Expr * PCF_Expr
	  | pcf_Or of PCF_Expr * PCF_Expr

    datatype PCF_ParseTree = PCF_Exp of PCF_Expr
          | PCF_Val of string * PCF_Expr
          | PCF_Load of string
          | PCF_Print of string
	  | PCF_Quit
	  | PCF_Empty

    end;
