(* Cell names Lexer Definition *)

structure Tokens = Tokens
structure Interface = Interface
open Interface

type pos = Interface.pos
type svalue = Tokens.svalue
type ('a,'b) token = ('a,'b) Tokens.token
type lexresult= (svalue,pos) token

(* Keep track of nested comments.  A rather ugly way of doing it. *)
val comment = ref false
val nesting = ref 0

val eof = fn () => if !comment = true 
                      then (comment := false;
                            raise CommentError "unclosed comment")
                   else Tokens.EOF(!line,!line)

%%
%header (functor CellLexFun(structure Tokens: Cell_TOKENS
			    structure Interface : INTERFACE) : LEXER);
alpha=[a-zA-Z];
extAlpha=[A-Za-z0-9_];
fileName=[^\"\n];
digit=[0-9];
ws = [\ \t];
%%
\n       => (next_line(); lex());
{ws}+    => (lex());
{digit}+ => (if !comment = true then lex() 
             else Tokens.INTEGER
                (revfold (fn (a,r) => ord(a)-ord("0")+10*r)
                         (explode yytext) 0,
                  !line,!line));

"(*"       => ((if !comment = true then nesting := !nesting + 1
                else comment := true); lex());
"*)"       => ((if !nesting = 0 andalso !comment = false
                   then raise CommentError "unmatched close comment"
                else if !nesting = 0 then comment := false
                else nesting := !nesting - 1); lex());

"or"       => (if !comment = true then lex() else Tokens.OR(!line,!line));
"and"	   => (if !comment = true then lex() else Tokens.AND(!line,!line));
"with"     => (if !comment = true then lex() else Tokens.WITH(!line,!line));

"valof"    => (if !comment = true then lex() else Tokens.VALOF(!line,!line));
"output"   => (if !comment = true then lex() else Tokens.OUTPUT(!line,!line));

","      => (if !comment = true then lex() else Tokens.COMMA(!line,!line));
"="      => (if !comment = true then lex() else Tokens.EQ(!line,!line));
";"      => (if !comment = true then lex() else Tokens.SEMI(!line,!line));
"."      => (if !comment = true then lex() else Tokens.APPLY(!line,!line));
"{"	 => (if !comment = true then lex() else Tokens.LC(!line,!line));
"}"	 => (if !comment = true then lex() else Tokens.RC(!line,!line));
"("	 => (if !comment = true then lex() else Tokens.LB(!line,!line));
")"	 => (if !comment = true then lex() else Tokens.RB(!line,!line));
"<"	 => (if !comment = true then lex() else Tokens.LA(!line,!line));
">"	 => (if !comment = true then lex() else Tokens.RA(!line,!line));
"["	 => (if !comment = true then lex() else Tokens.LS(!line,!line));
"]"	 => (if !comment = true then lex() else Tokens.RS(!line,!line));
"~"	 => (if !comment = true then lex() else Tokens.MINUS(!line,!line));
"+"	 => (if !comment = true then lex() else Tokens.PLUS(!line,!line));
"-"	 => (if !comment = true then lex() else Tokens.SUB(!line,!line));
"*"	 => (if !comment = true then lex() else Tokens.MULT(!line,!line));
"/"	 => (if !comment = true then lex() else Tokens.DIV(!line,!line));
"!"	 => (if !comment = true then lex() else Tokens.BANG(!line,!line));

{alpha}{extAlpha}*    => (if !comment = true then lex() 
                          else Tokens.ID(yytext,!line,!line));
"$"{alpha}{extAlpha}* => (if !comment = true then lex() 
                          else Tokens.VAR(yytext,!line,!line));

.	=> (if !comment = true then lex() else raise LexError);
