(* CDS Lexer Definition *)

structure Tokens = Tokens
structure Interface = Interface
open Interface

type pos = Interface.pos
type svalue = Tokens.svalue
type ('a,'b) token = ('a,'b) Tokens.token
type lexresult= (svalue,pos) token

(* Keep track of nested comments.  A rather ugly way of doing it. *)
val comment = ref false
val nesting = ref 0

val eof = fn () => if !comment = true 
                      then (comment := false;
                            raise CommentError "unclosed comment")
                   else Tokens.EOF(!line,!line)

%%
%header (functor CDSLexFun(structure Tokens: CDS_TOKENS
			    structure Interface : INTERFACE) : LEXER);
alpha=[a-zA-Z];
extAlpha=[A-Za-z0-9_];
fileName=[^\"\n];
digit=[0-9];
ws = [\ \t];
%%
\n       => (next_line(); lex());
{ws}+    => (lex());
{digit}+ => (if !comment = true then lex() 
             else Tokens.INTEGER
                (revfold (fn (a,r) => ord(a)-ord("0")+10*r)
                         (explode yytext) 0,
                  !line,!line));

"(*"       => ((if !comment = true then nesting := !nesting + 1
                else comment := true); lex());
"*)"       => ((if !nesting = 0 andalso !comment = false
                   then raise CommentError "unmatched close comment"
                else if !nesting = 0 then comment := false
                else nesting := !nesting - 1); lex());

"dcds"     => (if !comment = true then lex() else Tokens.DCDS(!line,!line));
"algo"     => (if !comment = true then lex() else Tokens.ALGO(!line,!line));
"end"      => (if !comment = true then lex() else Tokens.END(!line,!line));
"cell"     => (if !comment = true then lex() else Tokens.CELL(!line,!line));
"graft"    => (if !comment = true then lex() else Tokens.GRAFT(!line,!line));
"access"   => (if !comment = true then lex() else Tokens.ACCESS(!line,!line));
"values"   => (if !comment = true then lex() else Tokens.VALUES(!line,!line));
"or"       => (if !comment = true then lex() else Tokens.OR(!line,!line));
"and"	   => (if !comment = true then lex() else Tokens.AND(!line,!line));
"let"      => (if !comment = true then lex() else Tokens.LET(!line,!line));
"letrec"   => (if !comment = true then lex() else Tokens.LETREC(!line,!line));
"with"     => (if !comment = true then lex() else Tokens.WITH(!line,!line));
"local"    => (if !comment = true then lex() else Tokens.LOCAL(!line,!line));
"in"       => (if !comment = true then lex() else Tokens.IN(!line,!line));

"request"  => (if !comment = true then lex() else Tokens.REQUEST(!line,!line));
"do"       => (if !comment = true then lex() else Tokens.DO(!line,!line));
"from"     => (if !comment = true then lex() else Tokens.FROM(!line,!line));
"valof"    => (if !comment = true then lex() else Tokens.VALOF(!line,!line));
"is"       => (if !comment = true then lex() else Tokens.IS(!line,!line));
"output"   => (if !comment = true then lex() else Tokens.OUTPUT(!line,!line));

"fix"    => (if !comment = true then lex() else Tokens.FIX(!line,!line));
"curry"  => (if !comment = true then lex() else Tokens.CURRY(!line,!line));
"uncurry" => (if !comment = true then lex() else Tokens.UNCURRY(!line,!line));

"load"   => (if !comment = true then lex() else Tokens.LOAD(!line,!line));
"loadecho" => (if !comment = true then lex() 
               else Tokens.LOADECHO(!line,!line));
"print"	 => (if !comment = true then lex() else Tokens.PRINT(!line,!line));
"timer"	 => (if !comment = true then lex() else Tokens.TIMER(!line,!line));
"trace"	 => (if !comment = true then lex() else Tokens.TRACE(!line,!line));
"typing" => (if !comment = true then lex() else Tokens.TYPING(!line,!line));
"show"   => (if !comment = true then lex() else Tokens.SHOW(!line,!line));
"more"   => (if !comment = true then lex() else Tokens.MORE(!line,!line));
"hierarchy" => (if !comment = true then lex() 
                else Tokens.HIERARCHY(!line,!line));
"on"	 => (if !comment = true then lex() else Tokens.ON(!line,!line));
"off"	 => (if !comment = true then lex() else Tokens.OFF(!line,!line));
"env"	 => (if !comment = true then lex() else Tokens.ENV(!line,!line));
"omega"	 => (if !comment = true then lex() else Tokens.OMEGA(!line,!line));
"pcf"	 => (if !comment = true then lex() else Tokens.PCF(!line,!line));
"refine" => (if !comment = true then lex() else Tokens.REFINE(!line,!line));

","      => (if !comment = true then lex() else Tokens.COMMA(!line,!line));
"="      => (if !comment = true then lex() else Tokens.EQ(!line,!line));
";"      => (if !comment = true then lex() else Tokens.SEMI(!line,!line));
"."      => (if !comment = true then lex() else Tokens.APPLY(!line,!line));
":"	 => (if !comment = true then lex() else Tokens.COLON(!line,!line));
"{"	 => (if !comment = true then lex() else Tokens.LC(!line,!line));
"}"	 => (if !comment = true then lex() else Tokens.RC(!line,!line));
"("	 => (if !comment = true then lex() else Tokens.LB(!line,!line));
")"	 => (if !comment = true then lex() else Tokens.RB(!line,!line));
"|"	 => (if !comment = true then lex() else Tokens.COMPOSE(!line,!line));
"<"	 => (if !comment = true then lex() else Tokens.LA(!line,!line));
">"	 => (if !comment = true then lex() else Tokens.RA(!line,!line));
"["	 => (if !comment = true then lex() else Tokens.LS(!line,!line));
"]"	 => (if !comment = true then lex() else Tokens.RS(!line,!line));
"~"	 => (if !comment = true then lex() else Tokens.MINUS(!line,!line));
"+"	 => (if !comment = true then lex() else Tokens.PLUS(!line,!line));
"-"	 => (if !comment = true then lex() else Tokens.SUB(!line,!line));
"*"	 => (if !comment = true then lex() else Tokens.MULT(!line,!line));
"/"	 => (if !comment = true then lex() else Tokens.DIV(!line,!line));
"!"	 => (if !comment = true then lex() else Tokens.BANG(!line,!line));

{alpha}{extAlpha}*    => (if !comment = true then lex() 
                          else Tokens.ID(yytext,!line,!line));
"$"{alpha}{extAlpha}* => (if !comment = true then lex() 
                          else Tokens.VAR(yytext,!line,!line));
\"{fileName}*\" => (if !comment = true then lex() 
                    else Tokens.FILE
		(implode(tl(CDSBasic.take((size yytext)-1, explode yytext))),
		   !line,!line));

.	=> (if !comment = true then lex() else raise LexError);
