structure NaturalDeduction = struct
(*****************************************************************)
(*                          Definitions                          *)
(*****************************************************************)

  (* Commonly used hypothesis names are "u", "v", and "w" *)
  type hypothesis = string

  (* Using the common atom structure from logic.sml *)
  (* datatype logic.atom = AA | AB | AC | AD | AE *)
  open Atom

  (* Using the basic proposition structure from logic.sml *)
  (* Basic propositions consist of True, False, Atom, And, Or, Implication *)
  open BasicProp

  (* There is one judgment: that a proposition is true *)
  datatype judgment
    = JTrue of prop
    
  datatype rule
    = Hyp of hypothesis
    | TrueI
    | FalseE
    | AndI | AndE1 | AndE2
    | OrI1 | OrI2 | OrE of hypothesis * hypothesis
    | ImpI of hypothesis | ImpE

(* also defined : datatype proof = ProofTree of proof list * rule * judgment *)

(*****************************************************************)
(*                             Macros                            *)
(*****************************************************************)
  (* 
   * Macros for writing propositions
   * These will be infixed for you in a HW's starter code if it uses natural deduction 
   *)

  (* Atom proposition shortcuts *)
  val (A,B,C,D,E) = (PAtom AA,PAtom AB,PAtom AC,PAtom AD,PAtom AE)
  
  (*
   * Connective shortcuts in order of precedence
   * T for truth
   * F for false
   * ~ for negation (aka implying false)
   * ^ for and
   * v for or
   * > for implies
   *)
  val (op^,v, op>, ~, T, F) = (PAnd,POr,PImp,fn a => PImp(a,PFalse), PTrue, PFalse)

  (* 
   * Truth judgment magic to use with -- and ==
   * Example Usage:
     
     [proof_of_A_true,proof_of_B_true]
     --------- AndI
     tru A ^ B
     
   *)
  fun tru (f,p) = (f,JTrue p)


(*****************************************************************)
(*                            Printing                           *)
(*****************************************************************)

  fun rule_to_string rule = case rule of
      Hyp u => String.concat ["hyp_", u]
    | TrueI => "TI"
    | FalseE => "FE"
    | AndI => "^I"
    | AndE1 => "^E1"
    | AndE2 => "^E2"
    | OrI1 => "vI1"
    | OrI2 => "vI2"
    | OrE (u,v) => String.concat ["vE_", u, "_", v]
    | ImpI u => String.concat [">I_", u]
    | ImpE => ">E"
  
  fun judgment_to_string (JTrue p) = String.concat [prop_to_string p, " true"]


(*****************************************************************)
(*            See logic.sml for additional definitions           *)
(*****************************************************************)
  structure Logic = Logic(struct
    type judgment = judgment
    type rule = rule
    val judgment_to_string = judgment_to_string
    val rule_to_string = rule_to_string
    val name = "NaturalDeduction"
  end
  )
  open Logic
end