(* Each logic will define these properties, and call the functor below *)
signature SYNTAX =
sig
  type rule
  type judgment
  val rule_to_string : rule -> string
  val judgment_to_string : judgment -> string
  val name : string
end

functor Logic (Syntax : SYNTAX) = struct

  (* A proof is a collection of premises (proof list), a rule, and a conclusion *)
  datatype proof = ProofTree of proof list * Syntax.rule * Syntax.judgment

  (* 
   * Macros for writing proofs
   * Single lines are for inference rules
   * Plus sign indicase a single premise instead of a list
   * All line lengths are identical
   * These will be infixed for you in each HW's starter code
   *)
  fun -- (proofs, (rule, judgment)) = ProofTree (proofs , rule, judgment)
  fun +- (proof , (rule, judgment)) = ProofTree ([proof], rule, judgment)

  val (+--,----------) = (+-,--)
  val (+---,---------) = (+-,--)
  val (+----,--------) = (+-,--)
  val (+-----,-------) = (+-,--)
  val (+------,------) = (+-,--)
  val (+-------,-----) = (+-,--)
  val (+--------,----) = (+-,--)
  val (+---------,---) = (+-,--)


  (* Turn a proof into a string using the Printing backend *)
  val proof_to_string : proof -> string = 
    Printing.proof_to_string Syntax.rule_to_string Syntax.judgment_to_string (ProofTree, (fn ProofTree p => p))

  (* This will tell the SML/NJ repl how to display the type "proof" when you call Printing.activate() *)
  val _ = Printing.register (Syntax.name, proof_to_string)
end

structure Atom = struct
  (*
   * In logic, a metavariable is a symbol which belongs to a metalanguage and
   * stands for elements of some object language.
   *
   * We haven't defined what atomic facts our logic is talking about.
   * That means we need to use "metavariables". You don't need more
   * than five of these in the homeworks we give you.
   * https://en.wikipedia.org/wiki/Metavariable
   *)
  datatype atom = AA | AB | AC | AD | AE

  val atom_to_string = fn AA => "A" | AB => "B" | AC => "C" | AD => "D" | AE => "E"
end

structure BasicProp = struct
  open Atom

  datatype prop
    = PTrue
    | PFalse
    | PAtom of atom
    | PAnd of prop * prop
    | POr of prop * prop
    | PImp of prop * prop
  
  (* Used for printing *)
  fun prop_to_string prop = case prop of
      PTrue => "T"
    | PFalse => "F"
    | PAtom a => atom_to_string a
    | PAnd (a,b) => (fn (PAtom _|PTrue|PFalse) => prop_to_string a | _ => "(" ^ prop_to_string a ^ ")") a ^ " ^ " ^ (fn (PAtom _|PTrue|PFalse) => prop_to_string b | _ => "(" ^ prop_to_string b ^ ")") b
    | POr (a,b) => (fn (PAtom _|PTrue|PFalse|PAnd _) => prop_to_string a | _ => "(" ^ prop_to_string a ^ ")") a ^ " v " ^ (fn (PAtom _|PTrue|PFalse|PAnd _) => prop_to_string b | _ => "(" ^ prop_to_string b ^ ")") b
    | PImp (a as PImp _,b) => "(" ^ prop_to_string a ^ ") > " ^ prop_to_string b
    | PImp (a,b) => prop_to_string a ^ " > " ^ prop_to_string b
  
end

structure Void :> sig type void val abort : void -> 'a end = struct
  (* This type is empty. By definition, a value of type void can never be constructed. *)
  (* SML doesn't let us define a type with no constructors, so this is the best we can get. *)
  datatype void = Void of void
  (* Believe it or not this function is provably total *)
  fun abort (Void v) = abort v
end