structure InversionG4ip = struct
(*****************************************************************)
(*                          Definitions                          *)
(*****************************************************************)

  (* Commonly used hypothesis names are "u", "v", and "w" *)
  type hypothesis = string
  
  (* Using the common atom structure from logic.sml *)
  (* datatype logic.atom = AA | AB | AC | AD | AE *)
  open Atom

  (* Using the basic proposition structure from logic.sml *)
  (* Basic propositions consist of True, False, Atom, And, Or, Implication *)
  open BasicProp

  val context_to_string = Printing.context_to_string

  (* The type of positive propositions (aka right non-invertible propositions) *)
  datatype pos
    = PosOr of prop * prop
    | PosFalse
    | PosAtom of atom
  (* The type of negative propositions (aka left non-invertible propositions) *)
  datatype neg
    = NegImpAtom of atom * prop
    | NegImpImp of (prop * prop) * prop
    | NegAtom of atom
  

  (* Inversion G4IP has three judgments:
   * Right inversion
   * 
   *     Delta-;Omega -R-> C
   *
   * Left inversion
   *
   *     Delta-;Omega -L-> C+
   *
   * Search
   *
   *     Delta- -S-> C+
   *)
  datatype judgment
    = JInvR of neg list * prop list * prop
    | JInvL of neg list * prop list * pos
    | JSearch of neg list * pos
    

  datatype rule
    (* right inversion rules *)
    = ImpR | AndR | TrueR | Switch
    (* left inversion rules *)
    | OrL | TrueL | AndL | FalseL
    | ImpTrueL | ImpFalseL | ImpAndL | ImpOrL
    | Shift | Search
    (* search rules *)
    | Init | ImpAtomL | ImpImpL
    | OrR1 | OrR2

(* also defined : datatype proof = ProofTree of proof list * rule * judgment *)


(*****************************************************************)
(*                            Printing                           *)
(*****************************************************************)

  fun pos_to_string pos = case pos of
      PosOr p => prop_to_string (POr p)
    | PosFalse => prop_to_string (PFalse)
    | PosAtom a => prop_to_string (PAtom a)
  
  fun neg_to_string neg = case neg of
      NegImpAtom (a,b) => prop_to_string (PImp (PAtom a,b))
    | NegImpImp (a,b)=> prop_to_string (PImp (PImp a,b))
    | NegAtom a => prop_to_string (PAtom a)
  
  fun judgment_to_string judgment = case judgment of
      JInvR (D,O,C) => context_to_string neg_to_string D ^ ";" ^ context_to_string prop_to_string O ^ " -R-> " ^ prop_to_string C
    | JInvL (D,O,C) => context_to_string neg_to_string D ^ ";" ^ context_to_string prop_to_string O ^ " -L-> " ^ pos_to_string C
    | JSearch (D,C) => context_to_string neg_to_string D ^ " -S-> " ^ pos_to_string C
  
  fun rule_to_string rule = case rule of
      Init => "init"
    | TrueR => "TR"
    | TrueL => "TL"
    | FalseL => "FL"
    | AndR => "^R"
    | AndL => "^L2"
    | OrR1 => "vR1"
    | OrR2 => "vR2"
    | OrL => "vL"
    | ImpR => ">R"
    | ImpAtomL => "p>L"
    | ImpTrueL => "T>L"
    | ImpFalseL => "F>L"
    | ImpAndL => "^>L"
    | ImpOrL => "v>L"
    | ImpImpL => ">>L"
    | Switch => "switch"
    | Shift => "shift"
    | Search => "search"

(*****************************************************************)
(*            See logic.sml for additional definitions           *)
(*****************************************************************)
  structure Logic = Logic(struct
    type judgment = judgment
    type rule = rule
    val judgment_to_string = judgment_to_string
    val rule_to_string = rule_to_string
    val name = "InversionG4ip"
  end
  )
  open Logic
end