structure HExamples = struct
  open Harmony

  infixr 9 ^ ?
  infixr 8 v
  infixr 7 >

  infix 4 tru

  infix 3 -- +--------- ========== +=
  infix 3 --- +-------- ========= +==
  infix 3 ---- +------- ======== +===
  infix 3 ----- +------ ======= +====
  infix 3 ------ +----- ====== +=====
  infix 3 ------- +---- ===== +======
  infix 3 -------- +--- ==== +=======
  infix 3 --------- +-- === +========
  infix 3 ---------- +- == +=========

  (* We use ? to denote the relevent connective *)
  (* Rules are just labelled as Intro or Elim and unrestricted on number of hypotheses *)
  (* There are also numbered versions Intro1 Intro2 Elim1 Elim2 of the rules *)

  (* Soundness is proved with a reduction *)

  val and_soundness : Harmony.localreductions =
  [
    ( (* Reduction 1 *)
      (* lhs *)
      [
        given DD
        tru A
      ,
        given EE
        tru B
      ]
      ---------- Intro []
      tru A ? B
      +--------- Elim1 []
      tru A
    ,
      (* rhs *)
      given DD
      tru A
    )
  ,
    ( (* Reduction 2 *)
      (* lhs *)
      [
        given DD
        tru A
      ,
        given EE
        tru B
      ]
      ---------- Intro []
      tru A ? B
      +--------- Elim2 []
      tru B
    ,
      (* rhs *)
      given EE
      tru B
    )
  ]

  val and_completeness : Harmony.localexpansion =

  ( (* lhs *)
    given DD
    tru A ? B
  ,
    (* rhs *)
    [
      given DD
      tru A ? B
      +--------- Elim1 []
      tru A
    ,
      given DD
      tru A ? B
      +--------- Elim2 []
      tru B
    ]
    ---------- Intro []
    tru A ? B
  )

  val imp_soundness : Harmony.localreductions =
  [ (* Only one reduction *)
    ( (*lhs*)
      [
        given_hyps_tru [("x",A)]
        FF
        tru B
        +--------- Intro ["x"]
        tru A ? B
      ,
        given DD
        tru A
      ]
      ---------- Elim []
      tru B
    ,
      (* rhs *)
      subst [DD/"x"] FF
      tru B
    )

  ]

  val imp_completeness : Harmony.localexpansion =
  ( (* lhs *)
    given DD
    tru A ? B
  ,
    (* rhs *)
    [
      given DD
      tru A ? B
    ,
      []
      ---------- Hyp"u"
      tru A
    ]
    ---------- Elim []
    tru B
    +--------- Intro ["u"]
    tru A ? B
  )

  (* An example with multiple hypothesis in one premise *)
  val example_connective_intro : RuleCreation.nd_def =
    
    prem_hyps_tru [("x",A),("y",B)]
    tru C
    +-------- Intro ["x","y"]
    tru ??[A,B,C]
  
  val example_connective_elim : RuleCreation.nd_def =

    [
      prem
      tru ??[A,B,C]
    ,
      prem
      tru A
    ,
      prem
      tru B
    ]
    --------- Elim []
    tru C
  
  val example_connective_soundness : Harmony.localreductions =
  [ (* Only one reduction *)
    ( (*lhs*)
      [
        given_hyps_tru [("x",A),("y",B)]
        FF
        tru C
        +--------- Intro ["x","y"]
        tru ??[A,B,C]
      ,
        given DD
        tru A
      ,
        given EE
        tru B
      ]
      ---------- Elim []
      tru C
    ,
      (* rhs *)
      subst [DD/"x",EE/"y"] FF
      tru C
    )
  ]

  val example_connective_completeness : Harmony.localexpansion =
  ( (* lhs *)
    given DD
    tru ??[A,B,C]
  ,
    (* rhs *)
    [
      given DD
      tru ??[A,B,C]
    ,
      []
      ---------- Hyp"u"
      tru A
    ,
      []
      ---------- Hyp"v"
      tru B
    ]
    ---------- Elim []
    tru C
    +--------- Intro ["u","v"]
    tru ??[A,B,C]
  )
end
