structure Void = struct
  (* This type is empty. By definition, a value of type void can never be constructed. *)
  (* SML doesn't let us define a type with no constructors, so this is the best we can get. *)
  datatype void = Void of void
  (* Believe it or not this function is provably total *)
  fun abort (Void v) = abort v
end
structure Logic = struct
  open Void
  datatype either = datatype Either.either
  type 'a not = 'a -> void

  (* Commonly used hypothesis names are "u", "v", and "w" *)
  type hypothesis = string

  (*
   * A proof tree is either:
   * - A proof of a judgment using an inference rule and premises
   * - An invocation of a theorem to show a judgment
   * - A reference to a judgment which is considered given
   *)
  datatype ('given,'theorem,'rule,'judgment) abstractprooftree
    = ProofTree of ('given,'theorem,'rule,'judgment) abstractprooftree list * 'rule    * 'judgment
    | Theorem   of ('given,'theorem,'rule,'judgment) abstractprooftree list * 'theorem * 'judgment
    | Given     of  'given * 'judgment

  (*
   * In logic, a metavariable is a symbol which belongs to a metalanguage and
   * stands for elements of some object language.
   *
   * We haven't defined what atomic facts our logic is talking about.
   * That means we need to use "metavariables". You don't need more
   * than five of these in the homeworks we give you.
   * https://en.wikipedia.org/wiki/Metavariable
   *)
  datatype atom = AA | AB | AC | AD | AE
  
  datatype givenname = DD | EE | FF | GG | HH

  (* 
   * Macros for writing proofs
   * Single lines are for inference rules
   * Double lines are for invoking theorems
   * Plus sign indicase a single premise instead of a list
   * All line lengths are identical
   * These will be infixed for you in each HW's starter code
   *)
  (* fun -- (proofs, rule) judgment = ProofTree (proofs , rule, judgment)
  fun +- (proof , rule) judgment = ProofTree ([proof], rule, judgment)

  fun == (proofs, theorem) judgment = Theorem (proofs , theorem, judgment)
  fun += (proof , theorem) judgment = Theorem ([proof], theorem, judgment) *)

  fun -- (proofs, (rule, judgment)) = ProofTree (proofs , rule, judgment)
  fun +- (proof , (rule, judgment)) = ProofTree ([proof], rule, judgment)

  fun == (proofs, (theorem, judgment)) = Theorem (proofs , theorem, judgment)
  fun += (proof , (theorem, judgment)) = Theorem ([proof], theorem, judgment)

  val (+--,----------,+=========,===) = (+-,--,+=,==)
  val (+---,---------,+========,====) = (+-,--,+=,==)
  val (+----,--------,+=======,=====) = (+-,--,+=,==)
  val (+-----,-------,+======,======) = (+-,--,+=,==)
  val (+------,------,+=====,=======) = (+-,--,+=,==)
  val (+-------,-----,+====,========) = (+-,--,+=,==)
  val (+--------,----,+===,=========) = (+-,--,+=,==)
  val (+---------,---,+==,==========) = (+-,--,+=,==)

  datatype ('a,'b) slash = / of 'a * 'b
  type no_givens = void
  structure Theorem = struct
    (* These three common theorems sometimes extend logics *)

    (* each type argument should be void or unit *)
    datatype ('id,'cut,'weak) theorem_selection
      = TId of 'id
      | TCut of 'cut
      | TWeak of 'weak
    
    val Id   = TId ()
    val Cut  = TCut ()
    val Weak = TWeak ()

    local type void = Void.void in
    type no_thms       = (void,void,void) theorem_selection
    type id_thm        = (unit,void,void) theorem_selection
    type cut_thm       = (void,unit,void) theorem_selection
    type weak_thm      = (void,void,unit) theorem_selection
    type id_cut_thms   = (unit,unit,void) theorem_selection
    type id_weak_thms  = (unit,void,unit) theorem_selection
    type cut_weak_thms = (void,unit,unit) theorem_selection
    type all_thms      = (unit,unit,unit) theorem_selection

    (* Aliases for clarity to add id or cut to a task *)
    type allow_id = id_thm
    type allow_cut = cut_thm
    end
  end
  val Id   = Theorem.Id
  val Cut  = Theorem.Cut
  val Weak = Theorem.Weak
end
structure Theorem = Logic.Theorem
