structure HeytingArithmetic = struct
  open Logic

  type variable = hypothesis

  datatype object
    = Var of variable
    | Z
    | S of object

  (* Instead of just dealing with generic atoms we also have access to atoms
   * dependent on objects and the equality atom *)
  datatype atom
    = AA
    | AB
    | AC
    | AD
    | AE
    | AAof of object
    | ABof of object
    | ACof of object
    | ADof of object
    | AEof of object
    | Equals of object * object

  (* The only type of object we've defined is natural numbers *)
    datatype typ
    = Tyvar of variable
    | Nat

  (* We have all the normal propositions plus existentials and quantifiers*)
  datatype prop
    = PTrue
    | PFalse
    | PAtom of atom
    | PAnd of prop * prop
    | POr of prop * prop
    | PImp of prop * prop
    | PForall of variable * typ * prop
    | PExists of variable * typ * prop

  (* There are two judgements instead of one now *)
  datatype judgment
    = JTrue of prop
    | JOfTyp of object * typ

  datatype rule
    = Hyp of hypothesis
    | TrueI
    | FalseE
    | AndI | AndE1 | AndE2
    | OrI1 | OrI2 | OrE of hypothesis * hypothesis
    | ImpI of hypothesis | ImpE
    | ForallI of hypothesis | ForallE
    | ExistsI | ExistsE of hypothesis * hypothesis
    | NatI0 | NatIs | NatE of variable * hypothesis
    | EqualsI00 | EqualsIss | EqualsE0s | EqualsEs0 | EqualsEss

  type given = void
  type theorem = void
  type proof = (given, theorem, rule, judgment) abstractprooftree
  type ('given,'theorem) prooftree = ('given,'theorem, rule, judgment) abstractprooftree

  (*
   * Theorems
   * Because we showed the theorem Forall x:nat. x=x, we can augment the logic
   * with invocations of that theorem. These theorem augmented proof trees can
   * be converted into regular proof trees by filling in the known proof
   *)
  
  (*
   * Theorem Refl: forall x : nat. x = x
   *
   *  x : nat
   * ========== Refl
   * x = x true
   *
   * Proof contained in ha_examples.sml
   *)
  
  datatype refl_admissible = Refl
  type proof_refl_admissible = (void,refl_admissible) prooftree


  (*
   * Macros for writing propositions
   * These will be infixed for you in a HW's starter code if it uses natural deduction
   *)

  (* Atom proposition shortcuts *)
  val (A,B,C,D,E) = (PAtom AA,PAtom AB,PAtom AC,PAtom AD,PAtom AE)
  val (Aof,Bof,Cof,Dof,Eof) = (PAtom o AAof,PAtom o ABof,PAtom o ACof,PAtom o ADof,PAtom o AEof)
  fun eq (x,y) = PAtom (Equals (x,y))
  val (tau, tau1, tau2) = (Tyvar "t", Tyvar "t1", Tyvar "t2")

  (*
   * Connective shortcuts in order of precedence
   * T for truth
   * F for false
   * ~ for negation (aka implying false)
   * ^ for and
   * v for or
   * > for implies
   * exi for exists
   * for for forall
   *)
  val (op^,v, op>, ~, T, F) = (PAnd,POr,PImp,fn a => PImp(a,PFalse), PTrue, PFalse)
  fun forall x t p = PForall(x,t,p)
  fun exists x t p = PExists(x,t,p)
  val for = forall
  val exi = exists
  val var = Var

  (*
   * Truth judgment magic to use with -- and ==
   * Example Usage:

     [proof_of_A_true,proof_of_B_true]
     --------- AndI
     tru A ^ B

   *)
  fun tru (f,p) = (f,JTrue p)
  fun nat (f,p) = (f,JOfTyp (p,Nat))
  fun obj (f,p) = (f,p)
  val ::: = JOfTyp

  fun given_hyps hyps name prop = Given ((hyps,name),JTrue prop)
  fun given name prop = given_hyps [] name prop
end
