structure Out : sig

  val atom_to_string : Logic.atom -> string

  val prop_to_string : NaturalDeduction.prop -> string

  val term_to_string : ProofsAsPrograms.term -> string

  val context_to_string : ('a -> string) -> 'a list -> string

  val givenname_to_string : Logic.givenname -> string

  val ha_obj_to_string : HeytingArithmetic.object -> string
  val ha_typ_to_string : HeytingArithmetic.typ -> string
  val ha_prop_to_string : HeytingArithmetic.prop -> string
  val rc_prop_to_string : RuleCreation.prop -> string
  val nsc_prop_to_string : NegSequentCalculus.prop -> string
  val f_pos_to_string : Focusing.pos -> string
  val f_neg_to_string : Focusing.neg -> string
  val ll_prop_to_string : LinearLogic.prop -> string

  val rc_rule_to_string : RuleCreation.rule -> string

  val print_nd  : NaturalDeduction.proof -> unit
  val print_pap : ProofsAsPrograms.proof -> unit
  val print_vau : VerificationsAndUses.proof -> unit
  val print_ha  : HeytingArithmetic.proof -> unit
  val print_sc : SequentCalculus.proof -> unit

  val print_cnd : ContextNaturalDeduction.proof -> unit
  val print_nsc : NegSequentCalculus.proof -> unit
  val print_g4ip : InversionG4ip.proof -> unit
  val print_f : Focusing.proof -> unit

  val print_ll : LinearLogic.proof -> unit

  val print_rc_nd  : RuleCreation.nd_def -> unit
  val print_rc_vau : RuleCreation.vau_def -> unit

  val print_reductions : Harmony.localreductions -> unit
  val print_expansion : Harmony.localexpansion  -> unit

  val nd_judgment_to_string : NaturalDeduction.judgment -> string
  val pap_judgment_to_string : ProofsAsPrograms.judgment -> string
  val vau_judgment_to_string : VerificationsAndUses.judgment -> string
  val ha_judgment_to_string : HeytingArithmetic.judgment -> string
  val sc_judgment_to_string : SequentCalculus.judgment -> string

  val rc_nd_judgment_to_string : RuleCreation.nd_judgment -> string
  val rc_vau_judgment_to_string : RuleCreation.vau_judgment -> string
  val cnd_judgment_to_string : ContextNaturalDeduction.judgment -> string
  val nsc_judgment_to_string : NegSequentCalculus.judgment -> string
  val g4ip_judgment_to_string : InversionG4ip.judgment -> string
  val f_judgment_to_string : Focusing.judgment -> string

  val ll_judgment_to_string : LinearLogic.judgment -> string

  val formatting : bool ref
  
end = struct
  (* alias structures *)
  structure L = Logic
  structure ND = NaturalDeduction
  structure PAP = ProofsAsPrograms
  structure VAU = VerificationsAndUses
  structure SC = SequentCalculus
  structure RC = RuleCreation
  structure H = Harmony
  structure HA = HeytingArithmetic
  structure CND = ContextNaturalDeduction
  structure NSC = NegSequentCalculus
  structure G4IP = InversionG4ip
  structure F = Focusing
  structure LL = LinearLogic
  (* copy datatypes *)
  datatype atom = datatype ND.atom
  datatype prop = datatype ND.prop
  datatype rule = datatype ND.rule
  datatype term = datatype PAP.term

  (* formatting *)
  val formatting = ref true
  fun format code str = if !formatting then "\027[" ^ code ^ "m" ^ str ^ "\027[1;0m" else str
  val red  = format "31;1"
  val blue = format "34;1"
  val bold = format "1"


  fun givenname_to_string name = case name of
      L.DD => "DD"
    | L.EE => "EE"
    | L.FF => "FF"
    | L.GG => "GG"
    | L.HH => "HH"

  fun context_to_string p2s gamma = case gamma of
      [] => "•"
    | _ => String.concatWithMap ", " p2s gamma

  (* Natural Deduction *)
  fun nd_rule_to_string rule = case rule of
      Hyp u => "hyp_" ^ u
    | TrueI => "TI"
    | FalseE => "FE"
    | AndI => "^I"
    | AndE1 => "^E1"
    | AndE2 => "^E2"
    | OrI1 => "vI1"
    | OrI2 => "vI2"
    | OrE (u,v) => "vE_" ^ u ^ "_" ^ v
    | ImpI u => ">I_" ^ u
    | ImpE => ">E"
  fun atom_to_string atom = case atom of AA => "A" | AB => "B" | AC => "C" | AD => "D" | AE => "E"
  fun prop_to_string prop = case prop of
      PTrue => "T"
    | PFalse => "F"
    | PAtom a => atom_to_string a
    | PAnd (a,b) => (fn (PAtom _|PTrue|PFalse) => prop_to_string a | _ => "(" ^ prop_to_string a ^ ")") a ^ " ^ " ^ (fn (PAtom _|PTrue|PFalse) => prop_to_string b | _ => "(" ^ prop_to_string b ^ ")") b
    | POr (a,b) => (fn (PAtom _|PTrue|PFalse|PAnd _) => prop_to_string a | _ => "(" ^ prop_to_string a ^ ")") a ^ " v " ^ (fn (PAtom _|PTrue|PFalse|PAnd _) => prop_to_string b | _ => "(" ^ prop_to_string b ^ ")") b
    | PImp (a as PImp _,b) => "(" ^ prop_to_string a ^ ") > " ^ prop_to_string b
    | PImp (a,b) => prop_to_string a ^ " > " ^ prop_to_string b
  fun nd_judgment_to_string (ND.JTrue p) = prop_to_string p ^ " true"

  (* Proofs as Programs *)
  fun term_precedence term = case term of
      (Case _ | Lambda _) => 1
    | (Abort _ | Fst _ | Snd _ | Inl _ | Inr _ | App _) => 2
    | (Var _ | Unit | Tuple _) => 3
  fun parens ifleq term = if term_precedence term <= ifleq then "(" ^ term_to_string term ^ ")" else term_to_string term
  and term_to_string term = case term of
      Var u => u
    | Unit => "⟨⟩"
    | Abort e => "abort " ^ parens 2 e
    | Tuple (e1,e2) => "⟨" ^ parens 0 e1 ^ "," ^ parens 0 e2 ^ "⟩"
    | Fst e => "fst " ^ parens 2  e
    | Snd e => "snd " ^ parens 2  e
    | Inl e => "INL " ^ parens 2  e
    | Inr e => "INR " ^ parens 2  e
    | Case (e0,(x,e1),(y,e2)) => "case " ^ parens 0 e0 ^ " of INL " ^ x ^ " => " ^ parens 1 e1 ^ " | INR " ^ y ^ " => " ^ parens 1 e2
    | Lambda (x,e) => "λ" ^ x ^ "." ^ parens 0 e
    | App (e1,e2) => parens 1 e1 ^ " " ^ parens 2 e2
  fun pap_judgment_to_string (PAP.JHasType (t,p)) = term_to_string t ^ " : " ^ prop_to_string p

  (* Verifications and Uses *)
  fun vau_rule_to_string rule = case rule of
      VAU.Hyp u => "hyp_" ^ u
    | VAU.TrueI => "TI"
    | VAU.FalseE => "FE"
    | VAU.AndI => "^I"
    | VAU.AndE1 => "^E1"
    | VAU.AndE2 => "^E2"
    | VAU.OrI1 => "vI1"
    | VAU.OrI2 => "vI2"
    | VAU.OrE (u,v) => "vE_" ^ u ^ "_" ^ v
    | VAU.ImpI u => ">I_" ^ u
    | VAU.ImpE => ">E"
    | VAU.DownUp => "↓↑"
  fun vau_judgment_to_string judgment = case judgment of
      VAU.JUp p => prop_to_string p ^ " ↑"
    | VAU.JDown p => prop_to_string p ^ " ↓"
  
  (* Sequent Calculus *)
  fun sc_rule_to_string rule = case rule of
      SC.Init => "init"
    | SC.TrueR => "TR"
    | SC.FalseL => "FL"
    | SC.AndR => "^R"
    | SC.AndL1 => "^L1"
    | SC.AndL2 => "^L2"
    | SC.OrR1 => "vR1"
    | SC.OrR2 => "vR2"
    | SC.OrL => "vL"
    | SC.ImpR => ">R"
    | SC.ImpL => ">L"

  fun sc_judgment_to_string (SC.JSeq (gamma,c)) =
    context_to_string prop_to_string gamma ^ " ==> " ^ prop_to_string c

  (* Context Natural Deduction *)

  fun cnd_rule_to_string rule = case rule of
      CND.Hyp => "hyp"
    | CND.TrueI => "TI"
    | CND.FalseE => "FE"
    | CND.AndI => "^I"
    | CND.AndE1 => "^E1"
    | CND.AndE2 => "^E2"
    | CND.OrI1 => "vI1"
    | CND.OrI2 => "vI2"
    | CND.OrE => "vE"
    | CND.ImpI => ">I"
    | CND.ImpE => ">E"

  fun cnd_prop_to_string prop = case prop of
      CND.PTrue => "T"
    | CND.PFalse => "F"
    | CND.PAtom a => atom_to_string a
    | CND.PAnd (a,b) => (fn (CND.PAtom _|CND.PTrue|CND.PFalse) => cnd_prop_to_string a | _ => "(" ^ cnd_prop_to_string a ^ ")") a ^ " ^ " ^ (fn (CND.PAtom _|CND.PTrue|CND.PFalse) => cnd_prop_to_string b | _ => "(" ^ cnd_prop_to_string b ^ ")") b
    | CND.POr (a,b) => (fn (CND.PAtom _|CND.PTrue|CND.PFalse|CND.PAnd _) => cnd_prop_to_string a | _ => "(" ^ cnd_prop_to_string a ^ ")") a ^ " v " ^ (fn (CND.PAtom _|CND.PTrue|CND.PFalse|CND.PAnd _) => cnd_prop_to_string b | _ => "(" ^ cnd_prop_to_string b ^ ")") b
    | CND.PImp (a as CND.PImp _,b) => "(" ^ cnd_prop_to_string a ^ ") > " ^ cnd_prop_to_string b
    | CND.PImp (a,b) => cnd_prop_to_string a ^ " > " ^ cnd_prop_to_string b

  fun cnd_judgment_to_string (CND.JTrue (gamma,c)) =
    context_to_string cnd_prop_to_string gamma ^ " |- " ^ cnd_prop_to_string c

  (* Negation Sequent Calculus *)
  fun nsc_rule_to_string rule = case rule of
      NSC.Init => "init"
    | NSC.TrueR => "TR"
    | NSC.FalseL => "FL"
    | NSC.AndR => "^R"
    | NSC.AndL1 => "^L1"
    | NSC.AndL2 => "^L2"
    | NSC.OrR1 => "vR1"
    | NSC.OrR2 => "vR2"
    | NSC.OrL => "vL"
    | NSC.ImpR => ">R"
    | NSC.ImpL => ">L"
    | NSC.NotR => "~R"
    | NSC.NotL => "~L"

  fun nsc_prop_to_string prop = case prop of
      NSC.PTrue => "T"
    | NSC.PFalse => "F"
    | NSC.PAtom a => atom_to_string a
    | NSC.PNot a => "~" ^ (fn (NSC.PAtom _|NSC.PTrue|NSC.PFalse|NSC.PNot _) => nsc_prop_to_string a | _ => "(" ^ nsc_prop_to_string a ^ ")") a
    | NSC.PAnd (a,b) => (fn (NSC.PAtom _|NSC.PTrue|NSC.PFalse|NSC.PNot _) => nsc_prop_to_string a | _ => "(" ^ nsc_prop_to_string a ^ ")") a ^ " ^ " ^ (fn (NSC.PAtom _|NSC.PTrue|NSC.PFalse|NSC.PNot _) => nsc_prop_to_string b | _ => "(" ^ nsc_prop_to_string b ^ ")") b
    | NSC.POr (a,b) => (fn (NSC.PAtom _|NSC.PTrue|NSC.PFalse|NSC.PNot _|NSC.PAnd _) => nsc_prop_to_string a | _ => "(" ^ nsc_prop_to_string a ^ ")") a ^ " v " ^ (fn (NSC.PAtom _|NSC.PTrue|NSC.PFalse|NSC.PNot _|NSC.PAnd _) => nsc_prop_to_string b | _ => "(" ^ nsc_prop_to_string b ^ ")") b
    | NSC.PImp (a as NSC.PImp _,b) => "(" ^ nsc_prop_to_string a ^ ") > " ^ nsc_prop_to_string b
    | NSC.PImp (a,b) => nsc_prop_to_string a ^ " > " ^ nsc_prop_to_string b

  fun nsc_judgment_to_string (NSC.JSeq (gamma,c)) =
    context_to_string nsc_prop_to_string gamma ^ " ==> " ^ nsc_prop_to_string c

  (* G4IP *)
  fun g4ip_pos_to_string pos = case pos of
      G4IP.PosOr p => prop_to_string (POr p)
    | G4IP.PosFalse => prop_to_string (PFalse)
    | G4IP.PosAtom a => prop_to_string (PAtom a)
  
  fun g4ip_neg_to_string neg = case neg of
      G4IP.NegImpAtom (a,b) => prop_to_string (PImp (PAtom a,b))
    | G4IP.NegImpImp (a,b)=> prop_to_string (PImp (PImp a,b))
    | G4IP.NegAtom a => prop_to_string (PAtom a)
  
  fun g4ip_judgment_to_string judgment = case judgment of
      G4IP.JInvR (D,O,C) => context_to_string g4ip_neg_to_string D ^ ";" ^ context_to_string prop_to_string O ^ " -R-> " ^ prop_to_string C
    | G4IP.JInvL (D,O,C) => context_to_string g4ip_neg_to_string D ^ ";" ^ context_to_string prop_to_string O ^ " -L-> " ^ g4ip_pos_to_string C
    | G4IP.JSearch (D,C) => context_to_string g4ip_neg_to_string D ^ " -S-> " ^ g4ip_pos_to_string C
  
  fun g4ip_rule_to_string rule = case rule of
      G4IP.Init => "init"
    | G4IP.TrueR => "TR"
    | G4IP.TrueL => "TL"
    | G4IP.FalseL => "FL"
    | G4IP.AndR => "^R"
    | G4IP.AndL => "^L2"
    | G4IP.OrR1 => "vR1"
    | G4IP.OrR2 => "vR2"
    | G4IP.OrL => "vL"
    | G4IP.ImpR => ">R"
    | G4IP.ImpAtomL => "p>L"
    | G4IP.ImpTrueL => "T>L"
    | G4IP.ImpFalseL => "F>L"
    | G4IP.ImpAndL => "^>L"
    | G4IP.ImpOrL => "v>L"
    | G4IP.ImpImpL => ">>L"
    | G4IP.Switch => "switch"
    | G4IP.Shift => "shift"
    | G4IP.Search => "search"

  (* Rule Creation *)
  fun rc_rule_to_string rule =
    let
      datatype rule = datatype RC.rule
      val () = ()
    in
    case rule of
      Hyp u => "hyp_" ^ u
    | Intro  U => String.concatWith "_" ("?I" ::U)
    | Elim   U => String.concatWith "_" ("?E" ::U)
    | Intro1 U => String.concatWith "_" ("?I1"::U)
    | Intro2 U => String.concatWith "_" ("?I2"::U)
    | Elim1  U => String.concatWith "_" ("?E1"::U)
    | Elim2  U => String.concatWith "_" ("?E2"::U)
    end
  fun rc_prop_to_string prop = case prop of
      RC.PAtom a => atom_to_string a
    | RC.PNew [a] => "!" ^ rc_prop_to_string a
    | RC.PNew [a as RC.PNew _,b] => "(" ^ rc_prop_to_string a ^ ") ? " ^ rc_prop_to_string b
    | RC.PNew [a,b] => rc_prop_to_string a ^ " ? " ^ rc_prop_to_string b
    | RC.PNew L => "New(" ^ String.concatWithMap "," rc_prop_to_string L ^")"
  fun rc_nd_judgment_to_string (RC.JTrue p) = rc_prop_to_string p ^ " true"
  fun rc_vau_judgment_to_string judgment = case judgment of
      RC.JUp p => rc_prop_to_string p ^ " ↑"
    | RC.JDown p => rc_prop_to_string p ^ " ↓"

  (* Harmony *)
  fun subst_to_string (H./ (M,x)) = givenname_to_string M ^ "/" ^ x
  fun substs_to_string substs = "[" ^ String.concatWithMap "," subst_to_string substs ^ "]"
  fun givensubst_to_string given = case given of
      H.GivenName g => givenname_to_string g
    | H.GivenSubst (s,g) => substs_to_string s ^ givenname_to_string g
  
  (* Heyting Arithmetic *)
  datatype obj = datatype HA.object
  datatype typ = datatype HA.typ
  datatype prop = datatype HA.prop

  fun ha_typ_to_string typ = case typ of
      HA.Tyvar t => t
    | HA.Nat => "nat"
  
  fun ha_obj_to_string' obj a = case obj of
      HA.Var x => if a = 0 then x else Int.toString a ^ "+" ^ x
    | HA.Z => Int.toString a
    | HA.S(e) => ha_obj_to_string' e (a+1)
  
  fun ha_obj_to_string obj = ha_obj_to_string' obj 0

  fun ha_atom_to_string atom = case atom of
      HA.AA => "A" | HA.AB => "B" | HA.AC => "C" | HA.AD => "D" | HA.AE => "E"
    | HA.AAof obj => "A(" ^ ha_obj_to_string obj ^ ")"
    | HA.ABof obj => "B(" ^ ha_obj_to_string obj ^ ")"
    | HA.ACof obj => "C(" ^ ha_obj_to_string obj ^ ")"
    | HA.ADof obj => "D(" ^ ha_obj_to_string obj ^ ")"
    | HA.AEof obj => "E(" ^ ha_obj_to_string obj ^ ")"
    | HA.Equals (o1,o2) => ha_obj_to_string o1 ^ "=" ^ ha_obj_to_string o2
  
  fun ha_prop_precedence prop = case prop of
      (HA.PForall _ | HA.PExists _| HA.PAtom (HA.Equals _)) => 1
    | (HA.PImp _) => 2
    | (HA.POr _) => 3
    | (HA.PAnd _) => 4
    | (HA.PTrue | HA.PFalse | HA.PAtom _) => 5
  
  fun ha_rule_to_string rule = case rule of
      HA.Hyp u => "hyp_" ^ u
    | HA.TrueI => "TI"
    | HA.FalseE => "FE"
    | HA.AndI => "^I"
    | HA.AndE1 => "^E1"
    | HA.AndE2 => "^E2"
    | HA.OrI1 => "vI1"
    | HA.OrI2 => "vI2"
    | HA.OrE (u,v) => "vE_" ^ u ^ "_" ^ v
    | HA.ImpI u => ">I_" ^ u
    | HA.ImpE => ">E"
    | HA.ForallI u => "∀I_" ^ u
    | HA.ForallE => "∀E"
    | HA.ExistsI => "∃I"
    | HA.ExistsE (u,v) => "∃E_" ^ u ^ "_" ^ v
    | HA.NatI0 => "natI_0"
    | HA.NatIs => "natI_S"
    | HA.NatE (u,v) => "natE_" ^ u ^ "_" ^ v
    | HA.EqualsI00 => "=I_00"
    | HA.EqualsIss => "=I_SS"
    | HA.EqualsE0s => "=E_0S"
    | HA.EqualsEs0 => "=E_S0"
    | HA.EqualsEss => "=E_SS"

  fun parens ifleq prop = if ha_prop_precedence prop <= ifleq then "(" ^ ha_prop_to_string prop ^ ")" else ha_prop_to_string prop
  and ha_prop_to_string prop = case prop of
      HA.PAtom a => ha_atom_to_string a
    | HA.PTrue => "T"
    | HA.PFalse => "F"
    | HA.PAnd (a,b) => parens 4 a ^ "^" ^ parens 3 b
    | HA.POr (a,b) => parens 3 a ^ "v" ^ parens 2 b
    | HA.PImp (a,b) => parens 2 a ^ ">" ^ parens 1 b
    | HA.PForall (x,t,p) => "∀" ^ x ^ ":" ^ ha_typ_to_string t ^ "." ^ parens 0 p
    | HA.PExists (x,t,p) => "∃" ^ x ^ ":" ^ ha_typ_to_string t ^ "." ^ parens 0 p
  fun ha_judgment_to_string judgment = case judgment of
      HA.JTrue p => ha_prop_to_string p ^ " true"
    | HA.JOfTyp (obj,t) => ha_obj_to_string obj ^ " : " ^ ha_typ_to_string t

  (* Focusing *)
  
  fun f_pos_precedence prop = case prop of
      (F.POr _) => 3
    | (F.PAnd _) => 4
    | (F.PDown _) => 5
    | (F.PTrue | F.PFalse | F.PAtom _) => 6
  fun f_neg_precedence prop = case prop of
      (F.NImp _) => 2
    | (F.NAnd _) => 4
    | (F.NUp _) => 5
    | (F.NTrue | F.NAtom _) => 6
  
  fun pos_parens ifleq prop = if f_pos_precedence prop <= ifleq then "(" ^ f_pos_to_string prop ^ ")" else f_pos_to_string prop
  and neg_parens ifleq prop = if f_neg_precedence prop <= ifleq then "(" ^ f_neg_to_string prop ^ ")" else f_neg_to_string prop
  and f_pos_to_string prop = case prop of
      F.PAtom a => atom_to_string a ^ "+"
    | F.PTrue => "T+"
    | F.PFalse => "F"
    | F.PAnd (a,b) => pos_parens 4 a ^ "^+" ^ pos_parens 3 b
    | F.POr (a,b) => pos_parens 3 a ^ "v" ^ pos_parens 2 b
    | F.PDown a => "↓" ^ neg_parens 4 a
  and f_neg_to_string prop = case prop of
      F.NAtom a => atom_to_string a ^ "-"
    | F.NTrue => "T-"
    | F.NAnd (a,b) => neg_parens 4 a ^ "^-" ^ neg_parens 3 b
    | F.NImp (a,b) => pos_parens 2 a ^ ">" ^ neg_parens 1 b
    | F.NUp a => "↑" ^ pos_parens 4 a
  
  fun f_rule_to_string rule = case rule of
      F.IImpR => blue ">R"
    | F.IAndR => blue "^R"
    | F.ITrueR => blue "TR"
    | F.IAtomR => blue "pR"
    | F.IUpR => blue "↑R"
    | F.IOrL => blue "vL"
    | F.IFalseL => blue "FL"
    | F.IAndL => blue "^L"
    | F.ITrueL => blue "TL"
    | F.IAtomL => blue "pL"
    | F.IDownL => blue "↓L"
    | F.Stable => bold "stable"
    | F.FocusR => bold "focusR"
    | F.FocusL => bold "focusL"
    | F.COrR1 => red "vR1"
    | F.COrR2 => red "vR2"
    | F.CAndR => red "^R"
    | F.CTrueR => red "TR"
    | F.CAtomR => red "pR"
    | F.CDownR => red "↓R"
    | F.CImpL => red ">L"
    | F.CAndL1 => red "^L1"
    | F.CAndL2 => red "^L2"
    | F.CAtomL => red "pL"
    | F.CUpL => red "↑L"
  fun f_judgment_to_string judgment = case judgment of
      F.JInvR (d,g,c) => context_to_string f_neg_to_string d ^ "; " ^ context_to_string f_pos_to_string g ^ " =r=> " ^ blue(f_neg_to_string c)
    | F.JInvL (d,g,c) => context_to_string f_neg_to_string d ^ "; " ^ blue(context_to_string f_pos_to_string g) ^ " =l=> " ^ f_pos_to_string c
    | F.JStable (d,c) => context_to_string f_neg_to_string d ^ " ==> " ^ f_pos_to_string c
    | F.JFocusR (d,a) => context_to_string f_neg_to_string d ^ " ==> " ^ red("[" ^ f_pos_to_string a ^ "]")
    | F.JFocusL (d,a,c) => context_to_string f_neg_to_string d ^ "; " ^ red("[" ^ f_neg_to_string a ^ "]") ^ " ==> " ^ f_pos_to_string c

  (* Linear Logic *)

  fun ll_prop_precedence prop = case prop of
      (LL.PLolli _) => 2
    | (LL.PPlus _) => 3
    | (LL.PWith _) => 4
    | (LL.PTensor _) => 5
    | (LL.PTop | LL.POne | LL.PZero | LL.PAtom _) => 6
  
  fun ll_rule_to_string rule = case rule of
      LL.Init => "init"
    | LL.TensorR => "⊗R"
    | LL.TensorL => "⊗L"
    | LL.OneR => "1R"
    | LL.OneL => "1L"
    | LL.WithR => "&R"
    | LL.WithL1 => "&L1"
    | LL.WithL2 => "&L2"
    | LL.TopR => "⊤"
    | LL.PlusR1 => "⊕R1"
    | LL.PlusR2 => "⊕R2"
    | LL.PlusL => "⊕L"
    | LL.ZeroL => "0L"
    | LL.LolliR => "⊸R"
    | LL.LolliL => "⊸L"

  fun parens ifleq prop = if ll_prop_precedence prop <= ifleq then "(" ^ ll_prop_to_string prop ^ ")" else ll_prop_to_string prop
  and ll_prop_to_string prop = case prop of
      LL.PAtom a => atom_to_string a
    | LL.POne => "1"
    | LL.PZero => "0"
    | LL.PTop => "⊤"
    | LL.PTensor (a,b) => parens 5 a ^ "⊗" ^ parens 4 b
    | LL.PWith   (a,b) => parens 4 a ^ "&" ^ parens 3 b
    | LL.PPlus   (a,b) => parens 3 a ^ "⊕" ^ parens 2 b
    | LL.PLolli  (a,b) => parens 2 a ^ "⊸" ^ parens 1 b
  fun ll_judgment_to_string judgment = case judgment of
      LL.JSeq (d,c) => context_to_string ll_prop_to_string d ^ " ⊩ " ^ ll_prop_to_string c ^ " true"

  (* Printers *)
  val print_nd  : ND.proof  -> unit = PrettyPrinter.pretty_print_proof (nd_rule_to_string,nd_judgment_to_string)
  val print_pap : PAP.proof -> unit = PrettyPrinter.pretty_print_proof (nd_rule_to_string,pap_judgment_to_string)
  val print_vau : VAU.proof -> unit = PrettyPrinter.pretty_print_proof (vau_rule_to_string,vau_judgment_to_string)
  val print_sc  : SC.proof  -> unit = PrettyPrinter.pretty_print_proof (sc_rule_to_string,sc_judgment_to_string)
  val print_ha  : HA.proof  -> unit = PrettyPrinter.pretty_print_proof (ha_rule_to_string,ha_judgment_to_string)

  val print_cnd  : CND.proof  -> unit = PrettyPrinter.pretty_print_proof (cnd_rule_to_string,cnd_judgment_to_string)
  val print_nsc  : NSC.proof  -> unit = PrettyPrinter.pretty_print_proof (nsc_rule_to_string,nsc_judgment_to_string)
  val print_g4ip : G4IP.proof -> unit = PrettyPrinter.pretty_print_proof (g4ip_rule_to_string,g4ip_judgment_to_string)
  val print_f : F.proof -> unit = PrettyPrinter.pretty_print_proof (f_rule_to_string,f_judgment_to_string)

  val print_ll : LL.proof -> unit = PrettyPrinter.pretty_print_proof (ll_rule_to_string,ll_judgment_to_string)

  val print_rc_nd  : RC.nd_def -> unit = PrettyPrinter.pretty_print_prooftree_hyps (fn () => "",L.abort,rc_rule_to_string,rc_nd_judgment_to_string)
  val print_rc_vau : RC.vau_def -> unit = PrettyPrinter.pretty_print_prooftree_hyps (fn () => "",L.abort,rc_rule_to_string,rc_vau_judgment_to_string)

  val print_expansion  : H.localexpansion  -> unit = fn (l,r) : H.localexpansion => PrettyPrinter.pretty_print_prooftree_hyps_side_by_side_with (givensubst_to_string,L.abort,rc_rule_to_string,rc_nd_judgment_to_string) ["  ===>  ","E"] [l,r]
  val print_reductions : H.localreductions -> unit = List.app (fn (l,r) : H.localexpansion => PrettyPrinter.pretty_print_prooftree_hyps_side_by_side_with (givensubst_to_string,L.abort,rc_rule_to_string,rc_nd_judgment_to_string) ["  ===>  ","R"] [l,r])
end