structure Focusing = struct
  open Logic

  (*
   * The focused calculus makes a distinction between positive and negative
   * propositions and adds up/down shifts to make any proposition utterable
   *)  
  datatype pos
    = PTrue
    | PFalse
    | PAtom of atom
    | PAnd of pos * pos
    | POr of pos * pos
    | PDown of neg
    
  and neg
    = NTrue
    | NAtom of atom
    | NAnd of neg * neg
    | NImp of pos * neg
    | NUp of pos

  (* Focusing has five judgments:
   * Right inversion
   * 
   *     Gamma,Delta+ -R-> C-
   *
   * Left inversion
   *
   *     Gamma,Delta+ -R-> C+
   *
   * Stable sequent
   *
   *     Gamma --> C+
   *
   * Right focus
   * 
   *     Gamma -R-> [C+]
   * 
   * Left focus
   *
   *     Gamma,[A-] -L-> C+
   *)
  datatype judgment
    = JInvR of neg list * pos list * neg
    | JInvL of neg list * pos list * pos
    | JStable of neg list * pos
    | JFocusR of neg list * pos
    | JFocusL of neg list * neg * pos

  datatype rule
    (* right inversion rules *)
    = IImpR | IAndR | ITrueR | IAtomR | IUpR
    (* left inversion rules *)
    | IOrL | IFalseL | IAndL | ITrueL | IAtomL | IDownL
    (* stabilize and focus *)
    | Stable | FocusR | FocusL
    (* right chaining rules *)
    | COrR1 | COrR2 | CAndR | CTrueR | CAtomR | CDownR
    (* left chaining rules *)
    | CImpL | CAndL1 | CAndL2 | CAtomL | CUpL
    
  
  type proof = (void, void, rule, judgment) abstractprooftree
  type ('given,'theorem) prooftree = ('given,'theorem,rule,judgment) abstractprooftree
  
  (* 
   * Macros for writing propositions
   * These will be infixed for you in a HW's starter code if it uses natural deduction 
   *)

  (* Positive atom proposition shortcuts *)
  val (Ap,Bp,Cp,Dp,Ep) = (PAtom AA,PAtom AB,PAtom AC,PAtom AD,PAtom AE)
  (* Negative atom proposition shortcuts *)
  val (An,Bn,Cn,Dn,En) = (NAtom AA,NAtom AB,NAtom AC,NAtom AD,NAtom AE)
  (*
   * There are two connective we call True, both have the same right rule
   * Negative True in sequent calculus has no left rule
   * Positive True in sequent calculus has a left rule
   * Both are logically equivalent, however they are not equal
   *
   * There is only one False, and it behaves positively
   * Fun fact, there are systems where the rhs is a list of possible
   *  goals, instead of one, and they have two falses, both have the
   *  same left rule but negative false has a right rule that makes
   *  it just disappear from the right as a possible goal
   *)
  val (Tp,Tn) = (PTrue,NTrue)
  val F = PFalse


  val up   : pos -> neg = NUp
  val down : neg -> pos = PDown
  
  (*
   * Connective shortcuts in order of precedence
   * ^+ for positive and
   * ^- for negative and
   * v for or
   * > for implies
   *)
  val (^+,^-,v, op>,up,down) = (PAnd,NAnd,POr,NImp,NUp,PDown)

  structure Macros :> 
    sig
      type ('a,'b,'c) t type u
      val inv : neg list -> pos list -> 'b -> ('a,'c,'b) t -> ('a,'b,'b) t -> u
      val foc : neg list -> 'a       -> 'b -> ('a,'c,'b) t -> ('a,'b,'b) t -> u
      val right : rule * (('a -> ('b,'a,'a) t -> (pos,neg,'c ) t -> 'd) * 'a) -> rule * judgment
      val left  : rule * (('a -> ('b,'a,'a) t -> (neg,pos,pos) t -> 'd) * 'a) -> rule * judgment
    end
    = struct
    datatype ('a,'b,'c) ctxargs = Inv of (neg list * pos list * 'b) | Foc of (neg list * 'a * 'c) | U
    type ('a,'b,'c) t = ('a,'b,'c) ctxargs ref list type u = unit exception B
    fun inv x y z _ rs = (List.app (fn r => r := Inv(x,y,z)) rs)
    fun foc x y z r1 = ((List.app (fn r => r := Foc(x,y,z)) r1);(fn r2 => List.app (fn r => r := Foc(x,y,z)) r2))
    fun right (f,(c1,c2)) = (f,let val r = ref U in c1 c2 [] [r]; case !r of U => raise B | Inv x => JInvR x | Foc(d,p,_) => JFocusR(d,p) end)
    fun left (f,(c1,c2)) =  (f,let val r = ref U in c1 c2 [] [r]; case !r of U => raise B | Inv x => JInvL x | Foc x => JFocusL x end)
  end
  open Macros
  
  fun ==> (c,p) = (c,p)
  val _ = right (Stable,==>(inv [An,Bn] [Ap], Cn))
  val _ = left  (Stable,==>(inv [An,Bn] [Ap], Cp))
  val _ = right (Stable,==>(foc [An,Bn]     , Cp))
  val _ = left  (Stable,==>(foc [An,Bn]  An , Cp))
  fun stable (f,c) = (f,JStable c)

  (* 
   * Truth judgment magic to use with -- and ==
   * Example Usage:
     
      [proof_of_down_B]
      ------------------- OrR2
      left foc [up Ap, Ap > Bn] ==> Ap v down Bn
     
   *)
  type given = void
  type theorem = void
end