/*
 * TITLE:	Client.cc
 *
 * PURPOSE:	This file encapsulates the radio/vision client interfaces that 
 *              hide the socket interface to the vision/radio servers
 *
 * WRITTEN BY:  James R Bruce, Scott Lenser, Michael Bowling, Brett Browning
 */
/* LICENSE: */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>

#include "util.h"
#include "constants.h"
#include "socket.h"
#include "reality/net_vision.h"

#include "client.h"


bool Client::Initialize(char *hostname, int vport, int rport, int referee_port)
{
 if (vision_s.connect_client(hostname, vport) != Socket::Client)
   return (false);
 if (radio_s.connect_client(hostname, rport) != Socket::Client)
   return (false);

 if (referee_port > 0) {
   if (referee_s.connect_client(hostname, referee_port) != Socket::Client) {
     fprintf(stderr, "Client could not connect to referee port %d\n", referee_port);
     fprintf(stderr," Client will send control messages through vision\n");
   }
 }

 return (true);
}

void Client::Close(void)
{
  vision_s.disconnect();
  radio_s.disconnect();
  referee_s.disconnect();
}

bool Client::Configure(struct net_vconfig &vc)
{
  vc.msgtype = NET_VISION_CONFIG;
  if (referee_s.get_status() == Socket::Client) {
    NetRefereeMsg m(vc);
    if (!referee_s.ready_for_send()) {
      fprintf(stderr, "Referee socket not ready to send vconfig\n");
      return false;
    }
    referee_s.send((char *)&m, sizeof(m));
  } else {
    vision_s.send(&vc, sizeof(vc));
  }
  return(true);
}

bool Client::GetUpdate(net_vframe &vf)
{
  static char msg[net_vision_out_maxsize];
  char msgtype;

  if (vision_s.get_status() != Socket::Client)
    return (false);

  /* get the message and block until we do */
  if (vision_s.recv_type(msg, net_vision_out_maxsize, msgtype) <= 0)
    return (false);

  if (msgtype == NET_VISION_FRAME)
    memcpy((char *) &vf, msg, sizeof(vf));
  return(true);
}

bool Client::SendRef(char state) 
{
  net_vref c = {NET_VISION_REF, state};

  if (referee_s.get_status() == Socket::Client) {
    NetRefereeMsg m(c);
    if (!referee_s.ready_for_send()) {
      fprintf(stderr, "Referee socket not ready for send ('%c')\n", state);
      return false;
    }
    referee_s.send((char *)&m, sizeof(m));
  } else {
    if (!vision_s.ready_for_send()) {
      fprintf(stderr, "Cannot send yet\n");
      return false;
    }
    vision_s.send(&c, sizeof(c));
  }
  return true;
}

bool Client::MoveBall(double x, double y, double vx, double vy) 
{
  net_vsim c = {NET_VISION_SIM, VSIM_MOVEBALL};
  c.info.moveball = (vsim_moveball) {{x, y}, {vx, vy}};

  if (referee_s.get_status() == Socket::Client) {
    NetRefereeMsg m(c);
    referee_s.send((char *)&m, sizeof(m));
  } else {
    vision_s.send(&c, sizeof(c));
  }
  return true;
}

bool Client::MoveRobot(uchar team, uchar id, double x, double y, double angle) 
{
  net_vsim c = {NET_VISION_SIM, VSIM_MOVEROBOT};
  c.info.moverobot =  (vsim_moverobot) {team, id, {x, y}, angle};

  if (referee_s.get_status() == Socket::Client) {
    NetRefereeMsg m(c);
    referee_s.send((char *)&m, sizeof(m));
  } else
    vision_s.send(&c, sizeof(c));
  return true;
}


bool Client::EnableRadio(bool en)
{
  return RadioControl(en ? VCR_PLAY : VCR_PAUSE);
}

bool Client::RadioControl(char ctrl)
{
  net_rcontrol rc = (net_rcontrol) {NET_RADIO_CONTROL, ctrl};
  radio_s.send(&rc, sizeof(rc));
  return (true);
}

bool Client::SetCommand(double tstamp, char team, char rid, double vx, double vy, double va, 
		bool kick/* = false*/, bool drib/* = false*/, char priority/* = 1*/)

{
  // need to add command if we can...
  if (rcommands.nr_commands >= MAX_TEAM_ROBOTS) {
    fprintf(stderr, "ERROR: Cannot add any more commands!!!\n");
    fprintf(stderr, "DBG INFO: time %f team %i rid %i vel (%f, %f, %f), k/d (%i, %i) priority %i\n",
	    tstamp, team, rid, vx, vy, va, kick, drib, priority);
    return false;
  }

  // do a nan check first
  if (isnan(vx) || isnan(vy) || isnan(va)) {
    fprintf(stderr, "NAN's in velocity command: %f %f %f\n", vx, vy, va);
    return false;
  }

  // do an id sanity check
  if ((rid < 0) || (rid > MAX_ROBOT_ID)) {
    fprintf(stderr, "Bad Robot id %i\n", rid);
    return false;
  }

  //  fprintf(stderr, "\tAdding command t %i, r %i v(%f %f %f)\n", team, rid, vx, vy, va);

  int i = rcommands.nr_commands;

	rcommands.commands[i].mode     = 1; //RF_MODE_RUN;
  rcommands.commands[i].priority = priority;
  rcommands.commands[i].id       = rid;
  rcommands.commands[i].team     = team;
  rcommands.commands[i].type     = ROBOT_TYPE_ID(rid);
  rcommands.commands[i].dx       = (int) vx;
  rcommands.commands[i].dy       = (int) vy;
  rcommands.commands[i].da       = (int) (va*1000); // convert to mrad/sec
  rcommands.commands[i].kick     = kick;
  rcommands.commands[i].drib     = drib;

  // what do we do withthe timestamp??? For now set it to 
  // latest timestamp available
  rcommands.timestamp = max(rcommands.timestamp, tstamp);

  // add it to the list
  rcommands.nr_commands++;

  // all done
  return true;
}

bool Client::SetHalt(double tstamp, char team, char rid, char priority)
{
  // need to add command if we can...
  if (rcommands.nr_commands >= MAX_TEAM_ROBOTS) {
    fprintf(stderr, "ERROR: Cannot add any more commands!!!\n");
    fprintf(stderr, "DBG INFO: time %f team %i rid %i HALT\n",
	    tstamp, team, rid);
    return false;
  }

  // do an id sanity check
  if ((rid < 0) || (rid > MAX_ROBOT_ID)) {
    fprintf(stderr, "Bad Robot id %i\n", rid);
    return false;
  }

  int i = rcommands.nr_commands;

  rcommands.commands[i].mode     = 0; //RF_MODE_SLEEP;
  rcommands.commands[i].priority = priority;
  rcommands.commands[i].id       = rid;
  rcommands.commands[i].team     = team;
  rcommands.commands[i].type     = ROBOT_TYPE_ID(rid);
  rcommands.commands[i].dx       = (int) 0;
  rcommands.commands[i].dy       = (int) 0;
  rcommands.commands[i].da       = (int) 0;
  rcommands.commands[i].kick     = false;
  rcommands.commands[i].drib     = false;

  // what do we do withthe timestamp??? For now set it to 
  // latest timestamp available
  rcommands.timestamp = max(rcommands.timestamp, tstamp);

  // add it to the list
  rcommands.nr_commands++;

  // all done
  return true;
}

// send all the commands 
bool Client::Send(void)
{
  // check if there is anything to send
  if (rcommands.nr_commands == 0)
    return false;

  if (radio_s.get_status() != Socket::Client) {
    fprintf(stderr, "ERROR: Socket disconnected so cannot send!!!\n");
    return false;
  }

  bool rval = radio_s.send((net_rcommands *) &rcommands, rcommands.size());

  // refreshteh command structure so we can add more things to it
  rcommands.nr_commands = 0;
  return rval;
}

