/*=========================================================================
	UberSim Source Code Release
	-------------------------------------------------------------------------
	Copyright (C) 2002 Manuela Veloso, Brett Browning, Mike Bowling,
	James Bruce; {mmv, brettb, mhb, jbruce}@cs.cmu.edu
	Erick Tryzelaar {erickt}@andrew.cmu.edu
	School of Computer Science, Carnegie Mellon University
	-------------------------------------------------------------------------
	This software is distributed under the GNU General Public License,
	version 2.  If you do not have a copy of this licence, visit
	www.gnu.org, or write: Free Software Foundation, 59 Temple Place,
	Suite 330 Boston, MA 02111-1307 USA.  This program is distributed
	in the hope that it will be useful, but WITHOUT ANY WARRANTY,
	including MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
	-------------------------------------------------------------------------*/

#include <ode/ode.h>
#include <typeinfo>

#include "../utils/configreader.h"

#include "mmgr.h"

#include "BoxGeometry.h"
#include "CappedCylinderGeometry.h"
#include "CylinderGeometry.h"
#include "DiffRobot.h"
#include "Engine.h"
#include "Event.h"
#include "Geometry.h"
#include "Hinge2Joint.h"
#include "Node.h"
#include "RigidBody.h"
#include "RigidBodyCollider.h"
#include "SliderJoint.h"
#include "SphereGeometry.h"
#include "Surface.h"
#include "Timer.h"

CR_DECLARE (MASS);
CR_DECLARE (SHELL_LENGTH);
CR_DECLARE (SHELL_WIDTH);
CR_DECLARE (SHELL_HEIGHT);
CR_DECLARE (SHELL_POSITION);
CR_DECLARE (WHEEL_LENGTH);
CR_DECLARE (WHEEL_RADIUS);
CR_DECLARE (WHEEL_MU);
CR_DECLARE (WHEEL_MU2);
CR_DECLARE (WHEEL_SLIP1);
CR_DECLARE (WHEEL_SLIP2);
CR_DECLARE (WHEEL_MAX_FORCE);
CR_DECLARE (WHEEL_MASS);
CR_DECLARE (WHEEL1_POSITION);
CR_DECLARE (WHEEL2_POSITION);
CR_DECLARE (CASTER_RADIUS);
CR_DECLARE (CASTER_MU);
CR_DECLARE (FRONT_CASTER_POSITION);
CR_DECLARE (BACK_CASTER_POSITION);
CR_DECLARE (STOPPER_LENGTH);
CR_DECLARE (STOPPER_WIDTH);
CR_DECLARE (STOPPER_HEIGHT);
CR_DECLARE (LEFT_STOPPER_POSITION);
CR_DECLARE (RIGHT_STOPPER_POSITION);
CR_DECLARE (DRIBBLER_LENGTH);
CR_DECLARE (DRIBBLER_RADIUS);
CR_DECLARE (DRIBBLER_MU);
CR_DECLARE (DRIBBLER_SLIP1);
CR_DECLARE (DRIBBLER_SLIP2);
CR_DECLARE (DRIBBLER_MAX_FORCE);
CR_DECLARE (DRIBBLER_TARGET_VELOCITY);
CR_DECLARE (DRIBBLER_MASS);
CR_DECLARE (DRIBBLER_POSITION);
CR_DECLARE (KICKER_LENGTH);
CR_DECLARE (KICKER_WIDTH);
CR_DECLARE (KICKER_HEIGHT);
CR_DECLARE (KICKER_MAX_FORCE);
CR_DECLARE (KICKER_LOWER_LIMIT);
CR_DECLARE (KICKER_UPPER_LIMIT);
CR_DECLARE (KICKER_TARGET_VELOCITY);
CR_DECLARE (KICKER_MASS);
CR_DECLARE (KICKER_POSITION);
CR_DECLARE (KICKER_DELAY);

DiffRobot::DiffRobot (const Vector3& color)
{
	static bool isConfiged = false;

	if (!isConfiged)
	{
		isConfiged = true;

		CR_SETUP (DiffRobot, MASS, CR_DOUBLE);
		CR_SETUP (DiffRobot, SHELL_LENGTH, CR_DOUBLE);
		CR_SETUP (DiffRobot, SHELL_WIDTH, CR_DOUBLE);
		CR_SETUP (DiffRobot, SHELL_HEIGHT, CR_DOUBLE);
		CR_SETUP (DiffRobot, SHELL_POSITION, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL_LENGTH, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL_RADIUS, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL_MU, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL_MU2, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL_SLIP1, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL_SLIP2, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL_MAX_FORCE, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL_MASS, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL1_POSITION, CR_DOUBLE);
		CR_SETUP (DiffRobot, WHEEL2_POSITION, CR_DOUBLE);
		CR_SETUP (DiffRobot, CASTER_RADIUS, CR_DOUBLE);
		CR_SETUP (DiffRobot, CASTER_MU, CR_DOUBLE);
		CR_SETUP (DiffRobot, FRONT_CASTER_POSITION, CR_DOUBLE);
		CR_SETUP (DiffRobot, BACK_CASTER_POSITION, CR_DOUBLE);
		CR_SETUP (DiffRobot, STOPPER_LENGTH, CR_DOUBLE);
		CR_SETUP (DiffRobot, STOPPER_WIDTH, CR_DOUBLE);
		CR_SETUP (DiffRobot, STOPPER_HEIGHT, CR_DOUBLE);
		CR_SETUP (DiffRobot, LEFT_STOPPER_POSITION, CR_DOUBLE);
		CR_SETUP (DiffRobot, RIGHT_STOPPER_POSITION, CR_DOUBLE);
		CR_SETUP (DiffRobot, DRIBBLER_LENGTH, CR_DOUBLE);
		CR_SETUP (DiffRobot, DRIBBLER_RADIUS, CR_DOUBLE);
		CR_SETUP (DiffRobot, DRIBBLER_MU, CR_DOUBLE);
		CR_SETUP (DiffRobot, DRIBBLER_SLIP1, CR_DOUBLE);
		CR_SETUP (DiffRobot, DRIBBLER_SLIP2, CR_DOUBLE);
		CR_SETUP (DiffRobot, DRIBBLER_MAX_FORCE, CR_DOUBLE);
		CR_SETUP (DiffRobot, DRIBBLER_TARGET_VELOCITY, CR_DOUBLE);
		CR_SETUP (DiffRobot, DRIBBLER_MASS, CR_DOUBLE);
		CR_SETUP (DiffRobot, DRIBBLER_POSITION, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_LENGTH, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_WIDTH, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_HEIGHT, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_MAX_FORCE, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_LOWER_LIMIT, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_UPPER_LIMIT, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_TARGET_VELOCITY, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_MASS, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_POSITION, CR_DOUBLE);
		CR_SETUP (DiffRobot, KICKER_DELAY, CR_DOUBLE);
	}

	Real length = DVAR (SHELL_LENGTH);
	Real width  = DVAR (SHELL_WIDTH);
	Real height = DVAR (SHELL_HEIGHT);

	////

	Node* robot = new Node ("Diff Robot");
	this->model = robot;

	rigidBody        = new RigidBody ();
	motionController = rigidBody;
	rigidBody->setModel (robot);

	dMass bodyMass;
	dMassSetBox  (&bodyMass, 1, length, width, height);
	dMassAdjust  (&bodyMass, DVAR (MASS));
	dBodySetMass (rigidBody->getBody (), &bodyMass);

	////

	Geometry* shellGeometry = new BoxGeometry (length, width, height, "Shell");
	shellGeometry->setColor (color);
	robot->addChild                       (shellGeometry);
	shellGeometry->setCollisionController (new RigidBodyCollider (rigidBody));
	shellGeometry->setLocalPosition       (Vector3 (VDVAR (SHELL_POSITION)));

	////

	Node* wheels = new Node ("Wheels");
	robot->addChild (wheels);

	////

	Geometry* wheel1Geometry = new CylinderGeometry (DVAR (WHEEL_LENGTH), DVAR (WHEEL_RADIUS), "Left Wheel");
	wheels->addChild (wheel1Geometry);

	wheel1Body = new RigidBody ();
	wheel1Body->setModel (wheel1Geometry);	
	wheel1Body->setWorldRotation (Matrix33::rotateAboutX (Math::HALF_PI));
	wheel1Body->setLocalPosition (Vector3 (VDVAR (WHEEL1_POSITION)));

	dBodySetFiniteRotationMode (wheel1Body->getBody (), 1);

	Surface* wheel1Surface = new Surface ();
	dSurfaceParameters* sp = &wheel1Surface->getSurfaceParameters ();
	sp->mode  = dContactApprox1 | dContactFDir1 | dContactMu2 | dContactSlip1;// | dContactSlip2;
	sp->mu    = DVAR (WHEEL_MU);
	sp->mu2   = DVAR (WHEEL_MU2);
	sp->slip1 = DVAR (WHEEL_SLIP1);
	sp->slip2 = DVAR (WHEEL_SLIP2);

	wheel1Collider = new RigidBodyCollider (wheel1Body);
	wheel1Collider->setSurface (wheel1Surface);
	wheel1Geometry->setCollisionController (wheel1Collider);

	wheel1Motor = new Hinge2Joint (rigidBody, wheel1Body);
	wheel1Motor->setAnchor    (wheel1Body->getWorldPosition ());
	wheel1Motor->setAxis1     (Vector3 (0.0, 0.0, 1.0));
	wheel1Motor->setAxis2     (Vector3 (0.0, 1.0, 0.0));
	wheel1Motor->setMaxForce2 (DVAR (WHEEL_MAX_FORCE));

	dMass wheel1Mass;
	dMassSetSphere (&wheel1Mass, 1, DVAR (WHEEL_RADIUS));
	dMassAdjust    (&wheel1Mass, DVAR (WHEEL_MASS));
	dBodySetMass   (wheel1Body->getBody (), &wheel1Mass);

	////

	Geometry* wheel2Geometry = new CylinderGeometry ( DVAR (WHEEL_LENGTH), DVAR (WHEEL_RADIUS), "Right Wheel");
	wheels->addChild (wheel2Geometry);

	wheel2Body = new RigidBody ();
	wheel2Body->setModel (wheel2Geometry);
	wheel2Body->setWorldRotation (Matrix33::rotateAboutX (Math::HALF_PI));
	wheel2Body->setLocalPosition (Vector3 (VDVAR (WHEEL2_POSITION)));

	dBodySetFiniteRotationMode (wheel2Body->getBody (), 1);

	Surface* wheel2Surface = new Surface ();
	sp = &wheel2Surface->getSurfaceParameters ();
	sp->mode  = dContactApprox1 | dContactFDir1 | dContactMu2 | dContactSlip1;// | dContactSlip2;
	sp->mu    = DVAR (WHEEL_MU);
	sp->mu2   = DVAR (WHEEL_MU2);
	sp->slip1 = DVAR (WHEEL_SLIP1);
	sp->slip2 = DVAR (WHEEL_SLIP2);

	wheel2Collider = new RigidBodyCollider (wheel2Body);
	wheel2Collider->setSurface (wheel2Surface);
	wheel2Geometry->setCollisionController (wheel2Collider);

	wheel2Motor = new Hinge2Joint (rigidBody, wheel2Body);
	wheel2Motor->setAnchor    (wheel2Body->getWorldPosition ());
	wheel2Motor->setAxis1     (Vector3 (0.0, 0.0, 1.0));
	wheel2Motor->setAxis2     (Vector3 (0.0, 1.0, 0.0));
	wheel2Motor->setMaxForce2 (DVAR (WHEEL_MAX_FORCE));

	dMass wheel2Mass;
	dMassSetSphere (&wheel2Mass, 1, DVAR (WHEEL_RADIUS));
	dMassAdjust    (&wheel2Mass, DVAR (WHEEL_MASS));
	dBodySetMass   (wheel2Body->getBody (), &wheel2Mass);

	////
	
	Real r = DVAR (WHEEL_RADIUS);
	Real l = wheel1Geometry->getLocalPosition ().length ();

	transform = Matrix22 (
			-1.0 / r, -l / r,
			-1.0 / r,  l / r);

	////

	Node* casters = new Node ("Casters");
	robot->addChild (casters);

	////

	Geometry* frontCasterGeometry = new SphereGeometry (DVAR (CASTER_RADIUS), "Front Caster");
	casters->addChild (frontCasterGeometry);

	Surface* frontCasterSurface = new Surface ();;
	sp = &frontCasterSurface->getSurfaceParameters ();
	sp->mode = dContactSlip1 | dContactSlip2;
	sp->mu   = DVAR (CASTER_MU);
	sp->slip1 = 0;
	sp->slip2 = 0;

	RigidBodyCollider* frontCasterCollider = new RigidBodyCollider (rigidBody);
	frontCasterCollider->setSurface (frontCasterSurface);
	frontCasterGeometry->setCollisionController (frontCasterCollider);

	frontCasterGeometry->setLocalPosition (Vector3 (VDVAR (FRONT_CASTER_POSITION)));

	////

	Geometry* backCasterGeometry = new SphereGeometry (DVAR (CASTER_RADIUS), "Back Caster");
	casters->addChild (backCasterGeometry);

	Surface* backCasterSurface = new Surface ();;
	sp = &backCasterSurface->getSurfaceParameters ();
	sp->mode = dContactSlip1 | dContactSlip2;
	sp->mu   = DVAR (CASTER_MU);
	sp->slip1 = 0;
	sp->slip2 = 0;

	RigidBodyCollider* backCasterCollider = new RigidBodyCollider (rigidBody);
	backCasterCollider->setSurface (backCasterSurface);
	backCasterGeometry->setCollisionController (backCasterCollider);

	backCasterGeometry->setLocalPosition (Vector3 (VDVAR (BACK_CASTER_POSITION)));

	////

	Node* dribblerBar = new Node ("Dribbler Bar");
	robot->addChild (dribblerBar);

	////

	Geometry* leftStopperGeometry = new BoxGeometry (DVAR (STOPPER_LENGTH), DVAR (STOPPER_WIDTH), DVAR (STOPPER_HEIGHT), "Left Stopper");
	dribblerBar->addChild (leftStopperGeometry);
	leftStopperGeometry->setCollisionController (new RigidBodyCollider (rigidBody));

	leftStopperGeometry->setLocalPosition (Vector3 (VDVAR (LEFT_STOPPER_POSITION)));

	////

	Geometry* rightStopperGeometry = new BoxGeometry (DVAR (STOPPER_LENGTH), DVAR (STOPPER_WIDTH), DVAR (STOPPER_HEIGHT), "Right Stopper");
	dribblerBar->addChild (rightStopperGeometry);
	rightStopperGeometry->setCollisionController (new RigidBodyCollider (rigidBody));

	rightStopperGeometry->setLocalPosition (Vector3 (VDVAR (RIGHT_STOPPER_POSITION)));

	////

	Geometry* dribblerGeometry = new CappedCylinderGeometry (DVAR (DRIBBLER_LENGTH), DVAR (DRIBBLER_RADIUS), "Dribbler");
	dribblerBar->addChild (dribblerGeometry);

	RigidBody* dribblerBody = new RigidBody ();
	dribblerBody->setModel (dribblerGeometry);
	dribblerBody->setWorldRotation (Matrix33::rotateAboutX (Math::HALF_PI));
	dribblerBody->setWorldPosition (Vector3 (VDVAR (DRIBBLER_POSITION)));

	Surface* dribblerSurface = new Surface ();
	sp = &dribblerSurface->getSurfaceParameters ();
	//sp->mode  = dContactSlip1 | dContactSlip2;// | dContactApprox1;
	sp->mu    = DVAR (DRIBBLER_MU);
	//sp->slip1 = DVAR (DRIBBLER_SLIP1);
	//sp->slip2 = DVAR (DRIBBLER_SLIP2);

	RigidBodyCollider* dribblerCollider = new RigidBodyCollider (dribblerBody);
	dribblerCollider->setSurface (dribblerSurface);
	dribblerGeometry->setCollisionController (dribblerCollider);

	dribblerMotor = new Hinge2Joint (rigidBody, dribblerBody);
	dribblerMotor->setAnchor    (dribblerBody->getWorldPosition ());
	dribblerMotor->setAxis1     (Vector3 (0.0, 0.0, 1.0));
	dribblerMotor->setAxis2     (Vector3 (0.0, 1.0, 0.0));
	dribblerMotor->setMaxForce2 (DVAR (DRIBBLER_MAX_FORCE));

	dMass dribblerMass;
	dMassSetSphere (&dribblerMass, 1, DVAR (DRIBBLER_RADIUS));
	dMassAdjust    (&dribblerMass, DVAR (DRIBBLER_MASS));
	dBodySetMass   (dribblerBody->getBody (), &dribblerMass);

	////

	Geometry* kickerGeometry = new BoxGeometry (DVAR (KICKER_LENGTH), DVAR (KICKER_WIDTH), DVAR (KICKER_HEIGHT), "Kicker");
	robot->addChild (kickerGeometry);
	RigidBody* kickerBody = new RigidBody ();
	kickerBody->setModel (kickerGeometry);
	kickerBody->setWorldPosition (Vector3 (VDVAR (KICKER_POSITION)));

	/*
		 Surface* dribblerSurface = new Surface ();
		 sp = &dribblerSurface->getSurfaceParameters ();
		 sp->mu = Math::INFINITY;		
		 */

	RigidBodyCollider* kickerCollider = new RigidBodyCollider (kickerBody);
	kickerGeometry->setCollisionController (kickerCollider);

	kickerMotor = new SliderJoint  (rigidBody, kickerBody);
	kickerMotor->setAxis           (Vector3 (-1.0, 0.0, 0.0));
	kickerMotor->setMaxForce       (0.0);
	kickerMotor->setLimits         (0.0, 0.0);
	kickerMotor->setTargetVelocity (0.0);

	//kickerMotor->setMaxForce       (DVAR (KICKER_MAX_FORCE));
	//kickerMotor->setLimits         (DVAR (KICKER_LOWER_LIMIT), DVAR (KICKER_UPPER_LIMIT));
	//kickerMotor->setTargetVelocity (-DVAR (KICKER_TARGET_VELOCITY));

	dMass kickerMass;
	dMassSetBox  (&kickerMass, 1, DVAR (KICKER_LENGTH), DVAR (KICKER_WIDTH), DVAR (KICKER_HEIGHT));
	dMassAdjust  (&kickerMass, DVAR (KICKER_MASS));
	dBodySetMass (kickerBody->getBody (), &kickerMass);

	////

	Timer::getInstance ()->addEvent (this, new FrameEvent (), 0);
}

DiffRobot::~DiffRobot ()
{
	delete wheel1Motor;
	delete wheel2Motor;
	delete dribblerMotor;
	delete kickerMotor;
}

void DiffRobot::handleEvent (Event* event)
{
	if (!event)
	{
		return;
	}

	////
	
	FrameEvent* frameEvent = dynamic_cast<FrameEvent *> (event);

	if (frameEvent)
	{
		Matrix33 worldRotation = rigidBody->getWorldRotation () * Matrix33::rotateAboutX (Math::HALF_PI);

		dMatrix3 r;
		r[0]  = worldRotation[0][0];
		r[1]  = worldRotation[0][1];
		r[2]  = worldRotation[0][2];
		r[3]  = 0.0;
		r[4]  = worldRotation[1][0];
		r[5]  = worldRotation[1][1];
		r[6]  = worldRotation[1][2];
		r[7]  = 0.0;
		r[8]  = worldRotation[2][0];
		r[9]  = worldRotation[2][1];
		r[10] = worldRotation[2][2];
		r[11] = 0.0;

		dBodySetRotation (wheel1Body->getBody  (), r);
		dBodySetRotation (wheel2Body->getBody (), r);
		
		Vector3 fdir = rigidBody->getWorldRotation () * Vector3::UNIT_X;
		fdir.z       = 0;
		fdir.normalize ();

		wheel1Collider->getSurface ()->setFrictionDirection (fdir);
		wheel2Collider->getSurface ()->setFrictionDirection (fdir);

		delete frameEvent;

                // BUG: Memory leak here
		Timer::getInstance ()->addEvent (this, new FrameEvent (), 0.001);
		return;
	};

	////

	VelocityEvent* velocityEvent = dynamic_cast<VelocityEvent *> (event);

	if (velocityEvent)
	{
		//Engine* engine = Engine::getInstance ();
		//printf ("event: %f %f off: %f\n", velocityEvent->linearVelocity.x * 100, velocityEvent->linearVelocity.y * 100, engine->getFrameCount () / engine->getTargetFps ());

		targetVelocity.x = velocityEvent->linearVelocity.x;
		targetVelocity.y = velocityEvent->angularVelocity.z;

		Vector2 w = transform * targetVelocity;

		wheel1Motor->setTargetVelocity (w.x);
		wheel2Motor->setTargetVelocity (w.y);

		delete velocityEvent;
		return;
	}

	////
	
	KickerEvent* kickerEvent = dynamic_cast<KickerEvent *> (event);

	if (kickerEvent)
	{
		if (kickerEvent->isEnabled)
		{
			kickerMotor->setTargetVelocity (DVAR (KICKER_TARGET_VELOCITY));
			kickerMotor->setMaxForce       (DVAR (KICKER_MAX_FORCE));
			kickerMotor->setLimits         (DVAR (KICKER_LOWER_LIMIT), DVAR (KICKER_UPPER_LIMIT));
			Timer::getInstance ()->addEvent (this, new KickerEvent (false), DVAR (KICKER_DELAY));
		}
		else
		{
			kickerMotor->setTargetVelocity (0.0);
			kickerMotor->setMaxForce       (0.0);
			kickerMotor->setLimits         (0.0, 0.0);
		}

		delete kickerEvent;
		return;
	}

	////
	
	DribblerEvent* dribblerEvent = dynamic_cast<DribblerEvent *> (event);

	if (dribblerEvent)
	{
		if (dribblerEvent->isEnabled)
		{
			dribblerMotor-> setTargetVelocity (DVAR (DRIBBLER_TARGET_VELOCITY));
		}
		else
		{
			dribblerMotor-> setTargetVelocity (0.0);
		}

		delete dribblerEvent;
		return;
	}

	////
	
	RigidBodyEntity::handleEvent (event);
}
