/*****************************************************************************
**
**  graphic.cpp (needs OpenGL lib)
**	
**	  Graphic class for the Role Project.
**	
**	  This class is a major graphic part of the project.
**
**		See more in header graphic.h
**
**   Refer to md.h to see 3D models used in RoleEngine program.
**
**	
**	(c) 2001 by Byung Gil Yuh
**
***************************************************************************/

#include "graphic.h"
#include "image.h"
#include "glm.h"
#include "player.h"
#include "terrain.h"
#include "object.h"
#include "util.h"
#include "sdl.h"
#include "attribute.h"

GLMmodel* pmodel = NULL;
RoleFont font;

extern RoleUINT polycounter;
PFNGLMULTITEXCOORD2FARBPROC glMultiTexCoord2fARB_ptr = NULL;
PFNGLACTIVETEXTUREARBPROC glActiveTextureARB_ptr = NULL;

void extractFrustum(RoleFLOAT m_Frustum[6][4])
{
    RoleFLOAT proj[16];
    RoleFLOAT modl[16];
    RoleFLOAT clip[16];
	RoleFLOAT t;
    /* Get the current PROJECTION matrix from OpenGL */
    glGetFloatv(GL_PROJECTION_MATRIX, proj);
    /* Get the current MODELVIEW matrix from OpenGL */
    glGetFloatv(GL_MODELVIEW_MATRIX, modl);
    /* Combine the two matrices (multiply projection by modelview) */
/*
	rolePrintlog("Proj\n");
	rolePrintlog("%f,%f,%f,%f\n",proj[0],proj[4],proj[8],proj[12]);
	rolePrintlog("%f,%f,%f,%f\n",proj[1],proj[5],proj[9],proj[13]);
	rolePrintlog("%f,%f,%f,%f\n",proj[2],proj[6],proj[10],proj[14]);
	rolePrintlog("%f,%f,%f,%f\n",proj[3],proj[7],proj[11],proj[15]);

	rolePrintlog("Modl\n");
	rolePrintlog("%f,%f,%f,%f\n",modl[0],modl[4],modl[8],modl[12]);
	rolePrintlog("%f,%f,%f,%f\n",modl[1],modl[5],modl[9],modl[13]);
	rolePrintlog("%f,%f,%f,%f\n",modl[2],modl[6],modl[10],modl[14]);
	rolePrintlog("%f,%f,%f,%f\n",modl[3],modl[7],modl[11],modl[15]);
*/


//	doing clip = proj * modl;
	matrixMultiply4by4(proj,modl,clip);
/*
    clip[ 0] = modl[ 0] * proj[ 0] + modl[ 1] * proj[ 4] + modl[ 2] * proj[ 8] + modl[ 3] * proj[12];
    clip[ 1] = modl[ 0] * proj[ 1] + modl[ 1] * proj[ 5] + modl[ 2] * proj[ 9] + modl[ 3] * proj[13];
    clip[ 2] = modl[ 0] * proj[ 2] + modl[ 1] * proj[ 6] + modl[ 2] * proj[10] + modl[ 3] * proj[14];
    clip[ 3] = modl[ 0] * proj[ 3] + modl[ 1] * proj[ 7] + modl[ 2] * proj[11] + modl[ 3] * proj[15];
    clip[ 4] = modl[ 4] * proj[ 0] + modl[ 5] * proj[ 4] + modl[ 6] * proj[ 8] + modl[ 7] * proj[12];
    clip[ 5] = modl[ 4] * proj[ 1] + modl[ 5] * proj[ 5] + modl[ 6] * proj[ 9] + modl[ 7] * proj[13];
    clip[ 6] = modl[ 4] * proj[ 2] + modl[ 5] * proj[ 6] + modl[ 6] * proj[10] + modl[ 7] * proj[14];
    clip[ 7] = modl[ 4] * proj[ 3] + modl[ 5] * proj[ 7] + modl[ 6] * proj[11] + modl[ 7] * proj[15];
    clip[ 8] = modl[ 8] * proj[ 0] + modl[ 9] * proj[ 4] + modl[10] * proj[ 8] + modl[11] * proj[12];
    clip[ 9] = modl[ 8] * proj[ 1] + modl[ 9] * proj[ 5] + modl[10] * proj[ 9] + modl[11] * proj[13];
    clip[10] = modl[ 8] * proj[ 2] + modl[ 9] * proj[ 6] + modl[10] * proj[10] + modl[11] * proj[14];
    clip[11] = modl[ 8] * proj[ 3] + modl[ 9] * proj[ 7] + modl[10] * proj[11] + modl[11] * proj[15];
    clip[12] = modl[12] * proj[ 0] + modl[13] * proj[ 4] + modl[14] * proj[ 8] + modl[15] * proj[12];
    clip[13] = modl[12] * proj[ 1] + modl[13] * proj[ 5] + modl[14] * proj[ 9] + modl[15] * proj[13];
    clip[14] = modl[12] * proj[ 2] + modl[13] * proj[ 6] + modl[14] * proj[10] + modl[15] * proj[14];
    clip[15] = modl[12] * proj[ 3] + modl[13] * proj[ 7] + modl[14] * proj[11] + modl[15] * proj[15];
	*/
    /* Extract the numbers for the RIGHT plane */
    m_Frustum[0][0] = clip[ 3] - clip[ 0];
    m_Frustum[0][1] = clip[ 7] - clip[ 4];
    m_Frustum[0][2] = clip[11] - clip[ 8];
    m_Frustum[0][3] = clip[15] - clip[12];
    /* Normalize the result */
    t = sqrt(m_Frustum[0][0] * m_Frustum[0][0] + m_Frustum[0][1] * m_Frustum[0][1] + m_Frustum[0][2] * m_Frustum[0][2]);
    m_Frustum[0][0] /= t;
    m_Frustum[0][1] /= t;
    m_Frustum[0][2] /= t;
    m_Frustum[0][3] /= t;
    /* Extract the numbers for the LEFT plane */
    m_Frustum[1][0] = clip[ 3] + clip[ 0];
    m_Frustum[1][1] = clip[ 7] + clip[ 4];
    m_Frustum[1][2] = clip[11] + clip[ 8];
    m_Frustum[1][3] = clip[15] + clip[12];
    /* Normalize the result */
    t = sqrt(m_Frustum[1][0] * m_Frustum[1][0] + m_Frustum[1][1] * m_Frustum[1][1] + m_Frustum[1][2] * m_Frustum[1][2]);
    m_Frustum[1][0] /= t;
    m_Frustum[1][1] /= t;
    m_Frustum[1][2] /= t;
    m_Frustum[1][3] /= t;
    /* Extract the BOTTOM plane */
    m_Frustum[2][0] = clip[ 3] + clip[ 1];
    m_Frustum[2][1] = clip[ 7] + clip[ 5];
    m_Frustum[2][2] = clip[11] + clip[ 9];
    m_Frustum[2][3] = clip[15] + clip[13];
    /* Normalize the result */
    t = sqrt(m_Frustum[2][0] * m_Frustum[2][0] + m_Frustum[2][1] * m_Frustum[2][1] + m_Frustum[2][2] * m_Frustum[2][2]);
    m_Frustum[2][0] /= t;
    m_Frustum[2][1] /= t;
    m_Frustum[2][2] /= t;
    m_Frustum[2][3] /= t;
    /* Extract the TOP plane */
    m_Frustum[3][0] = clip[ 3] - clip[ 1];
    m_Frustum[3][1] = clip[ 7] - clip[ 5];
    m_Frustum[3][2] = clip[11] - clip[ 9];
    m_Frustum[3][3] = clip[15] - clip[13];
    /* Normalize the result */
    t = sqrt(m_Frustum[3][0] * m_Frustum[3][0] + m_Frustum[3][1] * m_Frustum[3][1] + m_Frustum[3][2] * m_Frustum[3][2]);
    m_Frustum[3][0] /= t;
    m_Frustum[3][1] /= t;
    m_Frustum[3][2] /= t;
    m_Frustum[3][3] /= t;
    /* Extract the FAR plane */
    m_Frustum[4][0] = clip[ 3] - clip[ 2];
    m_Frustum[4][1] = clip[ 7] - clip[ 6];
    m_Frustum[4][2] = clip[11] - clip[10];
    m_Frustum[4][3] = clip[15] - clip[14];
    /* Normalize the result */
    t = sqrt(m_Frustum[4][0] * m_Frustum[4][0] + m_Frustum[4][1] * m_Frustum[4][1] + m_Frustum[4][2] * m_Frustum[4][2]);
    m_Frustum[4][0] /= t;
    m_Frustum[4][1] /= t;
    m_Frustum[4][2] /= t;
    m_Frustum[4][3] /= t;
    /* Extract the NEAR plane */
    m_Frustum[5][0] = clip[ 3] + clip[ 2];
    m_Frustum[5][1] = clip[ 7] + clip[ 6];
    m_Frustum[5][2] = clip[11] + clip[10];
    m_Frustum[5][3] = clip[15] + clip[14];
    /* Normalize the result */
    t = sqrt(m_Frustum[5][0] * m_Frustum[5][0] + m_Frustum[5][1] * m_Frustum[5][1] + m_Frustum[5][2] * m_Frustum[5][2]);
    m_Frustum[5][0] /= t;
    m_Frustum[5][1] /= t;
    m_Frustum[5][2] /= t;
    m_Frustum[5][3] /= t;
}

RoleBOOL inline areaInFrustum(RoleFLOAT x, RoleFLOAT z, RoleFLOAT size, RoleINT sizey, RoleFLOAT m_Frustum[6][4])
{
    for(int p = 0; p < 6; p++)
    {
        if(m_Frustum[p][0] * (x - size) + m_Frustum[p][1] * (-sizey) + m_Frustum[p][2] * (z - size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x + size) + m_Frustum[p][1] * (-sizey) + m_Frustum[p][2] * (z - size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x - size) + m_Frustum[p][1] * (sizey) + m_Frustum[p][2] * (z - size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x + size) + m_Frustum[p][1] * (sizey) + m_Frustum[p][2] * (z - size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x - size) + m_Frustum[p][1] * (-sizey) + m_Frustum[p][2] * (z + size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x + size) + m_Frustum[p][1] * (-sizey) + m_Frustum[p][2] * (z + size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x - size) + m_Frustum[p][1] * (sizey) + m_Frustum[p][2] * (z + size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x + size) + m_Frustum[p][1] * (sizey) + m_Frustum[p][2] * (z + size) + m_Frustum[p][3] > 0)
            continue;
        return RoleFALSE;
    }
    return RoleTRUE;
}

RoleBOOL inline cubeInFrustum(RoleFLOAT x, RoleFLOAT y, RoleFLOAT z, RoleFLOAT size, RoleFLOAT m_Frustum[6][4])
{
    for(int p = 0; p < 6; p++)
    {
        if(m_Frustum[p][0] * (x - size) + m_Frustum[p][1] * (y - size) + m_Frustum[p][2] * (z - size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x + size) + m_Frustum[p][1] * (y - size) + m_Frustum[p][2] * (z - size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x - size) + m_Frustum[p][1] * (y + size) + m_Frustum[p][2] * (z - size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x + size) + m_Frustum[p][1] * (y + size) + m_Frustum[p][2] * (z - size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x - size) + m_Frustum[p][1] * (y - size) + m_Frustum[p][2] * (z + size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x + size) + m_Frustum[p][1] * (y - size) + m_Frustum[p][2] * (z + size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x - size) + m_Frustum[p][1] * (y + size) + m_Frustum[p][2] * (z + size) + m_Frustum[p][3] > 0)
            continue;
        if(m_Frustum[p][0] * (x + size) + m_Frustum[p][1] * (y + size) + m_Frustum[p][2] * (z + size) + m_Frustum[p][3] > 0)
            continue;
        return RoleFALSE;
    }
    return RoleTRUE;
}


void drawmodel(void)
{
    if (!pmodel) {
        pmodel = glmReadOBJ("knight.obj");
        if (!pmodel) exit(0);
        glmUnitize(pmodel);
        glmFacetNormals(pmodel);
        glmVertexNormals(pmodel, 90.0);
    }
    
    glmDraw(pmodel, GLM_SMOOTH | GLM_MATERIAL);
}

Graphic::~Graphic()
{
	delete particle;
	delete  model;
}

void Graphic::deleteTextures()
{
	for (int i = 0 ; i < tNameDB.size();i++)
	{
		glDeleteTextures(1,&(tNameDB[i]->tName));
		free(tNameDB[i]);
	}
}

Graphic::Graphic(string n,Object * o):name(n)
{
	nextUpdate =0;
	model = NULL;
	particle = NULL;
	ob = o;
	tName = 0;
	// from here we need some kind of object files.... or area script 
	// to save & load data
	if (name.compare(0,5,"W_bfg") == 0)
	{
		model = new RoleModel("W_bfg.md2","W_bfg.bmp");
	}
	else if (name.compare(0,5,"sword") == 0)
	{
		model = new RoleModel("sword.md2","sword.bmp");
	}
	else if (name.compare(0,4,"mace") == 0)
	{
		model = new RoleModel("mace.md2","mace.bmp");
	}
	else if (name.compare(0,6,"knight") == 0)
	{
		model = new RoleModel("knight.md2","knight.bmp");
//		model[1] = new RoleModel("W_bfg.md2","W_bfg.bmp");
		particle = new cParticleManager();
	    particle->Add(200, FIRE, cVector3(0.0, 0.0, 0));

	}
	else if (name.compare(0,7,"soldier") == 0)
	{
		model = new RoleModel("soldier.md2","soldier.bmp");
	}
	else if (name.compare(0,6,"dragon") == 0)
	{
		model = new RoleModel("dragon.md2","dragon_green.bmp");
	}
	else if (name.compare(0,7,"dknight") == 0)
	{
		model = new RoleModel("dknight.md2","dknight.bmp");
	}
	else if (name.compare(0,7,"chicken") == 0)
	{
		model = new RoleModel("chicken.md2","chicken.bmp");
	}
	else if (name.compare(0,5,"yohko") == 0)
	{
		model = new RoleModel("yohko.md2","yohko.bmp");
		particle = new cParticleManager();
	    particle->Add(200, FIRE, cVector3(0.0, 0.0, 0));
	}
	else if (name.compare(0,7,"w_yohko") == 0)
	{
		model = new RoleModel("yohko_w.md2","yohko_w.bmp");
	}
	else if (name.compare(0,6,"goblin")== 0)
	{
		model = new RoleModel("goblin.md2","goblin1.bmp");
	}
// Particle Systems..
	else if (name.compare(0,8,"fireball") == 0)
	{
		particle = new cParticleManager();
		particle->Add(700,FIREBALL,cVector3(0.0, 0.0, 0.0));
	}
	else if (name.compare(0,9,"lightning") == 0)
	{
		particle = new cParticleManager();
		particle->Add(1000,LIGHTNING,cVector3(0.0, 0.0, 0.0));
	}
	else if (name.compare(0,4,"fire") == 0)
	{
		particle = new cParticleManager();
	    particle->Add(200, FIRE, cVector3(0.0, 0.0, 0));
	}
	else if (name.compare(0,5,"spark") == 0)
	{
		particle = new cParticleManager();
		particle->Add(500,SPARK,cVector3(0.0, 0.0, 0.0));
	}
	else if (name.compare(0,9,"explosion") == 0)
	{
		particle = new cParticleManager();
		particle->Add(2000,EXPLOSION,cVector3(0.0, 0.0, 0.0));
	}

	else if (name.compare(0,13,"lightningball") == 0)
	{
		particle = new cParticleManager();
		particle->Add(300,LIGHTNINGBALL,cVector3(0.0, 0.0, 0.0));
	}
	else if (name.compare(0,10,"movingfire") == 0)
	{
		particle = new cParticleManager();
		particle->Add(500,MOVINGFIRE,cVector3(0.0, 0.0, 0.0));
	}


/*	else if (strncmp("al",name,2) == 0)
	{
		kind = AL;
		return;
	}
*/	else
	{
		TexUnit * t = getTexture(name,NORMAL);
//		TexUnit * t = getTexture(name,MIPMAP);
		if ( t == NULL)
		{
			rolePrintlog("Fatal error, exit\n");
			quit_role(1);
		}
		tName = t->tName;
		if (name.compare(0,4,"tree") == 0)
		{
	//		rolePrintlog ("tree\n");
		}
		else if (name.compare(0,6,"flower") == 0)
		{
	//		rolePrintlog ("tree\n");
		}
	}
}
vector<TexUnit *> Graphic::tNameDB;
const RoleINT Graphic::MIPMAP = 0;
const RoleINT Graphic::NORMAL = 1;


TexUnit * Graphic::getTexture(string fileName, RoleINT option)
{
	RoleUINT texName;
	for (int i = 0 ; i < tNameDB.size();i++)
	{
		
		if (fileName.compare(tNameDB[i]->name) == 0)
		{
//			rolePrintlog("save %i, %s\n",tNameDB[i]->tName,fileName);
			return tNameDB[i];
		}
	}
	TexUnit * temp = (TexUnit *) malloc (sizeof(TexUnit));
	fileName.copy(temp->name,  MAX_NAME < fileName.length() ? MAX_NAME : fileName.length());
	temp->name[MAX_NAME < fileName.length() ? MAX_NAME : fileName.length()] = 0;
	glGenTextures(1, &texName);
	loadTexture(texName, fileName,&(temp->width),&(temp->height),option);
	temp->tName = texName;
	tNameDB.push_back(temp);
	rolePrintlog ("texName:%i %s is loaded\n",texName,temp->name);
	return temp;
}

// idx is the index of body, weapon. I have not decided yet about idx...

void Graphic::switchModel(string model_name, string skin_name)
{
	model->switchModel(model_name, skin_name, model);
}

void Graphic::syncronizeFrame(Graphic * g)
{
	model->syncronizeFrame(g->getModel());
	nextUpdate=g->getNextUpdate();
}

RoleModel * Graphic::getModel() const
{
	return model;
}

void Graphic::setNextUpdate(RoleUINT time)
{
	nextUpdate = time;
}

RoleUINT Graphic::getNextUpdate() const
{
	return nextUpdate;
}

void Graphic::draw(Player * player)
{
	draw(ob->location[0],ob->location[1],ob->location[2],
		ob->direction[0],ob->direction[1],ob->direction[2],
		player);
}

void inline draw2D(RoleFLOAT x, RoleFLOAT y, RoleFLOAT z, Player * player) 
{
		glTranslatef(x,y+1.0f,z+0.3f);

		glDisable( GL_CULL_FACE );			// Enable Back-Face-Culling (Good For High Polycount)
		float camera[3] 
			= {player->getCameraCenterX(),player->getCameraCenterY(),player->getCameraCenterZ()};
		float cameral[3] = {player->getCameraX(),player->getCameraY(),player->getCameraZ()};

		float camerav1[3] = {cameral[0] - camera[0],0.0f,cameral[2] - camera[2]};
		float camerav[3];
		float direction[3] = {0,0,-1};
		float dc = sqrt (camerav1[0]*camerav1[0]+camerav1[1]*camerav1[1]+camerav1[2]*camerav1[2]);
		float dt = 1;
		
			
		crossProduct3f(direction,camerav1,camerav);
		
		RoleFLOAT temp = camerav1[0]*direction[0] + camerav1[1]*direction[1]+ camerav1[2]*direction[2];
		float angle = acos(temp/dt/dc)*180/PIE;
		glRotatef(angle,camerav[0],camerav[1],camerav[2]);



		if (camerav[1] > 0) 
		{
			rotateCoord(RIGHT,angle, cameral[0],cameral[1],cameral[2],
							&(camera[0]),&(camera[1]),&(camera[2]));
		}
		else 
		{
			rotateCoord(RIGHT,-angle, cameral[0],cameral[1],cameral[2],
							&(camera[0]),&(camera[1]),&(camera[2]));
		}
		camera[0] = cameral[0] - camera[0];
		camera[1] = cameral[1] - camera[1];
		camera[2] = cameral[2] - camera[2];
		crossProduct3f(direction,camera,camerav1);
		temp = camera[0]*direction[0] + camera[1]*direction[1]+ camera[2]*direction[2];
		dc = sqrt (camera[0]*camera[0]+camera[1]*camera[1]+camera[2]*camera[2]);
		angle = acos(temp/dt/dc)*180/PIE;;
		glRotatef(angle,camerav1[0],camerav1[1],camerav1[2]);

//		glScalef(1.0,-1.0,1.0);
		// to verify moving object location
}

void Graphic::draw(RoleFLOAT x, RoleFLOAT y, RoleFLOAT z, 
				   RoleFLOAT dx, RoleFLOAT dy, RoleFLOAT dz,Player * player)
{

	RoleFLOAT width = ob->size[0];
	RoleFLOAT height = ob->size[1];
	RoleFLOAT length  = ob->size[2]; 
	RoleINT type = ob->getType();
 // for orientation
				   
	if ((type == Object::NPC) ||
		(type == Object::PLAYER) ||
		(type == Object::MONSTER) ||
		(type == Object::WEAPON))
	{
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
		float h = sqrt(absolute(dz*dz)+absolute(dx*dx));
		glPushMatrix();
		if ((name.compare(0,5,"yohko")==0) ||
			(name.compare(0,7,"w_yohko")==0))
		{
			y+=0.3f;
		}
		glTranslatef(x,y+0.73f,z+0.3f);
		if (dz > 0)
			glRotated(asin(dx/h)*180/PIE,0.0,1.0,0.0);
		else if (dx > 0.999)
			glRotated(90,0.0,1.0,0.0);
		else if (dx < -0.999)
			glRotated(270,0.0,1.0,0.0);
		else
			glRotated(180-asin(dx/h)*180/PIE,0.0,1.0,0.0);
		// to verify moving object location
/*	glDisable (GL_TEXTURE_2D);
		if (type ==Object::NPC)
		{
			glColor3f(0.0f,0.6f,1.0f);
			drawBox(0.0f,height*1.5,0.0f,0.5f,1.5f,0.5f);
		}
		else if (type ==Object::PLAYER)
		{
			glColor3f(1.0f,0.3f,1.0f);
			drawBox(0.0f,height*1.5,0.0f,0.5f,1.5f,0.5f);
		}
		else if (type ==Object::MONSTER)
		{
			glColor3f(1.0f,0.6f,0.0f);
			drawBox(0.0f,height*1.5,0.0f,0.5f,1.5f,0.5f);
		}

		glColor3f(1.0f,1.0f,1.0f);
		glEnable (GL_TEXTURE_2D);
*/
		if ( SDL_GetTicks() > nextUpdate)
		{
			nextUpdate = SDL_GetTicks()+100;
//			model->advanceFrame(ob->getMode());
		}
		RoleFLOAT distance = player->getDistanceFromCamera(x,y,z);
		if (distance > width * 250.0f) {
			// for far object just draw a box with whatever texture?
			// or we can get small texture for each object...
			glPopMatrix();
		}
		else if (distance > width*150.0f)
		{
			glDisable (GL_TEXTURE_2D);
			if (type ==Object::NPC)
			{
				if (ob->getAttributePtr()->hp > 0) {
					glColor3f(0.0f,0.6f,1.0f);
					drawBox(0.0f,height*1.5,0.0f,0.5f,1.5f,0.5f);
				}
			}
			else if (type ==Object::PLAYER)
			{
				glColor3f(1.0f,0.3f,1.0f);
				drawBox(0.0f,height*1.5,0.0f,0.5f,1.5f,0.5f);
			}
			else if (type ==Object::MONSTER)
			{
				if (ob->getAttributePtr()->hp > 0) {
					glColor3f(1.0f,0.6f,0.0f);
					drawBox(0.0f,height*1.5,0.0f,0.5f,1.5f,0.5f);
				}
			}
	
			glColor3f(1.0f,1.0f,1.0f);
			glEnable (GL_TEXTURE_2D);
			glPopMatrix();
		}
		else
		{
			model->drawModel(distance);	
			glPopMatrix();
			glPushMatrix();
			draw2D(x,y+1.0f,z+0.0f,player);
			ob->drawStatus(0.0,0.0,0.0);
			if (ob->type == Object::PLAYER) {
				glPushMatrix();
				glScalef(1.2,1.2,1.2);
				if (ob->getAttributePtr() !=NULL) {
/*
					glDisable (GL_TEXTURE_2D);
					glColor3f(0.4f,0.4f,0.4f);
					glBegin(GL_QUADS);
					glVertex3f(x -0.5f, y + 0.15, z);
					glVertex3f(x -0.5f, y + 0.35,z);
					glVertex3f(x +0.5f, y + 0.35 , z);
					glVertex3f(x +0.5f, y +0.15, z);
					glEnd();
					glColor3f(1.0f,1.0f,1.0f);
					glEnable (GL_TEXTURE_2D);
*/				
					printChars3D(0.3,0.2,0.0,"Lv: %i ",ob->getAttributePtr()->level);
				}
				glPopMatrix();
			}
			glPopMatrix();
			if (ob->getWeapon() != NULL)
			{
//				rolePrintlog("drawing weapon\n");
				Object * weapon = ob->getWeapon();
				weapon->syncronizeLocDirFrame(ob);
				weapon->draw(player);
			}
		}

		if (type == Object::PLAYER && ob->getAttributePtr()->hp <= 0) {
			glPushMatrix();
			draw2D(x,y+1.0f,z+0.0f,player);
			glScalef(2.5,2.5,2.5);
			glDisable (GL_TEXTURE_2D);
			particle->Render();
			glEnable (GL_TEXTURE_2D);
			glEnable( GL_CULL_FACE );			// Enable Back-Face-Culling (Good For High Polycount)
			glDisable(GL_BLEND);
			glPopMatrix();
		}


	}
	else if (type == Object::FIREBALL || type == Object::LIGHTNING || type == Object::FIRE)
	{
		glPushMatrix();
		if (type == Object::LIGHTNING) {
			draw2D(x,y+1.5f,z+0.0f,player);
		}
		else
		{
			draw2D(x,y+1.0f,z+0.0f,player);
		}
		glScalef(2.5,2.5,2.5);
//		glRotated(180,0,0,1);
		glDisable (GL_TEXTURE_2D);
		particle->Render();
		glEnable (GL_TEXTURE_2D);
		glEnable( GL_CULL_FACE );			// Enable Back-Face-Culling (Good For High Polycount)
		glDisable(GL_BLEND);
		glColor3f(1.0,1.0,1.0);
		glPopMatrix();
	}
//	 this is obj format... not gonna use 
	else if (type == AL)
	{
		RoleFLOAT h = sqrt(absolute(dz*dz)+absolute(dx*dx));
		glDisable(GL_BLEND);
		glEnable(GL_LIGHTING);
		glDisable (GL_TEXTURE_2D);

		glPushMatrix();
			glTranslated(x,y,z);
			if (dz > 0)
				glRotated(asin(dx/h)*180/PIE,0.0,1.0,0.0);
			else if (dx > 0.999)
				glRotated(90,0.0,1.0,0.0);
			else if (dx < -0.999)
				glRotated(270,0.0,1.0,0.0);
			else
				glRotated(180-asin(dx/h)*180/PIE,0.0,1.0,0.0);
			drawmodel();
		glPopMatrix();
		glEnable (GL_TEXTURE_2D);
		glDisable(GL_LIGHTING);
		glColor3f(1.0,1.0,1.0);
		glBegin(GL_LINES);
			glVertex3f(x,y,z);
			glVertex3f(x+dx*5,y+dy*5,z+dz*5);
		glEnd();
		return;
	}
	else
	{
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

		glBindTexture(GL_TEXTURE_2D, tName);
		if ((type == Object::TREES)	&& (player->getDistanceFromCamera(x,y,z) < width*200.0f))
		{

			glDisable( GL_CULL_FACE );			// Enable Back-Face-Culling (Good For High Polycount)
			glEnable(GL_ALPHA_TEST);
			glAlphaFunc(GL_GREATER,0.0);
			glBegin(GL_QUADS);
				glNormal3f(0,1,0); // need to be cacluated
				glTexCoord2f(0.0f,0.0f); glVertex3f(x,y,z+length);
				glTexCoord2f(0.0f,1.0f); glVertex3f(x,y+height,z+length);
				glTexCoord2f(1.0f,1.0f); glVertex3f(x,y+height ,z-length);
				glTexCoord2f(1.0f,0.0f); glVertex3f(x,y,z-length);

				glNormal3f(0,1,0); // need to be cacluated
				glTexCoord2f(0.0f,0.0f); glVertex3f(x+width,y ,z);
				glTexCoord2f(0.0f,1.0f); glVertex3f(x+width,y+height,z);
				glTexCoord2f(1.0f,1.0f); glVertex3f(x-width,y+height,z);
				glTexCoord2f(1.0f,0.0f); glVertex3f(x-width,y,z);
			glEnd();
			glDisable(GL_ALPHA_TEST);
			glEnable( GL_CULL_FACE );			// Enable Back-Face-Culling (Good For High Polycount)
			polycounter+=4;
		}
		else
		{
			drawBox(x,y,z,width,height,length);
		}
	}
}


void Graphic::pick_draw(Player * player,RoleUINT pickID)
{
	RoleINT type = ob->getType();

	if ((type <= Object::ACTIONOBJBEGIN) 
		|| (type >= Object::ACTIONOBJEND) || (player->getControlObj() == ob))
	{
		return;
	}
	RoleFLOAT width = ob->size[0];
	RoleFLOAT height = ob->size[1];
	RoleFLOAT length  = ob->size[2]; 
	
	RoleFLOAT x = ob->location[0];
	RoleFLOAT y = ob->location[1];
	RoleFLOAT z = ob->location[2];
	RoleFLOAT dx = ob->direction[0];
	RoleFLOAT dy = ob->direction[1];
	RoleFLOAT dz = ob->direction[2];
	glLoadName(pickID);
	// this is bounding box for picking. For speed, this is better than drawing actual object
	drawPickBox(x,y,z,width,height,length);
	/*
	if ((kind == TWOMODEL) || (kind == ONEMODEL1) || (kind == ONEMODEL2))
	{

		float h = sqrt(absolute(dz*dz)+absolute(dx*dx));
		glPushMatrix();
		glTranslated(x,y+0.73f,z+0.3f);
		if (dz > 0)
			glRotated(asin(dx/h)*180/PIE,0.0,1.0,0.0);
		else if (dx > 0.999)
			glRotated(90,0.0,1.0,0.0);
		else if (dx < -0.999)
			glRotated(270,0.0,1.0,0.0);
		else
			glRotated(180-asin(dx/h)*180/PIE,0.0,1.0,0.0);
		// to verify moving object location
			drawBox(0.0f,height,0.0f,0.5f,1.5f,0.5f);
		if (player->getDistanceFromCamera(x,y,z) > width*200.0f)
		{
			// for far object just draw a box with whatever texture?
			// or we can get small texture for each object...
			glPopMatrix();
		}
		else
		{
			model->drawPickModel();	
			glPopMatrix();
			if (ob->getWeapon() != NULL)
			{
//				rolePrintlog("drawing weapon\n");
				Object * weapon = ob->getWeapon();
				weapon->syncronizeLocDirFrame(ob);
				weapon->pick_draw(player,pickID);
			}
		}
	}
	else
	{
		if (kind == TREE)
		{

			glDisable( GL_CULL_FACE );			// Enable Back-Face-Culling (Good For High Polycount)
			glBegin(GL_QUADS);
				glNormal3f(0,1,0); // need to be cacluated
				glVertex3f(x,y,z+length);
				glVertex3f(x,y+height,z+length);
				glVertex3f(x,y+height ,z-length);
				glVertex3f(x,y,z-length);

				glNormal3f(0,1,0); // need to be cacluated
				glVertex3f(x+width,y ,z);
				glVertex3f(x+width,y+height,z);
				glVertex3f(x-width,y+height,z);
				glVertex3f(x-width,y,z);
			glEnd();
			glEnable( GL_CULL_FACE );			// Enable Back-Face-Culling (Good For High Polycount)
			polycounter+=4;
		}
		else if (kind == FLOWER)
		{
		
			glDisable( GL_CULL_FACE );			// Enable Back-Face-Culling (Good For High Polycount)
			glBegin(GL_QUADS);
				glNormal3f(0,1,0); // need to be cacluated
				glVertex3f(x,y,z+length);
				glVertex3f(x,y+height,z+length);
				glVertex3f(x,y+height ,z-length);
				glVertex3f(x,y,z-length);
	
				glNormal3f(0,1,0); // need to be cacluated
				glVertex3f(x+width,y ,z);
				glVertex3f(x+width,y+height,z);
				glVertex3f(x-width,y+height,z);
				glVertex3f(x-width,y,z);
			glEnd();
			glEnable( GL_CULL_FACE );			// Enable Back-Face-Culling (Good For High Polycount)
			polycounter+=4;
		}
		else
		{
			drawPickBox(x,y,z,width,height,length);
		}
	}
*/
}

void inline drawPickBox(RoleFLOAT x,RoleFLOAT y,RoleFLOAT z,
					  RoleFLOAT width,RoleFLOAT height,RoleFLOAT length)
{
			glBegin(GL_QUADS);
				//left
				glNormal3f(-1,0,0); // need to be cacluated
				glVertex3f(x-width,y+height,z-length);
				glVertex3f(x-width,y,z-length);
				glVertex3f(x-width,y ,z+length);
				glVertex3f(x-width,y+height,z+length);
				//Right
				glNormal3f(1,0,0); // need to be cacluated
				glVertex3f(x+width,y ,z-length);
				glVertex3f(x+width,y+height,z-length);
				glVertex3f(x+width,y+height,z+length);
				glVertex3f(x+width,y,z+length);
	
				// top
				glNormal3f(0,1,0); // need to be cacluated
				glVertex3f(x+width,y+height ,z-length);
				glVertex3f(x-width,y+height,z-length);
				glVertex3f(x-width,y+height,z+length);
				glVertex3f(x+width,y+height,z+length);

				// bottom
				glNormal3f(0,-1,0); // need to be cacluated
				glVertex3f(x+width,y ,z-length);
				glVertex3f(x-width,y,z-length);
				glVertex3f(x-width,y,z+length);
				glVertex3f(x+width,y,z+length);

				// back
				glNormal3f(0,0,1); // need to be cacluated
				glVertex3f(x+width,y ,z+length);
				glVertex3f(x+width,y+height,z+length);
				glVertex3f(x-width,y+height,z+length);
				glVertex3f(x-width,y,z+length);
	
				// front
				glNormal3f(0,0,-1); // need to be cacluated
				glVertex3f(x-width,y ,z-length);
				glVertex3f(x-width,y+height,z-length);
				glVertex3f(x+width,y+height,z-length);
				glVertex3f(x+width,y,z-length);
			glEnd();
//			polycounter+=12; // six sides * 2 triangles;
}

void inline drawBox(RoleFLOAT x,RoleFLOAT y,RoleFLOAT z,
					  RoleFLOAT width,RoleFLOAT height,RoleFLOAT length)
{
			glBegin(GL_QUADS);
				//left
				glNormal3f(0,1.0f,0); // normal to get specail effect
//				glNormal3f(-1,0,0); // need to be cacluated
				glTexCoord2f(0.0f,1.0f); glVertex3f(x-width,y+height,z-length);
				glTexCoord2f(0.0f,0.0f); glVertex3f(x-width,y,z-length);
				glTexCoord2f(1.0f,0.0f); glVertex3f(x-width,y ,z+length);
				glTexCoord2f(1.0f,1.0f); glVertex3f(x-width,y+height,z+length);
				//Right
				glNormal3f(0,1.0f,0); // normal to get specail effect
//				glNormal3f(1,0,0); // need to be cacluated
				glTexCoord2f(0.0f,0.0f); glVertex3f(x+width,y ,z-length);
				glTexCoord2f(0.0f,1.0f); glVertex3f(x+width,y+height,z-length);
				glTexCoord2f(1.0f,1.0f); glVertex3f(x+width,y+height,z+length);
				glTexCoord2f(1.0f,0.0f); glVertex3f(x+width,y,z+length);
	
				// top
				glNormal3f(0,1.0f,0); // normal to get specail effect
//				glNormal3f(0,1,0); // need to be cacluated
				glTexCoord2f(0.0f,0.0f); glVertex3f(x+width,y+height ,z-length);
				glTexCoord2f(0.0f,1.0f); glVertex3f(x-width,y+height,z-length);
				glTexCoord2f(1.0f,1.0f); glVertex3f(x-width,y+height,z+length);
				glTexCoord2f(1.0f,0.0f); glVertex3f(x+width,y+height,z+length);

				// bottom
				glNormal3f(0,1.0f,0); // normal to get specail effect
//				glNormal3f(0,-1,0); // need to be cacluated
				glTexCoord2f(0.0f,0.0f); glVertex3f(x+width,y ,z-length);
				glTexCoord2f(0.0f,1.0f); glVertex3f(x-width,y,z-length);
				glTexCoord2f(1.0f,1.0f); glVertex3f(x-width,y,z+length);
				glTexCoord2f(1.0f,0.0f); glVertex3f(x+width,y,z+length);

				// back
				glNormal3f(0,1.0f,0); // normal to get specail effect
//				glNormal3f(0,0,1); // need to be cacluated
				glTexCoord2f(0.0f,0.0f); glVertex3f(x+width,y ,z+length);
				glTexCoord2f(0.0f,1.0f); glVertex3f(x+width,y+height,z+length);
				glTexCoord2f(1.0f,1.0f); glVertex3f(x-width,y+height,z+length);
				glTexCoord2f(1.0f,0.0f); glVertex3f(x-width,y,z+length);
	
				// front
				glNormal3f(0,1.0f,0); // normal to get specail effect
//				glNormal3f(0,0,-1); // need to be cacluated
				glTexCoord2f(0.0f,0.0f); glVertex3f(x-width,y ,z-length);
				glTexCoord2f(0.0f,1.0f); glVertex3f(x-width,y+height,z-length);
				glTexCoord2f(1.0f,1.0f); glVertex3f(x+width,y+height,z-length);
				glTexCoord2f(1.0f,0.0f); glVertex3f(x+width,y,z-length);
			glEnd();
			polycounter+=12; // six sides * 2 triangles;
}
/*************************************************************************************/
/*************************************************************************************/
/*** Must be changed..... this is loading only one texture					****/
/***        Load Bitmaps And Convert To Textures                                  ****/
/*************************************************************************************/
void inline Graphic::loadTexture(RoleUINT & texName, string fileName,
						RoleFLOAT * width,RoleFLOAT * height,
						RoleINT option) {	
    /* Load Texture*/
    Image *image;
    
    /* allocate space for texture*/
    image = (Image *) malloc(sizeof(Image));
    if (image == NULL) {
		rolePrintlog("Error allocating space for image %s\n",fileName.data());
		quit_role(1);
    }

	if (!ImageLoad(fileName.data(), image)) {
		rolePrintlog("error while loading image %s\n",fileName.data());
		quit_role(1);
	}
	/* Create Texture	*********************************************/
	glBindTexture(GL_TEXTURE_2D, texName);   /* 2d texture (x and y size)*/

	if (option == NORMAL)
	{
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR); /* scale linearly when image bigger than texture*/
// This is for the performance... Mipmap.
		//	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST);	// Linear Filtered + MIPMAPPING
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR); /* scale linearly when image smaller than texture*/
	//	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S, GL_REPEAT);
	//	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T, GL_REPEAT);
	
	    /* 2d texture, level of detail 0 (normal), 4 components (red, green, blue ,alpha), x size from image, y size from image, */
	    /* border 0 (normal), rgb color data, unsigned byte data, and finally the data itself.*/
	// choose between Mipmaps and Image2D
	//	gluBuild2DMipmaps(GL_TEXTURE_2D, 4, image->sizeX, image->sizeY, GL_RGBA, GL_UNSIGNED_BYTE, image->data);
		glTexImage2D(GL_TEXTURE_2D, 0, 4, image->sizeX, image->sizeY, 0, GL_RGBA, GL_UNSIGNED_BYTE, image->data);
	}
	else if (option ==MIPMAP)
	{
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR); /* scale linearly when image bigger than texture*/
//		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);	// Linear Filtered + MIPMAPPING
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST);	// Linear Filtered + MIPMAPPING
//		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_LINEAR);	// Linear Filtered + MIPMAPPING
		gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGBA, image->sizeX, image->sizeY, GL_RGBA, GL_UNSIGNED_BYTE, image->data);
	}

	*width = (RoleFLOAT) image->sizeX;
	*height = (RoleFLOAT) image->sizeY;
	free(image->data);
	free(image);
};

extern void inline loadFont(void)								// Build Our Font Display List
{
	TexUnit * t =  Graphic::getTexture("font.bmp",Graphic::NORMAL);
	font.fontID = t->tName;
}

extern void inline printChars(RoleINT x, RoleINT y, char * format, ...)	// Where The Printing Happens
{
	char buffer [255], *s;
	RoleINT i = 0;
	RoleFLOAT xTexOffset = 0;
	RoleFLOAT yTexOffset = 0;
	RoleINT xoffset = 0;
	RoleINT yoffset = 0;

	glBindTexture(GL_TEXTURE_2D, font.fontID);			// Select Our Font Texture

    va_list args;
	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);
	for (s = buffer; *s && (i< 256); s++) {
		xTexOffset = ((*s-32)%16)/16.0;
		yTexOffset = 1-((*s-32)/16)/8.0;
		xoffset = x+i*10;
		yoffset = y;
		glBegin(GL_QUADS);
			glTexCoord2f(xTexOffset ,yTexOffset-16.0f/128.0); glVertex2i(xoffset,yoffset);
			glTexCoord2f(xTexOffset ,yTexOffset); glVertex2i(xoffset,(yoffset+16));
			glTexCoord2f(xTexOffset+15/256.0 ,yTexOffset); glVertex2i((xoffset+15),(yoffset+16));
			glTexCoord2f(xTexOffset+15/256.0 ,yTexOffset-16.0f/128.0); glVertex2i((xoffset+15),yoffset);
		glEnd();
		i++;
	}
}

// need to be changed
extern void inline printChars3D(RoleFLOAT x, RoleFLOAT y, RoleFLOAT z,
						 char * format, ...)	// Where The Printing Happens
{
	RoleFLOAT size = 0.1f;
	char buffer [255], *s;
	RoleINT i = 0;
	RoleFLOAT xTexOffset = 0;
	RoleFLOAT yTexOffset = 0;
	RoleFLOAT xoffset = 0.0f;
	RoleFLOAT yoffset = 0.0f;

	glBindTexture(GL_TEXTURE_2D, font.fontID);			// Select Our Font Texture

    va_list args;
	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);
	for (s = buffer; *s && (i< 256); s++) {
		xTexOffset = ((*s-32)%16)/16.0;
		yTexOffset = 1-((*s-32)/16)/8.0;
		xoffset = x-i*size;
		yoffset = y;
		glBegin(GL_QUADS);
			glTexCoord2f(xTexOffset+15/256.0 ,yTexOffset-16.0f/128.0); glVertex3f(xoffset,yoffset,z);
			glTexCoord2f(xTexOffset+15/256.0 ,yTexOffset); glVertex3f(xoffset,(yoffset+size*1.6f),z);
			glTexCoord2f(xTexOffset ,yTexOffset); glVertex3f((xoffset+size*1.5f),(yoffset+size*1.6f),z);
			glTexCoord2f(xTexOffset ,yTexOffset-16.0f/128.0); glVertex3f((xoffset+size*1.5f),yoffset,z);
		glEnd();
		i++;
	}
}

/*
extern void printChar(INT x, INT y, char ch)
{

	glEnable (GL_TEXTURE_2D);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	glBindTexture(GL_TEXTURE_2D, font.fontID);
	glBegin(GL_QUADS);
//		glNormal3f(0,1,0); // need to be cacluated
		glTexCoord2f(xoffset ,yoffset); glVertex2f(x/640.0,y/480.0);
		glTexCoord2f(xoffset ,yoffset+1.0f/128.0); glVertex2f(x/640.0,(y+16)/480.0);
		glTexCoord2f(xoffset+1.0f/256.0 ,yoffset+1.0f/128.0); glVertex2f((x+16)/640.0,(y+16)/480.0);
		glTexCoord2f(xoffset+1.0f/256.0 ,yoffset); glVertex2f((x+16)/640.0,y/480.0);
	glEnd();
}
*/