#ifndef CADVPARTICLE_HEADER
#define CADVPARTICLE_HEADER

#include <vector>
#include <string>
#include <algorithm>
#include <functional>
#include <glut.h>
#include <iostream>
#include <cmath>
#include "cUtility.h"

//using namespace std;

///////////////////////////////
// Particle

struct cParticle
{
public:

    //constructor
    cParticle();

    //output
    void Render();

    //properties
    bool bAlive;
    bool bStretch;
    bool bSharp;
    //bool bEnergySize;   //if energy should affect size

    //location & movement
    cVector3 pos;
    cVector3 velocity;
    cVector3 acceleration;

    //appearance
    cVector3 color;
    float size;
    float energy;
};

///////////////////////////////
// Particle System - handles a set of particle

enum cSystemType {
    UNKNOWN = -1, 
    SPARK = 10, 
    EXPLOSION = 20,
    LIGHTNING = 30,
	LIGHTNINGBALL,
    FIRE = 40, 
	FIREBALL,
	MOVINGFIRE};

class cParticleSystem
{
friend class cParticleManager;

protected:

    cSystemType sysType;

    //texture of particle
    ////TO BE IMPLEMENTED
    cTexture* texture;
    //diff ways to blend
    ////TO BE IMPLEMENTED
    cBlendMode blendMode;
    //boudingBox to check if this should be rendered
    ////NOTE TO IMPLMENT THIS
    cBoundBox boundingBox;

    //number remaining alive
    int nrAlive;
    
    //position in real world
    cVector3 pos;

    //camera
    cVector3 cam;

    //store all the particles
    vector<cParticle> particles;
    int numParticle;
    //sidenote: 
    //be very careful, 
    //  numParticle might not necessarily equal particles.size()
    //  e.g. EXPLOSION, since it uses rotation to use less particle
    
public:

    //if this systerm is still alive
    bool bAlive;

    cParticleSystem();
    cParticleSystem(int num);

    virtual void Render(){;};
    virtual void Update(){;};
    virtual void Init(int num, cSystemType sysType){;};
};

///////////////////////////////
// Particle Manager - handles a set of Particle System

class cParticleManager
{
public:
    cParticleManager();
	~cParticleManager();
    //////////////
    // Interface

    //return the id
    int Add(int num, cSystemType sysType,
        cVector3 pos = cVector3(0.0, 0.0, 0.0),
        cVector3 cam = cVector3(0.0, 0.0, 1.0));
    ////NOTE IMPLEMENT CAMERA
    void Remove(int id);

    cParticleSystem* Get(int id);

    /////////////
    // Output
    void Render();
    void Reset(int id);

private:
    vector<cParticleSystem*> systems;
    
};

///////////////////////////////
// Buring - include Spark, Fire, Explosion

class cFire : public cParticleSystem
{
public:
    /////////////////
    // Fire Setting

    struct cInitInfo
    {
        ///////////////
        // Particle Property

        //basic properties
        cTexture* tex;
        cVector3 color;
        float size;
        cVector3 vel;
        
        //randomize the energy
        float minEnergy;
        float maxEnergy;
        float reduceFactor;
        float reduceAmount;

        bool bRejuvenate;

        ////////////////
        // General Property

        //shape of fire
        float height;
        float width;
        
        //transformation
        float direction; //0 is up, PI/2 is left
        cVector3 shear; ////TO BE IMPLEMENTED

        //when the fire start dying
        float fireAge;
        float step;

        ////////////////
        // Specific Property

        //explosion property
        float radius;
        int numRotate;

        //spark property
        cVector3 acc; //gravity

        //fire property
        float botRatio;
		cVector3 color2;

        void InitFire()
        {
            tex = NULL;
            color = cVector3(1.0, 0.1, 0.1);
            size = 0.05;

            bRejuvenate = true;

            height = 1.0;
            width = 0.1;

            direction = 0;
            
            fireAge = 1000000; //inf

            minEnergy = 0.0;
            maxEnergy = 1.0;
            reduceFactor = 1.0;
            reduceAmount = 0.03;

            step = ((maxEnergy - minEnergy) / reduceAmount);
            vel.y = height / step;
            vel.x = width / step;
            vel.x /= 2.0;

            radius = 0.0;   //not used for fire
            acc = cVector3(0.0, 0.0, 0.0);
            botRatio = 0.2;

			//interpolate color for fire, not used right now
			//color2 = cVector3(0.2, -0.3, 0.0);
			color2 = cVector3(0.0, 0.0, 0.0);

        };

        void InitSpark()
        {
            tex = NULL;
            color = cVector3(1.0, 0.1, 0.1);
            size = 0.005;

            bRejuvenate = true;

            direction = 0;
            height = 1.0;
            width = 1.0;
            
            fireAge = 1000000; //inf

            minEnergy = 0.0;
            maxEnergy = 1.0;
            reduceFactor = 1.0;
            reduceAmount = 0.03;

            step = ((maxEnergy - minEnergy) / reduceAmount);
            vel.y = height / step;
            vel.x = width / step;

            radius = 0.0;   //not used for spark
            acc = cVector3(0.0, -0.003, 0.0);
            botRatio = 0.0; //not used

			//for color only, thus set all to 0
			color2 = cVector3(0.0, 0.0, 0.0);
        };

        void InitExplosion()
        {
            tex = NULL;
            color = cVector3(1.0, 0.1, 0.1);
            size = 0.020;

            bRejuvenate = false;

            direction = 0;
            height = 1.0;
            width = 1.0;
            
            fireAge = 100000;   //inf

            minEnergy = 0.0;
            maxEnergy = 1.0;
            reduceFactor = 1.0;
            reduceAmount = 0.10;

            step = ((maxEnergy - minEnergy) / reduceAmount);
            vel.y = height / step;
            vel.x = width / step;

            radius = sqrt(vel.x * vel.x + vel.y * vel.y);
            acc = cVector3(0.0, 0.0, 0.0); //not used
            botRatio = 0.0; //not used
            numRotate = 2;

			//for color only, thus set all to 0
			color2 = cVector3(0.0, 0.0, 0.0);
        };

        void InitLightningBall()
        {
            tex = NULL;
            color = cVector3(0.1, 0.1, 1.0);
            size = 0.020;

            bRejuvenate = true;

            direction = 0;
            height = 0.7;
            width = 0.7;
            
            fireAge = 100000;   //inf

            minEnergy = 0.0;
            maxEnergy = 1.0;
            reduceFactor = 1.0;
            reduceAmount = 0.10;

            step = ((maxEnergy - minEnergy) / reduceAmount);
            vel.y = height / step;
            vel.x = width / step;

            radius = sqrt(vel.x * vel.x + vel.y * vel.y);
            acc = cVector3(0.0, 0.0, 0.0); //not used
            botRatio = 0.0; //not used
            numRotate = 2; //not used

			//for color only, thus set all to 0
			color2 = cVector3(0.0, 0.0, 0.0);
        };

        void InitFireBall()
        {
            tex = NULL;
            color = cVector3(1.0, 0.1, 0.1);
            size = 0.020;

            bRejuvenate = true;

            direction = 0;
            height = 0.25;
            width = 0.25;
            
            fireAge = 100000;   //inf

            minEnergy = 0.5;
            maxEnergy = 1.0;
            reduceFactor = 1.0;
            reduceAmount = 0.03;

            step = ((maxEnergy - minEnergy) / reduceAmount);
            vel.y = height / step / 3.0;
            vel.x = width / step / 3.0;

            radius = sqrt(vel.x * vel.x + vel.y * vel.y);
            acc = cVector3(0.0, 0.0, 0.0); //not used
            botRatio = 0.0; //not used
            numRotate = 1; //not used

			//for color only, thus set all to 0
			color2 = cVector3(0.0, 0.0, 0.0);
        };

        cInitInfo(){;};
    };
	cInitInfo& GetInfo()	{return info;};

    /////////////////
    // Constructor
    cFire(int num, cSystemType type);
    void Init(int num, cSystemType type);

    /////////////////
    // Modifier and Output
    void Update();
    void Render();

private:

    //all info about the fire
    cInitInfo info;
    //current age of the fire
    int age;
    //init one particle
    cParticle InitParticle();

};

///////////////////////////////
// Lightning - from Sky

class cLightning : public cParticleSystem
{
public:
    /////////////////
    // Settings

    struct cInitInfo
    {
        ///////////////
        // Precompute property
        
        //turning chances
        float tc1;
        float tc2;
        float tc3;

        //turning size
        float ts1;
        float ts2;
        float ts3;

        //turning bound;
        float maxAngle;
        float minAngle;

        //step size
        float diff;

        //split chance
        float split;  //incrase later
        float splitEnergyBoost;

        ///////////////
        // Particle Property

        //basic properties
        cTexture* tex;
        cVector3 color;
        float size;
        cVector3 vel;
        
        //randomize the energy
        float minEnergy;
        float maxEnergy;
        float reduceFactor;
        float reduceAmount;

        ////////////////
        // General Property

        float lightningAge; //how many frame
        
        //transformation
        float direction; //0 is down, PI/2 is right

        void InitLightning()
        {
            //turning chances
            tc1 = 0.80;
            tc2 = 0.18;
            tc3 = 0.02;

            //turning size
            ts1 = PI/36.0;    //-5~5 degrees
            ts2 = PI/9.0;     //-20~20 degrees
            ts3 = PI/4.5;     //-40~40 degrees

            //turning bound
            maxAngle = 2.0*PI - PI/4.0;
            minAngle = PI + PI/4.0;

            //step size
            diff = 0.005;

            //split chance
            split = 0.05;  //incrase later
            splitEnergyBoost = 1.1;

            //other properties

            tex = NULL;
            color = cVector3(0.1, 0.1, 1.0);
            size = 0.01;
            vel = cVector3(0, -0.1, 0.0);

            lightningAge = 5;
            
            minEnergy = 0.4;
            maxEnergy = 0.9;
            reduceFactor = 1.00;
            reduceAmount = 0.00;

            direction = 0;
        };

        cInitInfo(){;};
    };
	cInitInfo& GetInfo()	{return info;};

    /////////////////
    // Constructor
    cLightning(int num, cSystemType type);
    void Init(int num, cSystemType type);

    /////////////////
    // Modifier and Output
    void Update();
    void Render();

private:

    //precompute the location and the strenght of all particle
    //start with full energy
    void PreCalParticle(int num, float energy, float angle = 3*PI/2, cVector3 pos = cVector3(0.0, 0.0, 0.0));

    //time elasped since the beginning
    float age;

    //all info about the lightning
    cInitInfo info;

};

class cMovingFire : public cParticleSystem
{
public:

    struct cInitInfo
    {
        ///////////////
        // Particle Property

        //basic properties
        cTexture* tex;
        cVector3 color;
        float size;
        
        //randomize the energy
        float minEnergy;
        float maxEnergy;
        float reduceFactor;
        float reduceAmount;

        bool bRejuvenate;

		//step to disappear
		float stepParticle;

		//upward fire
		cVector3 gravity;

        ////////////////
        // General Property

        //shape of fire
		float length;
		cVector3 center;
		cVector3 tip;
		float radius;
        
        //when the fire start dying
        float fireAge;

		//moving direction
		cVector3 target;
		cVector3 vel;
		cVector3 acc;
		float stepTarget;

		//randomness in acceleration
		float randFactor;
		cVector3 detour;
		float maxVel;
		float maxAcc;

        void InitMovingFire()
        {
            tex = NULL;
            color = cVector3(1.0, 0.1, 0.1);
            size = 0.05;

            bRejuvenate = true;

            fireAge = 1000000; //inf

            minEnergy = 0.0;
            maxEnergy = 1.0;
            reduceFactor = 1.0;
            reduceAmount = 0.03;

            stepParticle = ((maxEnergy - minEnergy) / reduceAmount);
			gravity = cVector3(0.0, 0.01, 0.0);

			target = cVector3(0.5, -0.5, 0.0);
			center = cVector3(0.0, 0.0, 0.0);
			tip = cVector3(0.0, 0.0, 0.0);

			stepTarget = 1000;
			length = 0.3;

			acc = cVector3(0.0, 0.0, 0.0);
			vel = cVector3(0.0, 0.0, 0.0);
			
			radius = 0.05;
			randFactor = 0.5;

			detour = cVector3(0.00025, 0.0002, 0.0);
			maxVel = 0.2;
			maxAcc = 0.3;	//NOT USED
        };

		void UpdateMoving()
		{
			//tip = cVector3(0.0, 0.0, 0.0);
			
			center += vel;
			if ((vel+acc).Dist() > maxVel) return;
			
			vel += acc;
			
			acc = (target - center) / stepTarget;
			acc = cVector3(
				Rand(acc.x * (1 - randFactor), acc.x * (1 + randFactor)),
				Rand(acc.y * (1 - randFactor), acc.y * (1 + randFactor)),
				Rand(acc.z * (1 - randFactor), acc.z * (1 + randFactor)));
	
			acc.x += Rand(-detour.x, detour.x);
			acc.y += Rand(-detour.y, detour.y);
			acc.z += Rand(-detour.z, detour.z);

			tip = center - acc * length;

			
		};

        cInitInfo(){;};
    };
	cInitInfo& GetInfo()	{return info;};

    /////////////////
    // Constructor
    cMovingFire(int num, cSystemType type);
    void Init(int num, cSystemType type);

    /////////////////
    // Modifier and Output
    void Update();
    void Render();

private:

    //all info about the fire
    cInitInfo info;
    //current age of the fire
    int age;
    //init one particle
    cParticle InitParticle();

};




#endif