%
% Programmed and algorihtm by Chanwoo Kim (chanwook@cs.cmu.edu)
%
% This package is used for the INTERSPEECH 2009 paper
%
% Imporrant: The ITD threshold is for "4 cm" distance microphone and 16 kHz sampling
% rate. If the microphone distance and the sampling rate is different from
% this configuration you need to change dTH in the code.
%
% You can do select between PD and PDCW  by changning bPDCW option
%
% if bPDCW == 1 it works as PDCW if not it works as PD
%
%

function [ad_y, aadWeight] = PDCW(szOutFileName, szInFileName)
    %
    % 75 ms (0.1 s) frame length
    dFrameLen = 0.075;
    %
    % ITD threshold is for "4 cm" microphone with 16 kHz sampling rate case
    dITD_TH       = 0.2;
    
    %
    % Decides wheter to choose PDCW or PD
    %
    bPDCW = 1;
    
    %
    % This part for mswav
    %
    ad_x = wavread(szInFileName)';
    
    dSampRate      = 16000;
    iSpeechLen     = length(ad_x);
    iFrameLen      = round(dSampRate * dFrameLen);
    iFramePeriod   = round(dSampRate * dFrameLen / 2); % To meet the OLA constraint
    iFFTSize       = 2^ceil(log(iFrameLen) / log(2));
    adITD_TH     = linspace(dITD_TH, dITD_TH,  iFFTSize / 2);
    dEta         = 1e-2;
	dPowerFactor = 1;

    dOrgStd  = std(ad_x(1, :));
    ad_X_L   = zeros(1, iFFTSize);
    ad_X_R   = zeros(1, iFFTSize);
    adAngle1 = zeros(1, iFFTSize);
    adAngle2 = zeros(1, iFFTSize);
    ad_y     = zeros(1, length(ad_x));
    
    iNumFrames  = floor((iSpeechLen - iFrameLen) / iFramePeriod) + 1;
    aad_X_tilde = zeros(iFFTSize, iNumFrames);
    aad_mu      = zeros(iFFTSize / 2, iNumFrames);
    aad_mu_g    = zeros(iFFTSize / 2, iNumFrames);
 
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %
    % Analysis
    %
    iFI = 0;    
    adBuffer  = zeros(iFFTSize, 1);
    iNumFilts = 40;
    aad_H     = ComputeFilterResponse(iNumFilts, iFFTSize);
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %
    % Frame-by-frame processing
    %
    for i = 0 : iFramePeriod : iSpeechLen - 1 -  iFrameLen,
        iFI = iFI + 1;
        
        ad_x_L_Frame = ad_x(1, i + 1 : i + iFrameLen) .* hamming(iFrameLen)';
        ad_x_R_Frame = ad_x(2, i + 1 : i + iFrameLen) .* hamming(iFrameLen)';
        ad_X_L       = fft(ad_x_L_Frame, iFFTSize);
        ad_X_R       = fft(ad_x_R_Frame, iFFTSize);
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %
        % Now, let's calculate the phase difference
        %
        ad_X_LAngle = (angle(ad_X_L)); 
        ad_X_RAngle = (angle(ad_X_R ));
        adPhaseDiff1 = abs(ad_X_LAngle - ad_X_RAngle);
        adPhaseDiff2 = abs(ad_X_LAngle - ad_X_RAngle - 2 * pi);
        adPhaseDiff3 = abs(ad_X_LAngle - ad_X_RAngle + 2 * pi);
        adPhaseDiff = min([adPhaseDiff1; adPhaseDiff2; adPhaseDiff3]);
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %
        % ITD calculation from the phase difference
        %
        ad_ITD(1) = 0;
        ad_ITD(2 : iFFTSize / 2) = (adPhaseDiff(2 : iFFTSize / 2) ./ (2 * pi * (1 : iFFTSize / 2 - 1) / iFFTSize))';
        ad_X_Bar = (ad_X_L + ad_X_R) / 2;
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %
        %  time-frequency mask /mu
        %
        ad_mu            = zeros(iFFTSize / 2, 1);
        aiIndex          = find(ad_ITD < adITD_TH(1 : iFFTSize / 2));
        ad_mu(aiIndex)   = 1;
        ad_mu            = max(ad_mu, dEta);
        aad_mu( :, iFI)  = ad_mu;
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %
        % The processing using PDCW (Phase Difference Channel Weighting)
        %
        if bPDCW == 1
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            %
            % Calculation of w(i, m)
            %
            for j = 1 : iNumFilts,
                aad_w(j, iFI) = sum(ad_mu .*   abs (aad_H( : , j)  .*  ad_X_Bar(1 : iFFTSize / 2)').^ 1) ...
                                             / sum(abs(aad_H( :, j) .* ad_X_Bar(1 : iFFTSize / 2)').^ 1);
            end
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            %
            % Calculation of /mu_g(k, m) (time-channel mask)
            %
            for j = 1 : iNumFilts,
                aad_mu_g(:, iFI) = aad_mu_g(:, iFI) + aad_w(j, iFI) .^ 1 .* abs(aad_H(:, j)) .^ 1;
            end
            aad_mu_g(:, iFI) = ((aad_mu_g(:, iFI)) ...
                                           ./ sum(abs(aad_H' .^ 1))') .^ (1 / 1); 

            % Make the mu_g symmetric
            ad_mu_g_sym = [aad_mu_g(:, iFI); flipud(aad_mu_g( :, iFI))];
            ad_mu_g_sym = max(ad_mu_g_sym, dEta);
            aad_X_tilde( :, iFI) = (ad_X_Bar .*  ad_mu_g_sym')';
        else
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            %
            % The processing using PD (Phase Difference)
            %
            ad_mu     = max(ad_mu, dEta);
            ad_mu_sym = [aad_mu(:, iFI); flipud(aad_mu( :, iFI))];
            aad_X_tilde( :, iFI) = (ad_X_Bar .*  ad_mu_sym')';   
        end    
    end
        
    ad_y = zeros(1, length(ad_x));
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %
    % Resynthesis using OLA
    %
    iFI = 0;
     for i = 0 : iFramePeriod : iSpeechLen - 1 -  iFrameLen,
         iFI = iFI + 1;
         aad_X_tilde( :, iFI) = aad_X_tilde( :, iFI); 
         adBuffer = fliplr(ifft(aad_X_tilde( :, iFI))');
         ad_y(i + 1 : i + iFrameLen) = ad_y(i + 1 : i + iFrameLen) + adBuffer(1 : iFrameLen);
     end
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %
    % Selecting only the real part
    %
    ad_y = real(ad_y);
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %
    % Removing the first and the last frames.. there were not overlapped
    %
    ad_y(length(ad_y) - iFrameLen : length(ad_y)) = [];
    ad_y(1 : iFrameLen)                           = [];
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %
    % If you want to see the mask please use the following imagescs
    %
    if 0
        figure
        imagesc(aad_mu); axis xy;
        if (bPDCW == 1)
            figure
            imagesc(aad_w); axis xy;
            figure
            imagesc(aad_mu_g); axis xy;
        end
    end   
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %
    % Writing the data
    %
   
    %
    % Writing the data in MS wave
    %
    wavwrite(ad_y / max(abs(ad_y(:))) / 1.1, 16000, 16, szOutFileName)
   
end
