/*****************************************************************************/
/*
  parameters.c: manage simulation parameters
*/
/*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include "main.h"
#include "opt-parameters.h"

/*****************************************************************************/

void
init_default_parameters( SIM *s )
{

  /* Overall parameters */
  s->trial_duration = 10.0;
  s->time_step = 0.001;

  /* Objective function */
  s->desired_speed = 0.4;
  s->speed_penalty_weight = 1000.0; /* 10.0 to optimize launch parameters */
  s->torque_penalty_weight = 1/20000.0;

  /* Desireds */
  s->pitch_d = 0;
  s->pitchd_d = 0;
  /* need to do all the rest. */

  /* Initial posture */
  s->wait_duration = 0.1;
  s->wait_l_hip = -0.1;
  s->wait_l_knee = 0.0;
  s->wait_r_hip =  0.1;
  s->wait_r_knee = 0.0;

  /* Launch1 parameters: dip knees */
  s->l1_knee_feedforward = -0.5;
  s->l1_duration = 0.900306463242;
  s->l1_movement_duration = 0.499999552965;
  s->l1_lhip = -0.299919515848;
  s->l1_lknee = 0.293819665909;
  s->l1_rhip = -0.099923759699;
  s->l1_rknee = 0.300010144711;

  /* right leg pushoff */
  s->l2_movement_duration = 0.4;
  s->l2_rhip  = 0.25;
  s->l2_rknee = 0.05;
  s->l2_z_force_threshold = 5.0;

  /* right swing */
  s->l3_duration = 0.35;
  s->l3_rhip1_time = 0.6;
  s->l3_rhip1 = -0.4;
  s->l3_rhip2_time = 1.0;
  s->l3_rhip2 = -0.3;
  s->l3_rknee1_time = 0.4;
  s->l3_rknee1 = 0.7;
  s->l3_rknee2_time = 0.6;
  s->l3_rknee2 = 0.6;
  s->l3_rknee3_time = 0.7;
  s->l3_rknee3 = 0.3;
  s->l3_lhip_time = 1.0;
  s->l3_lhip = 0.2;
  s->l3_lknee1_time = 0.5;
  s->l3_lknee1 = 0.3;
  s->l3_lknee2_time = 1.0;
  s->l3_lknee2 = 0.0;

  /* Steady state gait parameters */
  s->swing_time = 0.31;
  s->swing_hip1 = -0.5;
  s->swing_hip1_time = 0.6;
  s->swing_hip2 = -0.4;
  s->swing_knee1 = 0.76;
  s->swing_knee1_time = 0.69;
  s->swing_knee2 = 0.5;
  s->swing_knee2_time = 0.897;
  s->swing_knee3 = 0.3;
  s->swing_knee3_time = 0.7;
  s->pushoff_duration = 0.2;
  s->stance_knee_top = 0.3;
  s->stance_knee_top_time = 0.15;

  s->knee_ff_peak_time = 0.04;
  s->knee_ff_peak = -2.0;
  s->knee_ff_time = 0.08;
  s->knee_ff = -1.0;

/* Things to adjust
ground speed match variables (velocity of last knot?)
K and B could be a function of state
  s->k_hip = 50.0;
  s->b_hip = 0.25;
  s->k_knee = 50.0;
  s->b_knee = 0.25;
launch variables
 */

/* model parameters
sdfast stuff
  s->gnd_k = 2000.0;
  s->gnd_b = 2000.0;
  s->gnd_k_x = 2000.0;
*/

  s->how_many_trials = 100;
  s->rand_scale = 0.25;
  s->rand_seed = 1;

  /* Garth limits
     k_hip 0.6
     b_hip 0.018
     k_knee 0.47
     b_knee 0.031
     torque_hip 1.8  could be 4
     torque_knee 5.4 could be 12
  */
  s->k_hip = 5.0;
  s->b_hip = 0.1;
  s->k_knee = 10.0;
  s->b_knee = 0.25;
  s->k_pitch = 5.0;
  s->b_pitch = 0.1;

  // contact impedance parameters
  s->gnd_k = 2000.0;
  s->gnd_b = 2000.0;
  s->gnd_k_x = 2000.0;

  s->hip_I_model = 0.01;
  s->crashed_penalty_increment = 0.001;

  s->hip_gain = -1.0;
  s->knee_gain = -0.5;
  /*
  s->hip_gain = 0.0;
  s->knee_gain = 0.0;
  */
  s->hip_target_max = -0.1;
  s->knee_target_min = 0;
  s->knee_target_max = 0.25;
}

/*****************************************************************************/

void
apply_parameters( float *p, SIM *s )
{
  s->swing_time = p[0];
  if ( s->swing_time < 0 )
    s->swing_time = 0;
  s->swing_knee3 = p[1];
  s->swing_hip2 = p[2];
  s->swing_hip1 = p[4];
  s->pitch_d = p[5];
  s->pushoff_duration = p[6];
  s->swing_hip1_time = p[7];
  s->swing_knee1 = p[8];
  s->swing_knee1_time = p[9];
  s->swing_knee2 = p[10];
  s->swing_knee2_time = p[11];
}

/*****************************************************************************/
/******************************************************************/

int process_parameters( OPT_PARAM *p, SIM *s, int verbose )
{
  int total = 0;

  for( ; ; )
    {
      if ( p == NULL )
	break;
      if ( strcmp( p->name, "desired_speed" ) == 0 )
	p->pointer = &(s->desired_speed);
      else if ( strcmp( p->name, "how_many_trials" ) == 0 )
	p->pointer = &(s->how_many_trials);
      else if ( strcmp( p->name, "trial_duration" ) == 0 )
	p->pointer = &(s->trial_duration);
      else if ( strcmp( p->name, "swing_time" ) == 0 )
	p->pointer = &(s->swing_time);
      else if ( strcmp( p->name, "swing_hip1" ) == 0 )
	p->pointer = &(s->swing_hip1);
      else if ( strcmp( p->name, "swing_hip1_time" ) == 0 )
	p->pointer = &(s->swing_hip1_time);
      else if ( strcmp( p->name, "swing_hip2" ) == 0 )
	p->pointer = &(s->swing_hip2);
      else if ( strcmp( p->name, "swing_knee1" ) == 0 )
	p->pointer = &(s->swing_knee1);
      else if ( strcmp( p->name, "swing_knee1_time" ) == 0 )
	p->pointer = &(s->swing_knee1_time);
      else if ( strcmp( p->name, "swing_knee2" ) == 0 )
	p->pointer = &(s->swing_knee2);
      else if ( strcmp( p->name, "swing_knee2_time" ) == 0 )
	p->pointer = &(s->swing_knee2_time);
      else if ( strcmp( p->name, "swing_knee3" ) == 0 )
	p->pointer = &(s->swing_knee3);
      else if ( strcmp( p->name, "swing_knee3_time" ) == 0 )
	p->pointer = &(s->swing_knee3_time);
      else if ( strcmp( p->name, "pushoff_duration" ) == 0 )
	p->pointer = &(s->pushoff_duration);
      else if ( strcmp( p->name, "stance_knee_top" ) == 0 )
	p->pointer = &(s->stance_knee_top);
      else if ( strcmp( p->name, "stance_knee_top_time" ) == 0 )
	p->pointer = &(s->stance_knee_top_time);
      else if ( strcmp( p->name, "speed_penalty_weight" ) == 0 )
	p->pointer = &(s->speed_penalty_weight);
      else if ( strcmp( p->name, "pitch_d" ) == 0 )
	p->pointer = &(s->pitch_d);
      else if ( strcmp( p->name, "l1_duration" ) == 0 )
	p->pointer = &(s->l1_duration);
      else if ( strcmp( p->name, "l1_movement_duration" ) == 0 )
	p->pointer = &(s->l1_movement_duration);
      else if ( strcmp( p->name, "l1_lhip" ) == 0 )
	p->pointer = &(s->l1_lhip);
      else if ( strcmp( p->name, "l1_lknee" ) == 0 )
	p->pointer = &(s->l1_lknee);
      else if ( strcmp( p->name, "l1_rhip" ) == 0 )
	p->pointer = &(s->l1_rhip);
      else if ( strcmp( p->name, "l1_rknee" ) == 0 )
	p->pointer = &(s->l1_rknee);
      else if ( strcmp( p->name, "l2_movement_duration" ) == 0 )
	p->pointer = &(s->l2_movement_duration);
      else if ( strcmp( p->name, "l2_rhip" ) == 0 )
	p->pointer = &(s->l2_rhip);
      else if ( strcmp( p->name, "l2_rknee" ) == 0 )
	p->pointer = &(s->l2_rknee);
      else if ( strcmp( p->name, "l2_z_force_threshold" ) == 0 )
	p->pointer = &(s->l2_z_force_threshold);
      else if ( strcmp( p->name, "l3_duration" ) == 0 )
	p->pointer = &(s->l3_duration);
      else if ( strcmp( p->name, "l3_rhip1_time" ) == 0 )
	p->pointer = &(s->l3_rhip1_time);
      else if ( strcmp( p->name, "l3_rhip1" ) == 0 )
	p->pointer = &(s->l3_rhip1);
      else if ( strcmp( p->name, "l3_rhip2_time" ) == 0 )
	p->pointer = &(s->l3_rhip2_time);
      else if ( strcmp( p->name, "l3_rhip2" ) == 0 )
	p->pointer = &(s->l3_rhip2);
      else if ( strcmp( p->name, "l3_rknee1_time" ) == 0 )
	p->pointer = &(s->l3_rknee1_time);
      else if ( strcmp( p->name, "l3_rknee1" ) == 0 )
	p->pointer = &(s->l3_rknee1);
      else if ( strcmp( p->name, "l3_rknee2_time" ) == 0 )
	p->pointer = &(s->l3_rknee2_time);
      else if ( strcmp( p->name, "l3_rknee2" ) == 0 )
	p->pointer = &(s->l3_rknee2);
      else if ( strcmp( p->name, "l3_rknee3_time" ) == 0 )
	p->pointer = &(s->l3_rknee3_time);
      else if ( strcmp( p->name, "l3_rknee3" ) == 0 )
	p->pointer = &(s->l3_rknee3);
      else if ( strcmp( p->name, "l3_lhip_time" ) == 0 )
	p->pointer = &(s->l3_lhip_time);
      else if ( strcmp( p->name, "l3_lhip" ) == 0 )
	p->pointer = &(s->l3_lhip);
      else if ( strcmp( p->name, "l3_lknee1_time" ) == 0 )
	p->pointer = &(s->l3_lknee1_time);
      else if ( strcmp( p->name, "l3_lknee1" ) == 0 )
	p->pointer = &(s->l3_lknee1);
      else if ( strcmp( p->name, "l3_lknee2_time" ) == 0 )
	p->pointer = &(s->l3_lknee2_time);
      else if ( strcmp( p->name, "l3_lknee2" ) == 0 )
	p->pointer = &(s->l3_lknee2);
      else if ( strcmp( p->name, "knee_ff_peak" ) == 0 )
	p->pointer = &(s->knee_ff_peak);
      else if ( strcmp( p->name, "knee_ff_time" ) == 0 )
	p->pointer = &(s->knee_ff_time);
      else if ( strcmp( p->name, "knee_ff" ) == 0 )
	p->pointer = &(s->knee_ff);
      else if ( strcmp( p->name, "k_hip" ) == 0 )
	p->pointer = &(s->k_hip);
      else if ( strcmp( p->name, "b_hip" ) == 0 )
	p->pointer = &(s->b_hip);
      else if ( strcmp( p->name, "k_knee" ) == 0 )
	p->pointer = &(s->k_knee);
      else if ( strcmp( p->name, "b_knee" ) == 0 )
	p->pointer = &(s->b_knee);
      else if ( strcmp( p->name, "k_pitch" ) == 0 )
	p->pointer = &(s->k_pitch);
      else if ( strcmp( p->name, "b_pitch" ) == 0 )
	p->pointer = &(s->b_pitch);
      else if ( strcmp( p->name, "rand_scale" ) == 0 )
	p->pointer = &(s->rand_scale);
      /*
      else if ( strcmp( p->name, "" ) == 0 )
	p->pointer = &(s->);
      */

      else 
	{
	  fprintf( stderr, "Don't recognize %s in process_parameters()\n", p->name );
	  exit( -1 );
	}
      *(p->pointer) = p->value;
      if ( verbose )
	printf( "%d: %s %lg", total, p->name, *(p->pointer) );
      if ( p->optimize )
	{
	  total++;
	  printf( " OPTIMIZE!" );
	}
      printf( "\n" );
      p = p->next;
    }

  if ( verbose )
    printf( "Optimizing %d parameters.\n", total );
  return total;
}

/************************************************************************/

void
parameters_to_vector( OPT_PARAM *p, float *v )
{
  int indx = 0;

  printf( "Parameters_to_vector:\n" );
  for( ; ; )
    {
      if ( p == NULL )
	break;
      if ( p->optimize )
	{
	  v[indx++] = *(p->pointer);
	  printf( "%d: %g\n", indx-1, v[indx-1] );
	}
      p = p->next;
    }
}

/************************************************************************/

void
vector_to_sim( float *v, int n_parameters, OPT_PARAM *p )
{
  int indx;

  for( indx = 0; indx < n_parameters; indx++ )
    {
      /* Find next parameter to optimize */
      for( ; ; )
	{
	  if ( p == NULL )
	    {
	      fprintf( stderr, "Failure in vector_to_parameters: %d\n",
		       indx );
	      exit( - 1 );
	    }
	  if ( p->optimize )
	    break;
	  p = p->next;
	}
      *(p->pointer) = v[indx];
      /*
      printf( "%d: %g\n", indx, v[indx] );
      */
      p = p->next;
    }
}

/******************************************************************/

write_param_file( char *filename, OPT_PARAM *p )
{
  FILE *stream;

  stream = fopen( filename, "w" );
  if ( stream == NULL )
    {
      fprintf( stderr, "Can't open %s for write.\n", filename );
      exit( -1 );
    }

  for( ; ; )
    {
      if ( p == NULL )
	break;
      fprintf( stream, "%s %18.12f", p->name, *(p->pointer) );
      if ( p->optimize )
	fprintf( stream, " opt" );
      fprintf( stream, " end\n" );
      p = p->next;
    }

  fclose( stream );
}

/*****************************************************************************/

