/*
handle sdfast flag correctly (only b1gstate sets it).
*/
/************************************************************************/
/*
  dynamics.c: This is where the numerical integration and SDFAST stuff is done.
*/
/************************************************************************/

#include <stdio.h>
#include <math.h>

#include "main.h"
#include "sdfast/b1g.h"

/************************************************************************/
SIM sim;
/************************************************************************/

/* SDFAST constants and variables */
#define NQ 7
#define NU 7
#define NSTATE (NQ+NU)
#define CTOL	1e-5	/* constraint tolerance */
#define GND -1
const double baumgarte=100.0; /* could be 10.0 - 1000.0 */

/* Here we define the state */
double xt[NSTATE];
/* Here we define the derivative of the state */
static double dxt[NSTATE];
/* Leg state machines */
const double GROUND_LEVEL = 0.0;
/************************************************************************/
/************************************************************************/

static void forward_kinematics( SIM *s )
{

  b1gpos(BODY_PELVIS, s->head_offset, s->head);    
  b1gpos(BODY_PELVIS, s->hip_offset, s->hip);    
  b1gpos(BODY_L_SHIN, s->knee_offset, &(s->knee[LEFT][0]));    
  b1gpos(BODY_R_SHIN, s->knee_offset, &(s->knee[RIGHT][0]));    
  b1gpos(BODY_L_SHIN, s->foot_offset, &(s->foot[LEFT][0]));    
  b1gpos(BODY_R_SHIN, s->foot_offset, &(s->foot[RIGHT][0]));    

  b1gvel(BODY_PELVIS, s->head_offset, s->headd);    
  b1gvel(BODY_PELVIS, s->hip_offset, s->hipd);    
  b1gvel(BODY_L_SHIN, s->knee_offset, &(s->kneed[LEFT][0]));    
  b1gvel(BODY_R_SHIN, s->knee_offset, &(s->kneed[RIGHT][0]));    
  b1gvel(BODY_L_SHIN, s->foot_offset, &(s->footd[LEFT][0]));    
  b1gvel(BODY_R_SHIN, s->foot_offset, &(s->footd[RIGHT][0]));    

  b1gpos(BODY_L_SHIN, s->foot_offset, s->footpos[LEFT]);    
  b1gpos(BODY_R_SHIN, s->foot_offset, s->footpos[RIGHT]);    

  b1gvel(BODY_L_SHIN, s->foot_offset, s->footvel[LEFT]);    
  b1gvel(BODY_R_SHIN, s->foot_offset, s->footvel[RIGHT]);    

  s->pitch = s->state_sdfast[Q_PITCH];
  s->hip_angle[LEFT] = s->state_sdfast[Q_L_HIP];
  s->hip_angle[RIGHT] = s->state_sdfast[Q_R_HIP];
  s->knee_angle[LEFT] = s->state_sdfast[Q_L_KNEE];
  s->knee_angle[RIGHT] = s->state_sdfast[Q_R_KNEE];

  s->pitchd = s->state_sdfast[QD_PITCH];
  s->hip_angled[LEFT] = s->state_sdfast[QD_L_HIP];
  s->hip_angled[RIGHT] = s->state_sdfast[QD_R_HIP];
  s->knee_angled[LEFT] = s->state_sdfast[QD_L_KNEE];
  s->knee_angled[RIGHT] = s->state_sdfast[QD_R_KNEE];
}

/************************************************************************/
/* Initialize the state vector */

void init_dynamics_state( SIM *s, double *state )
{
  int i, j;

  for( i = 0; i < NSTATE; i++ )
    {
      xt[i] = state[i];
      dxt[i] = 0;
    }

  for( i = 0, j = NQ; i < NU; i++, j++ )
    { 
      dxt[i] = xt[j];
    }

  b1gstate( 0.0, xt, dxt );

  forward_kinematics( s );
}

/************************************************************************/

b1gdump()
{
  double value;
  double vector[3];
  double inertia[3][3];

  b1ggetmass( BODY_PELVIS, &value );
  printf( "Pelvis mass: %g\n", value );
  b1ggetmass( BODY_L_THIGH, &value );
  printf( "L Thigh mass: %g\n", value );
  b1ggetmass( BODY_R_THIGH, &value );
  printf( "R Thigh mass: %g\n", value );
  b1ggetmass( BODY_L_SHIN, &value );
  printf( "L Calf mass: %g\n", value );
  b1ggetmass( BODY_R_SHIN, &value );
  printf( "R Calf mass: %g\n", value );

  b1ggetiner( BODY_PELVIS, inertia );
  printf( "Pelvis I: %g\n", inertia[YY][YY] );
  b1ggetiner( BODY_L_THIGH, inertia );
  printf( "L Thigh I: %g\n", inertia[YY][YY] );
  b1ggetiner( BODY_R_THIGH, inertia );
  printf( "R Thigh I: %g\n", inertia[YY][YY] );
  b1ggetiner( BODY_L_SHIN, inertia );
  printf( "L Shin I: %g\n", inertia[YY][YY] );
  b1ggetiner( BODY_R_SHIN, inertia );
  printf( "R Shin I: %g\n", inertia[YY][YY] );

  b1ggetbtj( BODY_L_THIGH, vector );
  printf( "L_THIGH BTJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetbtj( BODY_R_THIGH, vector );
  printf( "R_THIGH BTJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetbtj( BODY_L_SHIN, vector );
  printf( "L_Calf BTJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetbtj( BODY_R_SHIN, vector );
  printf( "R_Calf BTJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );

  b1ggetitj( BODY_L_THIGH, vector );
  printf( "L_THIGH ITJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetitj( BODY_R_THIGH, vector );
  printf( "R_THIGH ITJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetitj( BODY_L_SHIN, vector );
  printf( "L_Calf ITJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
  b1ggetitj( BODY_R_SHIN, vector );
  printf( "R_Calf ITJ: %g %g %g\n", vector[0],
	  vector[1], vector[2] );
}

/************************************************************************/
/* Initialize sdfast  */

void init_dynamics( SIM *s )
{
  int i, j;
  double vector[3];
  double inertia[3][3];

  init_kin_dyn_parameters( s );

  s->status = OK;

  b1gdump();

  b1gmass( BODY_PELVIS, s->pelvis_mass );
  b1gmass( BODY_L_THIGH, s->thigh_mass );
  b1gmass( BODY_R_THIGH, s->thigh_mass );
  b1gmass( BODY_L_SHIN, s->calf_mass );
  b1gmass( BODY_R_SHIN, s->calf_mass );

  for ( i = 0; i < 3; i++ )
    {
      vector[i] = 0;
      for ( j = 0; j < 3; j++ )
	{
	  if ( i == j )
	    inertia[i][i] = 1.0;
	  else
	    inertia[i][j] = 0;
	}
    }
  inertia[YY][YY] = s->pelvis_I;
  b1giner( BODY_PELVIS, inertia );
  inertia[YY][YY] = s->thigh_I;
  b1giner( BODY_L_THIGH, inertia );
  b1giner( BODY_R_THIGH, inertia );
  inertia[YY][YY] = s->calf_I;
  b1giner( BODY_L_SHIN, inertia );
  b1giner( BODY_R_SHIN, inertia );

  vector[ZZ] = s->thigh_cm;
  b1gbtj( BODY_L_THIGH, vector );
  b1gbtj( BODY_R_THIGH, vector );
  vector[ZZ] = s->calf_cm;
  b1gbtj( BODY_L_SHIN, vector );
  b1gbtj( BODY_R_SHIN, vector );

  vector[ZZ] = -(s->thigh_length - s->thigh_cm);
  b1gitj( BODY_L_SHIN, vector );
  b1gitj( BODY_R_SHIN, vector );

  b1gdump();

  b1ginit(); /* initialize SDFAST model */

  b1gstab(2.0*baumgarte, baumgarte*baumgarte); 

  for( i = 0; i < NSTATE; i++ )
    s->state_sdfast[i] = 0;

  b1gprinterr(stderr);
}

/************************************************************************/
/* Need to handle constraints and xd, zd, and pitchd requests correctly */

void init_state_one_foot_on_ground( SIM *s )
{
  double min;

  s->status = OK;

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = s->knee_angled[RIGHT];

  init_dynamics_state( s, s->state_sdfast );

  /* Replace with bogus code
  min = s->foot[LEFT][ZZ];
  if ( min > s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  if ( min < -0.003 )
    s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min - 0.003;
  */
  /*
  printf( "%g %g\n", s->foot[LEFT][ZZ], s->foot[RIGHT][ZZ] );
  */
  min = s->foot[LEFT][ZZ];
  if ( min < s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  if ( min < -0.000 )
    s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min - 0.000;

  init_dynamics_state( s, s->state_sdfast );
}

/************************************************************************/
/* Need to handle constraints and xd, zd, and pitchd requests correctly */

void init_state_two_feet_on_ground( SIM *s )
{
  double min;
  double offset;

  s->status = OK;

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = s->knee_angled[RIGHT];

  init_dynamics_state( s, s->state_sdfast );

  /* Need to handle case where two feet together */

  if ( s->foot[RIGHT][XX] - s->foot[LEFT][XX] > 0 )
    offset = atan2( s->foot[RIGHT][ZZ] - s->foot[LEFT][ZZ],
		    s->foot[RIGHT][XX] - s->foot[LEFT][XX] );
  else
    offset = atan2( s->foot[LEFT][ZZ] - s->foot[RIGHT][ZZ],
		    s->foot[LEFT][XX] - s->foot[RIGHT][XX] );

  /*
  printf( "two feet on ground: %g %g %g %g: %g\n", 
	  s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[LEFT][XX], offset );
  */

  s->state_sdfast[Q_PITCH] += offset;

  init_dynamics_state( s, s->state_sdfast );

  /*
  printf( "%g %g\n", s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ] );
  */

  min = s->foot[LEFT][ZZ];
  if ( min > s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min;

  init_dynamics_state( s, s->state_sdfast );

  /*
  printf( "%g %g\n", s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ] );
  */
}

/************************************************************************/
/* This is what is called on each integration step */

void integrate_one_time_step( SIM *s, double *action  )
{ 
  int i;
  int flag; /* set to 1 when things are changed or first call */
  int err; 
    /* { OK, DERIVATIVE_DISCONTINUITY, SYSTEM_LOCKED, CONSTRAINTS_ERR } */
  double errest;

  if ( s->hip[ZZ] < 0.2 )
    {
      s->status = CRASHED;
      s->time += s->time_step;
      return;
    }

  flag=1;

  // clear outstanding error flags
  b1gclearerr();

  for( i = 0; i < 20; i++ )
    {
      b1gfmotion(&(sim.time),xt,dxt,sim.time_step/20.0,CTOL,&flag,&errest,&err);
      b1gprinterr(stderr);
    }

  for( i = 0; i < NSTATE; i++ )
    s->state_sdfast[i] = xt[i];

  forward_kinematics( s );

  /*
  printf( "Press return to continue.\n" );
  getchar();
  */
}

/****************************************************************/

// Utility function to compute the ground force function.
static 
void compute_ground_force(double contactpos[3],  // contact position wrt world
			  double contact_zero[3], // original conact point.
			  double contactvel[3],  // contact velocity wrt world
			  double bodypos[3],     // contact point wrt body
			  int body,            
			  double force[3])       // resultant force vector wrt world
			  
{
  int i;

  // force only exists if foot is below ground level
  if (contactpos[2] < 0.0) {
    double fvec[3];  // ground force vector in body coordintes

    // Compute the desired ground force vector in the world
    // coordinates.  For now, the lateral force is just viscous.
    // It should really be a constraint equation.
    // CGA: Actually keep track of original contact point. */
    if ( contact_zero[ZZ] > 0.0 )
      for( i = 0; i < 3; i++ )
	contact_zero[i] = contactpos[i];

    force[0] = (contact_zero[0] - contactpos[0])*sim.gnd_k_x
						 -contactvel[0] * sim.gnd_b;
    force[1] = 0.0;   // not relevant
    force[2] = (-contactpos[2] * sim.gnd_k) - (contactvel[2] * sim.gnd_b);

    // ground can't pull down
    if (force[2] < 0) 
      {
	force[0] = 0.0;
	force[1] = 0.0;
	force[2] = 0.0;
	/* Slipping */
	for( i = 0; i < 3; i++ )
	  contact_zero[i] = contactpos[i];
      }

    // transform the vector into body coordinates
    b1gtrans(BODY_WORLD, force, body, fvec);

    // apply to the model
    b1gpointf(body, bodypos, fvec);

  } else {
    force[0] = force[1] = force[2] = 0.0;
    for( i = 0; i < 3; i++ )
      contact_zero[i] = contactpos[i];
  }
}
/************************************************************************/
/* SDFAST stuff */
/************************************************************************/
/* This is where the control (hip_torque) is applied. May be called many
times per integration step at any time or state. */

void b1guforce(double t, double *q, double *u)
{
  // find the position and velocity of each foot point in the world frame
  b1gpos(BODY_L_SHIN, sim.foot_offset, sim.footpos[LEFT]);    
  b1gpos(BODY_R_SHIN, sim.foot_offset, sim.footpos[RIGHT]);    

  b1gvel(BODY_L_SHIN, sim.foot_offset, sim.footvel[LEFT]);    
  b1gvel(BODY_R_SHIN, sim.foot_offset, sim.footvel[RIGHT]);    

  // Apply ground forces to each foot as point forces.

  compute_ground_force(sim.footpos[LEFT], sim.foot_zero[LEFT], sim.footvel[LEFT],
		       sim.foot_offset, BODY_L_SHIN, sim.gndforce[LEFT]);
  compute_ground_force(sim.footpos[RIGHT], sim.foot_zero[RIGHT], sim.footvel[RIGHT],
		       sim.foot_offset, BODY_R_SHIN, sim.gndforce[RIGHT]);

  b1ghinget( 1, 0, sim.hip_torque[LEFT] );
  b1ghinget( 2, 0, sim.knee_torque[LEFT] );
  b1ghinget( 3, 0, sim.hip_torque[RIGHT] );
  b1ghinget( 4, 0, sim.knee_torque[RIGHT] );
}

/************************************************************************/
/************************************************************************/
/************************************************************************/
/************************************************************************/
