/************************************************************************/
/*
dynamics-impact.c: 
This is where the numerical integration and SDFAST stuff is done.
Use momentum equations to analytically compute effect of impacts.
*/
/************************************************************************/

#include <stdio.h>
#include <math.h>

#include "main.h"
#include "sdfast/b1g.h"

/************************************************************************/

/* Dynamic model modes */
#define CONSTRAINTS 4
#define INITIALIZE 3
#define DOUBLE_SUPPORT 2
#define SINGLE_SUPPORT 1
#define IN_AIR 0

#define REF_HIP 0
#define REF_LFOOT 1
#define REF_RFOOT 2

#define EPSILON1 0.0 /* Fudge factor for ground level */
#define EPSILON2 0.0 /* -0.0001 Fudge factor for ignoring SS -> SS */

/************************************************************************/

SIM sim; /* The simulation interface */

/************************************************************************/
/* Generic SDFAST model b1g and b1c */
/* SDFAST constants and variables */
#define NQ 7
#define NU 7
#define NSTATE (NQ+NU)
#define CTOL	1e-5	/* constraint tolerance */
#define GND -1
const double GROUND_LEVEL = 0.00;
const double GROUND_LEVEL_X_INCREMENT = 0.00;
/* Here we define the state */
const double baumgarte=100.0; /* could be 10.0 - 1000.0 */

int interactive = 0;
int impact1_print = 0;
int impact2_print = 0;

/************************************************************************/
/* Double support SDFAST model b1ssgg */

#define DS_NQ 5
#define DS_NU 5
#define DS_NSTATE (DS_NQ+DS_NU)
#define DS_CTOL	1e-5	/* constraint tolerance */
#define DS_NJNT 20

#define DS_ANKLE1 0
#define DS_KNEE1 1
#define DS_HIP1 2
#define DS_HIP2 3
#define DS_KNEE2 4
#define DS_ANKLE1D 5
#define DS_KNEE1D 6
#define DS_HIP1D 7
#define DS_HIP2D 8
#define DS_KNEE2D 9

#define DS_GND_ID -1
#define DS_SHIN1_ID 0
#define DS_THIGH1_ID 1
#define DS_PELVIS_ID 2
#define DS_THIGH2_ID 3
#define DS_SHIN2_ID 4
#define DS_LOOP_ID 5

static double ds_joint_forces[DS_NJNT][3];
static double ds_joint_torques[DS_NJNT][3];

/************************************************************************/
/* Single support SDFAST model b1ss */

#define SS_NQ 5
#define SS_NU 5
#define SS_NSTATE (SS_NQ+SS_NU)
#define SS_CTOL	1e-5	/* constraint tolerance */
#define SS_NJNT 20

#define SS_ANKLE1 0
#define SS_KNEE1 1
#define SS_HIP1 2
#define SS_HIP2 3
#define SS_KNEE2 4
#define SS_ANKLE1D 5
#define SS_KNEE1D 6
#define SS_HIP1D 7
#define SS_HIP2D 8
#define SS_KNEE2D 9

#define SS_GND_ID -1
#define SS_SHIN1_ID 0
#define SS_THIGH1_ID 1
#define SS_PELVIS_ID 2
#define SS_THIGH2_ID 3
#define SS_SHIN2_ID 4

static double ss_joint_forces[SS_NJNT][3];
static double ss_joint_torques[SS_NJNT][3];

/************************************************************************/

static void generic_forward_kinematics( SIM *s )
{

  b1gpos(BODY_PELVIS, s->head_offset, s->head);    
  b1gpos(BODY_PELVIS, s->hip_offset, s->hip);    
  b1gpos(BODY_L_SHIN, s->knee_offset, &(s->knee[LEFT][0]));    
  b1gpos(BODY_R_SHIN, s->knee_offset, &(s->knee[RIGHT][0]));    
  b1gpos(BODY_L_SHIN, s->foot_offset, &(s->foot[LEFT][0]));    
  b1gpos(BODY_R_SHIN, s->foot_offset, &(s->foot[RIGHT][0]));    

  b1gvel(BODY_PELVIS, s->head_offset, s->headd);    
  b1gvel(BODY_PELVIS, s->hip_offset, s->hipd);    
  b1gvel(BODY_L_SHIN, s->knee_offset, &(s->kneed[LEFT][0]));    
  b1gvel(BODY_R_SHIN, s->knee_offset, &(s->kneed[RIGHT][0]));    
  b1gvel(BODY_L_SHIN, s->foot_offset, &(s->footd[LEFT][0]));    
  b1gvel(BODY_R_SHIN, s->foot_offset, &(s->footd[RIGHT][0]));    

  b1gacc(BODY_PELVIS, s->hip_offset, s->hipdd);    

  s->pitch = s->state_sdfast[Q_PITCH];
  s->hip_angle[LEFT] = s->state_sdfast[Q_L_HIP];
  s->hip_angle[RIGHT] = s->state_sdfast[Q_R_HIP];
  s->knee_angle[LEFT] = s->state_sdfast[Q_L_KNEE];
  s->knee_angle[RIGHT] = s->state_sdfast[Q_R_KNEE];

  s->pitchd = s->state_sdfast[QD_PITCH];
  s->hip_angled[LEFT] = s->state_sdfast[QD_L_HIP];
  s->hip_angled[RIGHT] = s->state_sdfast[QD_R_HIP];
  s->knee_angled[LEFT] = s->state_sdfast[QD_L_KNEE];
  s->knee_angled[RIGHT] = s->state_sdfast[QD_R_KNEE];

  s->pitchdd = s->state_sdfast2[QD_PITCH];
  s->hip_angledd[LEFT] = s->state_sdfast2[QD_L_HIP];
  s->hip_angledd[RIGHT] = s->state_sdfast2[QD_R_HIP];
  s->knee_angledd[LEFT] = s->state_sdfast2[QD_L_KNEE];
  s->knee_angledd[RIGHT] = s->state_sdfast2[QD_R_KNEE];
}

/************************************************************************/

static void constraints_forward_kinematics( SIM *s )
{

  b1cpos(BODY_PELVIS, s->head_offset, s->head);    
  b1cpos(BODY_PELVIS, s->hip_offset, s->hip);    
  b1cpos(BODY_L_SHIN, s->knee_offset, &(s->knee[LEFT][0]));    
  b1cpos(BODY_R_SHIN, s->knee_offset, &(s->knee[RIGHT][0]));    
  b1cpos(BODY_L_SHIN, s->foot_offset, &(s->foot[LEFT][0]));    
  b1cpos(BODY_R_SHIN, s->foot_offset, &(s->foot[RIGHT][0]));    

  b1cvel(BODY_PELVIS, s->head_offset, s->headd);    
  b1cvel(BODY_PELVIS, s->hip_offset, s->hipd);    
  b1cvel(BODY_L_SHIN, s->knee_offset, &(s->kneed[LEFT][0]));    
  b1cvel(BODY_R_SHIN, s->knee_offset, &(s->kneed[RIGHT][0]));    
  b1cvel(BODY_L_SHIN, s->foot_offset, &(s->footd[LEFT][0]));    
  b1cvel(BODY_R_SHIN, s->foot_offset, &(s->footd[RIGHT][0]));    

  b1cacc(BODY_PELVIS, s->hip_offset, s->hipdd);    

  s->pitch = s->state_sdfast[Q_PITCH];
  s->hip_angle[LEFT] = s->state_sdfast[Q_L_HIP];
  s->hip_angle[RIGHT] = s->state_sdfast[Q_R_HIP];
  s->knee_angle[LEFT] = s->state_sdfast[Q_L_KNEE];
  s->knee_angle[RIGHT] = s->state_sdfast[Q_R_KNEE];

  s->pitchd = s->state_sdfast[QD_PITCH];
  s->hip_angled[LEFT] = s->state_sdfast[QD_L_HIP];
  s->hip_angled[RIGHT] = s->state_sdfast[QD_R_HIP];
  s->knee_angled[LEFT] = s->state_sdfast[QD_L_KNEE];
  s->knee_angled[RIGHT] = s->state_sdfast[QD_R_KNEE];

  s->pitchdd = s->state_sdfast2[QD_PITCH];
  s->hip_angledd[LEFT] = s->state_sdfast2[QD_L_HIP];
  s->hip_angledd[RIGHT] = s->state_sdfast2[QD_R_HIP];
  s->knee_angledd[LEFT] = s->state_sdfast2[QD_L_KNEE];
  s->knee_angledd[RIGHT] = s->state_sdfast2[QD_R_KNEE];
}

/************************************************************************/

int other_side( int value )
{
  if ( value == LEFT )
    return RIGHT;
  else
    return LEFT;
}

/************************************************************************/

fix_positions( SIM *s, double v1[3], double v2[3] )
{
  v2[0] = v1[0] + s->x_offset;
  v2[1] = v1[2];
  v2[2] = v1[1];
}

/************************************************************************/

fix_yz( SIM *s, double v1[3], double v2[3] )
{
  v2[0] = v1[0];
  v2[1] = v1[2];
  v2[2] = v1[1];
}

/************************************************************************/
/* Double support: convert sdfast to sim coordinates. */

static void ds_forward_kinematics( SIM *s )
{
  double foot1[3];
  double knee1[3];
  double hip[3];
  double head[3];
  double knee2[3];
  double foot2[3];
  double offset[3];
  double foot1d[3];
  double knee1d[3];
  double hipd[3];
  double hipdd[3];
  double headd[3];
  double knee2d[3];
  double foot2d[3];
  int i;

  foot1[XX] = 0;
  foot1[YY] = 0;
  foot1[ZZ] = 0;
  foot1d[XX] = 0;
  foot1d[YY] = 0;
  foot1d[ZZ] = 0;
  offset[XX] = 0;
  offset[YY] = s->knee_offset[ZZ];
  offset[ZZ] = 0;
  b1dspos( DS_SHIN1_ID, offset, knee1 );
  b1dsvel( DS_SHIN1_ID, offset, knee1d );
  offset[YY] = s->hip_offset[ZZ];
  b1dspos( DS_PELVIS_ID, offset, hip );
  b1dsvel( DS_PELVIS_ID, offset, hipd );
  b1dsacc( DS_PELVIS_ID, offset, hipdd );
  offset[YY] = s->head_offset[ZZ];
  b1dspos( DS_PELVIS_ID, offset, head );
  b1dsvel( DS_PELVIS_ID, offset, headd );
  offset[YY] = s->knee_offset[ZZ];
  b1dspos( DS_SHIN2_ID, offset, knee2 );
  b1dsvel( DS_SHIN2_ID, offset, knee2d );
  offset[YY] = s->foot_offset[ZZ];
  b1dspos( DS_SHIN2_ID, offset, foot2 );
  b1dsvel( DS_SHIN2_ID, offset, foot2d ); 

  /* Hack to keep foot2 above ground. */
  if ( foot2[YY] < 0 )
    foot2[YY] = 0;

  fix_positions( s, head, s->head );
  fix_yz( s, headd, s->headd );
  fix_positions( s, hip, s->hip );
  fix_yz( s, hipd, s->hipd );
  fix_yz( s, hipdd, s->hipdd );
  fix_positions( s, knee1, s->knee[LEFT] );
  fix_yz( s, knee1d, s->kneed[LEFT] );
  fix_positions( s, foot1, s->foot[LEFT] );
  fix_yz( s, foot1d, s->footd[LEFT] );
  fix_positions( s, knee2, s->knee[RIGHT] );
  fix_yz( s, knee2d, s->kneed[RIGHT] );
  fix_positions( s, foot2, s->foot[RIGHT] );
  fix_yz( s, foot2d, s->footd[RIGHT] );

  s->pitch = -(s->ds_state[DS_ANKLE1] + s->ds_state[DS_KNEE1] + s->ds_state[DS_HIP1]);
  s->knee_angle[LEFT] = s->ds_state[DS_KNEE1];
  s->hip_angle[LEFT] = s->ds_state[DS_HIP1];
  s->hip_angle[RIGHT] = -s->ds_state[DS_HIP2];
  s->knee_angle[RIGHT] = -s->ds_state[DS_KNEE2];

  s->pitchd = -(s->ds_state[DS_ANKLE1D] + s->ds_state[DS_KNEE1D] + s->ds_state[DS_HIP1D]);
  s->knee_angled[LEFT] = s->ds_state[DS_KNEE1D];
  s->hip_angled[LEFT] = s->ds_state[DS_HIP1D];
  s->hip_angled[RIGHT] = -s->ds_state[DS_HIP2D];
  s->knee_angled[RIGHT] = -s->ds_state[DS_KNEE2D];

  s->pitchdd = -(s->state_sdfast2[DS_ANKLE1D] + s->state_sdfast2[DS_KNEE1D] + s->state_sdfast2[DS_HIP1D]);
  s->knee_angledd[LEFT] = s->state_sdfast2[DS_KNEE1D];
  s->hip_angledd[LEFT] = s->state_sdfast2[DS_HIP1D];
  s->hip_angledd[RIGHT] = -s->state_sdfast2[DS_HIP2D];
  s->knee_angledd[RIGHT] = -s->state_sdfast2[DS_KNEE2D];
}

/************************************************************************/
/* Double support: convert sdfast to sim coordinates. */

static void ss_forward_kinematics( SIM *s )
{
  double foot1[3];
  double knee1[3];
  double hip[3];
  double head[3];
  double knee2[3];
  double foot2[3];
  double offset[3];
  double foot1d[3];
  double knee1d[3];
  double hipd[3];
  double hipdd[3];
  double headd[3];
  double knee2d[3];
  double foot2d[3];
  int i;

  foot1[XX] = 0;
  foot1[YY] = 0;
  foot1[ZZ] = 0;
  foot1d[XX] = 0;
  foot1d[YY] = 0;
  foot1d[ZZ] = 0;
  offset[XX] = 0;
  offset[YY] = s->knee_offset[ZZ];
  offset[ZZ] = 0;
  b1sspos( SS_SHIN1_ID, offset, knee1 );
  b1ssvel( SS_SHIN1_ID, offset, knee1d );
  offset[YY] = s->hip_offset[ZZ];
  b1sspos( SS_PELVIS_ID, offset, hip );
  b1ssvel( SS_PELVIS_ID, offset, hipd );
  b1ssacc( SS_PELVIS_ID, offset, hipdd );
  offset[YY] = s->head_offset[ZZ];
  b1sspos( SS_PELVIS_ID, offset, head );
  b1ssvel( SS_PELVIS_ID, offset, headd );
  offset[YY] = s->knee_offset[ZZ];
  b1sspos( SS_SHIN2_ID, offset, knee2 );
  b1ssvel( SS_SHIN2_ID, offset, knee2d );
  offset[YY] = s->foot_offset[ZZ];
  b1sspos( SS_SHIN2_ID, offset, foot2 );
  b1ssvel( SS_SHIN2_ID, offset, foot2d ); 

  fix_positions( s, head, s->head );
  fix_yz( s, headd, s->headd );
  fix_positions( s, hip, s->hip );
  fix_yz( s, hipd, s->hipd );
  fix_yz( s, hipdd, s->hipdd );

  if ( s->ss_foot_down == LEFT )
    {
      fix_positions( s, knee1, s->knee[LEFT] );
      fix_yz( s, knee1d, s->kneed[LEFT] );
      fix_positions( s, foot1, s->foot[LEFT] );
      fix_yz( s, foot1d, s->footd[LEFT] );
      fix_positions( s, knee2, s->knee[RIGHT] );
      fix_yz( s, knee2d, s->kneed[RIGHT] );
      fix_positions( s, foot2, s->foot[RIGHT] );
      fix_yz( s, foot2d, s->footd[RIGHT] );

      s->pitch = -(s->ss_state[SS_ANKLE1] + s->ss_state[SS_KNEE1] + s->ss_state[SS_HIP1]);
      s->knee_angle[LEFT] = s->ss_state[SS_KNEE1];
      s->hip_angle[LEFT] = s->ss_state[SS_HIP1];
      s->hip_angle[RIGHT] = -s->ss_state[SS_HIP2];
      s->knee_angle[RIGHT] = -s->ss_state[SS_KNEE2];
      s->pitchd = -(s->ss_state[SS_ANKLE1D] + s->ss_state[SS_KNEE1D] + s->ss_state[SS_HIP1D]);
      s->knee_angled[LEFT] = s->ss_state[SS_KNEE1D];
      s->hip_angled[LEFT] = s->ss_state[SS_HIP1D];
      s->hip_angled[RIGHT] = -s->ss_state[SS_HIP2D];
      s->knee_angled[RIGHT] = -s->ss_state[SS_KNEE2D];
      s->pitchdd = -(s->state_sdfast2[SS_ANKLE1D] + s->state_sdfast2[SS_KNEE1D] + s->state_sdfast2[SS_HIP1D]);
      s->knee_angledd[LEFT] = s->state_sdfast2[SS_KNEE1D];
      s->hip_angledd[LEFT] = s->state_sdfast2[SS_HIP1D];
      s->hip_angledd[RIGHT] = -s->state_sdfast2[SS_HIP2D];
      s->knee_angledd[RIGHT] = -s->state_sdfast2[SS_KNEE2D];
    }
  else
    {
      fix_positions( s, knee1, s->knee[RIGHT] );
      fix_yz( s, knee1d, s->kneed[RIGHT] );
      fix_positions( s, foot1, s->foot[RIGHT] );
      fix_yz( s, foot1d, s->footd[RIGHT] );
      fix_positions( s, knee2, s->knee[LEFT] );
      fix_yz( s, knee2d, s->kneed[LEFT] );
      fix_positions( s, foot2, s->foot[LEFT] );
      fix_yz( s, foot2d, s->footd[LEFT] );

      s->pitch = -(s->ss_state[SS_ANKLE1] + s->ss_state[SS_KNEE1] + s->ss_state[SS_HIP1]);
      s->knee_angle[RIGHT] = s->ss_state[SS_KNEE1];
      s->hip_angle[RIGHT] = s->ss_state[SS_HIP1];
      s->hip_angle[LEFT] = -s->ss_state[SS_HIP2];
      s->knee_angle[LEFT] = -s->ss_state[SS_KNEE2];
      s->pitchd = -(s->ss_state[SS_ANKLE1D] + s->ss_state[SS_KNEE1D] + s->ss_state[SS_HIP1D]);
      s->knee_angled[RIGHT] = s->ss_state[SS_KNEE1D];
      s->hip_angled[RIGHT] = s->ss_state[SS_HIP1D];
      s->hip_angled[LEFT] = -s->ss_state[SS_HIP2D];
      s->knee_angled[LEFT] = -s->ss_state[SS_KNEE2D];
      s->pitchdd = -(s->state_sdfast2[SS_ANKLE1D] + s->state_sdfast2[SS_KNEE1D] + s->state_sdfast2[SS_HIP1D]);
      s->knee_angledd[RIGHT] = s->state_sdfast2[SS_KNEE1D];
      s->hip_angledd[RIGHT] = s->state_sdfast2[SS_HIP1D];
      s->hip_angledd[LEFT] = -s->state_sdfast2[SS_HIP2D];
      s->knee_angledd[LEFT] = -s->state_sdfast2[SS_KNEE2D];
    }
}

/************************************************************************/
/* Initialize the state vector */

void constraints_init_dynamics_state( SIM *s, double *state )
{
  int i, j;

  for( i = 0; i < NSTATE; i++ )
    {
      s->state_sdfast[i] = state[i];
      s->state_sdfast2[i] = 0;
    }

  for( i = 0, j = NQ; i < NU; i++, j++ )
    { 
      s->state_sdfast2[i] = s->state_sdfast[j];
    }

  b1cstate( 0.0, s->state_sdfast, s->state_sdfast2 );

  s->sdfast_flag = 1;

  constraints_forward_kinematics( s );
}

/************************************************************************/
/* Initialize sdfast  */

void init_dynamics( SIM *s )
{
  int i, j;
  double vector[3];
  double inertia[3][3];

  init_kin_dyn_parameters( s );

  s->status = OK;
  s->sdfast_last_mode = INITIALIZE;
  s->sdfast_mode = INITIALIZE;

  for( i = 0; i < DS_NJNT; i++ )
    {
      for( j = 0; j < 3; j++ )
	{
	  ds_joint_forces[i][j] = 0;
	  ds_joint_torques[i][j] = 0;
	}
    }

  s->sdfast_flag = 1;

  b1gmass( BODY_PELVIS, s->pelvis_mass );
  b1gmass( BODY_L_THIGH, s->thigh_mass );
  b1gmass( BODY_R_THIGH, s->thigh_mass );
  b1gmass( BODY_L_SHIN, s->calf_mass );
  b1gmass( BODY_R_SHIN, s->calf_mass );

  for ( i = 0; i < 3; i++ )
    {
      vector[i] = 0;
      for ( j = 0; j < 3; j++ )
	{
	  if ( i == j )
	    inertia[i][i] = 1.0;
	  else
	    inertia[i][j] = 0;
	}
    }
  inertia[YY][YY] = s->pelvis_I;
  b1giner( BODY_PELVIS, inertia );
  inertia[YY][YY] = s->thigh_I;
  b1giner( BODY_L_THIGH, inertia );
  b1giner( BODY_R_THIGH, inertia );
  inertia[YY][YY] = s->calf_I;
  b1giner( BODY_L_SHIN, inertia );
  b1giner( BODY_R_SHIN, inertia );

  vector[ZZ] = s->thigh_cm;
  b1gbtj( BODY_L_THIGH, vector );
  b1gbtj( BODY_R_THIGH, vector );
  vector[ZZ] = s->calf_cm;
  b1gbtj( BODY_L_SHIN, vector );
  b1gbtj( BODY_R_SHIN, vector );

  vector[ZZ] = -(s->thigh_length - s->thigh_cm);
  b1gitj( BODY_L_SHIN, vector );
  b1gitj( BODY_R_SHIN, vector );

  b1ginit(); /* initialize SDFAST model */
  b1gstab(2.0*baumgarte, baumgarte*baumgarte); 
  for( i = 0; i < NSTATE; i++ )
    s->state_sdfast[i] = 0;
  b1gprinterr(stderr);

  b1cmass( BODY_PELVIS, s->pelvis_mass );
  b1cmass( BODY_L_THIGH, s->thigh_mass );
  b1cmass( BODY_R_THIGH, s->thigh_mass );
  b1cmass( BODY_L_SHIN, s->calf_mass );
  b1cmass( BODY_R_SHIN, s->calf_mass );

  for ( i = 0; i < 3; i++ )
    {
      vector[i] = 0;
      for ( j = 0; j < 3; j++ )
	{
	  if ( i == j )
	    inertia[i][i] = 1.0;
	  else
	    inertia[i][j] = 0;
	}
    }
  inertia[YY][YY] = s->pelvis_I;
  b1ciner( BODY_PELVIS, inertia );
  inertia[YY][YY] = s->thigh_I;
  b1ciner( BODY_L_THIGH, inertia );
  b1ciner( BODY_R_THIGH, inertia );
  inertia[YY][YY] = s->calf_I;
  b1ciner( BODY_L_SHIN, inertia );
  b1ciner( BODY_R_SHIN, inertia );

  vector[ZZ] = s->thigh_cm;
  b1cbtj( BODY_L_THIGH, vector );
  b1cbtj( BODY_R_THIGH, vector );
  vector[ZZ] = s->calf_cm;
  b1cbtj( BODY_L_SHIN, vector );
  b1cbtj( BODY_R_SHIN, vector );

  vector[ZZ] = -(s->thigh_length - s->thigh_cm);
  b1citj( BODY_L_SHIN, vector );
  b1citj( BODY_R_SHIN, vector );

  b1cinit(); /* initialize SDFAST model */
  b1cstab(2.0*baumgarte, baumgarte*baumgarte); 
  for( i = 0; i < NSTATE; i++ )
    s->state_sdfast[i] = 0;
  b1cprinterr(stderr);
}

/************************************************************************/
/* Need to handle constraints and xd, zd, and pitchd requests correctly */

void init_state_one_foot_on_ground( SIM *s )
{
  double min;

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = s->knee_angled[RIGHT];

  constraints_init_dynamics_state( s, s->state_sdfast );

  min = s->foot[LEFT][ZZ];
  if ( min > s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min;

  constraints_init_dynamics_state( s, s->state_sdfast );
}

/************************************************************************/

void ds_init_double_support_state( SIM *s, int ref_flag )
{
  double r_foot[3];
  double inbtojoint[3];
  double distance;
  double tt = 0.0;
  double tol = 1e-5;
  double angle_offset;
  int lock[DS_NU];
  int maxevals = 10;
  int fcnt;
  int err;
  int i, j;
  double vector[3];
  double inertia[3][3];
  double foot1[3];
  double knee1[3];
  double hip[3];
  double head[3];
  double knee2[3];
  double foot2[3];
  double offset[3];
  static int firsttime = 0;
  double trash[100];

  s->sdfast_flag = 1;
  s->sdfast_mode = DOUBLE_SUPPORT;

  for ( i = 0; i < DS_NSTATE; i++ )
    s->ds_state[i] = 0;
  /*
    1) Figure out distance between feet.
    2) Figure out x for left foot.
  */
  /*
    printf( "Garth angles: %g %g %g %g %g\n",
	  s->pitch, s->hip_angle[LEFT],
	  s->knee_angle[LEFT],
	  s->hip_angle[RIGHT],
	  s->knee_angle[RIGHT] );

  printf( "foot-ds1: %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ] );
  */

  s->ds_state[DS_ANKLE1] = -(s->pitch + s->hip_angle[LEFT] + s->knee_angle[LEFT]);
  s->ds_state[DS_KNEE1] = s->knee_angle[LEFT];
  s->ds_state[DS_HIP1] = s->hip_angle[LEFT];
  s->ds_state[DS_HIP2] = -s->hip_angle[RIGHT];
  s->ds_state[DS_KNEE2] = -s->knee_angle[RIGHT];

  s->ds_state[DS_ANKLE1D] = -(s->pitchd + s->hip_angled[LEFT] + s->knee_angled[LEFT]);
  s->ds_state[DS_KNEE1D] = s->knee_angled[LEFT];
  s->ds_state[DS_HIP1D] = s->hip_angled[LEFT];
  s->ds_state[DS_HIP2D] = -s->hip_angled[RIGHT];
  s->ds_state[DS_KNEE2D] = -s->knee_angled[RIGHT];

  r_foot[XX] = -s->calf_length*sin( s->ds_state[DS_ANKLE1] ) 
    + -s->thigh_length*sin( s->ds_state[DS_ANKLE1] + s->ds_state[DS_KNEE1] ) 
    + s->thigh_length*sin( s->ds_state[DS_ANKLE1] + s->ds_state[DS_KNEE1] + s->ds_state[DS_HIP1] 
		   + s->ds_state[DS_HIP2] ) 
    + s->calf_length*sin( s->ds_state[DS_ANKLE1] + s->ds_state[DS_KNEE1] + s->ds_state[DS_HIP1] 
		  + s->ds_state[DS_HIP2] + s->ds_state[DS_KNEE2] );
  r_foot[YY] = 0;
  r_foot[ZZ] = s->calf_length*cos( s->ds_state[DS_ANKLE1] ) 
    + s->thigh_length*cos( s->ds_state[DS_ANKLE1] + s->ds_state[DS_KNEE1] ) 
    - s->thigh_length*cos( s->ds_state[DS_ANKLE1] + s->ds_state[DS_KNEE1] + s->ds_state[DS_HIP1] 
		   + s->ds_state[DS_HIP2] ) 
    - s->calf_length*cos( s->ds_state[DS_ANKLE1] + s->ds_state[DS_KNEE1] + s->ds_state[DS_HIP1] 
		  + s->ds_state[DS_HIP2] + s->ds_state[DS_KNEE2] );
  distance = sqrt(r_foot[XX]*r_foot[XX] + r_foot[ZZ]*r_foot[ZZ]);

  /* Need a signed distance */
  if ( r_foot[XX] < 0 )
    distance = -distance;

  /*
  printf( "Distance: %g\n", distance );
  */

  inbtojoint[0] = distance;
  inbtojoint[1] = 0;
  inbtojoint[2] = 0;
  /*
    second_leg_down_transition();
  */
  b1dsmass( DS_PELVIS_ID, s->pelvis_mass );
  b1dsmass( DS_THIGH1_ID, s->thigh_mass );
  b1dsmass( DS_THIGH2_ID, s->thigh_mass );
  b1dsmass( DS_SHIN1_ID, s->calf_mass );
  b1dsmass( DS_SHIN2_ID, s->calf_mass );

  for ( i = 0; i < 3; i++ )
    {
      vector[i] = 0;
      for ( j = 0; j < 3; j++ )
	{
	  if ( i == j )
	    inertia[i][i] = 1.0;
	  else
	    inertia[i][j] = 0;
	}
    }
  inertia[ZZ][ZZ] = s->pelvis_I;
  b1dsiner( DS_PELVIS_ID, inertia );
  inertia[ZZ][ZZ] = s->thigh_I;
  b1dsiner( DS_THIGH1_ID, inertia );
  b1dsiner( DS_THIGH2_ID, inertia );
  inertia[ZZ][ZZ] = s->calf_I;
  b1dsiner( DS_SHIN1_ID, inertia );
  b1dsiner( DS_SHIN2_ID, inertia );

  if ( firsttime )
    {
      for( i = 0; i <=5; i++ )
	{
	  b1dsgetbtj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
    }

  vector[YY] = -(s->calf_length - s->calf_cm);
  b1dsbtj( DS_SHIN1_ID, vector );
  vector[YY] = -(s->thigh_length - s->thigh_cm);
  b1dsbtj( DS_THIGH1_ID, vector );
  vector[YY] = s->thigh_cm;
  b1dsbtj( DS_THIGH2_ID, vector );
  vector[YY] = s->calf_cm;
  b1dsbtj( DS_SHIN2_ID, vector );
  vector[YY] = -(s->calf_length - s->calf_cm);
  b1dsbtj( DS_LOOP_ID, vector );

  if ( firsttime )
    {
      for( i = 0; i <=5; i++ )
	{
	  b1dsgetbtj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
    }

  if ( firsttime )
    {
      for( i = 0; i <=5; i++ )
	{
	  b1dsgetitj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
    }

  vector[YY] = s->calf_cm;
  b1dsitj( DS_THIGH1_ID, vector );
  vector[YY] = s->thigh_cm;
  b1dsitj( DS_PELVIS_ID, vector );
  vector[YY] = -(s->thigh_length - s->thigh_cm);
  b1dsitj( DS_SHIN2_ID, vector );

  if ( firsttime )
    {
      for( i = 0; i <=5; i++ )
	{
	  b1dsgetitj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
      printf( "press return to continue.\n" );
      getchar();
    }

  firsttime = 0;

  b1dsitj( DS_LOOP_ID, inbtojoint );
  b1dsprinterr(stderr);
  b1dsinit(); /* initialize SDFAST model two legs on the ground */
  b1dsprinterr(stderr);

  /*
  printf( "r_foot: %g %g\n", r_foot[XX], r_foot[ZZ] );
  */

  if ( r_foot[XX] > 0 )
    angle_offset = atan2( r_foot[ZZ], r_foot[XX] );
  else
    angle_offset = atan2( -r_foot[ZZ], -r_foot[XX] );
  s->ds_state[DS_ANKLE1] -= angle_offset;

  for( i = 0; i < DS_NU; i++ )
    lock[i] = 0;

  /*
  printf( "Before assemble: %g %g %g %g %g\n", 
	  s->ds_state[0], s->ds_state[1], s->ds_state[2], s->ds_state[3], s->ds_state[4] );
  */

  b1dsassemble( tt, s->ds_state, lock, tol, maxevals, &fcnt, &err );
  b1dsprinterr(stderr);

  /*
  printf( "After assemble: %d %d: %g %g %g %g %g %g %g %g %g %g\n", err, fcnt,
	  s->ds_state[0], s->ds_state[1], s->ds_state[2], s->ds_state[3], s->ds_state[4],
	  s->ds_state[5], s->ds_state[6], s->ds_state[7], s->ds_state[8], s->ds_state[9] );
  */

  b1dsinitvel( tt, s->ds_state, lock, tol, maxevals, &fcnt, &err );
  b1dsprinterr(stderr);

  /*
  printf( "After initvel: %d %d: %g %g %g %g %g %g %g %g %g %g\n", err, fcnt,
	  s->ds_state[0], s->ds_state[1], s->ds_state[2], s->ds_state[3], s->ds_state[4],
	  s->ds_state[5], s->ds_state[6], s->ds_state[7], s->ds_state[8], s->ds_state[9] );
  */

  b1dsderiv( trash, s->state_sdfast2 );
  b1dsprinterr(stderr);

  s->sdfast_mode = DOUBLE_SUPPORT;

  foot1[XX] = 0;
  foot1[YY] = 0;
  foot1[ZZ] = 0;
  offset[XX] = 0;
  offset[YY] = s->knee_offset[ZZ];
  offset[ZZ] = 0;
  b1dspos( DS_SHIN1_ID, offset, knee1 );
  offset[YY] = s->hip_offset[ZZ];
  b1dspos( DS_PELVIS_ID, offset, hip );
  offset[YY] = s->head_offset[ZZ];
  b1dspos( DS_PELVIS_ID, offset, head );
  offset[YY] = s->knee_offset[ZZ];
  b1dspos( DS_SHIN2_ID, offset, knee2 );
  offset[YY] = s->foot_offset[ZZ];
  b1dspos( DS_SHIN2_ID, offset, foot2 );

  /*
  printf( "hip: %g %g; %g %g\n", s->hip[XX], s->hip[ZZ], hip[XX], s->x_offset );
  */
  /*
  s->x_offset = s->foot[LEFT][XX];
  */
  if ( ref_flag == REF_HIP )
    s->x_offset = s->hip[XX] - hip[XX];
  else if ( ref_flag = REF_LFOOT )
    s->x_offset = s->foot[LEFT][XX] - foot1[XX];
  else if ( ref_flag = REF_RFOOT )
    s->x_offset = s->foot[RIGHT][XX] - foot2[XX];

  /*
  printf( "hip: %g %g; %g %g\n", s->hip[XX], s->hip[ZZ], hip[XX], s->x_offset );
  printf( "foot1: %g %g\n", foot1[XX] + s->x_offset, foot1[YY] );
  printf( "knee1: %g %g\n", knee1[XX] + s->x_offset, knee1[YY] );
  printf( "hip: %g %g\n", hip[XX] + s->x_offset, hip[YY] );
  printf( "head: %g %g\n", head[XX] + s->x_offset, head[YY] );
  printf( "knee2: %g %g\n", knee2[XX] + s->x_offset, knee2[YY] );
  printf( "foot2: %g %g\n", foot2[XX] + s->x_offset, foot2[YY] );
  */

  ds_forward_kinematics( s );

  /*
  printf( "foot-ds2: %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ] );
  printf( "Press return to continue.\n" );
  getchar();}
  */
}

/************************************************************************/
/* Need to handle constraints and xd, zd, and pitchd requests correctly */

void constrained_init_double_support_state( SIM *s )
{
  double min;
  double offset;
  double distance;

  s->sdfast_flag = 1;
  s->sdfast_mode = CONSTRAINTS;

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = s->knee_angled[RIGHT];

  constraints_init_dynamics_state( s, s->state_sdfast );

  distance = sqrt( (s->foot[RIGHT][XX] - s->foot[LEFT][XX])*(s->foot[RIGHT][XX] - s->foot[LEFT][XX])
		   + (s->foot[RIGHT][ZZ] - s->foot[LEFT][ZZ])*(s->foot[RIGHT][ZZ] - s->foot[LEFT][ZZ]) );
  /*
  printf( "Distance: %g\n", distance );
  */

  /* Need to handle case where two feet together */

  if ( s->foot[RIGHT][XX] - s->foot[LEFT][XX] > 0 )
    offset = atan2( s->foot[RIGHT][ZZ] - s->foot[LEFT][ZZ],
		    s->foot[RIGHT][XX] - s->foot[LEFT][XX] );
  else
    offset = atan2( s->foot[LEFT][ZZ] - s->foot[RIGHT][ZZ],
		    s->foot[LEFT][XX] - s->foot[RIGHT][XX] );

  /*
  printf( "two feet on ground: %g %g %g %g: %g\n", 
	  s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[LEFT][XX], offset );
  */

  s->state_sdfast[Q_PITCH] += offset;

  constraints_init_dynamics_state( s, s->state_sdfast );

  /*
  printf( "%g %g\n", s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ] );
  */

  min = s->foot[LEFT][ZZ];
  if ( min > s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min;

  constraints_init_dynamics_state( s, s->state_sdfast );

  /*
  printf( "%g %g\n", s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ] );
  */

  distance = sqrt( (s->foot[RIGHT][XX] - s->foot[LEFT][XX])*(s->foot[RIGHT][XX] - s->foot[LEFT][XX])
		   + (s->foot[RIGHT][ZZ] - s->foot[LEFT][ZZ])*(s->foot[RIGHT][ZZ] - s->foot[LEFT][ZZ]) );
  /*
  printf( "Distance: %g\n", distance );
  printf( "lfoot: %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ] );
  printf( "lknee: %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ] );
  printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
  printf( "rknee: %g %g\n", s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
  printf( "rfoot: %g %g\n", s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );
  */
}

/************************************************************************/
/* Need to handle constraints and xd, zd, and pitchd requests correctly */

void init_state_two_feet_on_ground( SIM *s )
{

  s->status = OK;

  ds_init_double_support_state( s, REF_HIP );

  /*
  constrained_init_double_support_state( s );
  */
}

/************************************************************************/

transition_to_constraints( SIM *s )
{
  int i, j;

  printf( "transition_to_constraints: %g\n", s->time );

  /*
  printf( "state_sdfast: " );
  for( i = 0; i < 7; i++ )
    printf( "%g \n", s->state_sdfast[i] );
  printf( "\n" );

  printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
  printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	  s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
  printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );

  printf( "pitch: %g\n", s->pitch );
  printf( "hip angles: %g %g\n", s->hip_angle[LEFT], s->hip_angle[RIGHT] );
  printf( "knee angles: %g %g\n", s->knee_angle[LEFT], s->knee_angle[RIGHT] );
  */

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = s->knee_angled[RIGHT];

  s->sdfast_mode = CONSTRAINTS;

  constraints_init_dynamics_state( s, s->state_sdfast );

  /*
  for( i = 0; i < NSTATE; i++ )
    {
      s->state_sdfast[i] = s->state_sdfast[i];
      s->state_sdfast2[i] = 0;
    }

  for( i = 0, j = NQ; i < NU; i++, j++ )
    { 
      s->state_sdfast2[i] = s->state_sdfast[j];
    }

  b1cstate( 0.0, xt, s->state_sdfast2 );
  */
  b1cprinterr(stderr);

  /*
  printf( "state_sdfast: " );
  for( i = 0; i < 7; i++ )
    printf( "%g \n", s->state_sdfast[i] );
  printf( "\n" );

  printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
  printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	  s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
  printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );

  printf( "pitch: %g\n", s->pitch );
  printf( "hip angles: %g %g\n", s->hip_angle[LEFT], s->hip_angle[RIGHT] );
  printf( "knee angles: %g %g\n", s->knee_angle[LEFT], s->knee_angle[RIGHT] );
  */

  /*
  printf( "Press return to continue.\n" );
  getchar();
  */
}

/************************************************************************/

transition_to_air( SIM *s )
{
  int i, j;

  if ( interactive )
    printf( "transition_to_air: %g\n", s->time );

  /*
  printf( "state_sdfast: " );
  for( i = 0; i < 7; i++ )
    printf( "%g \n", s->state_sdfast[i] );
  printf( "\n" );

  printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
  printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	  s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
  printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );

  printf( "pitch: %g\n", s->pitch );
  printf( "hip angles: %g %g\n", s->hip_angle[LEFT], s->hip_angle[RIGHT] );
  printf( "knee angles: %g %g\n", s->knee_angle[LEFT], s->knee_angle[RIGHT] );
  */

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = s->knee_angled[RIGHT];

  s->sdfast_mode = IN_AIR;

  for( i = 0; i < NSTATE; i++ )
    {
      s->state_sdfast[i] = s->state_sdfast[i];
      s->state_sdfast2[i] = 0;
    }

  for( i = 0, j = NQ; i < NU; i++, j++ )
    { 
      s->state_sdfast2[i] = s->state_sdfast[j];
    }

  b1gstate( 0.0, s->state_sdfast, s->state_sdfast2 );

  s->sdfast_flag = 1;

  generic_forward_kinematics( s );

  /*
  for( i = 0; i < NSTATE; i++ )
    {
      s->state_sdfast[i] = s->state_sdfast[i];
      s->state_sdfast2[i] = 0;
    }

  for( i = 0, j = NQ; i < NU; i++, j++ )
    { 
      s->state_sdfast2[i] = s->state_sdfast[j];
    }

  b1cstate( 0.0, xt, s->state_sdfast2 );
  */
  b1gprinterr(stderr);

  /*
  printf( "state_sdfast: " );
  for( i = 0; i < 7; i++ )
    printf( "%g \n", s->state_sdfast[i] );
  printf( "\n" );

  printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
  printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	  s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
  printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );

  printf( "pitch: %g\n", s->pitch );
  printf( "hip angles: %g %g\n", s->hip_angle[LEFT], s->hip_angle[RIGHT] );
  printf( "knee angles: %g %g\n", s->knee_angle[LEFT], s->knee_angle[RIGHT] );
  */

  /*
  printf( "Press return to continue.\n" );
  getchar();
  */
}

/************************************************************************/
/* Begin single support, right stance; from ds or in air */

ss_right_stance( SIM *s )
{
  int i, j;
  double vector[3];
  double inertia[3][3];
  double foot1[3];
  double knee1[3];
  double hip[3];
  double head[3];
  double knee2[3];
  double foot2[3];
  double offset[3];
  static int firsttime = 0;

  if ( interactive )
    printf( "ss_right_stance: %g\n", s->time );

  /*
  printf( "ss_state: " );
  for( i = 0; i < SS_NQ; i++ )
    printf( "%g \n", s->ss_state[i] );
  printf( "\n" );

  printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
  printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	  s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
  printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );

  printf( "pitch: %g\n", s->pitch );
  printf( "hip angles: %g %g\n", s->hip_angle[LEFT], s->hip_angle[RIGHT] );
  printf( "knee angles: %g %g\n", s->knee_angle[LEFT], s->knee_angle[RIGHT] );
  */

  b1ssmass( SS_PELVIS_ID, s->pelvis_mass );
  b1ssmass( SS_THIGH1_ID, s->thigh_mass );
  b1ssmass( SS_THIGH2_ID, s->thigh_mass );
  b1ssmass( SS_SHIN1_ID, s->calf_mass );
  b1ssmass( SS_SHIN2_ID, s->calf_mass );

  for ( i = 0; i < 3; i++ )
    {
      vector[i] = 0;
      for ( j = 0; j < 3; j++ )
	{
	  if ( i == j )
	    inertia[i][i] = 1.0;
	  else
	    inertia[i][j] = 0;
	}
    }
  inertia[ZZ][ZZ] = s->pelvis_I;
  b1ssiner( SS_PELVIS_ID, inertia );
  inertia[ZZ][ZZ] = s->thigh_I;
  b1ssiner( SS_THIGH1_ID, inertia );
  b1ssiner( SS_THIGH2_ID, inertia );
  inertia[ZZ][ZZ] = s->calf_I;
  b1ssiner( SS_SHIN1_ID, inertia );
  b1ssiner( SS_SHIN2_ID, inertia );

  if ( firsttime )
    {
      for( i = 0; i < 5; i++ )
	{
	  b1ssgetbtj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
    }

  vector[YY] = -(s->calf_length - s->calf_cm);
  b1ssbtj( DS_SHIN1_ID, vector );
  vector[YY] = -(s->thigh_length - s->thigh_cm);
  b1ssbtj( DS_THIGH1_ID, vector );
  vector[YY] = s->thigh_cm;
  b1ssbtj( DS_THIGH2_ID, vector );
  vector[YY] = s->calf_cm;
  b1ssbtj( DS_SHIN2_ID, vector );

  if ( firsttime )
    {
      for( i = 0; i < 5; i++ )
	{
	  b1ssgetbtj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
    }

  if ( firsttime )
    {
      for( i = 0; i < 5; i++ )
	{
	  b1ssgetitj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
    }

  vector[YY] = s->calf_cm;
  b1ssitj( DS_THIGH1_ID, vector );
  vector[YY] = s->thigh_cm;
  b1ssitj( DS_PELVIS_ID, vector );
  vector[YY] = -(s->thigh_length - s->thigh_cm);
  b1ssitj( DS_SHIN2_ID, vector );

  if ( firsttime )
    {
      for( i = 0; i < 5; i++ )
	{
	  b1ssgetitj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
      printf( "press return to continue.\n" );
      getchar();
    }

  firsttime = 0;

  b1ssinit(); /* initialize SDFAST model */
  b1ssprinterr(stderr);

  s->sdfast_flag = 1;
  s->sdfast_mode = SINGLE_SUPPORT;
  s->ss_foot_down = RIGHT;

  for ( i = 0; i < SS_NSTATE; i++ )
    s->ss_state[i] = 0;

  s->ss_state[SS_ANKLE1] = -(s->pitch + s->hip_angle[RIGHT] + s->knee_angle[RIGHT]);
  s->ss_state[SS_KNEE1] = s->knee_angle[RIGHT];
  s->ss_state[SS_HIP1] = s->hip_angle[RIGHT];
  s->ss_state[SS_HIP2] = -s->hip_angle[LEFT];
  s->ss_state[SS_KNEE2] = -s->knee_angle[LEFT];

  s->state_sdfast2[SS_ANKLE1] = s->ss_state[SS_ANKLE1D] = 
    -(s->pitchd + s->hip_angled[RIGHT] + s->knee_angled[RIGHT]);
  s->state_sdfast2[SS_KNEE1] = s->ss_state[SS_KNEE1D] = s->knee_angled[RIGHT];
  s->state_sdfast2[SS_HIP1] = s->ss_state[SS_HIP1D] = s->hip_angled[RIGHT];
  s->state_sdfast2[SS_HIP2] = s->ss_state[SS_HIP2D] = -s->hip_angled[LEFT];
  s->state_sdfast2[SS_KNEE2] = s->ss_state[SS_KNEE2D] = -s->knee_angled[LEFT];

  b1ssstate( 0.0, s->ss_state, s->state_sdfast2 );
  b1ssprinterr(stderr);

  offset[XX] = 0;
  offset[YY] = s->hip_offset[ZZ];
  offset[ZZ] = 0;
  b1sspos( SS_PELVIS_ID, offset, hip );
  s->x_offset = s->hip[XX] - hip[XX];
  /*
  printf( "%g %g %g\n", s->hip[XX], hip[XX], s->x_offset );
  */

  ss_forward_kinematics( s );

  /*
  printf( "after ss_forward_k: %g %g\n", s->hip[XX], s->hip[ZZ] );

  printf( "ss_state: " );
  for( i = 0; i < SS_NQ; i++ )
    printf( "%g \n", s->ss_state[i] );
  printf( "\n" );

  printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
  printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	  s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
  printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );

  printf( "pitch: %g\n", s->pitch );
  printf( "hip angles: %g %g\n", s->hip_angle[LEFT], s->hip_angle[RIGHT] );
  printf( "knee angles: %g %g\n", s->knee_angle[LEFT], s->knee_angle[RIGHT] );
  */
}

/************************************************************************/
/* Begin single support, left stance; from ds or in air */

ss_left_stance( SIM *s )
{ 
  int i, j;
  double vector[3];
  double inertia[3][3];
  double foot1[3];
  double knee1[3];
  double hip[3];
  double head[3];
  double knee2[3];
  double foot2[3];
  double offset[3];
  static int firsttime = 0;

  if ( interactive )
    printf( "ss_left_stance: %g\n", s->time );

  /*
  printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );
  */
  /*
  printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
  printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	  s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
  printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );

  printf( "pitch: %g\n", s->pitch );
  printf( "hip angles: %g %g\n", s->hip_angle[LEFT], s->hip_angle[RIGHT] );
  printf( "knee angles: %g %g\n", s->knee_angle[LEFT], s->knee_angle[RIGHT] );

  printf( "hipd: %g %g\n", s->hipd[XX], s->hipd[ZZ] );
  printf( "kneesd: %g %g; %g %g\n", s->kneed[LEFT][XX], s->kneed[LEFT][ZZ],
	  s->kneed[RIGHT][XX], s->kneed[RIGHT][ZZ] );
  printf( "feetd: %g %g; %g %g\n", s->footd[LEFT][XX], s->footd[LEFT][ZZ],
	  s->footd[RIGHT][XX], s->footd[RIGHT][ZZ] );

  printf( "pitchd: %g\n", s->pitchd );
  printf( "hip angular velocities: %g %g\n", s->hip_angled[LEFT], s->hip_angled[RIGHT] );
  printf( "knee angular velocities: %g %g\n", s->knee_angled[LEFT], s->knee_angled[RIGHT] );
  */

  b1ssmass( SS_PELVIS_ID, s->pelvis_mass );
  b1ssmass( SS_THIGH1_ID, s->thigh_mass );
  b1ssmass( SS_THIGH2_ID, s->thigh_mass );
  b1ssmass( SS_SHIN1_ID, s->calf_mass );
  b1ssmass( SS_SHIN2_ID, s->calf_mass );

  for ( i = 0; i < 3; i++ )
    {
      vector[i] = 0;
      for ( j = 0; j < 3; j++ )
	{
	  if ( i == j )
	    inertia[i][i] = 1.0;
	  else
	    inertia[i][j] = 0;
	}
    }
  inertia[ZZ][ZZ] = s->pelvis_I;
  b1ssiner( SS_PELVIS_ID, inertia );
  inertia[ZZ][ZZ] = s->thigh_I;
  b1ssiner( SS_THIGH1_ID, inertia );
  b1ssiner( SS_THIGH2_ID, inertia );
  inertia[ZZ][ZZ] = s->calf_I;
  b1ssiner( SS_SHIN1_ID, inertia );
  b1ssiner( SS_SHIN2_ID, inertia );

  if ( firsttime )
    {
      for( i = 0; i < 5; i++ )
	{
	  b1ssgetbtj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
    }

  vector[YY] = -(s->calf_length - s->calf_cm);
  b1ssbtj( DS_SHIN1_ID, vector );
  vector[YY] = -(s->thigh_length - s->thigh_cm);
  b1ssbtj( DS_THIGH1_ID, vector );
  vector[YY] = s->thigh_cm;
  b1ssbtj( DS_THIGH2_ID, vector );
  vector[YY] = s->calf_cm;
  b1ssbtj( DS_SHIN2_ID, vector );

  if ( firsttime )
    {
      for( i = 0; i < 5; i++ )
	{
	  b1ssgetbtj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
    }

  if ( firsttime )
    {
      for( i = 0; i < 5; i++ )
	{
	  b1ssgetitj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
    }

  vector[YY] = s->calf_cm;
  b1ssitj( DS_THIGH1_ID, vector );
  vector[YY] = s->thigh_cm;
  b1ssitj( DS_PELVIS_ID, vector );
  vector[YY] = -(s->thigh_length - s->thigh_cm);
  b1ssitj( DS_SHIN2_ID, vector );

  if ( firsttime )
    {
      for( i = 0; i < 5; i++ )
	{
	  b1ssgetitj( i, vector );
	  printf( "%d: %g %g %g\n", i, vector[0],
		  vector[1], vector[2] );
	}
      printf( "press return to continue.\n" );
      getchar();
    }

  firsttime = 0;

  b1ssinit(); /* initialize SDFAST model */
  b1ssprinterr(stderr);

  s->sdfast_flag = 1;
  s->sdfast_mode = SINGLE_SUPPORT;
  s->ss_foot_down = LEFT;

  for ( i = 0; i < SS_NSTATE; i++ )
    s->ss_state[i] = 0;

  s->ss_state[SS_ANKLE1] = -(s->pitch + s->hip_angle[LEFT] + s->knee_angle[LEFT]);
  s->ss_state[SS_KNEE1] = s->knee_angle[LEFT];
  s->ss_state[SS_HIP1] = s->hip_angle[LEFT];
  s->ss_state[SS_HIP2] = -s->hip_angle[RIGHT];
  s->ss_state[SS_KNEE2] = -s->knee_angle[RIGHT];

  s->state_sdfast2[SS_ANKLE1] = s->ss_state[SS_ANKLE1D] = 
    -(s->pitchd + s->hip_angled[LEFT] + s->knee_angled[LEFT]);
  s->state_sdfast2[SS_KNEE1] = s->ss_state[SS_KNEE1D] = s->knee_angled[LEFT];
  s->state_sdfast2[SS_HIP1] = s->ss_state[SS_HIP1D] = s->hip_angled[LEFT];
  s->state_sdfast2[SS_HIP2] = s->ss_state[SS_HIP2D] = -s->hip_angled[RIGHT];
  s->state_sdfast2[SS_KNEE2] = s->ss_state[SS_KNEE2D] = -s->knee_angled[RIGHT];

  b1ssstate( 0.0, s->ss_state, s->state_sdfast2 );
  b1ssprinterr(stderr);

  /*
  offset[XX] = 0;
  offset[YY] = s->hip_offset[ZZ];
  offset[ZZ] = 0;
  b1sspos( SS_THIGH1_ID, offset, hip );
  */
  s->x_offset = s->foot[LEFT][XX];

  ss_forward_kinematics( s );

  /*
  printf( "xfeet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );
  */

  /*
  printf( "after ss_forward_k\n" );
  printf( "ss_state: " );
  for( i = 0; i < SS_NQ; i++ )
    printf( "%g \n", s->ss_state[i] );
  printf( "\n" );

  printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
  printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	  s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
  printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );

  printf( "pitch: %g\n", s->pitch );
  printf( "hip angles: %g %g\n", s->hip_angle[LEFT], s->hip_angle[RIGHT] );
  printf( "knee angles: %g %g\n", s->knee_angle[LEFT], s->knee_angle[RIGHT] );

  printf( "hipd: %g %g\n", s->hipd[XX], s->hipd[ZZ] );
  printf( "kneesd: %g %g; %g %g\n", s->kneed[LEFT][XX], s->kneed[LEFT][ZZ],
	  s->kneed[RIGHT][XX], s->kneed[RIGHT][ZZ] );
  printf( "feetd: %g %g; %g %g\n", s->footd[LEFT][XX], s->footd[LEFT][ZZ],
	  s->footd[RIGHT][XX], s->footd[RIGHT][ZZ] );

  printf( "pitchd: %g\n", s->pitchd );
  printf( "hip angular velocities: %g %g\n", s->hip_angled[LEFT], s->hip_angled[RIGHT] );
  printf( "knee angular velocities: %g %g\n", s->knee_angled[LEFT], s->knee_angled[RIGHT] );
  */
}

/*************************************************************************/
/* Foot 1 is on the ground, and foot 2 just touched down */

#define SS_IMPACT1_N 17

float **a = NULL;
float *b = NULL;

ss_impact1( SIM *s, double *result )
{
  float **matrix();
  float *vector();
  float d;
  int n,indx[SS_IMPACT1_N];
  int i, j;
  double m1, I1, l1, r1;
  double a1, j1xdi, j1ydi, a1di;
  double c1, s1;
  double m2, I2, l2, r2;
  double a2, j2xdi, j2ydi, a2di;
  double c12, s12;
  double m3, I3, l3, r3;
  double a3, j3xdi, j3ydi, a3di;
  double c123, s123;
  double m4, I4, l4, r4;
  double a4, j4xdi, j4ydi, a4di;
  double c1234, s1234;
  double m5, I5, l5, r5;
  double a5, j5xdi, j5ydi, a5di;
  double c12345, s12345;

  m1 = s->calf_mass;
  I1 = s->calf_I;
  l1 = s->calf_length;
  r1 = s->calf_length - s->calf_cm;

  m2 = s->thigh_mass;
  I2 = s->thigh_I;
  l2 = s->thigh_length;
  r2 = s->thigh_length - s->thigh_cm;

  m3 = s->pelvis_mass;
  I3 = s->pelvis_I;
  l3 = 0.0;
  r3 = 0.0;

  m4 = s->thigh_mass;
  I4 = s->thigh_I;
  l4 = s->thigh_length;
  r4 = s->thigh_cm;

  m5 = s->calf_mass;
  I5 = s->calf_I;
  l5 = s->calf_length;;
  r5 = s->calf_cm;

  a1 = s->ss_state[0];
  a2 = s->ss_state[1];
  a3 = s->ss_state[2];
  a4 = s->ss_state[3];
  a5 = s->ss_state[4];
  a1di = s->ss_state[5];
  a2di = s->ss_state[6];
  a3di = s->ss_state[7];
  a4di = s->ss_state[8];
  a5di = s->ss_state[9];

  if ( impact1_print )
    printf( "%10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f\n",
	    a1, a2, a3, a4, a5, a1di, a2di, a3di, a4di, a5di );

  c1 = cos( a1 );
  s1 = sin( a1 );
  c12 = cos( a1 + a2 );
  s12 = sin( a1 + a2 );
  c123 = cos( a1 + a2 + a3 );
  s123 = sin( a1 + a2 + a3 );
  c1234 = cos( a1 + a2 + a3 + a4 );
  s1234 = sin( a1 + a2 + a3 + a4 );
  c12345 = cos( a1 + a2 + a3 + a4 + a5 );
  s12345 = sin( a1 + a2 + a3 + a4 + a5 );

  n = SS_IMPACT1_N;
  if ( a == NULL )
    a = matrix( 1, n, 1, n );
  if ( b == NULL )
    b = vector( 1, n );

  for( i = 1; i <= n; i++ )
    {
      b[i] = 0;
      for( j = 1; j <= n; j++ )
        {
          a[i][j] = 0;
        }
    }

  a[1][1] = -(m1*r1*c1);
  a[1][6] = 1;
  a[1][8] = -1;

  a[2][1] = -(m1*r1*s1);
  a[2][7] = 1;
  a[2][9] = -1;

  a[3][1] = -(l1*m2*c1) - m2*r2*c12;
  a[3][2] = -(m2*r2*c12);
  a[3][8] = 1;
  a[3][10] = -1;

  a[4][1] = -(l1*m2*s1) - m2*r2*s12;
  a[4][2] = -(m2*r2*s12);
  a[4][9] = 1;
  a[4][11] = -1;

  a[5][1] = -(l1*m3*c1) - l2*m3*c12 - m3*r3*c123;
  a[5][2] = -(l2*m3*c12) - m3*r3*c123;
  a[5][3] = -(m3*r3*c123);
  a[5][10] = 1;
  a[5][12] = -1;

  a[6][1] = -(l1*m3*s1) - l2*m3*s12 - m3*r3*s123;
  a[6][2] = -(l2*m3*s12) - m3*r3*s123;
  a[6][3] = -(m3*r3*s123);
  a[6][11] = 1;
  a[6][13] = -1;

  a[7][1] = -(l1*m4*c1) - l2*m4*c12 + m4*r4*c1234;
  a[7][2] = -(l2*m4*c12) + m4*r4*c1234;
  a[7][3] = m4*r4*c1234;
  a[7][4] = m4*r4*c1234;
  a[7][12] = 1;
  a[7][14] = -1;

  a[8][1] = -(l1*m4*s1) - l2*m4*s12 + m4*r4*s1234;
  a[8][2] = -(l2*m4*s12) + m4*r4*s1234;
  a[8][3] = m4*r4*s1234;
  a[8][4] = m4*r4*s1234;
  a[8][13] = 1;
  a[8][15] = -1;

  a[9][1] = -(l1*m5*c1) - l2*m5*c12 + l4*m5*c1234 + 
    m5*r5*c12345;
  a[9][2] = -(l2*m5*c12) + l4*m5*c1234 + 
    m5*r5*c12345;
  a[9][3] = l4*m5*c1234 + m5*r5*c12345;
  a[9][4] = l4*m5*c1234 + m5*r5*c12345;
  a[9][5] = m5*r5*c12345;
  a[9][14] = 1;
  a[9][16] = -1;

  a[10][1] = -(l1*m5*s1) - l2*m5*s12 + l4*m5*s1234 + 
    m5*r5*s12345;
  a[10][2] = -(l2*m5*s12) + l4*m5*s1234 + 
    m5*r5*s12345;
  a[10][3] = l4*m5*s1234 + m5*r5*s12345;
  a[10][4] = l4*m5*s1234 + m5*r5*s12345;
  a[10][5] = m5*r5*s12345;
  a[10][15] = 1;
  a[10][17] = -1;

  a[11][1] = I1;
  a[11][6] = r1*c1;
  a[11][7] = r1*s1;
  a[11][8] = l1*c1 - r1*c1;
  a[11][9] = l1*s1 - r1*s1;

  a[12][1] = I2;
  a[12][2] = I2;
  a[12][8] = r2*c12;
  a[12][9] = r2*s12;
  a[12][10] = l2*c12 - r2*c12;
  a[12][11] = l2*s12 - r2*s12;

  a[13][1] = I3;
  a[13][2] = I3;
  a[13][3] = I3;
  a[13][10] = r3*c123;
  a[13][11] = r3*s123;
  a[13][12] = -r3*c123;
  a[13][13] = -r3*s123;

  a[14][1] = I4;
  a[14][2] = I4;
  a[14][3] = I4;
  a[14][4] = I4;
  a[14][12] = -(r4*c1234);
  a[14][13] = -(r4*s1234);
  a[14][14] = -(l4*c1234) + r4*c1234;
  a[14][15] = -(l4*s1234) + r4*s1234;

  a[15][1] = I5;
  a[15][2] = I5;
  a[15][3] = I5;
  a[15][4] = I5;
  a[15][5] = I5;
  a[15][14] = -(r5*c12345);
  a[15][15] = -(r5*s12345);
  a[15][16] = -(l5*c12345) + r5*c12345;
  a[15][17] = -(l5*s12345) + r5*s12345;

  a[16][1] = -(l1*c1) - l2*c12 + l4*c1234 + 
    l5*c12345;
  a[16][2] = -(l2*c12) + l4*c1234  + 
    l5*c12345;
  a[16][3] = l4*c1234 + l5*c12345;
  a[16][4] = l4*c1234 + l5*c12345;
  a[16][5] = l5*c12345;

  a[17][1] = -(l1*s1) - l2*s12 + l4*s1234 + 
    l5*s12345;
  a[17][2] = -(l2*s12) + l4*s1234 + 
    l5*s12345;
  a[17][3] = l4*s1234 + l5*s12345;
  a[17][4] = l4*s1234 + l5*s12345;
  a[17][5] = l5*s12345;

  b[1] = -(a1di*m1*r1*c1);
  b[2] = -(a1di*m1*r1*s1);
  b[3] = -(a1di*l1*m2*c1) - a1di*m2*r2*c12 - a2di*m2*r2*c12;
  b[4] = -(a1di*l1*m2*s1) - a1di*m2*r2*s12 - a2di*m2*r2*s12;
  b[5] = -(a1di*l1*m3*c1) - a1di*l2*m3*c12 - 
    a2di*l2*m3*c12 - a1di*m3*r3*c123 - 
    a2di*m3*r3*c123 - a3di*m3*r3*c123;
  b[6] = -(a1di*l1*m3*s1) - a1di*l2*m3*s12 - 
    a2di*l2*m3*s12 - a1di*m3*r3*s123 - 
    a2di*m3*r3*s123 - a3di*m3*r3*s123;
  b[7] = -(a1di*l1*m4*c1) - a1di*l2*m4*c12 - 
    a2di*l2*m4*c12 + a1di*m4*r4*c1234 + 
    a2di*m4*r4*c1234 + a3di*m4*r4*c1234 + 
    a4di*m4*r4*c1234;
  b[8] = -(a1di*l1*m4*s1) - a1di*l2*m4*s12 - 
    a2di*l2*m4*s12 + a1di*m4*r4*s1234 + 
    a2di*m4*r4*s1234 + a3di*m4*r4*s1234 + 
    a4di*m4*r4*s1234;
  b[9] =  -(a1di*l1*m5*c1) - a1di*l2*m5*c12 - 
    a2di*l2*m5*c12 + a1di*l4*m5*c1234 + 
    a2di*l4*m5*c1234 + a3di*l4*m5*c1234 + 
    a4di*l4*m5*c1234 + 
    a1di*m5*r5*c12345 + 
    a2di*m5*r5*c12345 + 
    a3di*m5*r5*c12345 + 
    a4di*m5*r5*c12345 + 
    a5di*m5*r5*c12345;
  b[10] =  -(a1di*l1*m5*s1) - a1di*l2*m5*s12 - 
    a2di*l2*m5*s12 + a1di*l4*m5*s1234 + 
    a2di*l4*m5*s1234 + a3di*l4*m5*s1234 + 
    a4di*l4*m5*s1234 + 
    a1di*m5*r5*s12345 + 
    a2di*m5*r5*s12345 + 
    a3di*m5*r5*s12345 + 
    a4di*m5*r5*s12345 + 
    a5di*m5*r5*s12345;
  b[11] = a1di*I1;
  b[12] = a1di*I2 + a2di*I2;
  b[13] = a1di*I3 + a2di*I3 + a3di*I3;
  b[14] = a1di*I4 + a2di*I4 + a3di*I4 + a4di*I4;
  b[15] = a1di*I5 + a2di*I5 + a3di*I5 + a4di*I5 + a5di*I5;
  b[16] = 0;
  b[17] = 0;

  ludcmp(a,n,indx,&d);
  lubksb(a,n,indx,b);

  for( i = 1; i <= n; i++ )
    {
      /*
      printf( "%d: %g\n", i, b[i] );
      */
      result[i-1] = b[i];
    }

  /* Need to free a and b, or make them global */
}

/************************************************************************/
/* This impact analysis assumes robot in air and one
   foot hits */

#define SS_IMPACT2_N 17

ss_impact2( SIM *s, double *result )
{
  float **matrix();
  float *vector();
  float d;
  int n,indx[SS_IMPACT2_N];
  int i, j;
  double j1x, j1y;
  double m1, I1, l1, r1;
  double a1, j1xdi, j1ydi, a1di;
  double c1, s1;
  double m2, I2, l2, r2;
  double a2, j2xdi, j2ydi, a2di;
  double c12, s12;
  double m3, I3, l3, r3;
  double a3, j3xdi, j3ydi, a3di;
  double c123, s123;
  double m4, I4, l4, r4;
  double a4, j4xdi, j4ydi, a4di;
  double c14, s14;
  double m5, I5, l5, r5;
  double a5, j5xdi, j5ydi, a5di;
  double c145, s145;
  double footx, footy, footxd, footyd;

  m1 = s->pelvis_mass;
  I1 = s->pelvis_I;
  l1 = 0;
  r1 = 0.0;

  m2 = s->thigh_mass;
  I2 = s->thigh_I;
  l2 = s->thigh_length;
  r2 = s->thigh_cm;

  m3 = s->calf_mass;
  I3 = s->calf_I;
  l3 = s->calf_length;
  r3 = s->calf_cm;

  m4 = s->thigh_mass;
  I4 = s->thigh_I;
  l4 = s->thigh_length;
  r4 = s->thigh_cm;

  m5 = s->calf_mass;
  I5 = s->calf_I;
  l5 = s->calf_length;
  r5 = s->calf_cm;

  a1 = -s->pitch;
  j1x = s->hip[XX];
  j1y = s->hip[ZZ];
  a1di = -s->pitchd;
  j1xdi = s->hipd[XX];
  j1ydi = s->hipd[ZZ];
  if ( s->ss_foot_down == LEFT )
    {
      a2 = -s->hip_angle[RIGHT];
      a3 = -s->knee_angle[RIGHT];
      a4 = -s->hip_angle[LEFT];
      a5 = -s->knee_angle[LEFT];
      a2di = -s->hip_angled[RIGHT];
      a3di = -s->knee_angled[RIGHT];
      a4di = -s->hip_angled[LEFT];
      a5di = -s->knee_angled[LEFT];
    }
  else
    {
      a2 = -s->hip_angle[LEFT];
      a3 = -s->knee_angle[LEFT];
      a4 = -s->hip_angle[RIGHT];
      a5 = -s->knee_angle[RIGHT];
      a2di = -s->hip_angled[LEFT];
      a3di = -s->knee_angled[LEFT];
      a4di = -s->hip_angled[RIGHT];
      a5di = -s->knee_angled[RIGHT];
    }

  if ( impact2_print )
    printf( "%10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f %10.5f\n",
	    j1x, j1y, a1, a2, a3, a4, a5, j1xdi, j1ydi, a1di, a2di, a3di, a4di, a5di );

  c1 = cos( a1 );
  s1 = sin( a1 );
  c12 = cos( a1 + a2 );
  s12 = sin( a1 + a2 );
  c123 = cos( a1 + a2 + a3 );
  s123 = sin( a1 + a2 + a3 );
  c14 = cos( a1 + a4 );
  s14 = sin( a1 + a4 );
  c145 = cos( a1 + a4 + a5 );
  s145 = sin( a1 + a4 + a5 );

  n = SS_IMPACT2_N;
  if ( a == NULL )
    a = matrix( 1, n, 1, n );
  if ( b == NULL )
    b = vector( 1, n );

  for( i = 1; i <= n; i++ )
    {
      b[i] = 0;
      for( j = 1; j <= n; j++ )
        {
          a[i][j] = 0;
        }
    }

  a[1][1] = m1;
  a[1][3] = m1*r1*c1;
  a[1][8] = -1;
  a[1][14] = -1;

  a[2][2] = m1;
  a[2][3] = m1*r1*s1;
  a[2][9] = -1;
  a[2][15] = -1;

  a[3][1] = m2;
  a[3][3] = l1*m2*c1 + m2*r2*c12;
  a[3][4] = m2*r2*c12;
  a[3][8] = 1;
  a[3][10] = -1;

  a[4][2] = m2;
  a[4][3] = l1*m2*s1 + m2*r2*s12;
  a[4][4] = m2*r2*s12;
  a[4][9] = 1;
  a[4][11] = -1;

  a[5][1] = m3;
  a[5][3] = l1*m3*c1 + l2*m3*c12 + m3*r3*c123;
  a[5][4] = l2*m3*c12 + m3*r3*c123;
  a[5][5] = m3*r3*c123;
  a[5][10] = 1;
  a[5][12] = -1;

  a[6][2] = m3;
  a[6][3] = l1*m3*s1 + l2*m3*s12 + m3*r3*s123;
  a[6][4] = l2*m3*s12 + m3*r3*s123;
  a[6][5] = m3*r3*s123;
  a[6][11] = 1;
  a[6][13] = -1;

  a[7][1] = m4;
  a[7][3] = m4*r4*c14;
  a[7][6] = m4*r4*c14;
  a[7][14] = 1;
  a[7][16] = -1;

  a[8][2] = m4;
  a[8][3] = m4*r4*s14;
  a[8][6] = m4*r4*s14;
  a[8][15] = 1;
  a[8][17] = -1;

  a[9][1] = m5;
  a[9][3] = l4*m5*c14 + m5*r5*c145;
  a[9][6] = l4*m5*c14 + m5*r5*c145;
  a[9][7] = m5*r5*c145;
  a[9][16] = 1;

  a[10][2] = m5;
  a[10][3] = l4*m5*s14 + m5*r5*s145;
  a[10][6] = l4*m5*s14 + m5*r5*s145;
  a[10][7] = m5*r5*s145;
  a[10][17] = 1;

  a[11][3] = I1;
  a[11][8] = -(l1*c1) + r1*c1;
  a[11][9] = -(l1*s1) + r1*s1;
  a[11][14] = r1*c1;
  a[11][15] = r1*s1;

  a[12][3] = I2;
  a[12][4] = I2;
  a[12][8] = -(r2*c12);
  a[12][9] = -(r2*s12);
  a[12][10] = -(l2*c12) + r2*c12;
  a[12][11] = -(l2*s12) + r2*s12;

  a[13][3] = I3;
  a[13][4] = I3;
  a[13][5] = I3;
  a[13][10] = -(r3*c123);
  a[13][11] = -(r3*s123);
  a[13][12] = -(l3*c123) + r3*c123;
  a[13][13] = -(l3*s123) + r3*s123;

  a[14][3] = I4;
  a[14][6] = I4;
  a[14][14] = -(r4*c14);
  a[14][15] = -(r4*s14);
  a[14][16] = -(l4*c14) + r4*c14;
  a[14][17] = -(l4*s14) + r4*s14;

  a[15][3] = I5;
  a[15][6] = I5;
  a[15][7] = I5;
  a[15][16] = -(r5*c145);
  a[15][17] = -(r5*s145);

  a[16][1] = 1;
  a[16][3] = l1*c1 + l2*c12 + l3*c123;
  a[16][4] = l2*c12 + l3*c123;
  a[16][5] = l3*c123;

  a[17][2] = 1;
  a[17][3] = l1*s1 + l2*s12 + l3*s123;
  a[17][4] = l2*s12 + l3*s123;
  a[17][5] = l3*s123;

  b[1] = j1xdi*m1 + a1di*m1*r1*c1;
  b[2] = j1ydi*m1 + a1di*m1*r1*s1;
  b[3] = j1xdi*m2 + a1di*l1*m2*c1 + a1di*m2*r2*c12 + 
    a2di*m2*r2*c12;
  b[4] = j1ydi*m2 + a1di*l1*m2*s1 + a1di*m2*r2*s12 + 
    a2di*m2*r2*s12;
  b[5] = j1xdi*m3 + a1di*l1*m3*c1 + a1di*l2*m3*c12 + 
    a2di*l2*m3*c12 + a1di*m3*r3*c123 + 
    a2di*m3*r3*c123 + a3di*m3*r3*c123;
  b[6] = j1ydi*m3 + a1di*l1*m3*s1 + a1di*l2*m3*s12 + 
    a2di*l2*m3*s12 + a1di*m3*r3*s123 + 
    a2di*m3*r3*s123 + a3di*m3*r3*s123;
  b[7] = j1xdi*m4 + a1di*m4*r4*c14 + a4di*m4*r4*c14;
  b[8] = j1ydi*m4 + a1di*m4*r4*s14 + a4di*m4*r4*s14;
  b[9] = j1xdi*m5 + a1di*l4*m5*c14 + a4di*l4*m5*c14 + 
    a1di*m5*r5*c145 + a4di*m5*r5*c145 + 
    a5di*m5*r5*c145;
  b[10] = j1ydi*m5 + a1di*l4*m5*s14 + a4di*l4*m5*s14 + 
    a1di*m5*r5*s145 + a4di*m5*r5*s145 + 
    a5di*m5*r5*s145;
  b[11] = a1di*I1;
  b[12] = a1di*I2 + a2di*I2;
  b[13] = a1di*I3 + a2di*I3 + a3di*I3;
  b[14] = a1di*I4 + a4di*I4;
  b[15] = a1di*I5 + a4di*I5 + a5di*I5;
  b[16] = 0;
  b[17] = 0;

  ludcmp(a,n,indx,&d);
  lubksb(a,n,indx,b);

  for( i = 1; i <= n; i++ )
    {
      /*
      printf( "%d: %g\n", i, b[i] );
      */
      result[i-1] = b[i];
    }

  footx = j1x + l4*s14 + l5*s145;
  footy = j1y - l4*c14 - l5*c145;
  footxd = b[1] + (b[3] + b[6])*l4*c14
    + (b[3] + b[6] + b[7])*l5*c145;
  footyd = b[2] + (b[3] + b[6])*l4*s14
    + (b[3] + b[6] + b[7])*l5*s145;

  result[17] = footx;
  result[18] = footy;
  result[19] = footxd;
  result[20] = footyd;
  result[21] = b[1];
  result[22] = b[2];
  result[23] = b[1] + (b[3] + b[6])*l4*c14;
  result[24] = b[2] + (b[3] + b[6])*l4*s14;

  /* Need to free a and b, or make them global */
}

/************************************************************************/

ss_impact2_update( SIM *s, double *impact2_analysis )
{

  s->hipd[XX] = impact2_analysis[0];
  s->hipd[ZZ] = impact2_analysis[1];
  s->pitchd = -impact2_analysis[2];
  if ( s->ss_foot_down == LEFT )
    {
      s->hip_angled[RIGHT] = -impact2_analysis[3];
      s->knee_angled[RIGHT] = -impact2_analysis[4];
      s->hip_angled[LEFT] = -impact2_analysis[5];
      s->knee_angled[LEFT] = -impact2_analysis[6];
    }
  else
    {
      s->hip_angled[LEFT] = -impact2_analysis[3];
      s->knee_angled[LEFT] = -impact2_analysis[4];
      s->hip_angled[RIGHT] = -impact2_analysis[5];
      s->knee_angled[RIGHT] = -impact2_analysis[6];
    }
}

/************************************************************************/
/* Need to rerun impact analysis with foot1 free.*/

ss_touchdown_to_ss( SIM *s )
{
  double impact2_analysis[SS_IMPACT2_N+4+4];
  int i;

  if ( interactive )
    printf( "ss_touchdown_to_ss: %g %d\n", s->time, s->ss_foot_down );

  ss_impact2( s, impact2_analysis );

  if ( impact2_analysis[12] < 0 && interactive )
    {
      printf( "Negative impulse: ss_impact2????\n" );
      impact2_print = 1;
    }

  if ( impact2_print )
    {
      printf( "p, lh, lk, rh, rk: %g %g %g %g %g\n",
	      s->pitch, s->hip_angle[LEFT],
	      s->knee_angle[LEFT],
	      s->hip_angle[RIGHT],
	      s->knee_angle[RIGHT] );

      printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
      printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	      s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
      printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	      s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );
      printf( "feetd: %g %g; %g %g\n", s->footd[LEFT][XX], s->footd[LEFT][ZZ],
	      s->footd[RIGHT][XX], s->footd[RIGHT][ZZ] );
      printf( "force: %g %g; %g %g\n", s->gndforce[LEFT][XX], s->gndforce[LEFT][ZZ],
	      s->gndforce[RIGHT][XX], s->gndforce[RIGHT][ZZ] );

      for( i = 0; i < SS_IMPACT2_N + 4 + 4; i++ )
	{
	  printf( "%d: %g\n", i, impact2_analysis[i] );
	}
    }

  ss_impact2_update( s, impact2_analysis );

  if ( s->ss_foot_down == LEFT )
    ss_right_stance( s );
  else
    ss_left_stance( s );

  if ( impact2_print )
    {
      printf( "hipd: %g %g\n", s->hipd[XX], s->hipd[ZZ] );
      printf( "kneesd: %g %g; %g %g\n", s->kneed[LEFT][XX], s->kneed[LEFT][ZZ],
	      s->kneed[RIGHT][XX], s->kneed[RIGHT][ZZ] );
      printf( "feetd: %g %g; %g %g\n", s->footd[LEFT][XX], s->footd[LEFT][ZZ],
	      s->footd[RIGHT][XX], s->footd[RIGHT][ZZ] );
    }
  impact2_print = 0;
}

/************************************************************************/
/* Need to rerun impact analysis with foot1 free.*/

ss_touchdown_to_ds( SIM *s )
{
  double impact1_analysis[SS_IMPACT1_N];
  int i;

  if ( interactive )
    printf( "ss_touchdown_to_ds: %g %d\n", s->time, s->ss_foot_down );

  ss_impact1( s, impact1_analysis );

  if ( impact1_analysis[16] < -s->time_step*9.81 && interactive )
    {
      printf( "Negative impulse ss_impact1????\n" );
      impact1_print = 1;
    }

  /* We are allowing negative impulses here.
  if ( impact1_analysis[6] > s->time_step*9.81 )
    {
      printf( "Negative impulse 2 ss_impact1????\n" );
      impact1_print = 1;
    }
  */

  if ( impact1_print )
    {
      printf( "p, lh, lk, rh, rk: %g %g %g %g %g\n",
	      s->pitch, s->hip_angle[LEFT],
	      s->knee_angle[LEFT],
	      s->hip_angle[RIGHT],
	      s->knee_angle[RIGHT] );
      printf( "pd, lhd, lkd, rhd, rkd: %g %g %g %g %g\n",
	      s->pitchd, s->hip_angled[LEFT],
	      s->knee_angled[LEFT],
	      s->hip_angled[RIGHT],
	      s->knee_angled[RIGHT] );
      
      printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
      printf( "knees: %g %g; %g %g\n", s->knee[LEFT][XX], s->knee[LEFT][ZZ],
	      s->knee[RIGHT][XX], s->knee[RIGHT][ZZ] );
      printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
	      s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );
      printf( "hipd: %g %g\n", s->hipd[XX], s->hipd[ZZ] );
      printf( "kneesd: %g %g; %g %g\n", s->kneed[LEFT][XX], s->kneed[LEFT][ZZ],
	      s->kneed[RIGHT][XX], s->kneed[RIGHT][ZZ] );
      printf( "feetd: %g %g; %g %g\n", s->footd[LEFT][XX], s->footd[LEFT][ZZ],
	      s->footd[RIGHT][XX], s->footd[RIGHT][ZZ] );
      printf( "force: %g %g; %g %g\n", s->gndforce[LEFT][XX], s->gndforce[LEFT][ZZ],
	      s->gndforce[RIGHT][XX], s->gndforce[RIGHT][ZZ] );

      for( i = 0; i < SS_IMPACT1_N; i++ )
	{
	  printf( "%d: %g\n", i, impact1_analysis[i] );
	}
    }

  /*
  if ( s->ss_foot_down == LEFT )
    {
      ds_flag = (impact1_analysis[6] < s->time_step*s->gndforce[LEFT][ZZ]);
      printf( "ds? %d; %g %g; %g %g\n",
	      ds_flag, impact1_analysis[6], s->time_step*s->gndforce[LEFT][ZZ],
	      s->time_step, s->gndforce[LEFT][ZZ] );
    }
  else
    {
      ds_flag = (impact1_analysis[6] < s->time_step*s->gndforce[RIGHT][ZZ]);
      printf( "ds? %d; %g %g; %g %g\n",
	      ds_flag, impact1_analysis[6], s->time_step*s->gndforce[RIGHT][ZZ],
	      s->time_step, s->gndforce[RIGHT][ZZ] );
    }
  */


  for( i = 0; i < SS_NQ; i++ )
    s->ss_state[i+SS_NQ] = impact1_analysis[i];

  /*
    printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
    printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
    s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );
    printf( "hipd: %g %g\n", s->hipd[XX], s->hipd[ZZ] );
  */
  ss_forward_kinematics( s );
  /*
    printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
    printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
    s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );
    printf( "hipd: %g %g\n", s->hipd[XX], s->hipd[ZZ] );
  */
  if ( s->ss_foot_down == LEFT )
    ds_init_double_support_state( s, REF_LFOOT );
  else
    ds_init_double_support_state( s, REF_RFOOT );
  /*
    printf( "hip: %g %g\n", s->hip[XX], s->hip[ZZ] );
    printf( "feet: %g %g; %g %g\n", s->foot[LEFT][XX], s->foot[LEFT][ZZ],
    s->foot[RIGHT][XX], s->foot[RIGHT][ZZ] );
    printf( "hipd: %g %g\n", s->hipd[XX], s->hipd[ZZ] );
  */
  impact1_print = 0;
}

/************************************************************************/
/* Assumes foot1 is on ground, and foot2 just touched down.
Two things can happen.
A) foot1 can be jerked off the ground during the impact.
B) Both feet can end up on the ground: double support.
We first assume case A is what will happen, and check the
velocity of foot1. If it is downwards, it must be case B.
A few things can go wrong.
In either case the impulse for foot2 can be downwards. Need
to understand whether this is a bug or is correct.
It may be the impulse for foot1 in the double support case
has the wrong sign. Need
to understand whether this is a bug or is correct.
*/

ss_touchdown( SIM *s )
{
  double impact1_analysis[SS_IMPACT2_N+4+4];
  double impact2_analysis[SS_IMPACT2_N+4+4];
  int ds_flag;

  if ( interactive )
    printf( "\nss_touchdown: %g %d\n", s->time, s->ss_foot_down );

  /* Check whether the stance foot reaction force has right sign. */
  ss_impact1( s, impact1_analysis );
  /* Check whether the stance foot lifts off. */
  ss_impact2( s, impact2_analysis );

  if ( ( impact1_analysis[6] <= 0 ) != ( impact2_analysis[20] < 0 ) && interactive )
    {
      printf( "Conflict between predictions. %g %g\n", impact1_analysis[6], impact2_analysis[20] );
      printf( "Press return to continue.\n" );
      getchar();
    }

  /* Here we implement a threshold for double support. */
  if ( s->ss_foot_down == LEFT )
    {
      ds_flag = (impact1_analysis[6] < s->time_step*s->gndforce[LEFT][ZZ]);
      if ( interactive )
	printf( "ds? %d; %g %g; %g %g\n",
		ds_flag, impact1_analysis[6], s->time_step*s->gndforce[LEFT][ZZ],
		s->time_step, s->gndforce[LEFT][ZZ] );
    }
  else
    {
      ds_flag = (impact1_analysis[6] < s->time_step*s->gndforce[RIGHT][ZZ]);
      if ( interactive )
	printf( "ds? %d; %g %g; %g %g\n",
		ds_flag, impact1_analysis[6], s->time_step*s->gndforce[RIGHT][ZZ],
		s->time_step, s->gndforce[RIGHT][ZZ] );
    }

  /* everybody agrees this is double support */
  if ( ds_flag || impact2_analysis[20] < 0 )
    {
      ss_touchdown_to_ds( s );
      return;
    }

  ss_touchdown_to_ss( s );
}

/************************************************************************/

ss_touchdown_alt( SIM *s )
{
  double impact_analysis[SS_IMPACT2_N+4+4];

  printf( "\nss_touchdown: %g %d\n", s->time, s->ss_foot_down );

  /* Check whether the stance foot reaction force has right sign. */
  ss_impact1( s, impact_analysis );

  /* check impluse */
  if ( impact_analysis[6] <= 0 )
    ss_touchdown_to_ds( s );
  else ss_touchdown_to_ss( s );
}

/************************************************************************/

ss_touchdown_alt2( SIM *s )
{
  double impact_analysis[SS_IMPACT2_N+4+4];

  printf( "\nss_touchdown: %g %d\n", s->time, s->ss_foot_down );

  /* Check whether the stance foot lifts off. */
  ss_impact2( s, impact_analysis );

  /* check footyd */
  if ( impact_analysis[20] < 0 )
    ss_touchdown_to_ds( s );
  else ss_touchdown_to_ss( s );
}

/************************************************************************/
/* This is what is called on each integration step */

void integrate_one_time_step( SIM *s  )
{ 
  int i;
  int err; 
    /* { OK, DERIVATIVE_DISCONTINUITY, SYSTEM_LOCKED, CONSTRAINTS_ERR } */
  double errest;
  double gndforce[3];
  int subdividei = 1;
  int subdividec = 200;
  
  /*
  printf( "%g\n", s->time );
  */
  /* Give up if hip is too high or too low */
  if ( s->hip[ZZ] > 1.1*(s->calf_length + s->thigh_length)
       || s->hip[ZZ] < 1.5*s->calf_length )
    {
      s->status = CRASHED;
      s->time += s->time_step;
      return;
    }

  if ( s->sdfast_mode == CONSTRAINTS )
    {
      // clear outstanding error flags
      b1cclearerr();

      for( i = 0; i < subdividec; i++ )
	{
	  b1cfmotion(&(s->time),
		      s->state_sdfast,
		      s->state_sdfast2,
		      s->time_step/subdividec,
		      CTOL,&(s->sdfast_flag),&errest,&err);
	  b1cprinterr(stderr);
	}

      constraints_forward_kinematics( s );
    }

  if ( s->sdfast_mode == IN_AIR )
    {
      // clear outstanding error flags
      b1gclearerr();

      for( i = 0; i < 1; i++ )
	{
	  b1gfmotion(&(s->time),
		      s->state_sdfast,
		      s->state_sdfast2,
		      s->time_step,
		      CTOL,&(s->sdfast_flag),&errest,&err);
	  b1gprinterr(stderr);
	}

      generic_forward_kinematics( s );

      if ( s->foot[LEFT][ZZ] < 0 && s->foot[RIGHT][ZZ] < 0 )
	ds_init_double_support_state( s, REF_HIP );
      else if ( s->foot[LEFT][ZZ] < 0 )
	ss_left_stance( s );
      else if ( s->foot[RIGHT][ZZ] < 0 )
	ss_right_stance( s );
    }

  if ( s->sdfast_mode == DOUBLE_SUPPORT )
    {
      // clear outstanding error flags
      b1dsclearerr();

      for( i = 0; i < subdividei; i++ )
	{
	  b1dsfmotion( &(s->time), 
		       s->ds_state,
		       s->state_sdfast2,
		       s->time_step/subdividei,
		       DS_CTOL,&(s->sdfast_flag),&errest,&err);
	  b1dsprinterr(stderr);
	}

      b1dsreac( ds_joint_forces, ds_joint_torques );
      b1dstrans( DS_SHIN1_ID, ds_joint_forces[DS_ANKLE1], DS_GND_ID, gndforce );
      fix_yz( s, gndforce, s->gndforce[LEFT] );
      b1dstrans( DS_SHIN2_ID, ds_joint_forces[5], DS_GND_ID, gndforce );
      fix_yz( s, gndforce, s->gndforce[RIGHT] );
      ds_forward_kinematics( s );

      if ( s->gndforce[LEFT][ZZ] < 0
	   && s->gndforce[RIGHT][ZZ] < 0 )
	{
	  if ( interactive )
	    fprintf( stderr, "Need to handle biped in air.\n" );
	  transition_to_air( s );
	}

      if ( s->gndforce[LEFT][ZZ] < 0 )
	ss_right_stance( s );

      if ( s->gndforce[RIGHT][ZZ] < 0 )
	ss_left_stance( s );
    }

  if ( s->sdfast_mode == SINGLE_SUPPORT )
    {
      // clear outstanding error flags
      b1ssclearerr();

      for( i = 0; i < subdividei; i++ )
	{
	  b1ssfmotion( &(s->time), 
		       s->ss_state,
		       s->state_sdfast2,
		       s->time_step/subdividei,
		       SS_CTOL,&(s->sdfast_flag),&errest,&err);
	  b1ssprinterr(stderr);
	}

      b1ssreac( ss_joint_forces, ss_joint_torques );
      b1sstrans( SS_SHIN1_ID, ss_joint_forces[SS_ANKLE1], SS_GND_ID, gndforce );
      if ( s->ss_foot_down == LEFT )
	{
	  fix_yz( s, gndforce, s->gndforce[LEFT] );
	  for( i = 0; i < 3; i++ )
	    s->gndforce[RIGHT][i] = 0;
	}
      else
	{
	  fix_yz( s, gndforce, s->gndforce[RIGHT] );
	  for( i = 0; i < 3; i++ )
	    s->gndforce[LEFT][i] = 0;
	}
      ss_forward_kinematics( s );

      if ( s->ss_foot_down == LEFT )
	{
	  if ( s->gndforce[LEFT][ZZ] < 0 )
	    { /* ss_liftoff( s ); */
	      if ( interactive )
		fprintf( stderr, "SS1: Need to handle biped in air.\n" );
	      transition_to_air( s );
	    }
	  if ( s->foot[RIGHT][ZZ] < EPSILON1 )
	    ss_touchdown( s );
	  /*
	    transition_to_constraints( s );
	  */
	}
      else
	{
	  if ( s->gndforce[RIGHT][ZZ] < 0 )
	    { /* ss_liftoff( s ); */
	      if ( interactive )
		fprintf( stderr, "SS2: Need to handle biped in air.\n" );
	      transition_to_air( s );
	    }
	  if ( s->foot[LEFT][ZZ] < EPSILON1 )
	    ss_touchdown( s );
	  /*
	    transition_to_constraints( s );
	  */
	}
    }

  /*
  if ( s->time > 1.0 && s->sdfast_mode != CONSTRAINTS )
    transition_to_constraints( s );
  */

  /*
  printf( "Press return to continue.\n" );
  getchar();
  */
}

/************************************************************************/
/* SDFAST stuff */
/************************************************************************/
/* This is where the control (hip_torque) is applied. May be called many
times per integration step at any time or state. */

void b1guforce(double t, double *q, double *u)
{
  b1ghinget( 1, 0, sim.hip_torque[LEFT] );
  b1ghinget( 2, 0, sim.knee_torque[LEFT] );
  b1ghinget( 3, 0, sim.hip_torque[RIGHT] );
  b1ghinget( 4, 0, sim.knee_torque[RIGHT] );
}

/*************************************************************************/

void b1ssuforce(double t, double *q, double *u)
{
  if ( sim.ss_foot_down == LEFT )
    {
      b1sshinget( SS_KNEE1, 0, sim.knee_torque[LEFT] );
      b1sshinget( SS_HIP1, 0, sim.hip_torque[LEFT] );
      b1sshinget( SS_HIP2, 0, -sim.hip_torque[RIGHT] );
      b1sshinget( SS_KNEE2, 0, -sim.knee_torque[RIGHT] );
    }
  else
    {
      b1sshinget( SS_KNEE1, 0, sim.knee_torque[RIGHT] );
      b1sshinget( SS_HIP1, 0, sim.hip_torque[RIGHT] );
      b1sshinget( SS_HIP2, 0, -sim.hip_torque[LEFT] );
      b1sshinget( SS_KNEE2, 0, -sim.knee_torque[LEFT] );
    }
}

/*************************************************************************/

void b1dsuforce(double t, double *q, double *u)
{
  b1dshinget( DS_KNEE1, 0, sim.knee_torque[LEFT] );
  b1dshinget( DS_HIP1, 0, sim.hip_torque[LEFT] );
  b1dshinget( DS_HIP2, 0, -sim.hip_torque[RIGHT] );
  b1dshinget( DS_KNEE2, 0, -sim.knee_torque[RIGHT] );
}

/************************************************************************/
/* This is where the control (hip_torque) is applied. May be called many
times per integration step at any time or state. */

void b1cuforce(double t, double *q, double *u)
{
  b1chinget( 1, 0, sim.hip_torque[LEFT] );
  b1chinget( 2, 0, sim.knee_torque[LEFT] );
  b1chinget( 3, 0, sim.hip_torque[RIGHT] );
  b1chinget( 4, 0, sim.knee_torque[RIGHT] );
}

/************************************************************************/
/* SDFAST constraint stuff */

void b1cuperr(double t, double *q, double *errs) 
{
  double foot_position[2][3];

  errs[0]=errs[1]=errs[2]=errs[3] = 0;

  b1cpos( BODY_L_SHIN, sim.foot_offset, foot_position[LEFT] );
  b1cpos( BODY_R_SHIN, sim.foot_offset, foot_position[RIGHT] );

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL )
    {
      errs[0] = foot_position[LEFT][ZZ] - GROUND_LEVEL;
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL )
    {
      errs[1] = foot_position[RIGHT][ZZ] - GROUND_LEVEL;
    }
}

/************************************************************************/

void b1cuverr(double t, double *q, double *u, double *errs)
{
  double foot_position[2][3];
  double foot_velocity[2][3];

  errs[0]=errs[1]=errs[2]=errs[3] = 0;

  b1cpos( BODY_L_SHIN, sim.foot_offset, foot_position[LEFT] );
  b1cpos( BODY_R_SHIN, sim.foot_offset, foot_position[RIGHT] );
  b1cvel( BODY_L_SHIN, sim.foot_offset, foot_velocity[LEFT] );
  b1cvel( BODY_R_SHIN, sim.foot_offset, foot_velocity[RIGHT] );

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL )
    {
      if ( foot_velocity[LEFT][ZZ] < 0 )
	errs[0] = foot_velocity[LEFT][ZZ];
    }
  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      errs[2] = foot_velocity[LEFT][XX];
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL )
    {
      if ( foot_velocity[RIGHT][ZZ] < 0 )
	errs[1] = foot_velocity[RIGHT][ZZ];
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      errs[3] = foot_velocity[RIGHT][XX];
    }
}   

/************************************************************************/

void b1cuaerr(double t, double *q, double *u, double *udot, double *errs)
{
  double foot_position[2][3];
  double foot_acceleration[2][3];

  errs[0]=errs[1]=errs[2]=errs[3]= 0;

  b1cpos( BODY_L_SHIN, sim.foot_offset, foot_position[LEFT] );
  b1cpos( BODY_R_SHIN, sim.foot_offset, foot_position[RIGHT] );
  b1cacc( BODY_L_SHIN, sim.foot_offset, foot_acceleration[LEFT] );
  b1cacc( BODY_R_SHIN, sim.foot_offset, foot_acceleration[RIGHT] );

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL )
    {
      if ( foot_acceleration[LEFT][ZZ] < 0 )
	errs[0] = foot_acceleration[LEFT][ZZ];
    }
  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      errs[2] = foot_acceleration[LEFT][XX];
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL )
    {
      if ( foot_acceleration[RIGHT][ZZ] < 0 )
	errs[1] = foot_acceleration[RIGHT][ZZ];
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      errs[3] = foot_acceleration[RIGHT][XX];
    }
}

/************************************************************************/

void b1cuconsfrc(double t, double *q, double *u, double *m)
{
  double foot_position[2][3];
  double forces[2][3];
  int i;

  b1cpos( BODY_L_SHIN, sim.foot_offset, foot_position[LEFT] );
  b1cpos( BODY_R_SHIN, sim.foot_offset, foot_position[RIGHT] );

  for( i = 0; i < 3; i++ )
    {
      sim.gndforce[LEFT][i] = forces[LEFT][i] = 0;
      sim.gndforce[RIGHT][i] = forces[RIGHT][i] = 0;
    }

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL )
    {
      if ( m[0] >= 0  )
	{
	  sim.gndforce[LEFT][ZZ] = forces[LEFT][ZZ] = m[0];
	  sim.gndforce[LEFT][XX] = forces[LEFT][XX] = m[2];
	}
    }

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      ;
    }

  b1ctrans( GND, forces[LEFT], BODY_L_SHIN, forces[LEFT] );
  b1cpointf( BODY_L_SHIN, sim.foot_offset, forces[LEFT] );

  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL )
    {
      if ( m[1] >= 0  )
	{
	  sim.gndforce[RIGHT][ZZ] = forces[RIGHT][ZZ] = m[1];
	  sim.gndforce[RIGHT][XX] = forces[RIGHT][XX] = m[3];
	}
    }

  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      ;
    }

  b1ctrans( GND, forces[RIGHT], BODY_R_SHIN, forces[RIGHT] );
  b1cpointf( BODY_R_SHIN, sim.foot_offset, forces[RIGHT] );
}

/************************************************************************/
/************************************************************************/
