/*
handle sdfast flag correctly (only b1gstate sets it).
*/
/************************************************************************/
/*
  dynamics-constraints.c: 
This is where the numerical integration and SDFAST stuff is done.
This version uses SDFAST constraints
*/
/************************************************************************/

#include <stdio.h>
#include <math.h>

#include "main.h"
#include "sdfast/b1g.h"

/************************************************************************/
SIM sim;
/************************************************************************/

/* SDFAST constants and variables */
#define NQ 7
#define NU 7
#define NSTATE (NQ+NU)
#define CTOL	1e-5	/* constraint tolerance */
#define GND -1
const double baumgarte=100.0; /* could be 10.0 - 1000.0 */

/* Here we define the state */
double xt[NSTATE];
/* Here we define the derivative of the state */
static double dxt[NSTATE];
/* Leg state machines */
const double GROUND_LEVEL = 0.00;
const double GROUND_LEVEL_X_INCREMENT = 0.00;
/************************************************************************/

static void forward_kinematics( SIM *s )
{

  b1cpos(BODY_PELVIS, s->head_offset, s->head);    
  b1cpos(BODY_PELVIS, s->hip_offset, s->hip);    
  b1cpos(BODY_L_SHIN, s->knee_offset, &(s->knee[LEFT][0]));    
  b1cpos(BODY_R_SHIN, s->knee_offset, &(s->knee[RIGHT][0]));    
  b1cpos(BODY_L_SHIN, s->foot_offset, &(s->foot[LEFT][0]));    
  b1cpos(BODY_R_SHIN, s->foot_offset, &(s->foot[RIGHT][0]));    

  b1cvel(BODY_PELVIS, s->head_offset, s->headd);    
  b1cvel(BODY_PELVIS, s->hip_offset, s->hipd);    
  b1cvel(BODY_L_SHIN, s->knee_offset, &(s->kneed[LEFT][0]));    
  b1cvel(BODY_R_SHIN, s->knee_offset, &(s->kneed[RIGHT][0]));    
  b1cvel(BODY_L_SHIN, s->foot_offset, &(s->footd[LEFT][0]));    
  b1cvel(BODY_R_SHIN, s->foot_offset, &(s->footd[RIGHT][0]));    

  b1cpos(BODY_L_SHIN, s->foot_offset, s->footpos[LEFT]);    
  b1cpos(BODY_R_SHIN, s->foot_offset, s->footpos[RIGHT]);    

  b1cvel(BODY_L_SHIN, s->foot_offset, s->footvel[LEFT]);    
  b1cvel(BODY_R_SHIN, s->foot_offset, s->footvel[RIGHT]);    

  s->pitch = s->state_sdfast[Q_PITCH];
  s->hip_angle[LEFT] = s->state_sdfast[Q_L_HIP];
  s->hip_angle[RIGHT] = s->state_sdfast[Q_R_HIP];
  s->knee_angle[LEFT] = s->state_sdfast[Q_L_KNEE];
  s->knee_angle[RIGHT] = s->state_sdfast[Q_R_KNEE];

  s->pitchd = s->state_sdfast[QD_PITCH];
  s->hip_angled[LEFT] = s->state_sdfast[QD_L_HIP];
  s->hip_angled[RIGHT] = s->state_sdfast[QD_R_HIP];
  s->knee_angled[LEFT] = s->state_sdfast[QD_L_KNEE];
  s->knee_angled[RIGHT] = s->state_sdfast[QD_R_KNEE];
}

/************************************************************************/
/* Initialize the state vector */

void init_dynamics_state( SIM *s, double *state )
{
  int i, j;

  for( i = 0; i < NSTATE; i++ )
    {
      xt[i] = state[i];
      dxt[i] = 0;
    }

  for( i = 0, j = NQ; i < NU; i++, j++ )
    { 
      dxt[i] = xt[j];
    }

  b1cstate( 0.0, xt, dxt );

  forward_kinematics( s );
}

/************************************************************************/
/* Initialize sdfast  */

void init_dynamics( SIM *s )
{
  int i, j;
  double vector[3];
  double inertia[3][3];

  init_kin_dyn_parameters( s );

  b1cmass( BODY_PELVIS, s->pelvis_mass );
  b1cmass( BODY_L_THIGH, s->thigh_mass );
  b1cmass( BODY_R_THIGH, s->thigh_mass );
  b1cmass( BODY_L_SHIN, s->calf_mass );
  b1cmass( BODY_R_SHIN, s->calf_mass );

  for ( i = 0; i < 3; i++ )
    {
      vector[i] = 0;
      for ( j = 0; j < 3; j++ )
	{
	  if ( i == j )
	    inertia[i][i] = 1.0;
	  else
	    inertia[i][j] = 0;
	}
    }
  inertia[YY][YY] = s->pelvis_I;
  b1ciner( BODY_PELVIS, inertia );
  inertia[YY][YY] = s->thigh_I;
  b1ciner( BODY_L_THIGH, inertia );
  b1ciner( BODY_R_THIGH, inertia );
  inertia[YY][YY] = s->calf_I;
  b1ciner( BODY_L_SHIN, inertia );
  b1ciner( BODY_R_SHIN, inertia );

  vector[ZZ] = s->thigh_cm;
  b1cbtj( BODY_L_THIGH, vector );
  b1cbtj( BODY_R_THIGH, vector );
  vector[ZZ] = s->calf_cm;
  b1cbtj( BODY_L_SHIN, vector );
  b1cbtj( BODY_R_SHIN, vector );

  vector[ZZ] = -(s->thigh_length - s->thigh_cm);
  b1citj( BODY_L_SHIN, vector );
  b1citj( BODY_R_SHIN, vector );

  b1cinit(); /* initialize SDFAST model */

  b1cstab(2.0*baumgarte, baumgarte*baumgarte); 

  for( i = 0; i < NSTATE; i++ )
    s->state_sdfast[i] = 0;

  b1cprinterr(stderr);
}

/************************************************************************/
/* Need to handle constraints and xd, zd, and pitchd requests correctly */

void init_state_one_foot_on_ground( SIM *s )
{
  double min;

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = s->knee_angled[RIGHT];

  init_dynamics_state( s, s->state_sdfast );

  min = s->foot[LEFT][ZZ];
  if ( min > s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min;

  init_dynamics_state( s, s->state_sdfast );
}

/************************************************************************/
/* Need to handle constraints and xd, zd, and pitchd requests correctly */

void init_state_two_feet_on_ground( SIM *s )
{
  double min;
  double offset;

  s->state_sdfast[Q_X] = s->hip[XX];
  s->state_sdfast[Q_Z] = s->hip[ZZ];
  s->state_sdfast[Q_PITCH] = s->pitch;
  s->state_sdfast[Q_L_HIP] = s->hip_angle[LEFT];
  s->state_sdfast[Q_R_HIP] = s->hip_angle[RIGHT];
  s->state_sdfast[Q_L_KNEE] = s->knee_angle[LEFT];
  s->state_sdfast[Q_R_KNEE] = s->knee_angle[RIGHT];

  s->state_sdfast[QD_X] = s->hipd[XX];
  s->state_sdfast[QD_Z] = s->hipd[ZZ];
  s->state_sdfast[QD_PITCH] = s->pitchd;
  s->state_sdfast[QD_L_HIP] = s->hip_angled[LEFT];
  s->state_sdfast[QD_R_HIP] = s->hip_angled[RIGHT];
  s->state_sdfast[QD_L_KNEE] = s->knee_angled[LEFT];
  s->state_sdfast[QD_R_KNEE] = s->knee_angled[RIGHT];

  init_dynamics_state( s, s->state_sdfast );

  /* Need to handle case where two feet together */

  if ( s->foot[RIGHT][XX] - s->foot[LEFT][XX] > 0 )
    offset = atan2( s->foot[RIGHT][ZZ] - s->foot[LEFT][ZZ],
		    s->foot[RIGHT][XX] - s->foot[LEFT][XX] );
  else
    offset = atan2( s->foot[LEFT][ZZ] - s->foot[RIGHT][ZZ],
		    s->foot[LEFT][XX] - s->foot[RIGHT][XX] );

  /*
  printf( "two feet on ground: %g %g %g %g: %g\n", 
	  s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ],
	  s->foot[RIGHT][XX], s->foot[LEFT][XX], offset );
  */

  s->state_sdfast[Q_PITCH] += offset;

  init_dynamics_state( s, s->state_sdfast );

  /*
  printf( "%g %g\n", s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ] );
  */

  min = s->foot[LEFT][ZZ];
  if ( min > s->foot[RIGHT][ZZ] )
    min = s->foot[RIGHT][ZZ];
  s->state_sdfast[Q_Z] = s->state_sdfast[Q_Z] - min;

  init_dynamics_state( s, s->state_sdfast );

  /*
  printf( "%g %g\n", s->foot[RIGHT][ZZ], s->foot[LEFT][ZZ] );
  */
}

/************************************************************************/
/* This is what is called on each integration step */

void integrate_one_time_step( SIM *s )
{ 
  int i;
  int flag; /* set to 1 when things are changed or first call */
  int err; 
    /* { OK, DERIVATIVE_DISCONTINUITY, SYSTEM_LOCKED, CONSTRAINTS_ERR } */
  double errest;
  float subdivide = 100.0; /* 10000.0 */

  if ( s->hip[ZZ] < 0.2 )
    {
      s->time += s->time_step;
      return;
    }

  flag=1;

  // clear outstanding error flags
  b1cclearerr();

  for( i = 0; i < subdivide; i++ )
    {
      b1cfmotion(&(sim.time),xt,dxt,sim.time_step/subdivide,CTOL,&flag,&errest,&err);
      b1cprinterr(stderr);
    }

  for( i = 0; i < NSTATE; i++ )
    s->state_sdfast[i] = xt[i];

  forward_kinematics( s );

  /*
  printf( "Press return to continue.\n" );
  getchar();
  */
}

/****************************************************************/

// Utility function to compute the ground force function.
static 
void compute_ground_force(double contactpos[3],  // contact position wrt world
			  double contact_zero[3], // original conact point.
			  double contactvel[3],  // contact velocity wrt world
			  double bodypos[3],     // contact point wrt body
			  int body,            
			  double force[3])       // resultant force vector wrt world
			  
{
  int i;

  // force only exists if foot is below ground level
  if (contactpos[2] < 0.0) {
    double fvec[3];  // ground force vector in body coordintes

    // Compute the desired ground force vector in the world
    // coordinates.  For now, the lateral force is just viscous.
    // It should really be a constraint equation.
    // CGA: Actually keep track of original contact point. */
    if ( contact_zero[ZZ] > 0.0 )
      for( i = 0; i < 3; i++ )
	contact_zero[i] = contactpos[i];

    force[0] = (contact_zero[0] - contactpos[0])*sim.gnd_k_x
						 -contactvel[0] * sim.gnd_b;
    force[1] = 0.0;   // not relevant
    force[2] = (-contactpos[2] * sim.gnd_k) - (contactvel[2] * sim.gnd_b);

    // ground can't pull down
    if (force[2] < 0) 
      {
	force[0] = 0.0;
	force[1] = 0.0;
	force[2] = 0.0;
	/* Slipping */
	for( i = 0; i < 3; i++ )
	  contact_zero[i] = contactpos[i];
      }

    // transform the vector into body coordinates
    b1ctrans(BODY_WORLD, force, body, fvec);

    // apply to the model
    b1cpointf(body, bodypos, fvec);

  } else {
    force[0] = force[1] = force[2] = 0.0;
    for( i = 0; i < 3; i++ )
      contact_zero[i] = contactpos[i];
  }
}

/************************************************************************/
/* SDFAST stuff */
/************************************************************************/
/* This is where the control (hip_torque) is applied. May be called many
times per integration step at any time or state. */

void b1cuforce(double t, double *q, double *u)
{

  b1chinget( 1, 0, sim.hip_torque[LEFT] );
  b1chinget( 2, 0, sim.knee_torque[LEFT] );
  b1chinget( 3, 0, sim.hip_torque[RIGHT] );
  b1chinget( 4, 0, sim.knee_torque[RIGHT] );
}

/************************************************************************/
/* SDFAST constraint stuff */

void b1cuperr(double t, double *q, double *errs) 
{
  double foot_position[2][3];

  errs[0]=errs[1]=errs[2]=errs[3] = 0;

  b1cpos( BODY_L_SHIN, sim.foot_offset, foot_position[LEFT] );
  b1cpos( BODY_R_SHIN, sim.foot_offset, foot_position[RIGHT] );

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL )
    {
      errs[0] = foot_position[LEFT][ZZ] - GROUND_LEVEL;
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL )
    {
      errs[1] = foot_position[RIGHT][ZZ] - GROUND_LEVEL;
    }
}

/************************************************************************/

void b1cuverr(double t, double *q, double *u, double *errs)
{
  double foot_position[2][3];
  double foot_velocity[2][3];

  errs[0]=errs[1]=errs[2]=errs[3] = 0;

  b1cpos( BODY_L_SHIN, sim.foot_offset, foot_position[LEFT] );
  b1cpos( BODY_R_SHIN, sim.foot_offset, foot_position[RIGHT] );
  b1cvel( BODY_L_SHIN, sim.foot_offset, foot_velocity[LEFT] );
  b1cvel( BODY_R_SHIN, sim.foot_offset, foot_velocity[RIGHT] );

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL )
    {
      if ( foot_velocity[LEFT][ZZ] < 0 )
	errs[0] = foot_velocity[LEFT][ZZ];
    }
  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      errs[2] = foot_velocity[LEFT][XX];
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL )
    {
      if ( foot_velocity[RIGHT][ZZ] < 0 )
	errs[1] = foot_velocity[RIGHT][ZZ];
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      errs[3] = foot_velocity[RIGHT][XX];
    }
}   

/************************************************************************/

void b1cuaerr(double t, double *q, double *u, double *udot, double *errs)
{
  double foot_position[2][3];
  double foot_acceleration[2][3];

    errs[0]=errs[1]=errs[2]=errs[3]= 0;

  b1cpos( BODY_L_SHIN, sim.foot_offset, foot_position[LEFT] );
  b1cpos( BODY_R_SHIN, sim.foot_offset, foot_position[RIGHT] );
  b1cacc( BODY_L_SHIN, sim.foot_offset, foot_acceleration[LEFT] );
  b1cacc( BODY_R_SHIN, sim.foot_offset, foot_acceleration[RIGHT] );

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL )
    {
      if ( foot_acceleration[LEFT][ZZ] < 0 )
	errs[0] = foot_acceleration[LEFT][ZZ];
    }
  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      errs[2] = foot_acceleration[LEFT][XX];
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL )
    {
      if ( foot_acceleration[RIGHT][ZZ] < 0 )
	errs[1] = foot_acceleration[RIGHT][ZZ];
    }
  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      errs[3] = foot_acceleration[RIGHT][XX];
    }
}

/************************************************************************/

void b1cuconsfrc(double t, double *q, double *u, double *m)
{
  double foot_position[2][3];
  double forces[2][3];
  int i;

  b1cpos( BODY_L_SHIN, sim.foot_offset, foot_position[LEFT] );
  b1cpos( BODY_R_SHIN, sim.foot_offset, foot_position[RIGHT] );

  for( i = 0; i < 3; i++ )
    {
      sim.gndforce[LEFT][i] = forces[LEFT][i] = 0;
      sim.gndforce[RIGHT][i] = forces[RIGHT][i] = 0;
    }

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL )
    {
      if ( m[0] >= 0  )
	{
	  sim.gndforce[LEFT][ZZ] = forces[LEFT][ZZ] = m[0];
	  sim.gndforce[LEFT][XX] = forces[LEFT][XX] = m[2];
	}
    }

  if( foot_position[LEFT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      ;
    }

  b1ctrans( GND, forces[LEFT], BODY_L_SHIN, forces[LEFT] );
  b1cpointf( BODY_L_SHIN, sim.foot_offset, forces[LEFT] );

  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL )
    {
      if ( m[1] >= 0  )
	{
	  sim.gndforce[RIGHT][ZZ] = forces[RIGHT][ZZ] = m[1];
	  sim.gndforce[RIGHT][XX] = forces[RIGHT][XX] = m[3];
	}
    }

  if( foot_position[RIGHT][ZZ] <= GROUND_LEVEL + GROUND_LEVEL_X_INCREMENT )
    {
      ;
    }

  b1ctrans( GND, forces[RIGHT], BODY_R_SHIN, forces[RIGHT] );
  b1cpointf( BODY_R_SHIN, sim.foot_offset, forces[RIGHT] );
}

/************************************************************************/
/************************************************************************/
