/*****************************************************************************/
/*
  controller.c: control strategy.
*/
/*****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include "../trajectory/trajectory.h"
#include "main.h"

/*****************************************************************************/

#define N_BIPED_DOFS 8

/* DOFS */
/* positions */
#define L_HIP 0
#define L_KNEE 1
#define R_HIP 2
#define R_KNEE 3
/* torques */
#define L_HIP_T 4
#define L_KNEE_T 5
#define R_HIP_T 6
#define R_KNEE_T 7

/* States in state machine */
#define UNKNOWN_STATE 0
#define WAITING 1
#define LAUNCH1 2
#define LAUNCH2 3
#define LAUNCH3 4
#define L_SWING 5
#define RL_STANCE 6
#define R_SWING 7
#define LR_STANCE 8
#define ZERO_TORQUES 9
#define FREEZE 10

/*****************************************************************************/
/* Control variables */
int print = 0;

/* storage. */
double torque_score = 0;
double crashed_penalty = 0;
float last_l_foot_x = 0;
float last_r_foot_x = 0;
int top_detected = 0;

/* Trajectory info */
DOFS desired_trajectory[N_BIPED_DOFS];
DOFS current_desireds[N_BIPED_DOFS];

/*****************************************************************************/
/*****************************************************************************/

void reset_score( SIM *s )
{
  torque_score = 0;
  crashed_penalty = 0;
}

/*****************************************************************************/

void update_score( SIM *s )
{
  torque_score += s->hip_command[LEFT]*s->hip_command[LEFT];
  torque_score += s->hip_command[RIGHT]*s->hip_command[RIGHT];
  torque_score += s->knee_command[LEFT]*s->knee_command[LEFT];
  torque_score += s->knee_command[RIGHT]*s->knee_command[RIGHT];
  if ( s->status == CRASHED )
    crashed_penalty += s->crashed_penalty_increment;
}

/*****************************************************************************/
/*
Could normalize torque and other penalties w.r.t. distance
Could penalize deviations from symmetry (if necessary).
*/

float get_score( SIM *s )
{
  float speed,speed_penalty;

  speed = s->hip[XX]/s->time;
  speed_penalty = (s->desired_speed - speed)*(s->desired_speed - speed)
    *s->speed_penalty_weight;
  torque_score *= s->torque_penalty_weight;
  printf( "Torque score: %lg; distance score: %g; speed %g; crashed %g\n",
	  torque_score, speed_penalty, speed, crashed_penalty );
  return (float) (torque_score + speed_penalty + crashed_penalty);
}

/*****************************************************************************/
/* call this many times to restart a simulation */

reinit_controller( SIM *s )
{
  int i;

  s->controller_state = WAITING;
  s->state_start_time = s->time;
  s->state_elapsed_time = 0.0;
  s->desireds_index = s->time;
  s->last_time = s->time;
  s->step_length = 0;
  s->top_velocity = 0;

  for( i = 0; i < 2; i++ )
    {
      s->hip_angle_d[i] = s->hip_angle[i];
      s->knee_angle_d[i] = s->knee_angle[i];
      s->hip_angled_d[i] = s->hip_angled[i];
      s->knee_angled_d[i] = s->knee_angled[i];
      s->hip_angledd_d[i] = 0;
      s->hip_command[i] = 0;
      s->hip_feedforward[i] = 0;
      s->hip_torque[i] = 0;
      s->knee_command[i] = 0;
      s->knee_feedforward[i] = 0;
      s->knee_torque[i] = 0;
    }

  init_trajectory( desired_trajectory, N_BIPED_DOFS );
  init_trajectory( current_desireds, N_BIPED_DOFS );

  /* add actual position as knot to current desireds */
  add_knot_point( current_desireds, L_HIP, QUINTIC_SPLINE, 0.0, 
		  s->hip_angle_d[LEFT], s->hip_angled_d[LEFT], 0.0 );
  add_knot_point( current_desireds, L_KNEE, QUINTIC_SPLINE, 0.0,
		  s->knee_angle_d[LEFT], s->knee_angled_d[LEFT], 0.0 );
  add_knot_point( current_desireds, R_HIP, QUINTIC_SPLINE, 0.0, 
		  s->hip_angle_d[RIGHT], s->hip_angled_d[RIGHT], 0.0 );
  add_knot_point( current_desireds, R_KNEE, QUINTIC_SPLINE, 0.0, 
		  s->knee_angle_d[RIGHT], s->knee_angled_d[RIGHT], 0.0 );
  add_knot_point( current_desireds, L_HIP_T, QUINTIC_SPLINE, 0.0, 
		  s->hip_feedforward[LEFT], 0.0, 0.0 );
  add_knot_point( current_desireds, L_KNEE_T, QUINTIC_SPLINE, 0.0,
		  s->knee_feedforward[LEFT], 0.0, 0.0 );
  add_knot_point( current_desireds, R_HIP_T, QUINTIC_SPLINE, 0.0, 
		  s->hip_feedforward[RIGHT], 0.0, 0.0 );
  add_knot_point( current_desireds, R_KNEE_T, QUINTIC_SPLINE, 0.0, 
		  s->knee_feedforward[RIGHT], 0.0, 0.0 );

  reset_score( s );

  last_l_foot_x = s->foot[LEFT][XX];
  last_r_foot_x = s->foot[RIGHT][XX];
}

/*****************************************************************************/
/* call this many times to restart a simulation */

int reinit_sim( SIM *s )
{
  int i;
  double min;

  srand( s->rand_seed );
  /*
  printf( "seed: %d %g\n", s->rand_seed, s->rand_scale );
  */

  s->time = 0.0;

  s->hip[XX] = 0;
  s->hip[ZZ] = 0;
  s->pitch = 0;
  s->hipd[XX] = 0;
  s->hipd[ZZ] = 0;
  s->pitchd = 0;
  s->hip_angle[LEFT] = s->wait_l_hip;
  s->hip_angle[RIGHT] = s->wait_r_hip;
  s->knee_angle[LEFT] = s->wait_l_knee;
  s->knee_angle[RIGHT] = s->wait_r_knee;
  s->hip_angled[LEFT] = 0;
  s->hip_angled[RIGHT] = 0;
  s->knee_angled[LEFT] = 0;
  s->knee_angled[RIGHT] = 0;
  s->hip_torque[LEFT] = 0;
  s->hip_torque[RIGHT] = 0;
  s->knee_torque[LEFT] = 0;
  s->knee_torque[RIGHT] = 0;

  /* These are for original ground model */
  /* Indicate foot zero positions not set */
  s->foot_zero[LEFT][ZZ] = 1.0;
  s->foot_zero[RIGHT][ZZ] = 1.0;

  /*
  init_state_one_foot_on_ground( s );
  */
  init_state_two_feet_on_ground( s );

  reinit_controller( s );
}

/*****************************************************************************/
/* Call this once to do one time operations like memory allocation */

int init_sim( SIM *s )
{
  init_dynamics( s );
  reinit_sim( s );
}

/*****************************************************************************/
/*****************************************************************************/
  /* update current DESIRED positions based on DESIRED positions*/

void
reset_desireds_based_on_actuals( SIM *s, int dof )
{
  switch( dof )
    {
    case L_HIP:
      s->hip_angle_d[LEFT] = s->hip_angle[LEFT];
      s->hip_angled_d[LEFT] = s->hip_angled[LEFT];
      set_first_knot_to( current_desireds, L_HIP, QUINTIC_SPLINE, s->time,
			 s->hip_angle[LEFT], s->hip_angled[LEFT], 0.0 );
      break;
    case L_KNEE:
      s->knee_angle_d[LEFT] = s->knee_angle[LEFT];
      s->knee_angled_d[LEFT] = s->knee_angled[LEFT];
      set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, s->time,
			 s->knee_angle[LEFT], s->knee_angled[LEFT], 0.0 );
      break;
    case R_HIP:
      s->hip_angle_d[RIGHT] = s->hip_angle[RIGHT];
      s->hip_angled_d[RIGHT] = s->hip_angled[RIGHT];
      set_first_knot_to( current_desireds, R_HIP, QUINTIC_SPLINE, s->time,
			 s->hip_angle[RIGHT], s->hip_angled[RIGHT], 0.0 );
      break;
    case R_KNEE:
      s->knee_angle_d[RIGHT] = s->knee_angle[RIGHT];
      s->knee_angled_d[RIGHT] = s->knee_angled[RIGHT];
      set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, s->time,
			 s->knee_angle[RIGHT], s->knee_angled[RIGHT], 0.0 );
      break;
    }
}

/*****************************************************************************/

void
reset_desireds( SIM *s )
{
  /* update current DESIRED positions based on DESIRED positions.
   avoids problems with splines */
  set_first_knot_to( current_desireds, L_HIP, QUINTIC_SPLINE, s->time,
		     s->hip_angle_d[LEFT], s->hip_angled_d[LEFT], 0.0 );
  set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, s->time,
		     s->knee_angle_d[LEFT], s->knee_angled_d[LEFT], 0.0 );
  set_first_knot_to( current_desireds, R_HIP, QUINTIC_SPLINE, s->time,
		     s->hip_angle_d[RIGHT], s->hip_angled_d[RIGHT], 0.0 );
  set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, s->time,
		     s->knee_angle_d[RIGHT], s->knee_angled_d[RIGHT], 0.0 );
}

/*****************************************************************************/

void
set_feedforward( SIM *s, int dof, double value )
{
  switch( dof )
    {
    case L_HIP:
    case L_HIP_T:
      s->hip_feedforward[LEFT] = value;
      set_first_knot_to( current_desireds, L_HIP_T, QUINTIC_SPLINE, s->time,
			 value, 0.0, 0.0 );
      break;
    case L_KNEE:
    case L_KNEE_T:
      s->knee_feedforward[LEFT] = value;
      set_first_knot_to( current_desireds, L_KNEE_T, QUINTIC_SPLINE, s->time,
			 value, 0.0, 0.0 );
      break;
    case R_HIP:
    case R_HIP_T:
      s->hip_feedforward[RIGHT] = value;
      set_first_knot_to( current_desireds, R_HIP_T, QUINTIC_SPLINE, s->time,
			 value, 0.0, 0.0 );
      break;
    case R_KNEE:
    case R_KNEE_T:
      s->knee_feedforward[RIGHT] = value;
      set_first_knot_to( current_desireds, R_KNEE_T, QUINTIC_SPLINE, s->time,
			 value, 0.0, 0.0 );
      break;
    }
}

/*****************************************************************************/

void
zero_velocity_desireds( SIM *s, int side )
{
  /* When the leg has an impact, adjust DESIRED positions to reflect that. */
  if ( side == LEFT )
    {
      set_first_knot_to( current_desireds, L_HIP, QUINTIC_SPLINE, s->time,
			 s->hip_angle_d[LEFT], 0.0, 0.0 );
      set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, s->time,
			 s->knee_angle_d[LEFT], 0.0, 0.0 );
    }
  if ( side == RIGHT )
    {
      set_first_knot_to( current_desireds, R_HIP, QUINTIC_SPLINE, s->time,
			 s->hip_angle_d[RIGHT], 0.0, 0.0 );
      set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, s->time,
			 s->knee_angle_d[RIGHT], 0.0, 0.0 );
    }
}

/*****************************************************************************/
/*****************************************************************************/

void
run_trajectory( SIM *s, float time, int dof )
{
  float position, velocity, acceleration;

  if ( !lookup_spline3( desired_trajectory, dof, time,
			current_desireds,
			&position,
			&velocity,
			&acceleration ) )
    {
      printf( "Lookup failure.\n" );
      exit( -1 );
    }
  /* update current desired position */
  set_first_knot_to( current_desireds, dof, QUINTIC_SPLINE, time,
		     position, velocity, acceleration );
  switch( dof )
    {
    case L_HIP:
      s->hip_angle_d[LEFT] = position;
      s->hip_angled_d[LEFT] = velocity;
      s->hip_angledd_d[LEFT] = acceleration;
      break;
    case L_KNEE:
      s->knee_angle_d[LEFT] = position;
      s->knee_angled_d[LEFT] = velocity;
      break;
    case R_HIP:
      s->hip_angle_d[RIGHT] = position;
      s->hip_angled_d[RIGHT] = velocity;
      s->hip_angledd_d[RIGHT] = acceleration;
      break;
    case R_KNEE:
      s->knee_angle_d[RIGHT] = position;
      s->knee_angled_d[RIGHT] = velocity;
      break;
    case L_HIP_T:
      s->hip_feedforward[LEFT] = position;
      break;
    case L_KNEE_T:
      s->knee_feedforward[LEFT] = position;
      break;
    case R_HIP_T:
      s->hip_feedforward[RIGHT] = position;
      break;
    case R_KNEE_T:
      s->knee_feedforward[RIGHT] = position;
      break;
    }
}

/*****************************************************************************/

void
run_PD_servo( SIM *s, int dof )
{

  switch( dof )
    {
    case L_HIP:
      s->hip_command[LEFT] = 
	- s->k_hip*( s->hip_angle[LEFT] - s->hip_angle_d[LEFT] )
	- s->b_hip*( s->hip_angled[LEFT] - s->hip_angled_d[LEFT] )
	+ s->hip_feedforward[LEFT];
      break;
    case R_HIP:
      s->hip_command[RIGHT] = 
	- s->k_hip*( s->hip_angle[RIGHT] - s->hip_angle_d[RIGHT] )
	- s->b_hip*( s->hip_angled[RIGHT] - s->hip_angled_d[RIGHT] )
	+ s->hip_feedforward[RIGHT];
      break;
    case L_KNEE:
      s->knee_command[LEFT] = 
	- s->k_knee*( s->knee_angle[LEFT] - s->knee_angle_d[LEFT] )
	- s->b_knee*( s->knee_angled[LEFT] - s->knee_angled_d[LEFT] )
	+ s->knee_feedforward[LEFT];
      break;
    case R_KNEE:
      s->knee_command[RIGHT] = 
	- s->k_knee*( s->knee_angle[RIGHT] - s->knee_angle_d[RIGHT] )
	- s->b_knee*( s->knee_angled[RIGHT] - s->knee_angled_d[RIGHT] )
	+ s->knee_feedforward[RIGHT];
    }
}

/*****************************************************************************/

void
run_PD_servo_wrt_vertical( SIM *s, int dof )
{

  switch( dof )
    {
    case L_HIP:
      s->hip_command[LEFT] = 
	- s->k_hip*( s->hip_angle[LEFT] + s->pitch - s->hip_angle_d[LEFT] )
	- s->b_hip*( s->hip_angled[LEFT] + s->pitchd - s->hip_angled_d[LEFT] )
	+ s->hip_feedforward[LEFT];
      break;
    case R_HIP:
      s->hip_command[RIGHT] = 
	- s->k_hip*( s->hip_angle[RIGHT] + s->pitch - s->hip_angle_d[RIGHT] )
	- s->b_hip*( s->hip_angled[RIGHT] + s->pitchd - s->hip_angled_d[RIGHT] )
	+ s->hip_feedforward[RIGHT];
      break;
    default:
      fprintf( stderr, "Unknown dof %d in run_PD_servo_wrt_vertical\n", dof );
      exit( -1 );
    }
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

int waiting( SIM *s )
{
  if ( s->time >= s->wait_duration )
    {
      launch1( s );
      return;
    }
  /* desired position already set by initialization */
  run_PD_servo( s, L_HIP );
  run_PD_servo( s, L_KNEE );
  run_PD_servo( s, R_HIP );
  run_PD_servo( s, R_KNEE );
}

/*****************************************************************************/
/*****************************************************************************/
/* Bend legs */

int launch1( SIM *s )
{
  if ( s->controller_state != LAUNCH1 )
    {
      s->controller_state = LAUNCH1;
      s->state_start_time = s->time;
      if ( print )
	printf( "LAUNCH1 %g\n", s->time );
      clear_all_knots( desired_trajectory, N_BIPED_DOFS );
      add_knot_point( desired_trajectory, L_HIP, QUINTIC_SPLINE, 
		      s->l1_movement_duration + s->time,
		      s->l1_lhip, 0.0, 0.0 );
      add_knot_point( desired_trajectory, L_KNEE, QUINTIC_SPLINE, 
		      s->l1_movement_duration + s->time,
		      s->l1_lknee, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_HIP, QUINTIC_SPLINE, 
		      s->l1_movement_duration + s->time,
		      s->l1_rhip, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE, QUINTIC_SPLINE, 
		      s->l1_movement_duration + s->time,
		      s->l1_rknee, 0.0, 0.0 );
      add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
		      s->l1_movement_duration + s->time,
		      s->l1_knee_feedforward, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
		      s->l1_movement_duration + s->time,
		      s->l1_knee_feedforward, 0.0, 0.0 );
    }
  else if ( s->state_elapsed_time >= s->l1_duration )
    {
      launch2( s );
      return;
    }
  run_trajectory( s, s->time, L_HIP );
  run_trajectory( s, s->time, L_KNEE );
  run_trajectory( s, s->time, R_HIP );
  run_trajectory( s, s->time, R_KNEE );
  run_trajectory( s, s->time, L_KNEE_T );
  run_trajectory( s, s->time, R_KNEE_T );
  run_PD_servo( s, L_HIP );
  run_PD_servo( s, L_KNEE );
  run_PD_servo( s, R_HIP );
  run_PD_servo( s, R_KNEE );
}

/*****************************************************************************/
/*****************************************************************************/
/* Push off with right leg */

int launch2( SIM *s )
{
  if ( s->controller_state != LAUNCH2 )
    {
      s->controller_state = LAUNCH2;
      s->state_start_time = s->time;
      if ( print )
	printf( "LAUNCH2 %g\n", s->time );
      clear_all_knots( desired_trajectory, N_BIPED_DOFS );
      add_knot_point( desired_trajectory, R_HIP, QUINTIC_SPLINE, 
		      s->l2_movement_duration + s->time,
		      s->l2_rhip, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE, QUINTIC_SPLINE, 
		      s->l2_movement_duration + s->time,
		      s->l2_rknee, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
		      s->l2_movement_duration + s->time,
		      0.0, 0.0, 0.0 );
    }
  else if ( s->gndforce[RIGHT][ZZ] < s->l2_z_force_threshold ||
	    s->gndforce[RIGHT][XX] < 0.1 )
    {
      launch3( s );
      return;
    }
  run_trajectory( s, s->time, L_HIP );
  run_trajectory( s, s->time, L_KNEE );
  run_trajectory( s, s->time, R_HIP );
  run_trajectory( s, s->time, R_KNEE );
  run_trajectory( s, s->time, R_KNEE_T );
  /*
  run_trajectory( s, s->time, L_KNEE_T );
  */
  run_PD_servo( s, L_HIP );
  run_PD_servo( s, L_KNEE );
  run_PD_servo( s, R_HIP );
  run_PD_servo( s, R_KNEE );
}

/*****************************************************************************/
/*****************************************************************************/
/* swing right leg */

int launch3( SIM *s )
{
  if ( s->controller_state != LAUNCH3 )
    {
      s->controller_state = LAUNCH3;
      s->state_start_time = s->time;
      top_detected = 0;
      if ( print )
	printf( "LAUNCH3 %g\n", s->time );
      set_feedforward( s, R_KNEE_T, 0.0 );
      clear_all_knots( desired_trajectory, N_BIPED_DOFS );
      add_knot_point( desired_trajectory, R_HIP, QUINTIC_SPLINE, 
		      s->l3_rhip1_time*s->l3_duration + s->time,
		      s->l3_rhip1, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_HIP, QUINTIC_SPLINE, 
		      s->l3_rhip2_time*s->l3_duration + s->time,
		      s->l3_rhip2, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE, QUINTIC_SPLINE, 
		      s->l3_rknee1_time*s->l3_duration + s->time,
		      s->l3_rknee1, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE, QUINTIC_SPLINE, 
		      s->l3_rknee2_time*s->l3_duration + s->time,
		      s->l3_rknee2, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE, QUINTIC_SPLINE, 
		      s->l3_rknee3_time*s->l3_duration + s->time,
		      s->l3_rknee3, 0.0, 0.0 );
      /* swing left leg back NOT USED */
      /*
      add_knot_point( desired_trajectory, L_HIP, QUINTIC_SPLINE, 
		      s->l3_lhip_time*s->l3_duration + s->time,
		      s->l3_lhip, 0.0, 0.0 );
      */
      /*
      add_knot_point( desired_trajectory, L_KNEE, QUINTIC_SPLINE, 
		      s->stance_knee_top_time + s->time,
		      s->stance_knee_top, 0.0, 0.0 );
      */
    }
  /* Check for impact */
  else if ( ( s->time - s->state_start_time > 0.1 ) &&
	    ( s->gndforce[RIGHT][ZZ] > 0.0 ||
	      s->gndforce[RIGHT][XX] > 0.0 ) )
    {
      /*
      launch4( s );
      */
      lr_stance( s );
      return;
    }

  /* if after top, extend L knee, manage FF */
  if ( !top_detected )
    {
      if ( s->body_abs_angle[LEFT] >= 0 )
	{
	  if ( print )
	    printf( "Launch3 velocity: %g at %g\n", s->hipd[XX], s->time );
	  s->top_velocity = s->hipd[XX];
	  top_detected = 1;
	  /* zero any velocity */
	  set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, s->time,
			     s->knee_angle_d[LEFT], 0.0, 0.0 );
	  add_knot_point( desired_trajectory, L_KNEE, QUINTIC_SPLINE, 
			  s->pushoff_duration + s->time, 0.0, 0.0, 0.0 );
	  /* zero stance knee feedforward */
	  set_feedforward( s, L_KNEE_T, s->knee_feedforward[LEFT] );
	  add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
			  s->pushoff_duration + s->time, 0.0, 0.0, 0.0 );
	  /* fix spline problem */
	  add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
			  s->pushoff_duration + 0.01 + s->time, 0.0, 0.0, 0.0 );
	}
    }

  /* stance (L) HIP tries to keep pitch = pitch_d */
  /*
  run_trajectory( s, s->time, L_HIP );
  run_PD_servo( s, L_HIP );
  */
  run_trajectory( s, s->time, L_KNEE );
  run_trajectory( s, s->time, L_KNEE_T );
  run_trajectory( s, s->time, R_HIP );
  run_trajectory( s, s->time, R_KNEE );
  run_trajectory( s, s->time, R_KNEE_T );
  run_PD_servo( s, L_KNEE );
  run_PD_servo( s, R_KNEE );
  /*
  run_PD_servo_wrt_vertical( s, R_HIP );
  */
  s->hip_command[RIGHT] = 
    - s->k_hip*( s->hip_angle[RIGHT] + s->pitch - s->hip_angle_d[RIGHT] )
    - s->b_hip*( s->hip_angled[RIGHT] + s->pitchd - s->hip_angled_d[RIGHT] )
    + s->hip_I_model*s->hip_angledd_d[RIGHT]
    + s->hip_feedforward[RIGHT];
  s->hip_command[LEFT] = 
    + s->k_pitch*( s->pitch - s->pitch_d )
    + s->b_pitch*( s->pitchd - s->pitchd_d )
    + s->hip_command[RIGHT];
}

/*****************************************************************************/

int setup_swing_knots( SIM *s, int swing_hip, int swing_knee, double time )
{
  float desired_speed; /*0.2 to 0.5 */
  float hip_target;
  float knee_target;

  /* Convert from average desired speed to speed at top */
  desired_speed = 0.8214*s->desired_speed + 0.0227;

  clear_knots( desired_trajectory, swing_hip );
  clear_knots( desired_trajectory, swing_knee );

  add_knot_point( desired_trajectory, swing_hip, QUINTIC_SPLINE, 
		  s->swing_hip1_time*s->swing_time + time,
		  s->swing_hip1, 0.0, 0.0 );

  /* gain up to 2 */
  hip_target = s->swing_hip2 + s->hip_gain*(s->hipd[XX] - desired_speed);
  if ( hip_target > s->hip_target_max )
    hip_target = s->hip_target_max;
  if ( hip_target < s->swing_hip1 )
    hip_target = s->swing_hip1;
  add_knot_point( desired_trajectory, swing_hip, QUINTIC_SPLINE, 
		  s->swing_time + time,
		  hip_target, 0.0, 0.0 );

  add_knot_point( desired_trajectory, swing_knee, QUINTIC_SPLINE, 
		  s->swing_knee1_time*s->swing_time + time,
		  s->swing_knee1, 0.0, 0.0 );
  add_knot_point( desired_trajectory, swing_knee, QUINTIC_SPLINE, 
		  s->swing_knee2_time*s->swing_time + time,
		  s->swing_knee2, 0.0, 0.0 );

  /* Gain up to 1 */
  knee_target = s->swing_knee3 + s->knee_gain*(s->hipd[XX] - desired_speed);
  if ( knee_target < s->knee_target_min )
    knee_target = s->knee_target_min;
  if ( knee_target > s->knee_target_max )
    knee_target = s->knee_target_max;
  add_knot_point( desired_trajectory, swing_knee, QUINTIC_SPLINE, 
		  s->swing_knee3_time*s->swing_time + time,
		  knee_target, 0.0, 0.0 );
}

/*****************************************************************************/

/* Failed attempt to modulate stance knee top */
double stance_knee_top;
double stance_knee_top_gain = 0.0; /* 2.0? */
double desired_speed = 0.3;
double stance_knee_top_min = 0;
double stance_knee_top_max = 0.5;

int l_swing( SIM *s )
{
  if ( s->controller_state != L_SWING )
    {
      s->controller_state = L_SWING;
      s->state_start_time = s->time;
      top_detected = 0; /* reset top detector. */
      if ( print )
	printf( "L_SWING %g\n", s->time );
      last_l_foot_x = s->foot[LEFT][XX];
      reset_desireds_based_on_actuals( s, L_HIP );
      set_feedforward( s, L_KNEE_T, 0.0 );

      /* Handle stance knee. */
      stance_knee_top = s->stance_knee_top 
	+ stance_knee_top_gain*(s->hipd[XX] - desired_speed);
      if ( stance_knee_top > stance_knee_top_max )
	stance_knee_top = stance_knee_top_max;
      if ( stance_knee_top < stance_knee_top_min )
	stance_knee_top = stance_knee_top_min;
      add_knot_point( desired_trajectory, R_KNEE, QUINTIC_SPLINE, 
		      s->stance_knee_top_time + s->time,
		      stance_knee_top, 0.0, 0.0 );
      /*
      printf( "l_swing stance_knee_top: %g: %g %g\n", s->time,
	      stance_knee_top, s->hipd[XX] );
      */
      set_feedforward( s, R_KNEE_T, s->knee_feedforward[RIGHT] );
      add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_peak_time + s->time, s->knee_ff_peak, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_time + s->time, s->knee_ff, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_time + 0.01 + s->time, s->knee_ff, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_time + 0.02 + s->time, s->knee_ff, 0.0, 0.0 );
      add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_time + 0.03 + s->time, s->knee_ff, 0.0, 0.0 );
    }
  else if ( s->gndforce[LEFT][ZZ] > 0.0 && s->state_elapsed_time > s->swing_time/2 )
    {
      rl_stance( s );
      return;
    }

  /* if after top, extend stance knee, manage FF */
  if ( !top_detected )
    {
      setup_swing_knots( s, L_HIP, L_KNEE, s->state_start_time );

      if ( s->body_abs_angle[RIGHT] >= 0 )
	{
	  if ( print )
	    printf( "l_swing velocity: %g at %g\n", s->hipd[XX], s->time );
	  s->top_velocity = s->hipd[XX];
	  top_detected = 1;
	  clear_knots( desired_trajectory, R_KNEE );
	  /* zero any velocity */
	  set_first_knot_to( current_desireds, R_KNEE, QUINTIC_SPLINE, s->time,
			     s->knee_angle_d[RIGHT], 0.0, 0.0 );
	  add_knot_point( desired_trajectory, R_KNEE, QUINTIC_SPLINE, 
			  s->pushoff_duration + s->time, 0.0, 0.0, 0.0 );
	  /* zero stance knee feedforward */
	  clear_knots( desired_trajectory, R_KNEE_T );
	  add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
			  0.01 + s->time, s->knee_feedforward[RIGHT], 0.0, 0.0 );
	  add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
			  0.02 + s->time, s->knee_feedforward[RIGHT], 0.0, 0.0 );
	  add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
			  s->pushoff_duration + s->time, 0.0, 0.0, 0.0 );
	  add_knot_point( desired_trajectory, R_KNEE_T, QUINTIC_SPLINE, 
			  s->pushoff_duration + 0.1 + s->time, 0.0, 0.0, 0.0 );
	}
    }

  /* stance hip tries to keep pitch = pitch_d */
  run_trajectory( s, s->time, L_HIP );
  run_trajectory( s, s->time, L_KNEE );
  run_trajectory( s, s->time, L_KNEE_T );
  run_trajectory( s, s->time, R_KNEE );
  run_trajectory( s, s->time, R_KNEE_T );
  run_PD_servo( s, R_KNEE );
  run_PD_servo( s, L_KNEE );
  /*
  run_trajectory( s, s->time, R_HIP );
  run_PD_servo( s, R_HIP );
  run_PD_servo_wrt_vertical( s, L_HIP );
  */
  s->hip_command[LEFT] = 
    - s->k_hip*( s->hip_angle[LEFT] + s->pitch - s->hip_angle_d[LEFT] )
    - s->b_hip*( s->hip_angled[LEFT] + s->pitchd - s->hip_angled_d[LEFT] )
    + s->hip_I_model*s->hip_angledd_d[LEFT]
    + s->hip_feedforward[LEFT];
  s->hip_command[RIGHT] = 
    + s->k_pitch*( s->pitch - s->pitch_d )
    + s->b_pitch*( s->pitchd - s->pitchd_d )
    + s->hip_command[LEFT];
}

/*****************************************************************************/

int rl_stance( SIM *s )
{
  if ( s->controller_state != RL_STANCE )
    {
      s->controller_state = RL_STANCE;
      s->state_start_time = s->time;
      if ( print )
	printf( "RL_STANCE %g\n", s->time );
      zero_velocity_desireds( s, LEFT );
    }
  if ( s->gndforce[RIGHT][ZZ] < 0.1 ||
       s->gndforce[RIGHT][XX] < 0.1 )
    {
      r_swing( s );
      return;
    }
  /* Zero hip torques */
  s->hip_command[RIGHT] = 0;
  s->hip_command[LEFT] = 0;
  run_PD_servo( s, L_KNEE );
  run_PD_servo( s, R_KNEE );
}

/*****************************************************************************/

int r_swing( SIM *s )
{
  if ( s->controller_state != R_SWING )
    {
      s->controller_state = R_SWING;
      s->state_start_time = s->time;
      top_detected = 0; /* reset top detector. */
      if ( print )
	printf( "R_SWING %g\n", s->time );
      last_r_foot_x = s->foot[RIGHT][XX];
      reset_desireds_based_on_actuals( s, R_HIP );
      set_feedforward( s, R_KNEE_T, 0.0 );

      /* Handle stance knee. */
      stance_knee_top = s->stance_knee_top 
	+ stance_knee_top_gain*(s->hipd[XX] - desired_speed);
      if ( stance_knee_top > stance_knee_top_max )
	stance_knee_top = stance_knee_top_max;
      if ( stance_knee_top < stance_knee_top_min )
	stance_knee_top = stance_knee_top_min;
      add_knot_point( desired_trajectory, L_KNEE, QUINTIC_SPLINE, 
		      s->stance_knee_top_time + s->time,
		      stance_knee_top, 0.0, 0.0 );
      /*
      printf( "r_swing stance_knee_top: %g: %g %g\n", s->time,
	      stance_knee_top, s->hipd[XX] );
      */

      set_feedforward( s, L_KNEE_T, s->knee_feedforward[LEFT] );
      add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_peak_time + s->time, s->knee_ff_peak, 0.0, 0.0 );
      add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_time + s->time, s->knee_ff, 0.0, 0.0 );
      add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_time + 0.01 + s->time, s->knee_ff, 0.0, 0.0 );
      add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_time + 0.02 + s->time, s->knee_ff, 0.0, 0.0 );
      add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
		      s->knee_ff_time + 0.03 + s->time, s->knee_ff, 0.0, 0.0 );
    }
  else if ( s->gndforce[RIGHT][ZZ] > 0.0 && s->state_elapsed_time > s->swing_time/2 )
    {
      lr_stance( s );
      return;
    }

  /* if after top, extend stance knee, manage FF */
  if ( !top_detected )
    {
      setup_swing_knots( s, R_HIP, R_KNEE, s->state_start_time );

      if ( s->body_abs_angle[LEFT] >= 0 )
	{
	  if ( print )
	    printf( "r_swing velocity: %g at %g\n", s->hipd[XX], s->time );
	  s->top_velocity = s->hipd[XX];
	  top_detected = 1;
	  clear_knots( desired_trajectory, L_KNEE );
	  /* zero any velocity */
	  set_first_knot_to( current_desireds, L_KNEE, QUINTIC_SPLINE, s->time,
			     s->knee_angle_d[LEFT], 0.0, 0.0 );
	  add_knot_point( desired_trajectory, L_KNEE, QUINTIC_SPLINE, 
			  s->pushoff_duration + s->time, 0.0, 0.0, 0.0 );
	  /* zero stance knee feedforward */
	  clear_knots( desired_trajectory, L_KNEE_T );
	  add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
			  0.01 + s->time, s->knee_feedforward[LEFT], 0.0, 0.0 );
	  add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
			  0.02 + s->time, s->knee_feedforward[LEFT], 0.0, 0.0 );
	  add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
			  s->pushoff_duration + s->time, 0.0, 0.0, 0.0 );
	  add_knot_point( desired_trajectory, L_KNEE_T, QUINTIC_SPLINE, 
			  s->pushoff_duration + 0.1 + s->time, 0.0, 0.0, 0.0 );
	}
    }

  /* stance hip tries to keep pitch = pitch_d */
  run_trajectory( s, s->time, R_HIP );
  run_trajectory( s, s->time, R_KNEE );
  run_trajectory( s, s->time, R_KNEE_T );
  run_trajectory( s, s->time, L_KNEE );
  run_trajectory( s, s->time, L_KNEE_T );
  run_PD_servo( s, L_KNEE );
  run_PD_servo( s, R_KNEE );
  /*
  run_PD_servo_wrt_vertical( s, R_HIP );
  */
  s->hip_command[RIGHT] = 
    - s->k_hip*( s->hip_angle[RIGHT] + s->pitch - s->hip_angle_d[RIGHT] )
    - s->b_hip*( s->hip_angled[RIGHT] + s->pitchd - s->hip_angled_d[RIGHT] )
    + s->hip_I_model*s->hip_angledd_d[RIGHT]
    + s->hip_feedforward[RIGHT];
  s->hip_command[LEFT] = 
    + s->k_pitch*( s->pitch - s->pitch_d )
    + s->b_pitch*( s->pitchd - s->pitchd_d )
    + s->hip_command[RIGHT];
}

/*****************************************************************************/

int lr_stance( SIM *s )
{
  if ( s->controller_state != LR_STANCE )
    {
      s->controller_state = LR_STANCE;
      s->state_start_time = s->time;
      if ( print )
	printf( "LR_STANCE %g\n", s->time );
      zero_velocity_desireds( s, RIGHT );
    }
  if ( s->gndforce[LEFT][ZZ] < 0.1 ||
       s->gndforce[LEFT][XX] < 0.1 )
    {
      l_swing( s );
      return;
    }
  /* Zero hip torques */
  s->hip_command[LEFT] = 0;
  s->hip_command[RIGHT] = 0;
  run_PD_servo( s, L_KNEE );
  run_PD_servo( s, R_KNEE );
}

/*****************************************************************************/

int freeze( SIM *s )
{
  if ( s->controller_state != FREEZE )
    {
      s->controller_state = FREEZE;
      s->state_start_time = s->time;
      if ( print )
	printf( "FREEZE %g\n", s->time );
    }
  run_PD_servo( s, L_HIP );
  run_PD_servo( s, L_KNEE );
  run_PD_servo( s, R_HIP );
  run_PD_servo( s, R_KNEE );
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

controller( SIM *s )
{
  int i;

  /* Kinematics already computed by SDFAST */
  /* Forces already computed by SDFAST */
  /* Some additional kinematics */
  for( i = 0; i < 2; i++ )
    {
      s->body_abs_angle[i] = 
	atan2( s->hip[XX] - s->foot[i][XX], s->hip[ZZ] - s->foot[i][ZZ] );
      s->thigh_abs_angle[i] = 
	atan2( s->hip[XX] - s->knee[i][XX], s->hip[ZZ] - s->knee[i][ZZ] );
      s->calf_abs_angle[i] = 
	atan2( s->knee[i][XX] - s->foot[i][XX], s->knee[i][ZZ] - s->foot[i][ZZ] );
    }

  s->state_elapsed_time = s->time - s->state_start_time;

  switch( s->controller_state )
    {
    case WAITING:
      waiting( s );
      break;
    case LAUNCH1:
      launch1( s );
      break;
    case LAUNCH2:
      launch2( s );
      break;
    case LAUNCH3:
      launch3( s );
      break;
    case L_SWING:
      s->step_length = s->foot[RIGHT][XX] - last_r_foot_x;
      l_swing( s );
      break;
    case RL_STANCE:
      s->step_length = s->foot[LEFT][XX] - last_l_foot_x;
      rl_stance( s );
      break;
    case R_SWING:
      s->step_length = s->foot[LEFT][XX] - last_l_foot_x;
      r_swing( s );
      break;
    case LR_STANCE:
      s->step_length = s->foot[RIGHT][XX] - last_r_foot_x;
      lr_stance( s );
      break;
    case ZERO_TORQUES:
      break;
    case FREEZE:
      freeze( s );
      break;
    default:
      fprintf( stderr, "Unknown state %d in controller.\n",
	       s->controller_state );
      exit( -1 );
    }

  s->last_time = s->time;

  update_score( s );

  /* Add random noise to torques */
  s->hip_torque[LEFT] = s->hip_command[LEFT] + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  s->knee_torque[LEFT] = s->knee_command[LEFT] + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  s->hip_torque[RIGHT] = s->hip_command[RIGHT] + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);
  s->knee_torque[RIGHT] = s->knee_command[RIGHT] + s->rand_scale*((2.0*rand())/RAND_MAX - 1.0);

  if ( s->controller_state == ZERO_TORQUES )
    {
      s->hip_torque[LEFT] = 0;
      s->knee_torque[LEFT] = 0;
      s->hip_torque[RIGHT] = 0;
      s->knee_torque[RIGHT] = 0;
    }
}

/*****************************************************************************/
