/****************************************************************
Golf program with simple planner.

Assumes walls are rectangles alingned with the x and y axis

****************************************************************/

#include <math.h>
#include <string.h>
#include <ode/ode.h>
#include <drawstuff/drawstuff.h>
#include "texturepath.h"

/****************************************************************/

#ifdef _MSC_VER
#pragma warning(disable:4244 4305)  // for VC++, no precision loss complaints
#endif

/****************************************************************/

#define TRUE 1
#define FALSE 0

#define TIME_STEP 0.01

#define BALL_DIAMETER 0.035
#define HOLE_RADIUS 0.04
#define FLOOR_HEIGHT (3*BALL_DIAMETER)
#define WALL_HEIGHT FLOOR_HEIGHT

#define WINDOW_WIDTH 700
#define WINDOW_HEIGHT 500

#define FUDGE1 0.0001 // ball is actually launched in air
#define FUDGE2 0.0001 // space between floor tiles

// used to allocate space for floor tiles and walls
#define MAX_N_GEOMS 100

// object types
#define BALL 0
#define FLOOR 1
#define WALL 2
#define PLANE 2
#define UNKNOWN -1

/****************************************************************/

// select correct drawing functions (we are using double version)
#ifdef dDOUBLE
#define dsDrawBox dsDrawBoxD
#define dsDrawSphere dsDrawSphereD
#define dsDrawCylinder dsDrawCylinderD
#define dsDrawCapsule dsDrawCapsuleD
#define dsDrawConvex dsDrawConvexD
#define dsDrawTriangle dsDrawTriangleD
#endif

/****************************************************************/
// Typedefs

typedef struct rect
{
  double min_x;
  double min_y;
  double max_x;
  double max_y;
}
  RECT;

typedef struct hole
{
  RECT floor_size;

  dReal hole_x;
  dReal hole_y;

  dReal tee_x;
  dReal tee_y;

  // these are static ODE objects: just geometries
  dGeomID floor_tiles[MAX_N_GEOMS];
  int n_floor_tiles;
  RECT wall_rectangles[MAX_N_GEOMS];
  dGeomID the_walls[MAX_N_GEOMS];
  int n_walls;
}
  HOLE;

/****************************************************************/
// Globals

static dReal time = 0; // what time is it

static int move_flag = 0;  // has ball just been moved?

// ODE dynamics and collision objects
static dWorldID world;
static dSpaceID space;
static dJointGroupID contactgroup;

static dGeomID plane_geom; // default floor for simulation

// the ODE ball
static dBodyID ball_body;
static dGeomID ball_geom;

// the current hole
HOLE current_hole;

/****************************************************************/

// This function is called at the start of the simulation
//  to set up the point of view of the camera. 
static void start()
{
  float s = 1.5;
  float xyz[3] = {s*1.5f,s*1.5f,s*1.75f};
  float hpr[3] = {-150.000f,-50.0000f,0.0000f};
  dsSetViewpoint (xyz,hpr);
}

/****************************************************************/

void identify_contact( dGeomID o, int *type, const char **str, HOLE *h )
{
  int i;

  if ( o == plane_geom )
    {
      *type = PLANE;
      *str = "plane";
      return;
    }
  if ( o == ball_geom )
    {
      *type = BALL;
      *str = "ball";
      return;
    }
  for ( i = 0; i < h->n_floor_tiles; i++ )
    if ( o == h->floor_tiles[i] )
      {
	*type = FLOOR;
	*str = "floor";
	return;
      }
  for ( i = 0; i < h->n_walls; i++ )
    if ( o == h->the_walls[i] )
      {
	*type = WALL;
	*str = "wall";
	return;
      }
  *type = UNKNOWN;
  *str = "unknown";
}

/****************************************************************/

/*
When the collision system detects that two objects are colliding, it
calls this routine which determines the points of contact and creates
temporary joints. The surface parameters of the joint (friction,
bounce velocity, CFM, etc) are also set here.
*/
  // this is called by dSpaceCollide when two objects in space are
  // potentially colliding.
static void nearCallback( void *data, dGeomID o1, dGeomID o2 )
{
  dBodyID b1 = dGeomGetBody( o1 );
  dBodyID b2 = dGeomGetBody( o2 );
  dContact contact;
  int o1_type, o2_type, c_type;
  const char *o1_string, *o2_string, *c_string;

  // Need to identify contact:
  identify_contact( o1, &o1_type, &o1_string, &current_hole );
  identify_contact( o2, &o2_type, &o2_string, &current_hole );
  // printf( "%s %s\n", o1_string, o2_string );
  // give up if unknown object involved
  if ( o1_type == UNKNOWN )
    return;
  if ( o2_type == UNKNOWN )
    return;
  if ( o1_type == BALL )
    {
      c_type = o2_type;
      c_string = o2_string;
    }
  else if ( o2_type == BALL )
    {
      c_type = o1_type;
      c_string = o1_string;
    }
  else 
    return; // no ball involved, give up.

  if ( c_type == FLOOR )
    {
      contact.surface.mode = dContactBounce | dContactSoftCFM;
      // friction parameter
      contact.surface.mu = 0.5;
      // bounce is the amount of "bouncyness".
      contact.surface.bounce = 0.1; // 0.9 in bounce
      // bounce_vel is the minimum incoming velocity to cause a bounce
      contact.surface.bounce_vel = 0.01;
      // constraint force mixing parameter
      contact.surface.soft_cfm = 0.01;  // 0.001 in bounce
    }
  else if ( c_type == WALL )
    {
      contact.surface.mode = dContactBounce | dContactSoftCFM;
      // friction parameter
      contact.surface.mu = 1.0;
      // bounce is the amount of "bouncyness".
      contact.surface.bounce = 0.9; // 0.9 in bounce
      // bounce_vel is the minimum incoming velocity to cause a bounce
      contact.surface.bounce_vel = 0.01;
      // constraint force mixing parameter
      contact.surface.soft_cfm = 0.01;  // 0.001 in bounce
    }
  else if ( c_type == PLANE )
    {
      contact.surface.mode = dContactBounce | dContactSoftCFM;
      // friction parameter
      contact.surface.mu = dInfinity;
      // bounce is the amount of "bouncyness".
      contact.surface.bounce = 0.5; // 0.9 in bounce
      // bounce_vel is the minimum incoming velocity to cause a bounce
      contact.surface.bounce_vel = 0.1;
      // constraint force mixing parameter
      contact.surface.soft_cfm = 0.01;  // 0.001 in bounce
    }
  else
    return; // no floor or wall involved, give up

  if (int numc = dCollide (o1,o2,1,&contact.geom,sizeof(dContact)))
    {
      dJointID c = dJointCreateContact (world,contactgroup,&contact);
      dJointAttach (c,b1,b2);
    }
}

/****************************************************************/
// move the ball away from walls, and nudge it a little bit.

#define NUDGE 0.001
#define MOVE_SIZE 0.05

// this will have problems with multiple close walls or walls at
// non-right angles

int move_ball( dReal *pos, HOLE *h )
{
  int i, j;
  const dReal *wall_pos;
  const dReal *wall_R;
  dReal sides[3];
  int move_count = 0;

  // if just moved don't move it again
  if ( move_flag )
    return 0;

  // printf( "pos1: %g %g\n", pos[0], pos[1] );

  // first randomly move the ball
  for ( i = 0; i < 2; i++ )
    {
      pos[i] += 2*NUDGE*dRandReal() - NUDGE;
    }

  // printf( "pos2: %g %g\n", pos[0], pos[1] );

  // now move it away from any wall
  for ( i = 0; i < h->n_walls; i++ )
    {
      wall_pos = dGeomGetPosition( h->the_walls[i] );
      wall_R = dGeomGetRotation( h->the_walls[i] );
      dGeomBoxGetLengths( h->the_walls[i], sides );
      // ignore rotation R for now.
      for ( j = 0; j < 2; j++ )
	{
	  // make the lengths half lengths, and add the offset
	  sides[j] = sides[j]/2 + MOVE_SIZE;
	}
	  // check if close enough
      if ( fabs( (double) wall_pos[0] - pos[0] ) < sides[0]
	   && fabs( (double) wall_pos[1] - pos[1] ) < sides[1] )
	{
	  if ( sides[0] < sides[1] )
	    {
	      j = 0;
	      /*
	      printf( "contact %d %g %g %g\n",
		      j, pos[j], h->wall_pos[j], sides[j] );
	      */
	    }
	  else
	    {
	      j = 1;
	      /*
	      printf( "contact %d %g %g %g\n",
		      j, pos[j], h->wall_pos[j], sides[j] );
	      */
	    }
	  move_count++;
	  if ( pos[j] > wall_pos[j] )
	    pos[j] = wall_pos[j] + MOVE_SIZE;
	  else
	    pos[j] = wall_pos[j] - MOVE_SIZE;
	  // printf( "Too close: %d %d %g %g\n", i, j, wall_pos[j], pos[j] );
	}
    }
  if ( move_count > 0 )
    printf( "%g: ball moved to %g %g\n", time, pos[0], pos[1] );

  // lift ball a little bit to get it to move
  dBodySetPosition( ball_body, pos[0], pos[1], pos[2] + FUDGE1 );

  if ( move_count > 0 )
    move_flag = 1;

  return move_count;
}

/****************************************************************/
// Here is where we put in our player

void do_stuff( HOLE *h )
{
  int i;
  const dReal *pos;
  const dReal *vel;
  dReal vel_magnitude2 = 0;
  dReal pos_magnitude2 = 0;
  dReal dx, dy, vx, vy, d_magnitude;
  static dReal last_pos[3];
  static int stuck = FALSE;
  dReal new_pos[3];
  static dReal wait_until = 5.0;

  // return if we are stuck.
  if ( stuck )
    return;

  vel = dBodyGetLinearVel( ball_body );
  for ( i = 0; i < 3; i++ )
    vel_magnitude2 += vel[i]*vel[i];

  // if ball is moving return
  if ( vel_magnitude2 > 1e-4 )
    {
      wait_until = time + 5.0;
      return;
    }

  pos = dBodyGetPosition( ball_body );

  if ( pos[2] < FLOOR_HEIGHT )
    {
      printf( "%g: ball in hole at %g %g %g; hole at %g %g\n", time,
	      pos[0], pos[1], pos[2], h->hole_x, h->hole_y );
      stuck = TRUE;
      return;
    }

  // return if we are waiting (allows viewer to see ball stop).
  if ( time < wait_until )
    return;

  printf( "%g: ball at %g %g %g; hole at %g %g\n", time,
	  pos[0], pos[1], pos[2], h->hole_x, h->hole_y );

  for ( i = 0; i < 3; i++ )
    pos_magnitude2 += (pos[i] - last_pos[i])*(pos[i] - last_pos[i]);
  if ( pos_magnitude2 < 1e-7 && move_flag == 0 )
    {
      if ( !stuck )
	{
	  stuck = TRUE;
	  printf( "%g: I am stuck %g\n", time, pos_magnitude2 );
	  return;
	}
      else
	return;
    }

  // move the ball away from walls, and nudge it a little bit.
  for ( i = 0; i < 3; i++ )
    new_pos[i] = pos[i];

  if ( move_ball( new_pos, h ) > 0 )
    {
      for ( i = 0; i < 3; i++ )
	last_pos[i] = new_pos[i];
      wait_until = time + 5.0;
      return;
    }

  for ( i = 0; i < 3; i++ )
    last_pos[i] = new_pos[i];

  dx = h->hole_x - new_pos[0];
  dy = h->hole_y - new_pos[1];
  d_magnitude = 3*sqrt(dx*dx + dy*dy); // limit max velocity to 0.33
  vx = dx/d_magnitude;
  vy = dy/d_magnitude;

  printf( "%g: hit velocity: %g %g\n", time, vx, vy );
  dBodySetLinearVel( ball_body, (dReal) vx, (dReal) vy, 0.0 );

  move_flag = 0;
}

/****************************************************************/

/*
This is the main simulation loop that calls the collision detection
function, steps the simulation, resets the temporary contact joint
group, and redraws the objects at their new position.
*/

// simulation loop
static void simLoop (int pause)
{
  int i;
  const dReal *pos;
  const dReal *R;
  dReal sides[3];

  do_stuff( &current_hole );

  // find collisions and add contact joints
  dSpaceCollide( space, 0, &nearCallback );

  // step the simulation
  dWorldStep( world, TIME_STEP );  

  time += TIME_STEP;

  // remove all contact joints
  dJointGroupEmpty( contactgroup );

  // redraw sphere at new location
  pos = dGeomGetPosition( ball_geom );
  R = dGeomGetRotation( ball_geom );
  dsSetColor (1,0.5,0);
  dsDrawSphere( pos, R, dGeomSphereGetRadius( ball_geom ) );

  // redraw floor
  for ( i = 0; i < current_hole.n_floor_tiles; i++ )
    {
      pos = dGeomGetPosition( current_hole.floor_tiles[i] );
      R = dGeomGetRotation( current_hole.floor_tiles[i] );
      dGeomBoxGetLengths( current_hole.floor_tiles[i], sides );
      dsSetColor( 0.0, 0.75, 0.0 );
      dsDrawBox( pos, R, sides );
    }

  // redraw walls
  for ( i = 0; i < current_hole.n_walls; i++ )
    {
      pos = dGeomGetPosition( current_hole.the_walls[i] );
      R = dGeomGetRotation( current_hole.the_walls[i] );
      dGeomBoxGetLengths( current_hole.the_walls[i], sides );
      dsSetColor( 0.75, 0.5, 0.25 );
      dsDrawBox( pos, R, sides );
    }
}

/****************************************************************/

void update_min_max( HOLE *h, dReal x, dReal y )
{

  if ( x < h->floor_size.min_x )
    h->floor_size.min_x = x;
  if ( x > h->floor_size.max_x )
    h->floor_size.max_x = x;

  if ( y < h->floor_size.min_y )
    h->floor_size.min_y = y;
  if ( y > h->floor_size.max_y )
    h->floor_size.max_y = y;
}

/****************************************************************/

int read_hole_file( char *filename, HOLE *h )
{
  int i;
  FILE *stream;
  char buffer[1000];

  stream = fopen( filename, "r" );
  if ( stream == NULL )
    {
      fprintf( stderr, "Can't open hole file %s", filename );
      exit( -1 );
    }

  h->floor_size.min_x = 1e10;
  h->floor_size.min_y = 1e10;
  h->floor_size.max_x = -1e10;
  h->floor_size.max_y = -1e10;
  h->n_floor_tiles = 0;
  h->n_walls = 0;

  for( ; ; )
    {
      // read keyword
      if ( fscanf( stream, "%s", buffer ) < 1 )
	break; // if we didn't read anything we are done

      // handle a hole x y
      if ( strcmp( buffer, "hole" ) == 0 )
	{
	  if ( fscanf( stream, "%lg%lg", &(h->hole_x), &(h->hole_y) ) < 2 )
	    {
	      fprintf( stderr, "bad hole in hole file %s\n", filename );
	      exit( -1 );
	    }
	  update_min_max( h, h->hole_x, h->hole_y ); 
	  printf( "hole: %g %g\n", h->hole_x, h->hole_y );
	  continue;
	}
      
      // handle a tee x y
      if ( strcmp( buffer, "tee" ) == 0 )
	{
	  if ( fscanf( stream, "%lg%lg", &(h->tee_x), &(h->tee_y) ) < 2 )
	    {
	      fprintf( stderr, "bad tee in hole file %s\n", filename );
	      exit( -1 );
	    }
	  update_min_max( h, h->tee_x, h->tee_y ); 
	  printf( "tee: %g %g\n", h->tee_x, h->tee_y );
	  continue;
	}

      // handle a wall min_x min_y max_x max_y
      if ( strcmp( buffer, "wall" ) == 0 )
	{
	  i = h->n_walls;
	  if ( fscanf( stream, "%lg%lg%lg%lg",
		       &(h->wall_rectangles[i].min_x),
		       &(h->wall_rectangles[i].min_y),
		       &(h->wall_rectangles[i].max_x),
		       &(h->wall_rectangles[i].max_y) ) < 4 )
	    {
	      fprintf( stderr, "bad wall in hole file %s\n", filename );
	      exit( -1 );
	    }
	  update_min_max( h,
			  h->wall_rectangles[i].min_x,
			  h->wall_rectangles[i].min_y );
	  update_min_max( h,
			  h->wall_rectangles[i].max_x,
			  h->wall_rectangles[i].max_y );
	  printf( "wall: %g %g %g %g\n",
		  h->wall_rectangles[i].min_x,
		  h->wall_rectangles[i].min_y,
		  h->wall_rectangles[i].max_x,
		  h->wall_rectangles[i].max_y );
	  h->n_walls++;
	  continue;
	}
      
      fprintf( stderr, "bad keyword %s in hole file %s\n", buffer, filename );
      exit( -1 );
    }

  fclose( stream );
}

/****************************************************************/

void create_bodies( char *filename, HOLE *h )
{
  int i;
  dMass m;

  read_hole_file( filename, h );

  // create ball object
  ball_body = dBodyCreate( world );
  ball_geom = dCreateSphere( space, BALL_DIAMETER/2 );
  dMassSetSphere( &m, 1, BALL_DIAMETER/2 );
  dBodySetMass( ball_body, &m );
  dGeomSetBody( ball_geom, ball_body );
  // set initial position and velocity
  dBodySetPosition( ball_body, h->tee_x, h->tee_y,
		    FLOOR_HEIGHT + BALL_DIAMETER/2 + FUDGE1 );
  dBodySetLinearDamping( ball_body, 1e-4 ); 
  dBodySetLinearDampingThreshold( ball_body, 1e-7 );

  /*
    FROM FAQ:

  How can an immovable body be created?

  In other words, how can you create a body that doesn't move, but
  that interacts with other bodies? The answer is to create a geom
  only, without the corresponding rigid body object. The geom is
  associated with a rigid body ID of zero. Then in the contact
  callback when you detect a collision between two geoms with a
  nonzero body ID and a zero body ID, you can simply pass those two
  IDs to the dJointAttach function as normal. This will create a
  contact between the rigid body and the static environment.
  */

  // create floor objects

  printf( "%g %g\n",
			h->hole_x - h->floor_size.min_x - HOLE_RADIUS - FUDGE2,
				h->floor_size.max_y - h->floor_size.min_y
	  );
  h->floor_tiles[0] = dCreateBox( space,
			h->hole_x - h->floor_size.min_x - HOLE_RADIUS - FUDGE2,
				h->floor_size.max_y - h->floor_size.min_y,
				FLOOR_HEIGHT );
  dGeomSetPosition( h->floor_tiles[0],
		    h->hole_x - HOLE_RADIUS
		    - (h->hole_x - h->floor_size.min_x - HOLE_RADIUS)/2.0,
		    (h->floor_size.max_y + h->floor_size.min_y)/2.0,
		    FLOOR_HEIGHT/2.0 + FUDGE2 );

  printf( "%g %g\n",
			h->floor_size.max_x - h->hole_x - HOLE_RADIUS - FUDGE2,
				h->floor_size.max_y - h->floor_size.min_y
	  );
  h->floor_tiles[1] = dCreateBox( space,
			h->floor_size.max_x - h->hole_x - HOLE_RADIUS - FUDGE2,
				h->floor_size.max_y - h->floor_size.min_y,
				FLOOR_HEIGHT );
  dGeomSetPosition( h->floor_tiles[1],
		    h->hole_x + HOLE_RADIUS
		    + (h->floor_size.max_x - h->hole_x - HOLE_RADIUS)/2.0,
		    (h->floor_size.max_y + h->floor_size.min_y)/2.0,
		    FLOOR_HEIGHT/2.0 + FUDGE2 );

  printf( "%g %g\n",
				2*HOLE_RADIUS,
				h->hole_y - h->floor_size.min_y - HOLE_RADIUS
	  );
  h->floor_tiles[2] = dCreateBox( space,
				2*HOLE_RADIUS,
				h->hole_y - h->floor_size.min_y - HOLE_RADIUS,
				FLOOR_HEIGHT );
  dGeomSetPosition( h->floor_tiles[2],
		    h->hole_x,
		    h->hole_y - HOLE_RADIUS
		    - (h->hole_y - h->floor_size.min_y - HOLE_RADIUS)/2,
		    FLOOR_HEIGHT/2 + FUDGE2 );

  printf( "%g %g\n",
				2*HOLE_RADIUS,
				h->floor_size.max_y - h->hole_y - HOLE_RADIUS
	  );
  h->floor_tiles[3] = dCreateBox( space,
				2*HOLE_RADIUS,
				h->floor_size.max_y - h->hole_y - HOLE_RADIUS,
				FLOOR_HEIGHT );
  dGeomSetPosition( h->floor_tiles[3],
		    h->hole_x,
		    h->hole_y + HOLE_RADIUS
		    + (h->floor_size.max_y - h->hole_y - HOLE_RADIUS)/2,
		    FLOOR_HEIGHT/2 + FUDGE2 );

  h->n_floor_tiles = 4;

  // create walls

  for ( i = 0; i < h->n_walls; i++ )
    {
      printf( "%g %g\n",
		    h->wall_rectangles[i].max_x - h->wall_rectangles[i].min_x,
		    h->wall_rectangles[i].max_y - h->wall_rectangles[i].min_y
	  );
      h->the_walls[i] =
	dCreateBox( space,
		    h->wall_rectangles[i].max_x - h->wall_rectangles[i].min_x,
		    h->wall_rectangles[i].max_y - h->wall_rectangles[i].min_y,
		    WALL_HEIGHT );
      dGeomSetPosition( h->the_walls[i],
		(h->wall_rectangles[i].max_x + h->wall_rectangles[i].min_x)/2,
		(h->wall_rectangles[i].max_y + h->wall_rectangles[i].min_y)/2,
		FLOOR_HEIGHT + WALL_HEIGHT/2 + 2*FUDGE2 );
    }

  // plane that catches balls that leave course
  plane_geom = dCreatePlane( space, 0, 0, 1, 0 );
}


//============================

// called when a key pressed

static void command (int cmd)
{
  float vx, vy;

  // this is a leftover from past demo programs
  /*
  if ( cmd == 's' )
    {
      printf( "s pressed.\n" );
      const dReal *pos = dBodyGetPosition( ball_body );
      printf( "ball at %g %g %g; hole at %g %g\n",
	      pos[0], pos[1], pos[2],
	      current_hole.hole_x, current_hole.hole_y );
      printf( "enter ball velocity as two numbers: -0.05 -0.4\n" );
      scanf( "%g%g", &vx, &vy );
      printf( "you typed: %g %g\n", vx, vy );
      // lift ball a little bit to get it to move
      dBodySetPosition( ball_body, pos[0], pos[1], pos[2] + FUDGE1 );
      dBodySetLinearVel( ball_body, (dReal) vx, (dReal) vy, 0.0 );
    }
  */
}

/****************************************************************/

/*
Main program: When the program starts, the callbacks are set up,
everything is initialized, and then the simulation is started.
*/

int main( int argc, const char **argv )
{
  dReal erp, cfm;
  char *filename;

  if ( argc < 2 )
    filename = (char *) "c0000.txt";
  else
    filename = (char *) (argv[1]);
  printf( "Using hole file %s\n", filename );

  // setup pointers to drawstuff callback functions
  dsFunctions fn;
  fn.version = DS_VERSION;
  fn.start = &start;
  fn.step = &simLoop;
  fn.stop = 0;
  fn.command = &command;
  fn.path_to_textures = DRAWSTUFF_TEXTURE_PATH;
 
  dInitODE ();
  // create world
  world = dWorldCreate( );
  space = dHashSpaceCreate( 0 );
  dWorldSetGravity( world, 0.0, 0.0, -9.81 );
  dWorldSetCFM( world, 1e-5 );
  // dWorldSetERP (dWorldID, dReal erp);
  erp = dWorldGetERP( world );
  cfm = dWorldGetCFM( world );
  /*
  printf( "erp: %g, cfm: %g, kp: %g, kd: %g\n",
	  erp, cfm, erp/(cfm*TIME_STEP), (1 - erp)/cfm );
  */

  contactgroup = dJointGroupCreate( 0 );

  create_bodies( filename, &current_hole );

  // run simulation
  dsSimulationLoop( argc, (char **) argv, WINDOW_WIDTH, WINDOW_HEIGHT, &fn );

  // clean up
  dJointGroupDestroy (contactgroup);
  dSpaceDestroy (space);
  dWorldDestroy (world);
  dCloseODE();
  return 0;
}

/****************************************************************/
