/**********************************************************************

Run a position servo for the left motor.
Tell time by polling a clock instead of using interrupts.
Use a Kalman-filter-based velocity estimator to restore signs to the
 encoder signal, and estimate positions in radians and velocities in
 radians/second.
Print out data concurrently for capture by a terminal program (putty).

Running the servo is organized into "trials".
During a trial, the effect of the servo is gradually increased with a ramp
at the beginning, and gradually decreased with a ramp at the end.

[Ignore] means that you can safely ignore this item in trying to figure
out how this program works.

***********************************************************************/

#include <math.h>

/**********************************************************************/
// Defines

// Some constants
// Only increment integral for errors smaller than this limit
#define APPLY_INTEGRAL_LIMIT 0.1 

/**********************************************************************/
// Global variables

// motor enable
bool go = false;

// Wheel state
float left_wheel_angular_velocity_radians = 0;
float left_wheel_angle_radians = 0;

float angle_desired = 0;

// Motor commands
int left_command;
int right_command;

int my_debug = 0; // useful for debugging

/**********************************************************************/

#include "ignore.h" // [Ignore]

/**********************************************************************/
/**********************************************************************/
// note that int arguments have to be less than +/-32000

void run_trial( float angle_gain, // P (position) gain
                float angle_velocity_gain, // D (velocity) gain
		float angle_integral_gain, // I (integral) gain
                float angle_start, // Start angle
		int delay, // Delay after data collection starts to set goal
		float angle_goal, // goal can be position, or a range for a
		                  // sinusoid (angle_goal*sin())
		float frequency,  // Frequency (negative means no sinusoid)
                unsigned long run_time_limit, // how long to run trial
		int collect_data ) // Should data be collected?
{

#include "run_trial1.h" // [Ignore]

  // print out the angle gains
  if ( collect_data )
    {
      Serial.print( "Gains " );
      Serial.print( angle_gain );
      Serial.print( " " );
      Serial.print( angle_velocity_gain );
      Serial.print( " " );
      Serial.println( angle_integral_gain );
    }

  // print out the goal
  if ( collect_data )
    {
      Serial.print( "Goal " );
      Serial.println( angle_goal );
    }

  // print out the frequency
  if ( collect_data )
    {
      Serial.print( "Frequency " );
      Serial.println( frequency );
    }

  // current goal is current position
  angle_desired = left_wheel_angle_radians;

  // servo loop
  for( ; ; )
    {

#include "run_trial2.h" // [Ignore]

/*****************************************
Servo part */

      // delay is included in move_time
      if ( run_time >= move_time )
        {
          if ( frequency <= 0 )
	    angle_desired = angle_goal; // step response
	  else
	    angle_desired = angle_goal // generate sinusoidal desired angle
	       *sinf( frequency*(2*PI*0.001)*(run_time - move_time) );
  	    // (2*PI*0.001) makes frequency in Hz
        }


      float left_error = left_wheel_angle_radians - angle_desired;
      if ( abs( left_error ) < APPLY_INTEGRAL_LIMIT )
      	angle_error_integral_left -= left_error;

      float left_commandf = 0;
      
      if ( run_time < START_COLLECT )
        { // use safe gain during startup // currently all 0
          left_commandf = - safe_angle_gain*left_error
	                  - safe_angle_velocity_gain	
			    *left_wheel_angular_velocity_radians;
         		  + safe_angle_integral_gain
			    *angle_error_integral_left;
      	}
      else
        {
	  // use the gains given as arguments during data collection
    	  left_commandf = - angle_gain*left_error
	                  - angle_velocity_gain
			    *left_wheel_angular_velocity_radians;
    	                  + angle_integral_gain
			    *angle_error_integral_left;
       	}

      // ramp command up at beginning
      if ( ( run_time < START_COLLECT )
           && ( ramp < (1.0 - RAMP_INCREMENT) ) )
        ramp += RAMP_INCREMENT;

      // ramp command down at end
      if ( run_time > run_time_limit )
        ramp *= RAMP_TURNOFF_FACTOR; // gradually turn motor off

	left_commandf = ramp*left_commandf;

	// This check has to happen before conversion to an integer,
	// since there may be integer overflow.
	if ( left_commandf > MAX_COMMAND )
	  left_commandf = MAX_COMMAND;
	if ( left_commandf < -MAX_COMMAND )
          left_commandf = -MAX_COMMAND;

      motor_left_command( (int) round( left_commandf ) );

/*****************************************
Data printing part */

      // print out debugging info before data collection
      if ( ( debug_print_time > DEBUG_PRINT_INTERVAL )
           && ( ( run_time < START_COLLECT )
	        || ( run_time > run_time_limit ) ) )
        {
          Serial.print( servo_late );
          Serial.print( " " );
          Serial.print( max_servo_late );
          Serial.print( " " );
          Serial.print( left_error );
          Serial.print( " " );
          Serial.println( ramp );
          ProcessCommand();
          debug_print_time -= DEBUG_PRINT_INTERVAL;
        }

      // print out data
      if ( ( run_time >= START_COLLECT ) && ( run_time <= run_time_limit )
           && collect_data )
        {
          if ( !started )
            {
              servo_late = 0;
              max_servo_late = 0;
              started = 1;
              Serial.println( "Data" );
            }
          Serial.print( current_micros );
          Serial.print( " " );
          Serial.print( raw_left_encoder_count );
          Serial.print( " " );
          Serial.print( left_command );
          Serial.print( " " );
          Serial.print( (int) (1000.0*left_wheel_angle_radians) );
          Serial.print( " " );
          Serial.print( (int) (1000.0*left_wheel_angular_velocity_radians) );
          Serial.print( " " );        
          Serial.print( (int) (1000.0*angle_desired) ); 
          Serial.print( " " );        
          Serial.print( (int) (1000.0*angle_error_integral_left) ); 
          Serial.print( " " );
          Serial.println( my_debug );
        }
    }
}

/**********************************************************************/
// this subroutine is repeatedly called

void loop()
{

  // if motor is not enabled check for a command
  if ( !go )
    {
      motor_stop();
      delay(500);
      ProcessCommand();
      return;
    }

  // run trials if motor enabled

  // sinusoidal movement
  run_trial( 2000.0, // P (position) gain
  	     0.0, // D (velocity) gain
	     0.0, // I (integral) gain
  	     0.0, // Start angle
	     100, // Delay after data collection starts to set goal
	     PI, // Goal angle, or range for sinusoid
	     1.0, // Frequency (negative means no sinusoid)
	     10000, // Run time limit in milliseconds
	     1 ); // Should data be collected?

  delay( 1000 ); // take a 1s break

  run_trial( 2000.0, // P (position) gain
  	     0.0, // D (velocity) gain
	     0.0, // I (integral) gain
  	     PI, // Start angle
	     100, // Delay after data collection starts to set goal
	     0.0, // Goal angle, or range for sinusoid
             -1.0, // Frequency (negative means no sinusoid)
	     2000, // Run time limit in milliseconds
	     1 ); // Should data be collected?

  delay( 1000 ); // take a 1s break

  // disable the motor after a series of trials (an experiment)
  Serial.println( "Stop!" );
  go = false;
}
  
/**********************************************************************/
