/**********************************************************************/
// Includes

#include "PinChangeInterrupt.h"
#include "Pins.h"

/**********************************************************************/
// Defines you can safely ignore

// define sampling rate
#define SERVO_INTERVAL 5000 // microseconds

// maximum command to motor	
#define MAX_COMMAND 255

// start data collection after N milliseconds from go [Ignore]
#define START_COLLECT  5000 // milliseconds

// Safety ramp
// Duration of safety ramp turning on motor
#define RAMP_DURATION 1000 // milliseconds
// Safety ramp increment that gradually turns on motor
// at x/sample should take 1s to get to 1.0
#define RAMP_INCREMENT ((SERVO_INTERVAL*1e-3)/RAMP_DURATION)
// multiplier in ramp to turn motor off at end of run
#define RAMP_TURNOFF_FACTOR 0.95

// slop in counting late cycles
#define LATE_CYCLE_SLOP 50

// how often do we print debug information
#define DEBUG_PRINT_INTERVAL 500 // milliseconds, 

// the biggest unsigned 32 bit number
#define MAX_ULONG 0xFFFFFFFF

#define VOLTAGE_CONVERSION (1.1*(10+1.5)/(1.5*1024)) // WTF?

// 780 = 13 counts/rev*gear_ratio = 30*2 (count both up and down transitions)
#define ENCODER_TO_RADIANS (2*PI/780)  
#define ENCODER_VELOCITY_TO_RADIANS (ENCODER_TO_RADIANS*1.0e6/SERVO_INTERVAL)
// 1.0e6 is from SERVO_INTERVAL being in microseconds
// Eventually we should calcuate the actual microseconds between each tick

/**********************************************************************/
// Global variables you can safely ignore

// safety ramp to have smooth startup and shutdown of motor
float ramp = 0;

// For battery voltage measurements
int voltage_raw;
double voltage;

// The encoder interrupt service routines (ISRs) update these variables
volatile unsigned long int raw_right_encoder_count = 0;
volatile unsigned long int raw_left_encoder_count = 0;

// Velocity estimation
// used in velocity estimation
unsigned long int last_raw_left_encoder_count = 0;
// estimated wheel velocity
float left_wheel_angular_velocity_encoder_estimated = 0;
float left_wheel_angle_encoder_estimated = 0;
float a_vel_est = 0.9325;
float b_vel_est = 0.0045;
float k_vel_est = 0.1;

/**********************************************************************/
// Subroutines you can safely ignore

/**********************************************************************/
// Encoder interrupt service routines (ISRs)

// Getting right wheel encoder count [Ignore]
static void right_encoder_isr( void )
{
   raw_right_encoder_count++;
}

// Getting left wheel encoder count [Ignore]
static void left_encoder_isr( void )
{
   raw_left_encoder_count++;
}

/* 
This is based on the PinChangeInterrupt library and the example
program PinChangeInterrupt_LED.

Note that only one wire is connected from each encoder, so we can't figure
out direction.
*/

// [Ignore]
static void encoder_init()
{
  // set pin to input with a pullup
  pinMode( ENCODER_RIGHT_A_PIN, INPUT_PULLUP );

  Serial.print( "Encoder_init " );
  Serial.print( ENCODER_LEFT_A_PIN );
  Serial.print( " " );
  Serial.print( digitalPinToInterrupt(ENCODER_LEFT_A_PIN) );
  Serial.print( " " );
  Serial.print( ENCODER_RIGHT_A_PIN );
  Serial.print( " " );
  Serial.print( digitalPinToInterrupt(ENCODER_RIGHT_A_PIN) );
  Serial.println();
  delay(1000);
  // Left encoder is coming in on pin 2 which is an "interrupt" pin,
  // and attachInterrupt() works
  attachInterrupt( digitalPinToInterrupt(ENCODER_LEFT_A_PIN),
                   left_encoder_isr, CHANGE );
  Serial.print( "left encoder ISR set" );
  Serial.println();
  delay(1000);
  // Right encoder is coming in on pin 4 which is not an "interrupt" pin.
  // Attach PinChangeInterrupt instead.
  attachPCINT( digitalPinToPCINT( ENCODER_RIGHT_A_PIN ),
               right_encoder_isr, CHANGE );
  Serial.print( "right encoder ISR set" );
  Serial.println();
  delay(1000);
}

/*
void read_encoders()
{
  // ISRs set raw encoder counts directly.
}
*/

/**********************************************************************/

void motor_init()
{
  pinMode(AIN1, OUTPUT);
  pinMode(BIN1, OUTPUT);
  pinMode(PWMA_LEFT, OUTPUT);
  pinMode(PWMB_RIGHT, OUTPUT);
  digitalWrite(AIN1, HIGH);
  digitalWrite(BIN1, LOW);
  analogWrite(PWMA_LEFT, 0);
  analogWrite(PWMB_RIGHT, 0);
  pinMode(STBY_PIN, OUTPUT);
  digitalWrite(STBY_PIN, HIGH);
}

/**********************************************************************/

void motor_stop()
{
  digitalWrite(AIN1, HIGH);
  digitalWrite(BIN1, LOW);
  analogWrite(PWMA_LEFT, 0);
  analogWrite(PWMB_RIGHT, 0);
}

/**********************************************************************/

/**********************************************************************/
// Here is how you command the motors

void motor_left_command( int command )
{
  if ( command > MAX_COMMAND )
    command = MAX_COMMAND;
  if ( command < -MAX_COMMAND )
    command = -MAX_COMMAND;
  left_command = command;
  if ( command >= 0 )
    {
      digitalWrite( AIN1, 1 );
      analogWrite( PWMA_LEFT, command );
    }
  else
    {
      digitalWrite( AIN1, 0 );
      analogWrite( PWMA_LEFT, -command );
    }
}

// reverses the sign of "command"
void motor_right_command( int command )
{
  if ( command > MAX_COMMAND )
    command = MAX_COMMAND;
  if ( command < -MAX_COMMAND )
    command = -MAX_COMMAND;
  right_command = command;
  if ( command >= 0 )
    {
      digitalWrite( BIN1, 1 );
      analogWrite( PWMB_RIGHT, command );
    }
  else
    {
      digitalWrite( BIN1, 0 );
      analogWrite( PWMB_RIGHT, -command );
    }
}

/**********************************************************************/
/**********************************************************************/

void setup()
{
  go = false; // start with motors disabled
  
  Serial.begin( 115200 );

  analogReference( INTERNAL ); // obscure analog stuff

  while( !Serial ); // wait for serial stuff to start working
  Serial.println( "servo version 1" );
  delay(1000); // Delay to make sure above message prints out.

  // warm up reading of voltage
  voltage_raw = analogRead(VOL_MEASURE_PIN); // Read voltage value

  motor_init(); // initialize the motors
  Serial.println( "motor_init done." );
  delay(1000); // Delay to make sure above message prints out.

  encoder_init();
  Serial.println( "Initialized encoders" );

  Serial.println( "Wheels should be off the ground."  );
  Serial.println( "Type g <return> to run test, s <return> to stop."  );
  Serial.println( "Typing window is at the top of the Arduino serial monitor window." );
  Serial.println( "Type into the main window of a Putty serial monitor window." );
  delay(1000); // Delay to make sure above message prints out.

  // 2nd read of voltage gets it right.
  voltage_raw = analogRead(VOL_MEASURE_PIN); // Read voltage value
  voltage = VOLTAGE_CONVERSION*voltage_raw;
  Serial.print( "Current voltage check: " );
  Serial.print( voltage );
  Serial.print( " " );
  Serial.println( voltage_raw );
}

/******************************************************************/

// Take user input
void ProcessCommand()
{

  if ( Serial.available() <= 0 )
    return;

  int c = Serial.read();
  switch (c)
    {
      case 'G': case 'g':
        Serial.println( "Go!" );
        go = true;
        break;
      case '\n': case '\r': case '\a': case '\b': case '\e': case '\f': case '\t': case '\v': case ' ': // ignore white space
        break;
      case 'S': case 's':
      default: // anything other than g or G is a stop command
        Serial.println( "Stop!" );
        go = false;
    }
}

/**********************************************************************/
// deal with erratic encoder readings by reading several times and
// finding the median.

unsigned long int read_left_encoder()
{

  unsigned long int rlec1 = raw_left_encoder_count; // try reading.
  unsigned long int rlec2 = raw_left_encoder_count; // try reading again.
  unsigned long int rlec3 = raw_left_encoder_count; // one more time.
  
  my_debug = 0;

  // try 1st pair
  if ( rlec1 == rlec2 ) // readings agree, go with them.
    {
      my_debug = 1;
      return rlec1;
    }

  // try 2nd pair
  if ( rlec2 == rlec3 ) // readings agree, go with them.
    {
      my_debug = 2;
      return rlec2;
    }

  // find median
  if ( rlec1 < rlec2 )
    {
      if ( rlec2 < rlec3 )
        { // rlec1 < rlec2 < rlec3
          my_debug = 3;
	  return rlec2;
        }
      else
        {
          // rlec1 < rlec2 && rlec3 < rlec2
          if ( rlec1 < rlec3 )
	    { // rlec1 < rlec3 < rlec2
	      my_debug = 4;
	      return rlec3;
	    }
          else
	    { // rlec3 < rlec1 < rlec2
	      my_debug = 5;
	      return rlec1;
	    }
        }
    }
  else // rlec2 < rlec1
    {
      if ( rlec1 < rlec3 )
        { // rlec2 < rlec1 < rlec3
          my_debug = 6;
	  return rlec1;
        }
      else
        { // rlec2 < rlec1 && rlec3 < rlec1
          if ( rlec2 < rlec3 )
	    { // rlec2 < rlec3 < rlec1
	      my_debug = 7;
	      return rlec3;
	    }
          else
	    { // rlec3 < rlec2 < rlec1
	      my_debug = 8;
	      return rlec2;
	    }
        }
    }

  // should never get here
  my_debug = 9;
  return rlec3;
}     

/**********************************************************************/
