/************************************************************************/
/************************************************************************/
// maze1.c: code to simulate marble maze.

/************************************************************************/
/************************************************************************/
/************************************************************************/
/* Includes */

#include "stdafx.h" /* Windows stuff, make empty stdafx.h under Linux */
#ifdef LINUX
#include <sys/time.h>
#endif
#ifndef LINUX
#include <windows.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include <GL/glut.h> /* For some reason this include gets rid of lots
			of spurious compiler warnings, even though there
			is no GLUT used here. */
#include "maze1.h"

/************************************************************************/
/* Defines */

#define MAX_STRING_LENGTH 100

/************************************************************************/
/* Globals */

#ifdef LINUX
/* Stuff for figuring out what time it is on LINUX */
struct timeval tv_val;
struct timezone tz_val;
#endif

SHAPE_REF *free_shape_ref_list = NULL;

/* debuging flags */
int debug_find_collision = 0;
int debug_handle_collision = 0;
int debug_constraints = 0;

/************************************************************************/

BOARD *create_board()
{
  BOARD *b;

  b = (BOARD *) malloc( sizeof( BOARD ) );
  if ( b == NULL )
    {
      fprintf( stderr, "Can't allocate BOARD.\n" );
      exit( -1 );
    }
  return b;
}

/************************************************************************/

int shape_ref_count = 0;

SHAPE_REF *create_shape_ref()
{
  SHAPE_REF *s;

  if ( free_shape_ref_list != NULL )
    {
      s = free_shape_ref_list;
      free_shape_ref_list = s->next;
    }
  else
    {
      s = (SHAPE_REF *) malloc( sizeof( SHAPE_REF ) );
      if ( s == NULL )
	{
	  fprintf( stderr, "Can't allocate SHAPE_REF.\n" );
	  exit( -1 );
	}
      shape_ref_count++;
    }
  s->next = NULL;
  return s;
}

/************************************************************************/

void free_shape_refs( SHAPE_REF *s )
{
  SHAPE_REF *s2;

  for( ; s != NULL; )
    {
      s2 = s->next;
      s->next = free_shape_ref_list;
      free_shape_ref_list = s;
      s = s2;
    }
}

/************************************************************************/

int enforce_constraints( BOARD *b )
{
  int i;

  if ( b->board.type != RECTANGLE )
    {
      fprintf( stderr, "In %s: board must be a rectangle.\n", b->name );
      exit( -1 );
    }
  if ( b->goal.type != RECTANGLE )
    {
      fprintf( stderr, "In %s: goal must be a rectangle.\n", b->name );
      exit( -1 );
    }
  for( i = 0; i < b->n_obstacles; i++ )
    if ( b->obstacles[i].type != RECTANGLE )
      {
	fprintf( stderr, "In %s: wall %d must be a rectangle.\n", b->name, i );
	exit( -1 );
      }
  for( i = 0; i < b->n_holes; i++ )
    if ( b->holes[i].type != CIRCLE )
      {
	fprintf( stderr, "In %s: hole %d must be a circle.\n", b->name, i );
	exit( -1 );
      }
  return TRUE;
}

/************************************************************************/

int inside_rectangle( float *p, float *r )
{
  if ( r[LEFT] <= p[XX]
       && p[XX] <= r[RIGHT]
       && r[BOTTOM] <= p[YY]
       && p[YY] <= r[TOP] )
    return TRUE;
  return FALSE;
}

/************************************************************************/

int line_inside_rectangle( float *l, float *r )
{
  float intersection;

  /* Check if one of the line points is inside */
  if ( r[LEFT] <= l[X1]
       && l[X1] <= r[RIGHT]
       && r[BOTTOM] <= l[Y1]
       && l[Y1] <= r[TOP] )
    return TRUE;

  /* Now check for any intersections. */
  if ( l[X1] < r[LEFT] && l[X2] >= r[LEFT] )
    {
      intersection = (l[Y2] - l[Y1])*(r[LEFT] - l[X1])/(l[X2] - l[X1]) 
	+ l[Y1];
      if ( r[BOTTOM] <= intersection && intersection <= r[TOP] )
	return TRUE;
    }

  if ( l[X1] > r[RIGHT] && l[X2] <= r[RIGHT] )
    {
      intersection = (l[Y2] - l[Y1])*(r[RIGHT] - l[X1])/(l[X2] - l[X1]) 
	+ l[Y1];
      if ( r[BOTTOM] <= intersection && intersection <= r[TOP] )
	return TRUE;
    }

  if ( l[Y1] < r[BOTTOM] && l[Y2] >= r[BOTTOM] )
    {
      intersection = (l[X2] - l[X1])*(r[BOTTOM] - l[Y1])/(l[Y2] - l[Y1]) 
	+ l[X1];
      if ( r[LEFT] <= intersection && intersection <= r[RIGHT] )
	return TRUE;
    }

  if ( l[Y1] > r[TOP] && l[Y2] <= r[TOP] )
    {
      intersection = (l[X2] - l[X1])*(r[TOP] - l[Y1])/(l[Y2] - l[Y1]) 
	+ l[X1];
      if ( r[LEFT] <= intersection && intersection <= r[RIGHT] )
	return TRUE;
    }

  return FALSE;
}

/************************************************************************/

int inside_circle( float *p, float *c )
{
  float d2;

  d2 = (p[XX] - c[XX])*(p[XX] - c[XX])
    + (p[YY] - c[YY])*(p[YY] - c[YY]);
  if ( d2 <= c[RADIUS]*c[RADIUS] )
    return TRUE;
  return FALSE;
}

/************************************************************************/
/************************************************************************/

int state_to_map_indices( BOARD *b, float *state, int *indices )
{
  int i;

  for( i = 0; i < 2; i++ )
    {
      indices[i] = (state[i] - b->board.p[i])*MAP_RESOLUTION
	/(b->board.p[i+2] - b->board.p[i]);
      if ( state[i] == b->board.p[i+2] )
	indices[i] = MAP_RESOLUTION - 1;
      if ( indices[i] < 0 || indices[i] >= MAP_RESOLUTION )
	return FALSE;
    }
  return TRUE;
}

/************************************************************************/

void add_obstacle_to_cell( BOARD *b, int o, int ix, int iy )
{
  SHAPE_REF *sr;

  sr = create_shape_ref();
  sr->s = &(b->obstacles[o]);
  sr->next = b->obstacle_map[ix][iy];
  b->obstacle_map[ix][iy] = sr;
}

/************************************************************************/

int map_obstacles( BOARD *b )
{
  int obstacle;
  float x_increment, y_increment;
  float rect[4];
  float line[4];
  float grown_obstacle[4];
  int ix, iy;

  x_increment = (b->board.p[RIGHT] - b->board.p[LEFT])/MAP_RESOLUTION;
  y_increment = (b->board.p[TOP] - b->board.p[BOTTOM])/MAP_RESOLUTION;

  for( ix = 0; ix < MAP_RESOLUTION; ix++ )
    {
      rect[X1] = ix*x_increment + b->board.p[LEFT];
      rect[X2] = rect[X1] + x_increment;
      for( iy = 0; iy < MAP_RESOLUTION; iy++ )
	{
	  rect[Y1] = iy*y_increment + b->board.p[BOTTOM];
	  rect[Y2] = rect[Y1] + y_increment;
	  b->obstacle_map[ix][iy] = NULL;
	  for( obstacle = 0; obstacle < b->n_obstacles; obstacle++ )
	    {
	      grown_obstacle[LEFT] = b->obstacles[obstacle].p[LEFT] 
		- b->puck_radius;
	      grown_obstacle[RIGHT] = b->obstacles[obstacle].p[RIGHT] 
		+ b->puck_radius;
	      grown_obstacle[BOTTOM] = b->obstacles[obstacle].p[BOTTOM] 
		- b->puck_radius;
	      grown_obstacle[TOP] = b->obstacles[obstacle].p[TOP] 
		+ b->puck_radius;
	      /* Is obstacle inside cell =
		 Is lower left corner of obstacle in cell? */
	      if ( inside_rectangle( grown_obstacle, rect ) )
		{
		  add_obstacle_to_cell( b, obstacle, ix, iy );
		  continue;
		}

	      /* Is cell inside obstacle =
		 Is lower left corner of cell in obstacle? */
	      if ( inside_rectangle( rect, grown_obstacle ) )
		{
		  add_obstacle_to_cell( b, obstacle, ix, iy );
		  continue;
		}

	      /* At this point cell is not entirely inside the obstacle,
		 and the obstable is not entirely inside the cell.
		 Now we need to check of any walls intersect. */
	      line[X1] = rect[X1];
	      line[Y1] = rect[Y1];
	      line[X2] = rect[X1];
	      line[Y2] = rect[Y2];
	      if ( line_inside_rectangle( line, grown_obstacle ) )
		{
		  add_obstacle_to_cell( b, obstacle, ix, iy );
		  continue;
		}
	      line[X1] = rect[X1];
	      line[Y1] = rect[Y2];
	      line[X2] = rect[X2];
	      line[Y2] = rect[Y2];
	      if ( line_inside_rectangle( line, grown_obstacle ) )
		{
		  add_obstacle_to_cell( b, obstacle, ix, iy );
		  continue;
		}
	      line[X1] = rect[X2];
	      line[Y1] = rect[Y2];
	      line[X2] = rect[X2];
	      line[Y2] = rect[Y1];
	      if ( line_inside_rectangle( line, grown_obstacle ) )
		{
		  add_obstacle_to_cell( b, obstacle, ix, iy );
		  continue;
		}
	      line[X1] = rect[X2];
	      line[Y1] = rect[Y1];
	      line[X2] = rect[X1];
	      line[Y2] = rect[Y1];
	      if ( line_inside_rectangle( line, grown_obstacle ) )
		{
		  add_obstacle_to_cell( b, obstacle, ix, iy );
		  continue;
		}
	    }
	}
    }
  return TRUE;
}

/************************************************************************/

void add_hole_to_cell( BOARD *b, int h, int ix, int iy )
{
  SHAPE_REF *sr;

  sr = create_shape_ref();
  sr->s = &(b->holes[h]);
  sr->next = b->hole_map[ix][iy];
  b->hole_map[ix][iy] = sr;
}

/************************************************************************/

int map_holes( BOARD *b )
{
  int hole;
  float x_increment, y_increment;
  float rect[4];
  float point[2];
  int ix, iy;

  x_increment = (b->board.p[RIGHT] - b->board.p[LEFT])/MAP_RESOLUTION;
  y_increment = (b->board.p[TOP] - b->board.p[BOTTOM])/MAP_RESOLUTION;

  for( ix = 0; ix < MAP_RESOLUTION; ix++ )
    {
      rect[X1] = ix*x_increment + b->board.p[LEFT];
      rect[X2] = rect[X1] + x_increment;
      for( iy = 0; iy < MAP_RESOLUTION; iy++ )
	{
	  rect[Y1] = iy*y_increment + b->board.p[BOTTOM];
	  rect[Y2] = rect[Y1] + y_increment;
	  b->hole_map[ix][iy] = NULL;
	  for( hole = 0; hole < b->n_holes; hole++ )
	    {
	      /* Is hole inside cell =
		 Is hole center in cell? */
	      if ( inside_rectangle( b->holes[hole].p, rect ) )
		{
		  add_hole_to_cell( b, hole, ix, iy );
		  continue;
		}

	      /* Is any corner of cell in hole? */
	      point[XX] = rect[X1];
	      point[YY] = rect[Y1];
	      if ( inside_circle( point, b->holes[hole].p ) )
		{
		  add_hole_to_cell( b, hole, ix, iy );
		  continue;
		}
	      point[XX] = rect[X1];
	      point[YY] = rect[Y2];
	      if ( inside_circle( point, b->holes[hole].p ) )
		{
		  add_hole_to_cell( b, hole, ix, iy );
		  continue;
		}
	      point[XX] = rect[X2];
	      point[YY] = rect[Y1];
	      if ( inside_circle( point, b->holes[hole].p ) )
		{
		  add_hole_to_cell( b, hole, ix, iy );
		  continue;
		}
	      point[XX] = rect[X2];
	      point[YY] = rect[Y2];
	      if ( inside_circle( point, b->holes[hole].p ) )
		{
		  add_hole_to_cell( b, hole, ix, iy );
		  continue;
		}

	      /* Does hole enter any wall of cell =
		 Is any NEWS point on hole perimenter in cell? */
	      point[XX] = b->holes[hole].p[XX] + b->holes[hole].p[RADIUS];
	      point[YY] = b->holes[hole].p[YY];
	      if ( inside_rectangle( point, rect ) )
		{
		  add_hole_to_cell( b, hole, ix, iy );
		  continue;
		}
	      point[XX] = b->holes[hole].p[XX];
	      point[YY] = b->holes[hole].p[YY] + b->holes[hole].p[RADIUS];
	      if ( inside_rectangle( point, rect ) )
		{
		  add_hole_to_cell( b, hole, ix, iy );
		  continue;
		}
	      point[XX] = b->holes[hole].p[XX] - b->holes[hole].p[RADIUS];
	      point[YY] = b->holes[hole].p[YY];
	      if ( inside_rectangle( point, rect ) )
		{
		  add_hole_to_cell( b, hole, ix, iy );
		  continue;
		}
	      point[XX] = b->holes[hole].p[XX];
	      point[YY] = b->holes[hole].p[YY] - b->holes[hole].p[RADIUS];
	      if ( inside_rectangle( point, rect ) )
		{
		  add_hole_to_cell( b, hole, ix, iy );
		  continue;
		}
	    }
	}
    }
  return TRUE;
}

/************************************************************************/

int debug_hole_map( BOARD *b )
{
  int ix, iy;

  for( iy = MAP_RESOLUTION; iy > 0; iy-- )
    {
      for( ix = 0; ix < MAP_RESOLUTION; ix++ )
	{
	  if ( b->hole_map[ix][iy] != NULL )
	    printf( "*" );
	  else
	    printf( " " );
	}
      printf( "\n" );
    }
  return TRUE;
}

/************************************************************************/

void readni( FILE *stream, int n, int *array )
{
  int i;

  for( i = 0; i < n; i++ )
    {
      if ( fscanf( stream, "%d", &(array[i]) ) < 1 )
	{
	  fprintf( stderr, "Couldn't read integer.\n" );
	  exit( -1 );
	}
    }   
}

/************************************************************************/

void readnf( FILE *stream, int n, float *array )
{
  int i;

  for( i = 0; i < n; i++ )
    {
      if ( fscanf( stream, "%g", &(array[i]) ) < 1 )
	{
	  fprintf( stderr, "Couldn't read float.\n" );
	  exit( -1 );
	}
    }   
}

/************************************************************************/

void read_shape( FILE *stream, SHAPE *s )
{
  char buffer[MAX_STRING_LENGTH];

  if ( fscanf( stream, "%s", buffer ) < 1 )
    {
      fprintf( stderr, "Could't read shape type.\n" );
      exit( -1 );
    }
  switch( buffer[0] )
    {
    case 'r': /* rectangle */
      s->type = RECTANGLE;
      readnf( stream, 4, s->p );
      break;
    case 'c': /* circle */
      s->type = CIRCLE;
      readnf( stream, 3, s->p );
      break;
    default:
      fprintf( stderr, "Can't handle shape of type %s\n", buffer );
      exit( -1 );
    }
}

/************************************************************************/

int initialize_maze( char *board_file, BOARD *b )
{
  FILE *stream;
  char buffer[MAX_STRING_LENGTH];

#ifdef LINUX
  gettimeofday( &tv_val, &tz_val );
  srand( (unsigned int) tv_val.tv_sec );
#else
  srand( (int) GetCurrentTime() );
#endif

  b->name = strdup( board_file );

  b->coefficient_of_restitution = 0.5;
  b->plastic_impact_velocity_threshold = 0.1;
  b->puck_radius = 0.005;
  b->drag = 0.1;
  b->max_force = 1.0;
  b->random_force_0 = 0.1;

  b->puck[XX] = 0.0;
  b->puck[YY] = 0.0;
  b->puck[XD] = 0.0;
  b->puck[YD] = 0.0;
  b->global_time = 0.0;
  b->force[XX] = 0.0;
  b->force[YY] = 0.0;

  b->board.type = RECTANGLE;
  b->board.p[LEFT] = 0;
  b->board.p[BOTTOM] = 0;
  b->board.p[RIGHT] = 100;
  b->board.p[TOP] = 100;

  b->start[XX] = 0;
  b->start[YY] = 0;
  b->start[XD] = 0;
  b->start[YD] = 0;

  b->goal.type = RECTANGLE;
  b->goal.p[LEFT] = 0;
  b->goal.p[BOTTOM] = 0;
  b->goal.p[RIGHT] = 1;
  b->goal.p[TOP] = 1;

  b->n_obstacles = 0;
  b->n_holes = 0;

  b->constraint_state = FREE;
  b->constraint1 = NULL;
  b->constraint2 = NULL;

  stream = fopen( board_file, "r" );
  if ( stream == NULL )
    {
      fprintf( stderr, "Can't open file %s for read\n", board_file );
      exit( -1 );
    }
  for( ; ; )
    {
      if ( fscanf( stream, "%s", buffer ) < 1 )
	break;
      if ( strcmp( "board", buffer ) == 0 )
	{
	  read_shape( stream, &(b->board) );
	  continue;
	}
      if ( strcmp( "start", buffer ) == 0 )
	{
	  readnf( stream, 2, b->start );
	  continue;
	}
      if ( strcmp( "goal", buffer ) == 0 )
	{
	  read_shape( stream, &(b->goal) );
	  continue;
	}
      if ( strcmp( "wall", buffer ) == 0 )
	{
	  if ( b->n_obstacles >= MAX_N_OBSTACLES )
	    {
	      fprintf( stderr,
	       "Too many obstacles: %d, increase MAX_N_OBSTACLES.\n",
		       MAX_N_OBSTACLES );
	      exit( -1 );
	    }
	  read_shape( stream, &(b->obstacles[b->n_obstacles]) );
	  b->n_obstacles++;
	  continue;
	}
      if ( strcmp( "hole", buffer ) == 0 )
	{
	  if ( b->n_holes >= MAX_N_HOLES )
	    {
	      fprintf( stderr,
		       "Too many holes: %d > %d, increase MAX_N_HOLES.\n",
		       b->n_holes, MAX_N_HOLES );
	      exit( -1 );
	    }
	  read_shape( stream, &(b->holes[b->n_holes]) );
	  b->n_holes++;
	  continue;
	}
      fprintf( stderr, "Error in board file %s", buffer );
      exit( -1 );
    }
  fclose( stream );

  b->puck[XX] = b->start[0];
  b->puck[YY] = b->start[1];
  
  enforce_constraints( b );
  map_obstacles( b );
  map_holes( b );
  /*
  debug_hole_map( b );
  */
  return TRUE;
}

/************************************************************************/
/************************************************************************/
/************************************************************************/
/* Copy a shape_ref */

SHAPE_REF *copy_shape_ref( SHAPE_REF *s1 )
{
  SHAPE_REF *result;

  result = create_shape_ref();
  result->s = s1->s;
  result->next = NULL;
  return result;
}

/************************************************************************/
/* Detect if s1 has a duplicate on s2 */

int already_on( SHAPE_REF *s1, SHAPE_REF *s2 )
{
  for( ; s2 != NULL; s2 = s2->next )
    if ( s1->s == s2->s )
      return TRUE;
  return FALSE;
}

/************************************************************************/
/* Copy S1, and append S2. Do not copy items in S1 already in S1 or S2 */

SHAPE_REF *merge_lists( SHAPE_REF *s1, SHAPE_REF *s2 )
{
  SHAPE_REF *result = NULL;
  SHAPE_REF *tmp = NULL;

  if ( s1 == NULL )
    return s2;

  result = s2;

  for( ; s1 != NULL; s1 = s1->next )
    {
      if ( already_on( s1, result ) )
	continue;
      tmp = copy_shape_ref( s1 );
      tmp->next = result;
      result = tmp;
    }
  return result;
}

/************************************************************************/

SHAPE_REF *process_column( BOARD *b, int ix, float y1, float y2, 
			   SHAPE_REF *result )
{
  float min_y, max_y;
  int iy, iy1, iy2;

  if ( y1 <= y2 )
    {
      min_y = y1;
      max_y = y2;
    }
  else
    {
      min_y = y2;
      max_y = y1;
    }

  iy1 = (min_y - b->board.p[BOTTOM])*MAP_RESOLUTION
	/(b->board.p[TOP] - b->board.p[BOTTOM]);
  iy2 = (max_y - b->board.p[BOTTOM])*MAP_RESOLUTION
	/(b->board.p[TOP] - b->board.p[BOTTOM]);
  if ( max_y == b->board.p[TOP] )
    iy2 = MAP_RESOLUTION - 1;
  if ( iy1 < 0 )
    iy1 = 0;
  if ( iy2 >= MAP_RESOLUTION )
    iy2 = MAP_RESOLUTION - 1;
  /*
  printf( "process column: %g %g %d %d %d\n", y1, y2, ix, iy1, iy2 );
  */
  for( iy = iy1; iy <= iy2; iy++ )
    result = merge_lists( b->obstacle_map[ix][iy], result );
  return result;
}

/************************************************************************/
/* We need to merge a bunch of shape_ref lists. */

SHAPE_REF *find_all_obstacles( BOARD *b, float *p1, float *p2 )
{
  float start[2], goal[2];
  int start_x, goal_x;
  SHAPE_REF *result = NULL;
  float x_increment, y_increment;
  int ix;
  float min_y, max_y;
  float x_left, x_right;

  x_increment = (b->board.p[RIGHT] - b->board.p[LEFT])/MAP_RESOLUTION;
  y_increment = (b->board.p[TOP] - b->board.p[BOTTOM])/MAP_RESOLUTION;

  if ( p1[XX] <= p2[XX] )
    {
      start[XX] = p1[XX];
      start[YY] = p1[YY];
      goal[XX] = p2[XX];
      goal[YY] = p2[YY];
    }
  else
    {
      start[XX] = p2[XX];
      start[YY] = p2[YY];
      goal[XX] = p1[XX];
      goal[YY] = p1[YY];
    }
  /*
  printf( "find_all_obstacles: %g %g %g %g\n",
	  p1[XX], p1[YY], p2[XX], p2[YY] );
  */
  start_x = (start[XX] - b->board.p[LEFT])*MAP_RESOLUTION
	/(b->board.p[RIGHT] - b->board.p[LEFT]);
  if ( start_x < 0 )
    start_x = 0;
  if ( start[XX] == b->board.p[RIGHT] )
    start_x = MAP_RESOLUTION - 1;

  goal_x = (goal[XX] - b->board.p[LEFT])*MAP_RESOLUTION
	/(b->board.p[RIGHT] - b->board.p[LEFT]);
  if ( goal_x >= MAP_RESOLUTION )
    goal_x = MAP_RESOLUTION - 1;

  if ( start_x > goal_x )
    return NULL;

  /*
  printf( "%g %g %g %g -> %d %d\n", 
	  start[XX], start[YY], goal[XX], goal[YY], start_x, goal_x );
  */

  /* Get first row. */
  ix = start_x;
  min_y = start[YY];
  x_right = (ix + 1)*x_increment + b->board.p[LEFT];
  max_y = (goal[YY] - start[YY])*(x_right - start[XX])/(goal[XX] - start[XX])
    + start[YY];
  /*
  printf( "calling pc: %d %g %g\n", ix, min_y, max_y );
  */
  result = process_column( b, ix, min_y, max_y, result );
  ix++;
  if ( ix > goal_x )
    return result;

  /* Get all middle rows. */
  for( ; ix < goal_x; ix++ )
    {
      x_left = ix*x_increment + b->board.p[LEFT];
      min_y = (goal[YY] - start[YY])*(x_left - start[XX])
	/(goal[XX] - start[XX]) + start[YY];
      x_right = (ix + 1)*x_increment + b->board.p[LEFT];
      max_y = (goal[YY] - start[YY])*(x_right - start[XX])
	/(goal[XX] - start[XX]) + start[YY];
      result = process_column( b, ix, min_y, max_y, result );
    }

  /* Get last row. */
  x_left = ix*x_increment + b->board.p[LEFT];
  min_y = (goal[YY] - start[YY])*(x_left - start[XX])
    /(goal[XX] - start[XX]) + start[YY];
  max_y = goal[YY];
  result = process_column( b, ix, min_y, max_y, result );
  return result;
}

/************************************************************************/

int find_collision( BOARD *b, float *next_state, 
		    float *collision_time, SHAPE **first_obstacle, int *side )
{
  float this_collision_time;
  float collision_point;
  int result = FALSE;
  SHAPE_REF *obstacles = NULL;
  SHAPE_REF *free_me;
  float *state;
  
  state = b->puck;

  obstacles = find_all_obstacles( b, state, next_state );
  if ( obstacles == NULL )
    return FALSE;
  /*
  printf( "find: maybe hit obstacle.\n" );
  getchar();
  */
  free_me = obstacles;

  for( ; obstacles != NULL; obstacles = obstacles->next )
    {
      /* Did we cross any borders of this obstacle? */
      if ( state[XX] + b->puck_radius <= obstacles->s->p[LEFT]
	  && next_state[XX] + b->puck_radius > obstacles->s->p[LEFT] )
	{
	  this_collision_time = 
	    (obstacles->s->p[LEFT] - (state[XX] + b->puck_radius))
	    /(next_state[XX] - state[XX]);
	  collision_point = this_collision_time*(next_state[YY] - state[YY])
	    + state[YY];
	  if ( this_collision_time < *collision_time 
	       && collision_point > obstacles->s->p[BOTTOM] - b->puck_radius
	       && collision_point < obstacles->s->p[TOP] + b->puck_radius )
	    {
	      *collision_time = this_collision_time;
	      *first_obstacle = obstacles->s;
	      *side = LEFT;
	      result = TRUE;
	    }
	}
      if ( state[XX] - b->puck_radius >= obstacles->s->p[RIGHT]
	  && next_state[XX] - b->puck_radius < obstacles->s->p[RIGHT] )
	{
	  this_collision_time = 
	    (obstacles->s->p[RIGHT] - (state[XX] - b->puck_radius))
	    /(next_state[XX] - state[XX]);
	  collision_point = this_collision_time*(next_state[YY] - state[YY])
	    + state[YY];
	  if ( this_collision_time < *collision_time 
	       && collision_point > obstacles->s->p[BOTTOM] - b->puck_radius
	       && collision_point < obstacles->s->p[TOP] + b->puck_radius )
	    {
	      *collision_time = this_collision_time;
	      *first_obstacle = obstacles->s;
	      *side = RIGHT;
	      result = TRUE;
	    }
	}
      if ( state[YY] + b->puck_radius <= obstacles->s->p[BOTTOM]
	  && next_state[YY] + b->puck_radius > obstacles->s->p[BOTTOM] )
	{
	  this_collision_time = 
	    (obstacles->s->p[BOTTOM] - (state[YY] + b->puck_radius))
	    /(next_state[YY] - state[YY]);
	  collision_point = this_collision_time*(next_state[XX] - state[XX])
	    + state[XX];
	  if ( this_collision_time < *collision_time 
	       && collision_point > obstacles->s->p[LEFT] - b->puck_radius
	       && collision_point < obstacles->s->p[RIGHT] + b->puck_radius )
	    {
	      *collision_time = this_collision_time;
	      *first_obstacle = obstacles->s;
	      *side = BOTTOM;
	      result = TRUE;
	    }
	}
      if ( /* state[YD] <= 0.0 && */
	  state[YY] - b->puck_radius >= obstacles->s->p[TOP]
	  && next_state[YY] - b->puck_radius < obstacles->s->p[TOP] )
	{
	  this_collision_time = 
	    (obstacles->s->p[TOP] - (state[YY] - b->puck_radius))
	    /(next_state[YY] - state[YY]);
	  collision_point = this_collision_time*(next_state[XX] - state[XX])
	    + state[XX];
	  if ( this_collision_time < *collision_time 
	       && collision_point > obstacles->s->p[LEFT] - b->puck_radius
	       && collision_point < obstacles->s->p[RIGHT] + b->puck_radius )
	    {
	      *collision_time = this_collision_time;
	      *first_obstacle = obstacles->s;
	      *side = TOP;
	      result = TRUE;
	    }
	}
    }
  if ( result )
    {
      if ( *collision_time > 1.0 || *collision_time < 0.0 )
	{
	  printf( "find_collision ERROR (%g) %d: %g\n", b->global_time,
		  *side, *collision_time );
	  printf( "%g %g %g %g\n", (*first_obstacle)->p[0],
		  (*first_obstacle)->p[1], (*first_obstacle)->p[2],
		  (*first_obstacle)->p[3] );
	  printf( "%g %g %g %g; %g %g %g %g\n", 
		  state[XX], state[YY], state[XD], state[YD],
		  next_state[XX], next_state[YY],
		  next_state[XD], next_state[YD] );
	  exit( -1 );
	}
      /*
      printf( "find_collision (%g) %d: %g\n", b->global_time,
	      *side, *collision_time );
      printf( "%g %g %g %g\n", (*first_obstacle)->p[0],
	      (*first_obstacle)->p[1], (*first_obstacle)->p[2],
	      (*first_obstacle)->p[3] );
      printf( "%g %g %g %g; %g %g %g %g\n", 
	      state[XX], state[YY], state[XD], state[YD],
	      next_state[XX], next_state[YY],
	      next_state[XD], next_state[YD] );
      */
    }
  free_shape_refs( free_me );
  return result;
}

/************************************************************************/
/* Currently fails when inside more than one obstacle */

void fix_penetrating_obstacle( BOARD *b )
{
  float epsilon = 1e-7;
  int first = -1;
  int side = -1;
  float distance = 0;
  SHAPE_REF *sr = NULL;
  int map_indices[2];
  float *state;

  state = b->puck;

  if ( !state_to_map_indices( b, state, map_indices ) )
    return;
  sr = b->obstacle_map[map_indices[XX]][map_indices[YY]];
  /*
  if ( sr != NULL )
    {
      printf( "fix: maybe hit something.\n" );
      getchar();
    }
  */
  for( ; sr != NULL; sr = sr->next )
    {
      if ( state[XX] + b->puck_radius > sr->s->p[LEFT]
	   && state[XX] - b->puck_radius < sr->s->p[RIGHT]
	   && state[YY] + b->puck_radius > sr->s->p[BOTTOM]
	   && state[YY] - b->puck_radius < sr->s->p[TOP] )
	{
	  /* First figure out which way to push ball */
	  if ( state[XX] < sr->s->p[LEFT] )
	    {
	      side = LEFT;
	      distance = sr->s->p[LEFT] - state[XX];
	    }
	  else if ( state[XX] > sr->s->p[RIGHT] )
	    {
	      side = RIGHT;
	      distance = state[XX] - sr->s->p[RIGHT];
	    }
	  if ( state[YY] < sr->s->p[BOTTOM] )
	    {
	      if ( sr->s->p[BOTTOM] - state[YY] > distance )
		{
		  side = BOTTOM;
		}
	    }
	  else if ( state[YY] > sr->s->p[TOP] )
	    {
	      if ( state[YY] - sr->s->p[TOP] > distance )
		{
		  side = TOP;
		}
	    }
	  if ( side < 0 )
	    return;
	  /* Reset constraints, given screwup */
	  switch( side )
	    {
	    case LEFT:
	      state[XX] = sr->s->p[LEFT] 
		- b->puck_radius - epsilon;
	      if ( state[XD] > 0 )
		state[XD] = 0;
	      b->constraint_state = E;
	      b->constraint1 = sr->s;
	      b->constraint2 = NULL;
	      if ( debug_constraints )
		printf( "fix E\n" );
	      break;
	    case RIGHT:
	      state[XX] = sr->s->p[RIGHT] 
		+ b->puck_radius + epsilon;
	      if ( state[XD] < 0 )
		state[XD] = 0;
	      b->constraint_state = W;
	      b->constraint1 = sr->s;
	      b->constraint2 = NULL;
	      if ( debug_constraints )
		printf( "fix W\n" );
	      break;
	    case BOTTOM:
	      state[YY] = sr->s->p[BOTTOM] 
		- b->puck_radius - epsilon;
	      if ( state[YD] > 0 )
		state[YD] = 0;
	      b->constraint_state = N;
	      b->constraint1 = sr->s;
	      b->constraint2 = NULL;
	      if ( debug_constraints )
		printf( "fix N\n" );
	      break;
	    case TOP:
	      state[YY] = sr->s->p[TOP] 
		+ b->puck_radius + epsilon;
	      if ( state[YD] < 0 )
		state[YD] = 0;
	      b->constraint_state = S;
	      b->constraint1 = sr->s;
	      b->constraint2 = NULL;
	      if ( debug_constraints )
		printf( "fix S\n" );
	      break;
	    }
	}
    }
}

/************************************************************************/

void handle_collision( BOARD *b, float *puck_next, 
		       float time, SHAPE *obstacle, int side )
{
  float epsilon = 1e-7;

  /*
  printf( "handle collision: %g %d: %g %g %g %g; %g %g %g %g: %g %g %g %g\n", 
	  time, side, b->puck[XX], b->puck[YY], b->puck[XD], b->puck[YD],
	  puck_next[XX], puck_next[YY], puck_next[XD], puck_next[YD],
	  obstacle->p[LEFT], obstacle->p[BOTTOM], obstacle->p[RIGHT],
	  obstacle->p[TOP] );
  getchar();
  */

  puck_next[XX] = b->puck[XX] + (puck_next[XX] - b->puck[XX])*(time - epsilon);
  puck_next[YY] = b->puck[YY] + (puck_next[YY] - b->puck[YY])*(time - epsilon);
  switch( side )
    {
    case LEFT:
      puck_next[XX] = obstacle->p[LEFT] - b->puck_radius - epsilon;
      puck_next[XD] = -b->coefficient_of_restitution*puck_next[XD];
      if ( fabsf( puck_next[XD] ) < b->plastic_impact_velocity_threshold )
	{
	  puck_next[XD] = 0.0;
	  switch( b->constraint_state )
	    {
	    case FREE:
	      b->constraint_state = E;
	      b->constraint1 = obstacle;
	      b->constraint2 = NULL;
	      if ( debug_constraints )
		printf( "E\n" );
	      break;
	    case N:
	      b->constraint_state = NE;
	      b->constraint2 = obstacle;
	      if ( debug_constraints )
		printf( "NE\n" );
	      break;
	    case S:
	      b->constraint_state = SE;
	      b->constraint2 = obstacle;
	      if ( debug_constraints )
		printf( "SE\n" );
	      break;
	    default:
	      fprintf( stderr, "handle_collision LEFT: %d\n",
		       b->constraint_state );
	      exit( -1 );
	    }
	}
      break;
    case RIGHT:
      puck_next[XX] = obstacle->p[RIGHT] + b->puck_radius + epsilon;
      puck_next[XD] = -b->coefficient_of_restitution*puck_next[XD];
      if ( fabsf( puck_next[XD] ) < b->plastic_impact_velocity_threshold )
	{
	  puck_next[XD] = 0.0;
	  switch( b->constraint_state )
	    {
	    case FREE:
	      b->constraint_state = W;
	      b->constraint1 = obstacle;
	      b->constraint2 = NULL;
	      if ( debug_constraints )
		printf( "W\n" );
	      break;
	    case N:
	      b->constraint_state = NW;
	      b->constraint2 = obstacle;
	      if ( debug_constraints )
		printf( "NW\n" );
	      break;
	    case S:
	      b->constraint_state = SW;
	      b->constraint2 = obstacle;
	      if ( debug_constraints )
		printf( "SW\n" );
	      break;
	    default:
	      fprintf( stderr, "handle_collision RIGHT: %d\n",
		       b->constraint_state );
	      exit( -1 );
	    }
	}
      break;
    case BOTTOM:
      puck_next[YY] = obstacle->p[BOTTOM] - b->puck_radius - epsilon;
      puck_next[YD] = -b->coefficient_of_restitution*puck_next[YD];
      if ( fabsf( puck_next[YD] ) < b->plastic_impact_velocity_threshold )
	{
	  puck_next[YD] = 0.0;
	  switch( b->constraint_state )
	    {
	    case FREE:
	      b->constraint_state = N;
	      b->constraint1 = obstacle;
	      b->constraint2 = NULL;
	      if ( debug_constraints )
		printf( "N\n" );
	      break;
	    case E:
	      b->constraint_state = NE;
	      b->constraint2 = b->constraint1;
	      b->constraint1 = obstacle;
	      if ( debug_constraints )
		printf( "NE\n" );
	      break;
	    case W:
	      b->constraint_state = NW;
	      b->constraint2 = b->constraint1;
	      b->constraint1 = obstacle;
	      if ( debug_constraints )
		printf( "NW\n" );
	      break;
	    default:
	      fprintf( stderr, "handle_collision BOTTOM: %d\n",
		       b->constraint_state );
	      exit( -1 );
	    }
	}
      break;
    case TOP:
      puck_next[YY] = obstacle->p[TOP] + b->puck_radius + epsilon;
      puck_next[YD] = -b->coefficient_of_restitution*puck_next[YD];
      if ( fabsf( puck_next[YD] ) < b->plastic_impact_velocity_threshold )
	{
	  puck_next[YD] = 0.0;
	  switch( b->constraint_state )
	    {
	    case FREE:
	      b->constraint_state = S;
	      b->constraint1 = obstacle;
	      b->constraint2 = NULL;
	      if ( debug_constraints )
		printf( "S\n" );
	      break;
	    case E:
	      b->constraint_state = SE;
	      b->constraint2 = b->constraint1;
	      b->constraint1 = obstacle;
	      if ( debug_constraints )
		printf( "SE\n" );
	      break;
	    case W:
	      b->constraint_state = SW;
	      b->constraint2 = b->constraint1;
	      b->constraint1 = obstacle;
	      if ( debug_constraints )
		printf( "SW\n" );
	      break;
	    default:
	      fprintf( stderr, "handle_collision TOP: %d\n",
		       b->constraint_state );
	      exit( -1 );
	    }
	}
      break;
    }
  if ( debug_handle_collision )
    {
      printf( "handle collision2: %g %d: %g %g %g %g; %g %g %g %g\n", 
	      time, side, b->puck[XX], b->puck[YY], b->puck[XD], b->puck[YD],
	      puck_next[XX], puck_next[YY], puck_next[XD], puck_next[YD] );
    }
}

/************************************************************************/
/* Only need position, not velocities, in state */

int in_obstacle( BOARD *b, float *state )
{
  int map_indices[2];
  SHAPE_REF *sr;

  if ( !state_to_map_indices( b, state, map_indices ) )
    return FALSE;
  sr = b->obstacle_map[map_indices[XX]][map_indices[YY]];
  for( ; sr != NULL; sr = sr->next )
    {
      if ( state[XX] + b->puck_radius > sr->s->p[LEFT]
	   && state[XX] - b->puck_radius < sr->s->p[RIGHT]
	   && state[YY] + b->puck_radius > sr->s->p[BOTTOM]
	   && state[YY] - b->puck_radius < sr->s->p[TOP] )
	return TRUE;
    }
  return FALSE;
}

/************************************************************************/
/* Only need position, not velocities, in state */

int in_hole( BOARD *b, float *state )
{
  int map_indices[2];
  SHAPE_REF *sr;

  if ( !state_to_map_indices( b, state, map_indices ) )
    return FALSE;
  sr = b->hole_map[map_indices[XX]][map_indices[YY]];
  for( ; sr != NULL; sr = sr->next )
    {
      if ( inside_circle( state, sr->s->p ) )
	return TRUE;
    }
  return FALSE;
}

/************************************************************************/
/* Only need position, not velocities, in state */

int in_goal( BOARD *b, float *state )
{
  return inside_rectangle( state, b->goal.p );
}

/************************************************************************/
/* return FALSE if in HOLE or OBSTACLE, TRUE if state set correctly */

int set_state( BOARD *b, float *state, int *status )
{
  if ( in_obstacle( b, state ) )
    {
      *status = OBSTACLE;
      return FALSE;
    }
  if ( in_hole( b, state ) )
    {
      *status = HOLE;
      return FALSE;
    }

  b->puck[XX] = state[XX];
  b->puck[YY] = state[YY];
  b->puck[XD] = state[XD];
  b->puck[YD] = state[YD];
  *status = 0;
  b->constraint_state = FREE;
  b->constraint1 = NULL;
  b->constraint2 = NULL;
  return TRUE;
}

/************************************************************************/
/* Could get an error here if something comes out of a corner and has
   a very short ledge */

float check_for_running_off_constraint( BOARD *b, float *puck_next, 
					float time )
{
  float s;
  float epsilon = 1e-7;

  if ( b->constraint1 == NULL )
    {
      fprintf( stderr, "check_xxx: no active constraint: %d\n",
	       b->constraint_state );
      exit( -1 );
    }

  /*
  printf( "check_xxx: %d\n", b->constraint_state );
  */

  switch( b->constraint_state )
    {
    case FREE:
    case NE:
    case NW:
    case SE:
    case SW:
      return time;
      break;
    case N:
    case S:
      if ( puck_next[XX] + b->puck_radius < b->constraint1->p[LEFT] )
	{
	  if ( b->puck[XX] + b->puck_radius < b->constraint1->p[LEFT] )
	    {
	      fprintf( stderr, "check_xxx: N,S LEFT: %g %g: %g %g %g\n",
		       puck_next[XX], b->puck[XX], 
		       puck_next[XX] + b->puck_radius, 
		       b->puck[XX] + b->puck_radius, 
		       b->constraint1->p[LEFT] );
	      exit( -1 );
	    }
	  if ( debug_constraints )
	    printf( "off LEFT\n" );
	  s = (b->constraint1->p[LEFT] - (b->puck[XX] + b->puck_radius))
	    /(puck_next[XX] - b->puck[XX]);
	  puck_next[XX] = b->constraint1->p[LEFT] - b->puck_radius - epsilon;
	  puck_next[XD] = s*(puck_next[XD] - b->puck[XD]) + b->puck[XD];
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  return s*time;
	}
      if ( b->constraint1->p[RIGHT] < puck_next[XX] - b->puck_radius )
	{
	  if ( b->constraint1->p[RIGHT] < b->puck[XX] - b->puck_radius  )
	    {
	      fprintf( stderr, "check_xxx: N,S RIGHT: %g %g: %g %g %g\n",
		       puck_next[XX], b->puck[XX], 
		       puck_next[XX] - b->puck_radius, 
		       b->puck[XX] - b->puck_radius, 
		       b->constraint1->p[RIGHT] );
	      exit( -1 );
	    }
	  if ( debug_constraints )
	    printf( "off RIGHT\n" );
	  s = (b->constraint1->p[RIGHT] - (b->puck[XX] - b->puck_radius))
	    /(puck_next[XX] - b->puck[XX]);
	  puck_next[XX] = b->constraint1->p[RIGHT] + b->puck_radius + epsilon;
	  puck_next[XD] = s*(puck_next[XD] - b->puck[XD]) + b->puck[XD];
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  return s*time;
	}
      return time;
      break;
    case E:
    case W:
      if ( puck_next[YY] + b->puck_radius < b->constraint1->p[BOTTOM] )
	{
	  if ( b->puck[YY] + b->puck_radius < b->constraint1->p[BOTTOM] )
	    {
	      fprintf( stderr, "check_yyx: E,W BOTTOM: %g %g: %g %g %g\n",
		       puck_next[YY], b->puck[YY], 
		       puck_next[YY] + b->puck_radius, 
		       b->puck[YY] + b->puck_radius, 
		       b->constraint1->p[BOTTOM] );
	      exit( -1 );
	    }
	  if ( debug_constraints )
	    printf( "off BOTTOM\n" );
	  s = (b->constraint1->p[BOTTOM] - (b->puck[YY] + b->puck_radius))
	    /(puck_next[YY] - b->puck[YY]);
	  puck_next[YY] = b->constraint1->p[BOTTOM] - b->puck_radius - epsilon;
	  puck_next[YD] = s*(puck_next[YD] - b->puck[YD]) + b->puck[YD];
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  return s*time;
	}
      if ( b->constraint1->p[TOP] < puck_next[YY] - b->puck_radius )
	{
	  if ( b->constraint1->p[TOP] < b->puck[YY] - b->puck_radius )
	    {
	      fprintf( stderr, "check_yyx: E,W TOP: %g %g: %g %g %g\n",
		       puck_next[YY], b->puck[YY], 
		       puck_next[YY] - b->puck_radius, 
		       b->puck[YY] - b->puck_radius, 
		       b->constraint1->p[TOP] );
	      exit( -1 );
	    }
	  if ( debug_constraints )
	    printf( "off TOP\n" );
	  s = (b->constraint1->p[TOP] - (b->puck[YY] - b->puck_radius))
	    /(puck_next[YY] - b->puck[YY]);
	  puck_next[YY] = b->constraint1->p[TOP] + b->puck_radius + epsilon;
	  puck_next[YD] = s*(puck_next[YD] - b->puck[YD]) + b->puck[YD];
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  return s*time;
	}
      return time;
      break;
    default:
      fprintf( stderr, "Unknown constraint_state %d in do_simulation.\n",
	       b->constraint_state );
      exit( -1 );
      break;
    }
  fprintf( stderr, "check_xxx: Shouldn't get here.\n" );
  exit( -1 );
}

/************************************************************************/

int big_count = 0;

float do_simulation( BOARD *b, float dt )
{
  int i;
  float puck_next[4];
  float dt_done;
  int collision = FALSE;
  float collision_time;
  SHAPE *obstacle;
  int side;

  /*
  printf( "do_simulation: %f: %g %g %g %g; %g %g\n",
	  dt, 
	  b->puck[XX], b->puck[YY], b->puck[XD], b->puck[YD],
	  b->force[XX], b->force[YY] );
  */

  if ( b->force[XX] > b->max_force )
    b->force[XX] = b->max_force;
  if ( b->force[XX] < -b->max_force )
    b->force[XX] = -b->max_force;
  if ( b->force[YY] > b->max_force )
    b->force[YY] = b->max_force;
  if ( b->force[YY] < -b->max_force )
    b->force[YY] = -b->max_force;

  if ( b->random_force_0 > 0.0 )
    {
      b->force[XX] += b->random_force_0*((2.0*rand())/RAND_MAX - 1.0);
      b->force[YY] += b->random_force_0*((2.0*rand())/RAND_MAX - 1.0);
    }

  switch( b->constraint_state )
    {
    case FREE:
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt) + b->force[XX]*dt;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt) + b->force[YY]*dt;
      puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
      puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
      break;
    case N:
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt) + b->force[XX]*dt;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt) + b->force[YY]*dt;
      puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
      if ( puck_next[YD] + b->puck[YD] >= 0 )
	{
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	}
      else
	{
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "off N\n" );
	}
      break;
    case S:
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt) + b->force[XX]*dt;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt) + b->force[YY]*dt;
      puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
      if ( puck_next[YD] + b->puck[YD] <= 0 )
	{
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	}
      else
	{
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "off S\n" );
	}
      break;
    case E:
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt) + b->force[XX]*dt;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt) + b->force[YY]*dt;
      puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
      if ( puck_next[XD] + b->puck[XD] >= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	}
      else
	{
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  if ( debug_constraints )
	    printf( "off E\n" );
	}
      break;
    case W:
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt) + b->force[XX]*dt;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt) + b->force[YY]*dt;
      puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
      if ( puck_next[XD] + b->puck[XD] <= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	}
      else
	{
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  if ( debug_constraints )
	    printf( "off W\n" );
	}
      break;
    case NE:
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt) + b->force[XX]*dt;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt) + b->force[YY]*dt;
      if ( puck_next[YD] + b->puck[YD] >= 0 
	   && puck_next[XD] + b->puck[XD] >= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	}
      else if ( puck_next[YD] + b->puck[YD] >= 0 )
	{
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	  b->constraint_state = N;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  if ( debug_constraints )
	    printf( "NE -> N\n" );
	}
      else if ( puck_next[XD] + b->puck[XD] >= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	  b->constraint_state = E;
	  b->constraint1 = b->constraint2;
	  b->constraint2 = NULL;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "NE -> E\n" );
	}
      else
	{
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "off NE\n" );
	}
      break;
    case NW:
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt) + b->force[XX]*dt;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt) + b->force[YY]*dt;
      if ( puck_next[YD] + b->puck[YD] >= 0 
	   && puck_next[XD] + b->puck[XD] <= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	}
      else if ( puck_next[YD] + b->puck[YD] >= 0 )
	{
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	  b->constraint_state = N;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  if ( debug_constraints )
	    printf( "NW -> N\n" );
	}
      else if ( puck_next[XD] + b->puck[XD] <= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	  b->constraint_state = W;
	  b->constraint1 = b->constraint2;
	  b->constraint2 = NULL;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "NW -> W\n" );
	}
      else
	{
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "off NW\n" );
	}
      break;
    case SE:
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt) + b->force[XX]*dt;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt) + b->force[YY]*dt;
      if ( puck_next[YD] + b->puck[YD] <= 0 
	   && puck_next[XD] + b->puck[XD] >= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	}
      else if ( puck_next[YD] + b->puck[YD] <= 0 )
	{
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	  b->constraint_state = S;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  if ( debug_constraints )
	    printf( "SE -> S\n" );
	}
      else if ( puck_next[XD] + b->puck[XD] >= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	  b->constraint_state = E;
	  b->constraint1 = b->constraint2;
	  b->constraint2 = NULL;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "SE -> E\n" );
	}
      else
	{
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "off SE\n" );
	}
      break;
    case SW:
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt) + b->force[XX]*dt;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt) + b->force[YY]*dt;
      if ( puck_next[YD] + b->puck[YD] <= 0 
	   && puck_next[XD] + b->puck[XD] <= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	}
      else if ( puck_next[YD] + b->puck[YD] <= 0 )
	{
	  puck_next[YY] = b->puck[YY];
	  puck_next[YD] = 0;
	  b->constraint_state = S;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  if ( debug_constraints )
	    printf( "SW -> S\n" );
	}
      else if ( puck_next[XD] + b->puck[XD] <= 0 )
	{
	  puck_next[XX] = b->puck[XX];
	  puck_next[XD] = 0;
	  b->constraint_state = W;
	  b->constraint1 = b->constraint2;
	  b->constraint2 = NULL;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "SW -> W\n" );
	}
      else
	{
	  b->constraint_state = FREE;
	  b->constraint1 = NULL;
	  b->constraint2 = NULL;
	  puck_next[XX] = b->puck[XX] + 0.5*(puck_next[XD] + b->puck[XD])*dt;
	  puck_next[YY] = b->puck[YY] + 0.5*(puck_next[YD] + b->puck[YD])*dt;
	  if ( debug_constraints )
	    printf( "off SW\n" );
	}
      break;
    default:
      fprintf( stderr, "Unknown constraint_state %d in do_simulation.\n",
	       b->constraint_state );
      exit( -1 );
      break;
    }
  dt_done = dt;
  if ( b->constraint_state != FREE )
    dt_done = check_for_running_off_constraint( b, puck_next, dt );

  collision_time = 1e30;
  collision = find_collision( b, puck_next, &collision_time,
			  &obstacle, &side );
  if ( collision )
    {
      /* Get collision time in real time units */
      dt_done = dt*collision_time;
      /* Fix velocities */
      puck_next[XD] = b->puck[XD]*(1 - b->drag*dt_done) + b->force[XX]*dt_done;
      puck_next[YD] = b->puck[YD]*(1 - b->drag*dt_done) + b->force[YY]*dt_done;
      /* And go do collision */
      handle_collision( b, puck_next, collision_time,
			obstacle, side );
    }
  for( i = 0; i < N_STATES; i++ )
    b->puck[i] = puck_next[i];
  b->global_time += dt_done;

  /* A hack to fix bugs */
  fix_penetrating_obstacle( b );

  /*
  big_count++;
  if ( big_count > 10000 )
    {
      big_count = 0;
      printf( "shape_ref_count: %d\n", shape_ref_count );
    }
  */

  return dt_done;
}

/************************************************************************/
/* Return FALSE if in hole, TRUE if action applied correctly */

int apply_action( BOARD *b, float *u, float time_step, float *next_state,
		  int *status )
{
  float time_done = 0;
  int i;

  if ( in_hole( b, b->puck ) )
    {
      *status = HOLE;
      return FALSE;
    }

  b->force[XX] = u[XX];
  b->force[YY] = u[YY];
  for( i = 0; i < 1000; i++ ) /* limit number of iterations */
    {
      /*
	if ( b->puck[YD] < -0.024 )
	printf( " apply_action: %g %g %g\n",
	time_done, time_step, b->puck[YD] );
      */
      time_done = do_simulation( b, time_step );
      time_step -= time_done;
      if ( in_hole( b, b->puck ) )
	{
	  *status = HOLE;
	  return FALSE;
	}
      if ( time_step < 1e-4 )
	break;
    }
  if ( next_state != NULL )
    {
      for( i = 0; i < N_STATES; i++ )
	next_state[i] = b->puck[i];
      /*
      printf( "X1 %g %g %g %g\n",
			  b->puck[0], b->puck[1], b->puck[2], b->puck[3] );
      */
    }
  *status = 0;
  return TRUE;
}

/************************************************************************/
