#include <cthreads.h>

#include <mach/exc_server.h>
#include <mach/std_types.h>
#include <mach/port.h>
#include <mach/mach_traps.h>
#include <mach/host_info.h>
#include <mach/host_reboot.h>
#include <mach/mach_host.h>
#include <mach/mach_interface.h>
#include <mach/bootstrap.h>

#include <device/device_types.h>

#include "jiffies.h"

void setup_server ( int argc, char** argv );
unsigned long size_memory ( void );
void init_ports ( void );

void show_statistics ( void );
void show_paging ( void );
void show_clock ( void );

void console_init ( void );
int console_getc ( void );
void console_putc ( char c );

host_priority_info_data_t host_pri_info;
unsigned long osfmach3_mem_size = 0;

security_token_t server_security_token;

mach_port_t privileged_host_port;
mach_port_t device_server_port;
mach_port_t root_ledger_wired;
mach_port_t root_ledger_paged;
mach_port_t security_port;
mach_port_t default_pager_port;
mach_port_t bootstrap_port;
mach_port_t console_port;

mach_port_t host_port;

mutex_t console_in_mutex; /* only one person should get console input
			   at a time? */

void* spinner ( void* p ) { 
	int i;
	while(1) { 
/*	  printf("Hello, I'm spinner #%d\n", (int)p); */
		for ( i = 0; i < 1000000; i++ ) { 
		/*	cthread_yield(); */
		}
	}
	return NULL;
}

void main ( int argc, char** argv ) { 
	int 		i;
	kern_return_t 	kr;
	
	/* 
	 * Setup all of the important server structures
	 */
	setup_server(argc, argv);

	jiffies_init();

	/* 
	 * Spawn off a couple of idle threads
	 */
	for ( i = 0; i < 2; i++ )
		cthread_detach(cthread_fork(spinner, (void*)i));

	console_init();

	printf("Hello, I'm the server\n");
	
	while(1) { 
	  int c;

	  c = console_getc();
	  if ( c == -1 ) panic("console_getc() error");
	  else if ( c == '1' ) show_statistics();
	  else if ( c == '2' ) show_paging();
	  else if ( c == '3' ) show_clock();
	  else if ( c == '0' ) host_reboot(privileged_host_port, 
			     		   HOST_REBOOT_DEBUGGER);
	  else
	    console_putc(c);
	}
}

void console_putc ( char c ) { 
  if ( c == '\n' ) 
	printf("\n\r");
  else if ( c == '\r' ) 
	printf("\n\r");
  else printf("%c", c);
}

unsigned long size_memory ( void ) { 
	kern_return_t rc;
	int host_buff[HOST_BASIC_INFO_COUNT];
	mach_msg_type_number_t host_buff_size = HOST_BASIC_INFO_COUNT;
	
	rc = host_info(mach_host_self(), HOST_BASIC_INFO, 
		host_buff, &host_buff_size );
	if ( rc != KERN_SUCCESS ) { 
		panic("size_memory: host_info failed");
	}

	printf("cpu_type = %d\n", 
	       ((struct host_basic_info*)host_buff)->cpu_type);
	printf("cpu_subtype = %d\n", 
	       ((struct host_basic_info*)host_buff)->cpu_subtype);
	printf("max_cpus = %d\n", 
	       ((struct host_basic_info*)host_buff)->max_cpus);
	printf("avail_cpus = %d\n", 
	       ((struct host_basic_info*)host_buff)->avail_cpus);

	return ((struct host_basic_info*) host_buff)->memory_size;
}

void* mempcy ( void* dest, const void* src, size_t n ){ 
	bcopy ( dest, src, n );
	return dest;
}

void* memset ( void* dest, int c, size_t count ) { 
	char *xs = (char*) dest;
	while( count-- ) 
		*xs++ = c;
	return dest;
}

void setup_server ( int argc, char** argv ) 
{
	kern_return_t	kr;
	mach_msg_type_number_t	count;

	init_ports();

	/*
	 * Figure out how much memory we've got
	 */
	osfmach3_mem_size = size_memory();
	printf("Available memory : %dK/%dM\n", 
		osfmach3_mem_size/1024, (osfmach3_mem_size/1024)/1024);

	/*
	 * Get some host priority info - default priorities
	 * for kernel/user processes, etc.
	 */
	count = HOST_PRIORITY_INFO_COUNT;
	kr = host_info(mach_host_self(), HOST_PRIORITY_INFO,
			(host_info_t) &host_pri_info, &count);
	if ( kr != KERN_SUCCESS )
	  panic("host_info HOST_PRIORITY_INFO");
	
	/* 
	 * Make server unswappable (but it's still pageable).
	 */
	kr = task_swappable(privileged_host_port, mach_task_self(), FALSE);
	if ( kr != KERN_SUCCESS) { 
		panic("task_swappable %d", kr);
	}
}

void init_ports ( void ) { 
	kern_return_t	kr;
	unsigned int security_token_size;

	/* 
	 * Find the bootstrap port
	 */
	kr = task_get_special_port(mach_task_self(),
				TASK_BOOTSTRAP_PORT,
				&bootstrap_port );
	if ( kr != KERN_SUCCESS )
		panic("get bootstrap port %d", kr);

	/* 
	 * Ask for the privileged ports
	 */
	kr = bootstrap_ports ( bootstrap_port,
			&privileged_host_port,
			&device_server_port,
			&root_ledger_wired,
			&root_ledger_paged,
			&security_port);
	if ( kr != KERN_SUCCESS )
		panic("bootstrap_ports %d", kr );

	/*
	 * Get default_pager port
	 */
	kr = vm_set_default_memory_manager(privileged_host_port,
				&default_pager_port);
	if ( kr != KERN_SUCCESS ) 
		panic("getting default_pager port");


	/*
	 * Get the security token 
	 */
	security_token_size = TASK_SECURITY_TOKEN_COUNT;
	kr = task_info(mach_task_self(), TASK_SECURITY_TOKEN,
		       (task_info_t) &server_security_token,
		       &security_token_size);
	if ( kr != KERN_SUCCESS )
	  panic("server security token: %d", kr);

	host_port = mach_host_self();
}


void console_init ( void ) { 
  kern_return_t kr;
  static int console_initialized = 0;
  if ( !console_initialized ) { 
    kr = device_open(device_server_port, MACH_PORT_NULL,
		     D_READ|D_WRITE, server_security_token,
		     "console", &console_port );
    if ( kr != KERN_SUCCESS ) 
      panic("console_init");

    console_in_mutex = mutex_alloc();
    mutex_set_name(console_in_mutex, "console input mutex");
  }
}

int console_getc ( void ) { 
  int n;
  char c;
  kern_return_t kr;
  
  mutex_lock(console_in_mutex);
  kr = device_read_inband ( console_port, 0, 0, sizeof c, &c, &n );
  mutex_unlock(console_in_mutex);

  if ( kr != D_SUCCESS ) 
    panic("console_getc: read error %d", kr);
  else if ( n <= 0 ) 
    panic("console_getc: read returned %d bytes", n );

  return ((int)c) & 0xff;
}

void show_statistics ( void ) { 
	mach_msg_type_number_t count;
	kern_return_t kr;
	struct host_load_info hli;
	int i;
	integer_t avenrun;

	count = HOST_LOAD_INFO_COUNT;
	kr = host_statistics(privileged_host_port,
			HOST_LOAD_INFO,
			(host_info_t) &hli,
			&count);
	if ( kr != KERN_SUCCESS || count != HOST_LOAD_INFO_COUNT) {
		printf("No statistics found\n");
		return;
	}

	printf("Load info:\n");
	for ( i = 0; i < 3; i++ ) { 
		avenrun = hli.avenrun[i];
		if ( avenrun < LOAD_SCALE ) avenrun = 0;
		else avenrun -= LOAD_SCALE;

		printf("\tavenrun[%d] = %d\n", i, avenrun );
	}
	printf("\n");
}

void show_paging ( void ) { 
	kern_return_t kr;
	host_paging_info_data_t hpi;
	mach_msg_type_number_t count;

	count = HOST_VM_INFO_COUNT;
	kr = host_statistics ( privileged_host_port,
			HOST_VM_INFO,
			(host_info_t) &hpi,
			&count );
	if ( kr != KERN_SUCCESS || count != HOST_VM_INFO_COUNT ) {
		printf("no paging info found\n");
		return;
	}
	
	printf("Paging Info:\n");
	printf("\tsample_time\t: %d\n", hpi.sample_time );
	printf("\treset_time\t: %d\n", hpi.reset_time );
	printf("\tpagesize\t: %d\n", hpi.pagesize );
	printf("\tfree_count\t: %d\n", hpi.free_count );
	printf("\tactive_count\t: %d\n", hpi.active_count );
	printf("\tinactive_count\t: %d\n", hpi.inactive_count );
	printf("\twire_count\t: %d\n", hpi.wire_count );
	
	printf("\tfaults\t: %d\n", hpi.faults);
	printf("\tzero_fill_count\t: %d\n", hpi.zero_fill_count );
	printf("\tpageins\t: %d\n", hpi.pageins );
	printf("\tpages_pagedin\t: %d\n", hpi.pages_pagedin );
	printf("\tcow_faults\t: %d\n", hpi.cow_faults );
	printf("\treactivations\t: %d\n", hpi.reactivations );

	printf("\tlookups\t: %d\n", hpi.lookups );
	printf("\thits\t: %d\n", hpi.hits );
	printf("\tpageouts\t: %d\n", hpi.pageouts );
	printf("\tpages_pagedout\t: %d\n", hpi.pages_pagedout );
	printf("\n");
	
}

void show_clock ( void ) { 
	tvalspec_t curtime;
	curtime = jiffies_curtime();
	printf("Current Time:\n\tsec: %d (%d min)\n\tnsec: %d\n\tJiffies: %d\n", 
		cur_time.tv_sec, 
		(cur_time.tv_sec)/60, 
		cur_time.tv_nsec,
		jiffies_count());
	printf("\n");
} 
