/* katcpUtil.c -- API routines for TCP protocol
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

/* August 24, 1995 Modified by Teresa Webster- 
 *--------------------------------------------------------
 * The modifications fix special problems that occur when
 * calling KAPI functions from Lucid Lisp- 
 * THESE MODIFICATIONS MUST BE PORTED TO NEW VERSIONS
 * OF KATCPUTIL.C in order 
 * to run KAPI reliably from Lisp. The problem
 * is that Lisp periodically sends a SIGALARM signal to
 * itself. This will terminate any long running calls to Unix.
 * In particular a blocking BSD "read" and a long BSD
 * "connect" will terminate. There are two fixes:
 * (1) I have added a continue commands after the call
 *     to read in "TCPreadline".
 * (2) I block signals before the call to connect in
 *     in TCPconnect and then restore the signals.
 *
 * All of my added lines of code are marked with "taw"
 *-----------------------------------------------------------------
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <signal.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <fcntl.h>
#include <kapi.h>
#include <kapi_int.h>

#include <errno.h>  /*taw*/


#define  TCPIP_GATEWAY_PORT    9078

typedef struct sockaddr_in Sockaddr;

Sockaddr *resolveHost();

int
#if defined(__STDC__) || defined(__cplusplus)
TCPreadline(int socket, char *buf, int maxline)
#else
TCPreadline(socket, buf, maxline)
int socket, maxline;
char *buf;
#endif
{
  int i;
  char *localbuf = buf;
  char tmp;
  static fd_set fdset;
  static struct timeval timeout;
  timeout.tv_sec = 0.0;
  timeout.tv_usec = 0.0;


  for (i=0; (maxline==-1 || i<maxline); i++) {
    if (read(socket, localbuf, 1) < 0) {
      if(errno==EINTR) continue;                 /*taw*/
      else perror("Read socket");                /*taw*/
      return 0;
    };

    switch (*localbuf) {
      case MSG_EOF: /* i.e., ctrl-D -- single char (no buffer) from sender */
        *localbuf = '\0';
	FD_ZERO(&fdset);
	FD_SET(socket, &fdset);
	if (select(FD_SETSIZE, &fdset, NULL, NULL, &timeout) > 0) {
	  recv(socket, &tmp, 1, MSG_PEEK);
	  if (tmp == '\000') {
	    recv(socket, &tmp, 1, 0);
	    return(1);
	  }
	}
	return(2);
      case '\000':
        return(0);
      case EOF: 
        *localbuf = '\0'; 
        return 0;
    };
    localbuf++;
  };

  return -1;
}

/*
 * getUDPSocketAddr: to get the udp socket and bind it to the specified port
 * (for the server)
 */

int getUDPSocketAddr(port)
int port;
{
      int      socket_fd;
      Sockaddr socket_addr;

      bzero((char *)&socket_addr, sizeof(Sockaddr));
      socket_addr.sin_family = AF_INET;
      socket_addr.sin_addr.s_addr = 0;
      socket_addr.sin_port = htons(port);

      if ((socket_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	DPRINT((stderr, "getUDPSocketAddr: TCPIP Agent could not get socket.\n"));
	return -1;
      }

      if (bind(socket_fd, (struct sockaddr*)&socket_addr, sizeof(Sockaddr)) < 0) {
	DERROR( "getUDPSocketAddr: Socket bind" );
	DPRINT((stderr, "getUDPSocketAddr: TCPIP Agent could not bind socket.\n"));
	DPRINT((stderr, "\n\tWaiting 1 min before trying one more time.\n"));

	sleep(60); /* Try this once more time */

	if (bind(socket_fd, (struct sockaddr*)&socket_addr, sizeof(Sockaddr)) < 0) {
	  DERROR( "getUDPSocketAddr: Socket bind" );
	  return -1;
        } else {
	  DPRINT((stderr, "getUDPSocketAddr: Successfully bound socket.\n"));
        }
      } 

      return(socket_fd);
}


/*
 * UDPsend: to send the message to agent at receiver machine
 */

UDPsend(receiver, message, socket_fd, rmtPort)
char    *receiver, 
        *message;
int     socket_fd,
	rmtPort;
{
        Sockaddr  *recvaddr, rmtAddr;
	int msglen;

	msglen = strlen(message);

	if ((recvaddr = resolveHost(receiver)) == (Sockaddr *)NULL) {
	  DPRINT((stderr, 
		  "UDPsend: TCPIP Agent could not resolve receiver %s.\n", 
		  receiver));
	  return -1;
	}
	bcopy((caddr_t)recvaddr, (caddr_t)&rmtAddr, sizeof(Sockaddr));

	rmtAddr.sin_port = rmtPort;

	if (sendto(socket_fd, message, msglen, 0, (struct sockaddr*)&rmtAddr, sizeof(rmtAddr)) < 0) {
	  DPRINT((stderr, "UDPsend: TCPIP Agent could not send message.\n"));
	  return -1;
	}
	
	return 1;
}
	

/*
 * UDPrecv: to receive a message from an agent at receiver machine
 */

Sockaddr UDPrecv(message, msglen, socket_fd)
char    *message;
int     msglen, socket_fd;
{
        Sockaddr  rmtAddr;
	int       rmtlen;

	rmtlen = sizeof(rmtAddr);
	if (recvfrom(socket_fd, message, msglen, 0, (struct sockaddr*)&rmtAddr,&rmtlen) < 0) {
	  DPRINT((stderr, "UDPrecv: TCPIP Agent could not receive message.\n"));
	}
	
	return(rmtAddr);
}


/*
 * getTCPSocketAddr: to get the tcp socket and bind it to the specified port
 * (for the server)
 */

int getTCPSocketAddr(port)
int port;
{
      int      socket_fd;
      Sockaddr socket_addr;

      bzero((char *)&socket_addr, sizeof(Sockaddr));
      socket_addr.sin_family = AF_INET;
      socket_addr.sin_addr.s_addr = 0;
      socket_addr.sin_port = htons(port);

      if ((socket_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	DPRINT((stderr, "getTCPSocketAddr: TCPIP Agent could not get socket.\n"));
	return(-1);
      }

      if (bind(socket_fd, (struct sockaddr*)&socket_addr, sizeof(Sockaddr)) < 0) {
	  DERROR( "getTCPSocketAddr: Socket bind" );
	  DPRINT((stderr, "getTCPSocketAddr: TCPIP Agent could not bind socket. "));
	  DPRINT((stderr, "\n\tWaiting 1 min before trying one more time.\n"));

	  sleep(60); /* Try this once more time */

	  if (bind(socket_fd, (struct sockaddr*)&socket_addr, sizeof(Sockaddr)) < 0) {
	      DPRINT((stderr, 
	            "getTCPSocketAddr: TCPIP Agent still couldn't bind socket.\n"));
	      DERROR( "getTCPSocketAddr: Socket bind" );
	      return -1;
	  } else {
	    DPRINT((stderr, "getTCPSocketAddr: Successfully bound socket.\n"));
	  }
      } 

      return(socket_fd);
} 


/*
 * TCPlisten: ready to service client requests (for server)
 */

int TCPlisten(socket_fd)
int socket_fd;
{
      return(listen(socket_fd,5));
} 


/*
 * TCPaccept: accept a connection on a socket.
 */

int TCPaccept(socket_fd)
int socket_fd;
{
  int newsocket_fd;

  if ((newsocket_fd = accept(socket_fd, 0, 0)) == -1)
    {
	DPRINT((stderr, 
 	   "TCPaccept: TCPIP Agent could not accept a connection on the socket.\n"));
	DERROR( "TCPaccept" );
    }

  return newsocket_fd;
} 


/*
 * TCPconnect: to get the tcp socket and connect to the port of a specified
 * machine.
 */

int TCPconnect(hostname, port)
char* hostname;
int port;
{
      int      socket_fd;
      Sockaddr socket_addr;

      struct hostent *hp;

/* bitmaps of signals blocked from delivery */
#define NEW_SIGHANDLING
#ifndef NEW_SIGHANDLING
      int maskinit, mask;  /*taw*/
#else
		sigset_t set, oldset; /* dtm */
#endif

      if ((hp=gethostbyname(hostname)) == 0)
	{
	    DPRINT((stderr, 
		    "TCPconnect: TCPIP Agent could not find host name: %s.\n", 
		    hostname));
	  return -1;
	}

      bzero((char *)&socket_addr, sizeof(Sockaddr));
      socket_addr.sin_family = AF_INET;
      socket_addr.sin_addr.s_addr = ((struct in_addr *)(hp->h_addr))->s_addr;
      socket_addr.sin_port = htons(port);

      if ((socket_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	DPRINT((stderr, "TCPconnect: TCPIP Agent could not get socket.\n"));
	DERROR( "TCPconnect" );
	return -1;
      }

/*turn off signals before calling connect, because the resulting  */
/* interrupt causes connect() to fail */
#ifndef NEW_SIGHANDLING
      mask= sigmask(SIGALRM);      /*taw*/
      maskinit = sigblock(mask);   /*taw*/
      mask = sigmask(SIGINT);      /*taw*/
      sigblock(mask);              /*taw*/
      mask = sigmask(SIGTERM);     /*taw*/
      sigblock(mask);              /*taw*/
      mask = sigmask(SIGURG);      /*taw*/
      sigblock(mask);              /*taw*/
#else
			sigemptyset( &set );
			sigaddset( &set, SIGALRM );
			sigaddset( &set, SIGINT );
			sigaddset( &set, SIGTERM );
			sigaddset( &set, SIGURG );
			sigprocmask( SIG_BLOCK, &set, &oldset );
#endif

      if (connect(socket_fd, (struct sockaddr*)&socket_addr, sizeof(Sockaddr)) < 0) {
	DPRINT((stderr, "TCPconnect: TCPIP Agent could not connect to socket.\n"));
	DERROR( "TCPconnect" );
/* return to the previous level of signal blocking */
#ifndef NEW_SIGHANDLING
        sigsetmask(maskinit);      /*taw*/
#else
				sigprocmask( SIG_SETMASK, &oldset, 0 ); /* dtm */
#endif
	return -1;
      } 
/* return to the previous level of signal blocking */
#ifndef NEW_SIGHANDLING
      sigsetmask(maskinit);       /*taw*/
#else
				sigprocmask( SIG_SETMASK, &oldset, 0 ); /* dtm */
#endif
      return(socket_fd);
} 


/*
 * TCPsend: to send the message to agent at receiver machine
 */

int TCPsend(message, socket_fd)
char    *message;
int     socket_fd;
{
    int msglen;
    char terminal = MSG_EOF;
    msglen = strlen(message);

    DPRINT((stderr, "TCPsend sending: %s\n", message));

    signal(SIGPIPE, SIG_IGN);

    if (send(socket_fd, message, msglen, 0) < 0 ||
	send(socket_fd, &terminal, 1, 0) < 0) {
	DPRINT((stderr, "TCPsend: TCPIP Agent could not send message.\n"));
	signal(SIGPIPE, SIG_DFL);
	return -1;
    }
    else {
	signal(SIGPIPE, SIG_DFL);
	return 1;
    }
}
	

/*
 * TCPrecv: to receive a message from an agent at receiver machine
 */

int TCPrecv(message, msglen, socket_fd)
char    *message;
int     msglen, socket_fd;
{
  int flg;

  bzero(message, msglen);
  flg = TCPreadline(socket_fd, message, msglen);

  DPRINT((stderr, "TCPrecv receiving: %s\n", message));

  return(flg);
}


/*
 * Resolve the specified host name into an internet address.  The "name" may
 * be either a character string name, or an address in the form a.b.c.d where
 * the pieces are octal, decimal, or hex numbers.  Returns a pointer to a
 * sockaddr_in struct (note this structure is statically allocated and must
 * be copied), or NULL if the name is unknown.
 */

Sockaddr* resolveHost(name)
register char *name;
{
	register struct hostent *fhost;
	struct in_addr fadd;
	static Sockaddr sa;

	if ((fhost = gethostbyname(name)) != NULL) {
		sa.sin_family = fhost->h_addrtype;
		sa.sin_port = 0;
		bcopy(fhost->h_addr, &sa.sin_addr, fhost->h_length);
	} else {
		fadd.s_addr = inet_addr(name);
		if (fadd.s_addr != -1) {
			sa.sin_family = AF_INET;	/* grot */
			sa.sin_port = 0;
			sa.sin_addr.s_addr = fadd.s_addr;
		} else
			return(NULL);
	}
	return(&sa);
}
