/*
 * kahttp.c -- API routines for reading HTTP protocol
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <sys/types.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <kapi.h>
#include <kapi_int.h>

/*
 * HTTPreadline: 
 */

int
#if defined(__STDC__) || defined(__cplusplus)
HTTPreadline(int socket, char *buf, int maxline)
#else
HTTPreadline(socket, buf, maxline)
int socket, maxline;
char *buf;
#endif
{
  int i;

  for (i=0; (maxline==-1 || i<maxline); i++) {
    if (read(socket, buf, 1) < 0) {
      if ( errno != EWOULDBLOCK ) {
	  perror("Read socket");
      }
      return 0;
    };

    switch (*buf) {
      case '\r': *buf='\0'; buf--; break;
      case '\n': *buf='\0'; return 1;
      case MSG_EOF: /* i.e., ctrl-D -- single char (no buffer) from sender */
      case '\0':
      case EOF: *buf='\0'; return 0;
    };
    buf++;
  };
  return 1;
};

/*
 * HTTPrecv: to receive a message from an agent at receiver machine using
 * HTTP protocol
 */

int
#if defined(__STDC__) || defined(__cplusplus)
HTTPrecv(char *message, int msglen, int socket_fd)
#else
HTTPrecv(message, msglen, socket_fd)
char    *message;
int     msglen, socket_fd;
#endif
{
  char line[K_MAX_MESSAGE_SIZE], buf[K_MAX_MESSAGE_SIZE], *p;
  int content_length = -1;
  int line_length = 0,
      body_length = 0,
      message_length = 0,
      status = 0,
      answer_p = 0,
      sval = -1;

  line[0] = buf[0] = '\0';

  HTTPreadline(socket_fd, line, 80);

  strcpy(buf, line);

  p = strtok(buf, WHITESPACE);
  p = strtok(0, WHITESPACE);
  p = strtok(0, WHITESPACE);

  message_length = 0;

  /* Find out if this HTTP request is a full HTTP request */
  if (!p) {
    strcpy(message, line);
    strcat(message, "\n");
    DPRINT((stderr, "HTTPrecv receiving simple request: %s\n", message));
    message_length = strlen(line) + 1;
    return message_length;
  }

  if (strncmp (line, "HTTP/", 5) == 0) answer_p = 1;
  strcpy(message, line);
  strcat(message, "\n");

  message_length += strlen(line) + 1;

  /* Get header lines */
  while (HTTPreadline(socket_fd, line, 80)) {
    strcpy(buf, line);

    if ((p = strtok(buf, ":")) != (char*)NULL) {
      if (strcasecmp("Content-length", buf) == 0) {
	if ((p = strtok(0, WHITESPACE)) != (char*)NULL) 
	  content_length = atoi(p);
      }
      strcat(message, line);
      message_length += strlen(line);
    } else {
      strcat(message, "\n");
      message_length++;
      break; /* New line has been found */
    }

    strcat(message, "\n");
    message_length++;
    bzero(line, sizeof(line));
  }

  /* Get message body - Need to make sure that content_length is smaller than
     K_MAX_MESSAGE_SIZE and the addition of content to message is smaller than
     msglen */

  body_length = 0;

  status = 1;

  /* If this is an http request, and we don't have any information on 
   * the content-length, we want to continue reading, but don't want to 
   * block on the socket (since the socket isn't closed by the requester). 
   * With the O_NDELAY flag set, status will be set to zero when there's 
   * nothing more to read.
   */
  if (!answer_p && (content_length == -1)) {
      sval = fcntl(socket_fd, F_GETFL, 0);
      fcntl(socket_fd, F_SETFL, (sval | O_NDELAY)); 
  }

  while (((body_length < content_length) || (content_length == -1))
	 && status) {
    if (content_length > K_MAX_MESSAGE_SIZE) {
      status = HTTPreadline(socket_fd, line, K_MAX_MESSAGE_SIZE);
      content_length = content_length - K_MAX_MESSAGE_SIZE;
    }
    else status = HTTPreadline(socket_fd, 
			       line, 
			       (content_length == -1) ? 
			       K_MAX_MESSAGE_SIZE : content_length);

    line_length = strlen(line);

    body_length += line_length;

    if ((strlen(message) + line_length) > msglen) {
      DPRINT((stderr, 
	      "HTTPrecv: Error: Return buffer too small for message.\n"));
      return message_length;
    };

    strcat(message, line);
    strcat(message, "\n");
    message_length += strlen(line) + 1;
    bzero(line, sizeof(line));
  };

  /* Restore the original state of the socket, if it was changed. */
  if ( sval != -1 ) {
      fcntl(socket_fd, F_SETFL, sval);
  }

  DPRINT((stderr, "HTTPrecv buffer: %s", message));

  return message_length;
}
