/*
 * kaUtil.c -- KQML API Utility routines
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

/* kapi.h MUST be included last for ifndef MB_API_H in kapi.h to work */
#include <kapi.h>
#include <kapi_int.h>

/*
 *----------------------------------------------------------------------
 *
 * ParseURL --
 *
 *	Parses URL string into a ParsedURL struct.  Calls transport
 *      specific parsing function.
 *
 * Results:
 *	ParsedURL if successful
 *      NULL for failure
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
ParseURL(char* URLstring)
#else
ParseURL(URLstring)
     char* URLstring;
#endif
{
  char *s, *strtok();
  int TransType;
  ParsedURL *(*ParseFunc) _ANSI_ARGS_((char *, int));
  /* Make sure that it is a copy. */
  char url[MAXURLLENGTH];
  ParsedURL *purl;

  if ((URLstring == NULL) || (strlen(URLstring) > MAXURLLENGTH)) {
      return NULL;
  }

  strcpy(url, URLstring);

  s = strtok(url, ":");
  if (s == NULL || (TransType = TransportID(s)) == 0) 
    return(NULL); /* error condition <go>*/

  ParseFunc = Transports[TransType]->ParseURL;

  if ((purl = (*ParseFunc)(URLstring, TransType)) == NULL)
    return NULL;

  return(purl);
}


/*
 *----------------------------------------------------------------------
 *
 * ParsedMessToString --
 *
 *	Converts ParsedMessage struct into string
 *
 * Results:
 *	string if successful
 *      NULL for failure
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

char *
#if defined(__STDC__) || defined(__cplusplus)
ParsedMessToString(ParsedKQMLMess *pmessage, char *buffer)
#else
ParsedMessToString(pmessage, buffer)
     ParsedKQMLMess *pmessage;
     char *buffer;
#endif
{
  char tmpbuffer[K_MAX_MESSAGE_SIZE];
  KeyList *kl = NULL;

  tmpbuffer[0] = '\0';

  if (pmessage == NULL) return NULL;

  if (pmessage->performative == NULL) return NULL;

  sprintf(buffer, "(%s", pmessage->performative);

  if (pmessage->receiver != NULL) {
    sprintf(tmpbuffer, " :receiver %s", pmessage->receiver);
    strcat(buffer, tmpbuffer);
  }

  if (pmessage->sender != NULL) {
    sprintf(tmpbuffer, " :sender %s", pmessage->sender);
    strcat(buffer, tmpbuffer);
  }

  kl = pmessage->other_pairs;

  while(kl != NULL) {
    sprintf(tmpbuffer," %s %s",
	    kl->keyword,
	    kl->value);
    strcat(buffer, tmpbuffer);
    kl = kl->prev;
  }
  strcat(buffer,")");

  return(buffer);
}


/*
 *----------------------------------------------------------------------
 *
 * DeletePMessage --
 *
 *	Delete parsed message struct
 *
 * Results:
 *	none
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

void
#if defined(__STDC__) || defined(__cplusplus)
DeletePMessage(ParsedKQMLMess *pmessage)
#else
DeletePMessage(pmessage)
     ParsedKQMLMess *pmessage;
#endif
{
  if (pmessage == NULL)
    return;

  if (pmessage->performative)
    free(pmessage->performative);
  if (pmessage->receiver)
    free(pmessage->receiver);
  if (pmessage->sender)
    free(pmessage->sender);

  deleteKeyList(pmessage->other_pairs);

  free(pmessage);
}

/*
 *----------------------------------------------------------------------
 *
 * DeletePurl --
 *
 *	Delete parsed URL struct
 *
 * Results:
 *	none
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

void
#if defined(__STDC__) || defined(__cplusplus)
DeletePurl(ParsedURL *purl)
#else
DeletePurl(purl)
     ParsedURL *purl;
#endif
{
  ParsedURL *purl1 = NULL, *purl2 = NULL;
  
  purl1 = purl;

  while(purl1 != NULL) {
      if (( purl1->fd > 0 ) && ( purl->state != SHARED_CON )) {
	  close( purl1->fd );
      }
      purl2 = purl1->alternate;
      if (purl1->transinfo != NULL) {
	  free(purl1->transinfo);
      }
      free(purl1);
      purl1 = purl2;
  }
  return;
}


/*
 *----------------------------------------------------------------------
 *
 * kqml2mime --
 *
 *	Convert KQML string into a MIME string
 *
 * Results:
 *	returns MIME string ptr (malloc'd)
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

char *
#if defined(__STDC__) || defined(__cplusplus)
kqml2mime(char *buf)
#else
kqml2mime(buf)
char *buf;
#endif
  {
    int  length;
    char buffer[K_MAX_MESSAGE_SIZE],
         content_type[80], 
         content_length[10];
    char *performative=NULL;
    char *object=NULL;
    char *language=NULL;
    char *content=NULL;

    KeyList *kl = NULL, *tmpkl;
   
    /* The following approach is used to keep non-ANSI compilers happy.
       gcc allows direct initializtion of the KTable as in
    KeydPair KTable[] = {
      {":object", &object},
      {":language", &language},
      {":content", &content},
      {(char *) NULL, (char **) NULL}
    }; */

    KeydPair KTable[4];
    char *key1 = ":content";
    char *key2 = ":language";
    char *key3 = ":object";
    KTable[0].keyword = key1;
    KTable[0].value_address = &content;
    KTable[1].keyword = key2;
    KTable[1].value_address = &language;
    KTable[2].keyword = key3;
    KTable[2].value_address = &object;
    KTable[3].keyword = (char *) NULL;
    KTable[3].value_address = (char **) NULL;

    strcpy(buffer, buf);

    if ((KParse(buffer, &performative, KTable, &kl) != K_OK) || performative == NULL)
      /* Parse failure */
      return NULL;

    if (object) {
      sprintf(buffer, "%s %s HTTPv1.0\n", performative, object);
    } else 
      sprintf(buffer, "%s - HTTPv1.0\n", performative);

    tmpkl = kl;
    while (tmpkl != NULL) {
      strcat(buffer, &((tmpkl->keyword)[1]));
      strcat(buffer, ": ");
      strcat(buffer, tmpkl->value);
      strcat(buffer, "\n");
      tmpkl = tmpkl->prev;
    }

    sprintf(content_type, "kb/kif");
    if (language)
      strcpy(content_type, language);

    strcat(buffer, "Content-type: ");
    strcat(buffer, content_type);
    strcat(buffer, "\n");

    if (content) {
      strcat(buffer, "Content-length: ");
      length = strlen(content);
      sprintf(content_length, "%d", length);
      strcat(buffer, content_length);
      strcat(buffer, "\n\n");
      strncat(buffer, content, length);
      strcat(buffer, "\n");
    }
    else 
      strcat(buffer, "\n");
    
    if (content) free(content);
    if (language) free(language);
    if (object) free(object);
    if (performative) free(performative);
    deleteKeyList(kl);

    return strdup(buffer);
}


/*
 *----------------------------------------------------------------------
 *
 * mime2kqml --
 *
 *	Convert MIME string into a KQML string
 *
 * Results:
 *	returns KQML string ptr (malloc'd)
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

char *
#if defined(__STDC__) || defined(__cplusplus)
mime2kqml(char *buf)
#else
mime2kqml(buf)
char *buf;
#endif
  {
		static char kqml[K_MAX_MESSAGE_SIZE];
    char buffer[K_MAX_MESSAGE_SIZE],
         line[K_MAX_MESSAGE_SIZE],
         request[80], name[80], value[80],
         httpversion[80], 
         *p, *q, *r;

    strcpy(buffer, buf);

    p = buffer;

    /* Handling first line */
    q = p;
    while (p && *p != '\0') {
      if (*p == '\n') {
	p++;
	break;
      }
      p++;
    };
            
    line[0] = '\0';
    strncpy(line, q, p-q-1); /* Don't want the newline */

    r = strtok(line, WHITESPACE);

    if (r && strcmp(r, "") == 0) /* If r didn't have anything, then need to
                                    extract the next keyword. This is for the
				    case where the buffer was: " tell ..." */
      r = strtok(0, WHITESPACE);

    strcpy(request, r);

    strcpy(kqml, "(");
    strcat(kqml, request);

    if ((r = strtok(0, WHITESPACE)) && strncasecmp(r,"-",1)) {
      strcat(kqml, " :object ");
      strcat(kqml, r);
    }

    httpversion[0] = '\0';
    if ((r = strtok(0, WHITESPACE)) != (char*)NULL) {
      strcpy(httpversion, r);
    }

    if (*p == '\0') {
      strcat(kqml, ")");
      return kqml;
    }

    /* Skip over any initial parentheses and whitespace */
    while (*p != '\0' || *p != '\n') {

      q = p;
      while (p && *p != '\0') {
	if (*p == '\n') {
	  p++;
	  break;
	}
	p++;
      };
            
      if (p-q == 1) break; /* Hack to locate blank line */

      bzero(line, sizeof(line));
      strncpy(line, q, p-q-1); /* Don't want the newline */

      name[0] = '\0';
      if ((r = strtok(line, ":")) != (char*)NULL) {
	strcpy(name, r);
      }

      value[0] = '\0';

      /* Read to end of line */

      r = strtok(0, WHITESPACE);
      while (r) {
	strcat(value, r);
	r = strtok(0, WHITESPACE);
	if (r)
	  strcat(value, " ");
      }

      if (name && strcasecmp(name, "content-type") == 0) {
	strcat(kqml, " :language ");
	strcat(kqml, value);
      }
      else if (name && strcasecmp(name, "content-length") == 0) {
	strcat(kqml, " :content-length ");
	strcat(kqml, value);
      }
      else if (value && strcmp(value, "") && name &&
	       strcmp(name, "")) {
	strcat(kqml, " :");
	strcat(kqml, name);
	strcat(kqml, " ");
	strcat(kqml, value);
      }
    }

    /* Stick mime body into :content */

    q=p;
    if (*q) {
      strcat(kqml, " :content (");
      while (*p) p++;

      strncat(kqml, q, p-q-1);
      strcat(kqml, ")");
    }

    strcat(kqml, ")");

    return strdup(kqml);
  }


/*
 *----------------------------------------------------------------------
 *
 * flushPurlCache --
 *
 *      Deletes all entries in the given cache (hash table).
 *
 * Results:
 *	K_OK.
 *
 * Side effects:
 *      Hash table is empty.
 *
 *----------------------------------------------------------------------
 */

int 
#if defined(__STDC__) || defined(__cplusplus)
flushPurlCache( HashTable* table )
#else
flushPurlCache( table )
HashTable* table;
#endif
{
    HashSearch* searchPtr = (HashSearch *) calloc(1, sizeof(HashSearch));
    HashEntry*  entry = (HashEntry *) FirstHashEntry(table, searchPtr);

    while (entry != NULL) {
	DeletePurl( (ParsedURL*) GetHashValue(entry) );
	DeleteHashEntry(entry);
	entry = NextHashEntry(searchPtr);
    }

    free(searchPtr);
    return( K_OK );
}


/*
 *----------------------------------------------------------------------
 *
 * deletePurlCacheEntry --
 *
 *      Deletes entry in the cache.
 *
 * Results:
 *	K_OK if entry is found.
 *      K_ERROR if entry is not found.
 *
 * Side effects:
 *      Hash table entry is deleted.
 *
 *----------------------------------------------------------------------
 */

int 
#if defined(__STDC__) || defined(__cplusplus)
deletePurlCacheEntry( char* key, HashTable* table )
#else
deletePurlCacheEntry( key, table )
char      *key;
HashTable *table;
#endif
{
    HashEntry *entry;

    if ((entry = FindHashEntry(table, key)) != NULL) {
	DeletePurl( (ParsedURL *) GetHashValue(entry) );
	DeleteHashEntry(entry);
	return( K_OK );
    }

    return( K_ERROR );
}


/*
 *----------------------------------------------------------------------
 *
 * deletePurlCacheEntryURL --
 *
 *      Removes URL from given agent's entry in cache.
 *
 * Results:
 *	K_OK if entry is found.
 *      K_ERROR if entry is not found.
 *
 * Side effects:
 *      Hash table entry is modified.
 *
 *----------------------------------------------------------------------
 */

int 
#if defined(__STDC__) || defined(__cplusplus)
deletePurlCacheEntryURL( char* aname, char* url, HashTable* table )
#else
deletePurlCacheEntryURL( aname, url, table )
char      *aname;
char      *url;
HashTable *table;
#endif
{
    HashEntry *entry;

    if ((entry = FindHashEntry(table, aname)) != NULL) {
	return( deleteURLFromEntry(url, entry) );
    } 

    return( K_ERROR );
}


/*
 *----------------------------------------------------------------------
 *
 * deletePurlCacheEntriesURL --
 *
 *      Removes URL from given all agent's entries in cache.
 *
 * Results:
 *	K_OK.
 *
 * Side effects:
 *      Hash table entries are modified.
 *
 *----------------------------------------------------------------------
 */

int 
#if defined(__STDC__) || defined(__cplusplus)
deletePurlCacheEntriesURL( char* url, HashTable* table )
#else
deletePurlCacheEntriesURL( url, table )
char      *url;
HashTable *table;
#endif
{
    HashSearch *searchPtr = (HashSearch *) calloc(1, sizeof(HashSearch));
    HashEntry *entry = (HashEntry *) FirstHashEntry(table, searchPtr);

    while (entry != NULL) {
	deleteURLFromEntry(url, entry);
	entry = NextHashEntry(searchPtr);
    }

    free(searchPtr);
    return K_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * deleteURLFromEntry --
 *
 *      Removes the ParsedURL struct with the matching url from the
 *      hash table entry.
 *
 * Results:
 *	K_OK if successful.
 *      K_ERROR if url not found.
 *
 * Side effects:
 *      Hash table entries are modified.
 *
 *----------------------------------------------------------------------
 */

int 
#if defined(__STDC__) || defined(__cplusplus)
deleteURLFromEntry( char* url, HashEntry* entry )
#else
deleteURLFromEntry( url, entry )
char      *url;
HashEntry *entry;
#endif
{
    ParsedURL* purl     = (ParsedURL *) GetHashValue(entry);
    ParsedURL* lastpurl = NULL;
    
    while(purl != NULL) {
	if (strcmp(purl->url, url) == 0) {
	    if ((lastpurl == NULL) && (purl->alternate == NULL)) {
		/* Only purl in entry */
		DeletePurl(purl);
		DeleteHashEntry(entry);

	    } else if (lastpurl == NULL) { 
		/* Head of list */
		SetHashValue(entry,purl->alternate);
		purl->alternate->hashed = 1;
		purl->alternate = NULL;
		DeletePurl( purl );

	    } else { 
		lastpurl->alternate = purl->alternate;
		purl->alternate = NULL;
		DeletePurl( purl );
	    }
	    return( K_OK );
	}

	lastpurl = purl;
	purl = purl->alternate;
    } 

    return( K_ERROR );
}


/*
 *----------------------------------------------------------------------
 *
 * printCache --
 *
 *      Prints the contents of cache to stdout.
 *      
 *
 * Results:
 *	K_OK.
 *
 * Side effects:
 *      Hash table entries are printed.
 *
 *----------------------------------------------------------------------
 */

int 
#if defined(__STDC__) || defined(__cplusplus)
printCache( HashTable* table )
#else
printCache( table )
HashTable *table;
#endif
{
    HashSearch* searchPtr = (HashSearch *) calloc(1, sizeof(HashSearch));
    HashEntry*  entry = (HashEntry *) FirstHashEntry(table, searchPtr);
    ParsedURL*  purl;
    int ent = 1;
    int sub;

    while (entry != NULL) {
	purl = (ParsedURL *) GetHashValue(entry);
	sub = 0;
	while(purl != NULL) {
	    printf( "%2d.%d Name: %s\n", ent, sub, purl->name );
	    printf( "\tURL:   %s\n\tFdesc: %d\n\tHash:  %d\n", 
		    purl->url, purl->fd, purl->hashed );

	    switch ( purl->transtype ) {
	      case 1:
		printf( "\tTrans: smail\n" );
		break;
	      case 2:
		printf( "\tTrans: mbus\n" );
		break;
	      case 3:
		printf( "\tTrans: tcp\n" );
		break;
	      case 4:
		printf( "\tTrans: http\n" );
		break;
	      default:
		printf( "\tTrans: unknown\n" );
	    }

	    switch ( purl->state ) {
	      case OPEN_CON:
		printf( "\tState: OPEN_CON\n" );
		break;
	      case CLOSED_CON:
		printf( "\tState: CLOSED_CON\n" );
		break;
	      case LISTENING_CON:
		printf( "\tState: LISTENING_CON\n" );
		break;
	      case CONNECTIONLESS:
		printf( "\tState: CONNECTIONLESS\n" );
		break;
	      case CONNECTION_ERROR:
		printf( "\tState: CONNECTION_ERROR\n" );
		break;
	      case SHARED_CON:
		printf( "\tState: SHARED_CON\n" );
		break;
	      default:
		printf( "\tState: %d\n", purl->state );		
	    }

	    purl = purl->alternate;
	    sub++;
	}
	ent++;
	entry = NextHashEntry(searchPtr);
    }

    free(searchPtr);
    return K_OK;
}


static char* 
#if defined(__STDC__) || defined(__cplusplus)
genName(char* rootname)
#else
genName(rootname)
     char* rootname;
#endif
{ 
    static int id        = 0;
    static int id_max_sz = 16;  /* Should calculate this from sizeof(int) */
    
    char* newname = (char*)malloc( strlen( rootname ) + 2 + id_max_sz );
    
    sprintf( newname, "%s;%d", rootname, ++id );

    return( newname );
}


char* 
#if defined(__STDC__) || defined(__cplusplus)
canonicalName(char* agentname)
#else
canonicalName(agentname)
     char* agentname;
#endif
{
    char* name = strdup( agentname );

    return( strtok( name, ";" ) );
}


int
#if defined(__STDC__) || defined(__cplusplus)
duplicatedName( char* purlname, char* sender )
#else
duplicatedName( purlname, sender )
     char* purlname;
     char* sender;
#endif
{
    int len = strlen( sender );

    if ( strcmp( purlname, sender ) == 0 ) {
	return( 0 );
    }

    if (( strstr( purlname, sender ) == purlname ) &&
	( purlname[ len ] == ';' )) {
	return( 1 );
    }
    
    return( 0 );
}


ParsedURL* 
#if defined(__STDC__) || defined(__cplusplus)
insertConnectionTable(char* agentname, ParsedURL* purl)
#else
insertConnectionTable(agentname, purl)
     char* agentname;
     ParsedURL* purl;
#endif
{ 
    HashEntry* entry   = NULL;
    ParsedURL* altpurl = NULL;
    char* hashkey      = NULL;
    int   flg     = 0;
    int   retries = 0;

    if ((agentname==NULL) || (purl==NULL)) {
	return( NULL );
    }

    if ( purl->hashed ) {  /* Already in connection table */
	return( purl );
    }

    hashkey = strdup( agentname );

    while (( !flg ) && ( retries++ < 10 )) {
	entry = CreateHashEntry( &OpenConns, hashkey, &flg );

	if (!flg) {
	    DPRINT((stderr, 
		    "insertConnectionTable: Duplicate entry for %s.\n", 
		    hashkey ));
	    free( hashkey );
	    hashkey = genName( agentname );

	} else {
	    strcpy( purl->name, hashkey );
	    SetHashValue( entry, purl );
	    altpurl = purl;
	    while( altpurl != NULL ) {
		altpurl->hashed = 1;
		altpurl = altpurl->alternate;
	    }
	    return( purl );
	}
    }

    free( hashkey );
    return( NULL );
}
