/*
 * kaTCP.c -- KQML API TCP transport specific routines
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifdef DO_TCP_TRANS

#include <kapi_int.h>
#include <kaTCP.h>


/*
 *----------------------------------------------------------------------
 *
 * tcpParseURL --
 *
 *	Parses URLs beginning with 'tcp://'
 *
 * Results:
 *	ParsedURL if successful
 *      NULL for failure
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
tcpParseURL(char *urlstring, int ttype)
#else
tcpParseURL(urlstring, ttype)
     char *urlstring;
     int ttype;
#endif
{
  ParsedURL *purl;
  TCPInfo *info;
  char tmpbuffer[MAXURLLENGTH], *t = NULL;

  tmpbuffer[0] = '\0';

  if (urlstring == NULL || strncasecmp(urlstring, "tcp://", 6)) {
    return(NULL);
  }

  purl = (ParsedURL *) calloc(1,sizeof(ParsedURL));
  info = (TCPInfo *) calloc(1,sizeof(TCPInfo));

  strcpy(tmpbuffer, &urlstring[6]); /* Skip "tcp://" part */
  strcpy(info->host, strtok(tmpbuffer, ":"));

  if ((t = strtok(NULL,"")) && 
      (info->port = atoi(t))) {
    /* specified port is fine */
  }
  else {
    DPRINT((stderr,"Bad TCP URL: port required.\n"));
    free(purl);
    free(info);
    return(NULL);
  }
  purl->transtype = ttype;
  purl->state = CLOSED_CON;
  purl->fd = -1;
  purl->transinfo = (void *) info;
  purl->hashed = 0;

  strcpy(purl->url, urlstring);
  strcpy(purl->name, "none");

  purl->alternate = NULL;
  
  return(purl);
}


/*
 *----------------------------------------------------------------------
 *
 * tcpListen --
 *
 *	Listen on tcp url host & port
 *
 * Results:
 *	1 if successful
 *      0 for failure
 *
 * Side effects:
 *      Possible establishment of connection
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
tcpListen(ParsedURL *purl)
#else
tcpListen(purl)
     ParsedURL *purl;
#endif
{
  TCPInfo *info = (TCPInfo *) purl->transinfo;
  char hostname[MAXNAMESIZE];

  hostname[0] = '\0';

  if (purl->fd > 0) {
    close(purl->fd);    /* avoid a leak if we're already open */
    purl->fd = -1;
  }

  /* if the given host is the local host, then bind to the given
     port number; otherwise, I don't know what to do */

  if (gethostname(hostname, MAXNAMESIZE) < 0) {
    DPRINT((stderr, "tcpListen: can't find hostname\n"));
    DERROR( "tcpListen" );
    purl->state = CONNECTION_ERROR;
    return(0);
  }

  if (strncmp(hostname, info->host, strlen(hostname)) != 0) {
    DPRINT((stderr, "tcpListen: Attempt to listen on foreign host <%s> aborted\n", 
	    info->host));
    purl->state = CONNECTION_ERROR;
    return(0);
  }

  if ((purl->fd = getTCPSocketAddr(info->port)) != -1) {
    if (TCPlisten(purl->fd) != -1) {
      purl->state = LISTENING_CON;
    } else {
      DPRINT((stderr, "tcpListen: Can't listen to socket.\n"));
      DERROR( "tcpListen" );
      purl->state = CONNECTION_ERROR;
      close(purl->fd);
      purl->fd = -1;
      return(0);
    }
  }
  else {
    DPRINT((stderr, "tcpListen: Can't find socket.\n"));
    DERROR( "tcpListen" );
    purl->state = CONNECTION_ERROR;
    return(0);
  }
    
  return(1);
}


/*
 *----------------------------------------------------------------------
 *
 * tcpConnect --
 *
 *	Accept connection on tcp url host & port
 *
 * Results:
 *	parsedURL of new connection
 *      NULL for failure
 *
 * Side effects:
 *      Possible establishment of connection.
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
tcpConnect(ParsedURL *purl, int ttype)
#else
tcpConnect(purl, ttype)
     ParsedURL *purl;
     int ttype;
#endif
{
  int fd;
  ParsedURL *newpURL = NULL;
  TCPInfo   *info    = NULL;

  fd = TCPaccept(purl->fd);
  if (fd < 0)
    return NULL;

  newpURL = (ParsedURL *) calloc(1,sizeof(ParsedURL));
  info = (TCPInfo *) calloc(1,sizeof(TCPInfo));

  newpURL->fd = fd;
  strcpy(info->host, "unknown");
  info->port = 0;

  newpURL->state = OPEN_CON;
  newpURL->transinfo = info;
  newpURL->transtype = ttype;
  newpURL->hashed = 0;

  return(newpURL);
}


/*
 *----------------------------------------------------------------------
 *
 * tcpDisConnect --
 *
 *      Terminate connection on tcp url host & port
 *
 * Results:
 *	returns 1
 *
 * Side effects:
 *      Removal of connection.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
tcpDisconnect(ParsedURL *purl)
#else
tcpDisconnect(purl)
     ParsedURL *purl;
#endif
{
  struct linger l;

  if (purl->state == OPEN_CON) {
    l.l_onoff = 0;
    l.l_linger = 0.0;
    setsockopt(purl->fd, SOL_SOCKET, SO_LINGER, (char *) &l, sizeof(l));
	       
    DPRINT((stderr, "Closing TCP file descriptor on port %d\n",
	    ((TCPInfo *)(purl->transinfo))->port));
    close(purl->fd);
    purl->fd = -1;
    purl->state = CLOSED_CON;
  }

  return(1);
}


/*
 *----------------------------------------------------------------------
 *
 * tcpGet --
 *
 *      Get message on tcp url.
 *
 * Results:
 *	returns message if successful.
 *      NULL for failure.
 *
 * Side effects:
 *      Possible removal of connection, updating of OpenConns hashtable.
 *
 *----------------------------------------------------------------------
 */

char *
#if defined(__STDC__) || defined(__cplusplus)
tcpGet(ParsedURL *purl, char *str)
#else
tcpGet(purl, str)
     ParsedURL *purl;
     char *str;
#endif
{
  int flg;
  HashEntry *entry = NULL;

  if ((flg = TCPrecv(str, K_MAX_MESSAGE_SIZE, purl->fd)) < 2) {
    /* End of file received.  Delete connection. */
    purl->state = CLOSED_CON;
    close(purl->fd);
    purl->fd = -1;
    if (( purl->hashed ) &&
	((entry = FindHashEntry(&OpenConns, purl->name)) != NULL)) {
	DeleteHashEntry(entry);
    }
    if (flg < 1)
      return(NULL);
  }
  return(str);
}


/*
 *----------------------------------------------------------------------
 *
 * tcpSend --
 *
 *      Send message on tcp url
 *
 * Results:
 *	returns 1 if successful
 *      -1 for failure
 *
 * Side effects:
 *      Possible new connection.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
tcpSend(ParsedURL *purl, char *buffer)
#else
tcpSend(purl, buffer)
     ParsedURL *purl;
     char *buffer;
#endif
{
  TCPInfo *info = (TCPInfo *) purl->transinfo;

  if (purl->state != OPEN_CON) {
    purl->fd = TCPconnect(info->host, info->port);

    if (purl->fd == -1)
      return -1;

    purl->state = OPEN_CON;
  }

  TCPsend(buffer, purl->fd);
    
  return(1);
}

#endif
