/*
 * kaSend.c -- KQML API send routines
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <kapi.h>
#include <kapi_int.h>


/*
 *----------------------------------------------------------------------
 *
 * sendOnTransport --
 *
 *	Sends given message to given parsed url destination
 *
 * Results:
 *	1  - if successful
 *      -1 - for failure
 *
 * Side effects:
 *      OpenConns hash table of open connections may be updated, and new
 *      connections may be established.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
sendOnTransport(ParsedURL *purl, char *buf)
#else
sendOnTransport(purl, buf)
     ParsedURL *purl;
     char *buf;
#endif
{
  int (*SendFunc) _ANSI_ARGS_((ParsedURL *, char *));
  SendFunc = Transports[purl->transtype]->Send;
  return((*SendFunc)(purl, buf));
}


/*
 *----------------------------------------------------------------------
 *
 * KSendString --
 *
 *	Sends given message.  Parses message to find appropriate destination.
 *
 * Results:
 *	K_OK = success
 *      K_BAD_KQML = could not send bad kqml message
 *      K_NO_RECEIVER = no receiver was specified in kqml msg
 *      K_NO_TRANSPORT = could't find the transport for that agent name
 *
 * Side effects:
 *      OpenConns hash table of open connections may be updated, and new
 *      connections may be established.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KSendString(char* message)
#else
KSendString(message)
     char *message;        /* this is a KQML message to send */
#endif
{
    char buffer[K_MAX_MESSAGE_SIZE];
    ParsedURL *purl = NULL;
    char *receiver  = NULL;
    ParsedKQMLMess *pmessage = 
	(ParsedKQMLMess *) calloc(1, sizeof(ParsedKQMLMess));
   
    /* The following approach is used to keep non-ANSI compilers happy.
       gcc allows direct initializtion of the KTable as in
       KeydPair KTable[] = {
       {":sender", &(pmessage->sender)},
       {":receiver", &receiver},
       {(char *) NULL, (char **) NULL}
       }; */
    KeydPair KTable[3];
    char *key1 = ":sender";
    char *key2 = ":receiver";
    KTable[0].keyword = key1;
    KTable[0].value_address = &(pmessage->sender);
    KTable[1].keyword = key2;
    KTable[1].value_address = &receiver;
    KTable[2].keyword = (char *) NULL;
    KTable[2].value_address = (char **) NULL;
    
    buffer[0] = '\0';

    if ((KParse(message, 
	       &(pmessage->performative), 
	       KTable, 
	       &(pmessage->other_pairs)) != K_OK) || 
	*(pmessage->performative) == NULL) {

	/* Parse failure. Trying to send invalid KQML message */
	DPRINT((stderr,
		"Attempt to send unparsable KQML message. Send Aborted\n"));
	DeletePMessage(pmessage);
	if (receiver) free (receiver);
	return(K_BAD_KQML);
    }

    /* Must decide which transport to use to send string buf. */
    if (receiver == NULL) {
	DeletePMessage(pmessage);
	DPRINT((stderr, "No receiver in message. Send Aborted.\n"));
	return K_NO_RECEIVER;
    }

    purl = getReceiverURL( receiver );

    /* Remove any bookkeeping info from the name before replying */
    pmessage->receiver = canonicalName( receiver );

    while (purl != NULL) {
	if (sendOnTransport(purl, ParsedMessToString(pmessage,buffer)) != -1) {
	    DeletePMessage(pmessage);
	    if (receiver) free(receiver);

#if (DEBUG>0) 	    
	    printf( "------------- Connections Cache -----------------\n" );
	    printf( "-------------------------------------------------\n" );
	    printCache( &OpenConns );
	    printf( "-------------------------------------------------\n" );
#endif
	    return(K_OK);
	}
	purl = purl->alternate;
    }

    DeletePMessage(pmessage);
    if (receiver) free(receiver);

    DPRINT((stderr, "Send failure: Could not find a suitable transport.\n"));

    return K_NO_TRANSPORT;
}


/*
 *----------------------------------------------------------------------
 *
 * getReceiverURL --
 *
 *	Finds a parsed url corresponding to given agent identifier.  Searches
 *      in the following sequence:
 *         1)  Look for open connection
 *         2)  See if agent id is a direct address (URL)
 *         3)  Query ANS for agent id
 *
 * Results:
 *	returns parsed url if successful
 *      returns NULL for failure
 *
 * Side effects:
 *      Results of ANS query are cached in hash table.
 *
 *----------------------------------------------------------------------
 */

ParsedURL *
#if defined(__STDC__) || defined(__cplusplus)
getReceiverURL(char *receiver)
#else
getReceiverURL(receiver)
char *receiver;
#endif
{
  HashEntry *entry = NULL;
  ParsedURL *newURL = NULL;

  if (receiver == NULL)
    return(NULL);

  /* Check to see if connection to receiver is open */
  if((entry = FindHashEntry(&OpenConns, receiver)) != NULL)
    return((ParsedURL *) GetHashValue(entry));

  /* Check to see if receiver is a direct address */
  if ((newURL = ParseURL(receiver)) != NULL) {
      return( insertConnectionTable( receiver, newURL ) );
  }

  /* Query ANS (Agent Name Server) if available */
  DPRINT((stderr, "getReceiverURL: Querying ANS.\n"));

  if (gANSpurl == NULL) {
      DPRINT((stderr, "getReceiverURL: No ANS available. Abort send.\n"));
      return(NULL);
  }

  if((newURL = ANSLookup(receiver, gANSpurl)) == NULL) {
    DPRINT((stderr, 
	    "getReceiverURL: No transport for agent '%s'. Abort send.\n",
	    receiver));
    return(NULL);
  }

  return( insertConnectionTable( receiver, newURL ) );
}
