/*
 * kaRouter.c -- KQML API interface to the JATLite Router service
 *
 * Copyright (c)  1993, 1994 Enterprise Integration Technologies Corporation
 * and Lockheed Missiles and Space Company, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name of
 * Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company may not be used in any advertising or publicity
 * relating to the software without the specific, prior written permission
 * of Enterprise Integration Technologies Corporation and Lockheed Missiles
 * and Space Company.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL ENTERPRISE INTEGRATION TECHNOLOGIES CORPORATION OR
 * LOCKHEED MISSILES AND SPACE COMPANY BE LIABLE FOR ANY SPECIAL,
 * INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY
 * THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Added by Heecheol Jeon, jhc@cdr.stanford.edu
 * 03/01/97
 */

/* kapi.h MUST be included last for ifndef MB_API_H in kapi.h to work */
#include <kapi.h>
#include <kapi_int.h>

/************************************************************************
 Convert address string to Address object, defined in kapi.h

 Address string KQML format :
 (agent-address
  :name <agent-name>
  :password <password>
  :host <agent's internet address(not IP)>
  :port <port number>
  :message-method <message method>
  :email <email address>
  :description <agent description>
  :KQML-extensions <kqml extensions>
  )

 For more information about address, refer to :
 http://cdr.stanford.edu/ProcessLink/kqml-proposed.html

 Heecheol Jeon
 March 01, 1997
 ********************************************************************/
Address*
#if defined(__STDC__) || defined(__cplusplus)
ConvertKQMLToAddress(char* addr_str)
#else
ConvertKQMLToAddress(addr_str)
     char *addr_str;        
#endif
{
  Address* myaddress = (Address *)calloc(1,sizeof(Address));
  char* name = myaddress->name;
  char* password = myaddress->password;
  char* host = myaddress->host;
  char* port = myaddress->port;
  char* message_method = myaddress->message_method;
  char* email = myaddress->email;
  char* kqml_extensions = myaddress->kqml_extensions;

  KGetField( addr_str, ":name", &name, 0 );
  KGetField( addr_str, ":password", &password, 0 );
  KGetField( addr_str, ":host", &host, 0 );
  KGetField( addr_str, ":port", &port, 0 );
  KGetField( addr_str, ":message-method", &message_method, 0 );
  KGetField( addr_str, ":email", &email, 0 );
  KGetField( addr_str, ":KQML-extensions", &kqml_extensions, 0 );

  if((name == NULL) || (password == NULL) || (host == NULL) ||
     (port == NULL)  || (message_method == NULL)) {
    return NULL;
  }

  strcpy(myaddress->name,name);
  strcpy(myaddress->password,password);
  strcpy(myaddress->host,host);
  strcpy(myaddress->port,port);
  strcpy(myaddress->message_method,message_method);
  if(email != NULL) {
    strcpy(myaddress->email,email);
  }
  
  if(kqml_extensions != NULL) {
    strcpy(myaddress->kqml_extensions,kqml_extensions);
  }
  
  return myaddress;
}


/*
 *----------------------------------------------------------------------
 *
 * KRouterRegisterWithString : Register to the Router with address string
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = can't parse url
 *      K_NO_ANS_RESPONSE = no response from the name server
 *      K_ERROR = to use AgentLookup you must compile with HTTP support,
 *                or another error occured
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KRouterRegisterWithString(char* addr_str, char *routername)
#else
KRouterRegisterWithString(addr_str, routername)
     char *addr_str;        /* the name of the agent to lookup */
     char *routername; /* the Router name */
#endif
{
  int retval;
  Address* myaddress = (Address *)ConvertKQMLToAddress(addr_str);
  if(myaddress == NULL) {
    return (K_BAD_KQML);
  }

  retval = KRouterRegister(myaddress,routername);
  free(myaddress);
  return retval;
} 
 
/*
 *----------------------------------------------------------------------
 *
 * KRouterRegister : Register to the Router with address object
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = can't parse url
 *      K_NO_ANS_RESPONSE = no response from the name server
 *      K_ERROR = to use AgentLookup you must compile with HTTP support,
 *                or another error occured
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */
int
#if defined(__STDC__) || defined(__cplusplus)
KRouterRegister(Address *myaddress, char *routername)
#else
KRouterRegister(myaddress, routername)
     Address *myaddress;        /* the name of the agent to lookup */
     char *routername; /* the Router name */
#endif
{
  char* responsePerf = NULL;
  ParsedURL *routerURL = NULL;
  char buffer[K_MAX_MESSAGE_SIZE];

  bzero(buffer, K_MAX_MESSAGE_SIZE);

  routerURL = (ParsedURL*)getReceiverURL(routername);
printf("%s\n","0");

  /* Kapi is always client initiate message, and all of them are server*/
  strcpy(buffer,"(register-agent :sender ");
  strcat(buffer,myaddress->name);
  strcat(buffer," :password ");
  strcat(buffer,myaddress->password);
  strcat(buffer," :receiver ");
  strcat(buffer,routername);
  strcat(buffer,")");
printf("%s\n",buffer);
  if(!KSendString(buffer) == K_OK) {
    return K_ERROR;
  }

  if (!(routerURL && getFromTransport(routerURL, buffer))) {
printf("%s\n","Input error");
    return (K_NO_ANS_RESPONSE);
  }
printf("%s\n","1");  
  bzero(buffer, K_MAX_MESSAGE_SIZE);
  if(!strncmp(buffer,"201 AMR",7)) {
printf("%s\n","Incompatible Router");
    return (K_ERROR);
  }

  bzero(buffer, K_MAX_MESSAGE_SIZE);
  /* wait "identify-self" message */
  if (!getFromTransport(routerURL, buffer)) {
printf("%s\n","Identify self expected");
    return (K_NO_ANS_RESPONSE);
  }

  KGetField( buffer,NULL, &responsePerf, 0 );

printf("Response : %s",responsePerf);

  if(!(strncmp(responsePerf,"identify-self",13) == 0)) {
    return K_ERROR;
  }

  free(responsePerf);

printf("%s\n","identify-self");
  /* send "whoiam" message */
  strcpy(buffer,"(whoiam :sender ");
  strcat(buffer,myaddress->name);
  strcat(buffer," :receiver ");
  strcat(buffer,routername);
  strcat(buffer," :message-method ");
  strcat(buffer,myaddress->message_method);
  strcat(buffer," :content (contact-information :host ");
  strcat(buffer,myaddress->host);
  strcat(buffer," :port ");
  strcat(buffer,myaddress->port);
  if(myaddress->email != NULL) {
    strcat(buffer," :email ");
    strcat(buffer,myaddress->email);
  }

  if(myaddress->kqml_extensions != NULL) {
    strcat(buffer," :KQML-extensions ");
    strcat(buffer,myaddress->kqml_extensions);
  }

  strcat(buffer,"))");
printf("%s\n",buffer);
  if(KSendString(buffer) != K_OK) {
    return K_ERROR;
  }

  bzero(buffer, K_MAX_MESSAGE_SIZE);
  /* wait "register-accepted" message */
  if (!getFromTransport(routerURL, buffer)) {
    return K_ERROR;
  }

  KGetField( buffer,NULL, &responsePerf, 0 );
  if(responsePerf == NULL) {
    return K_ERROR;
  }

  if(!(strncmp(responsePerf,"register-accepted",17) == 0)) {
    if(responsePerf)  free(responsePerf);
    return K_ERROR;
  }

  if(responsePerf)  free(responsePerf);
  return (K_OK);
}

/*
 *----------------------------------------------------------------------
 *
 * KRouterConnect : Connect to the Router with address string
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = can't parse url
 *      K_NO_ANS_RESPONSE = no response from the name server
 *      K_ERROR = to use AgentLookup you must compile with HTTP support,
 *                or another error occured
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */
int
#if defined(__STDC__) || defined(__cplusplus)
KRouterConnectWithString(char* addr_str, char *routername)
#else
KRouterConnectWithString(addr_str, routername)
     char *addr_str;        /* the name of the agent to lookup */
     char *routername; /* the Router name */
#endif
{
  int retval;
  Address* myaddress = (Address *)ConvertKQMLToAddress(addr_str);
  if(myaddress == NULL) {
    return (K_BAD_KQML);
  }

  retval = KRouterConnect(myaddress,routername);
  free(myaddress);
  return retval;
} 

/*
 *----------------------------------------------------------------------
 *
 * KRouterConnect : Connect to the Router
 *
 * Results:
 *	K_OK = success
 *      K_BAD_URL = can't parse url
 *      K_NO_ANS_RESPONSE = no response from the name server
 *      K_ERROR = to use AgentLookup you must compile with HTTP support,
 *                or another error occured
 *
 * Side effects:
 *      none
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KRouterConnect(Address *myaddress, char *routername)
#else
KRouterConnect(myaddress, routername)
     Address *myaddress;        /* the name of the agent to lookup */
     char *routername; /* the Router name */
#endif
{
  ParsedURL *routerURL = NULL;
  char buffer[K_MAX_MESSAGE_SIZE];
  char* responsePerf;

  bzero(buffer, K_MAX_MESSAGE_SIZE);

  routerURL = (ParsedURL*)getReceiverURL(routername);

  /* always should send first */
  strcpy(buffer,"(reconnect-agent :sender ");
  strcat(buffer,myaddress->name);
  strcat(buffer," :password ");
  strcat(buffer,myaddress->password);
  strcat(buffer," :receiver ");
  strcat(buffer,routername);
  strcat(buffer," :host ");
  strcat(buffer,myaddress->host);
  strcat(buffer," :port ");
  strcat(buffer,myaddress->port);
  strcat(buffer,")");

  if(!KSendString(buffer) == K_OK) {
    return K_ERROR;
  }

  if (!(routerURL && getFromTransport(routerURL, buffer))) {
    return (K_NO_ANS_RESPONSE);
  }

  bzero(buffer, K_MAX_MESSAGE_SIZE);
  if(!strncmp(buffer,"201 AMR",7)) {
printf("%s\n","Wrong initial connection message is received from the Roturer");
    return (K_ERROR);
  }

  bzero(buffer, K_MAX_MESSAGE_SIZE);
  if (!getFromTransport(routerURL, buffer)) {
printf("%s\n",buffer);
    return (K_NO_ANS_RESPONSE);
  }

  KGetField(buffer,NULL,&responsePerf,0);

  if(!(strncmp(responsePerf,"connection-accepted",19) == 0)) {
printf("%s\n","No, Connection to the Router");
    if(responsePerf != NULL)  free(responsePerf);
    return (K_ERROR);
  }

  if(responsePerf != NULL)  free(responsePerf);
  return (K_OK);
}


/*
 *----------------------------------------------------------------------
 *
 * KSendStringToAMR
 *
 * Send message to the router. The router will parse the KQML message
 * and deliver it to the receiver.
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KSendStringToAMR(char* message,char* routername)
#else
KSendStringToAMR(message,routername)
     char *message;        /* this is a KQML message to send */
     char *routername;
#endif
{
    ParsedURL *purl = NULL;
    char* sender = NULL;
    char* receiver = NULL;

    KGetField(message,":receiver",&receiver,0);
    KGetField(message,":sender",&sender,0);

    if((sender == NULL) || (receiver == NULL)) {
      return K_NO_RECEIVER;
    }


#if ADD_CONN_INFO
    if ((entry = FindHashEntry(&OpenConns, routername)) != NULL &&
	(purl = (ParsedURL *) GetHashValue(entry)) != NULL &&
	(purl->state == OPEN_CON) &&
	sendOnTransport(purl, message) != -1) {
        if(sender != NULL)  free(sender);
        if(receiver != NULL)  free(receiver);	
	return(K_OK);
    }
#endif

    /* Connection is closed. Reconnect */
    purl = (ParsedURL*)getReceiverURL(routername);
    while (purl != NULL) {
	if (sendOnTransport(purl,message) != -1) {
	  if(sender != NULL)  free(sender);
	  if(receiver != NULL)  free(receiver);	
	  return(K_OK);
	}
	purl = purl->alternate;
    }

    if(sender != NULL)  free(sender);
    if(receiver != NULL)  free(receiver);	

#if (DEBUG > 0)
    fprintf(stderr, "Send failure: Could not find a suitable transport.\n");
#endif
    return K_NO_TRANSPORT;
}

/*
 *----------------------------------------------------------------------
 *
 * getRouterMail --
 *
 *	Retrieves a raw string
 *
 * Results:
 *	received raw string message if successful
 *      NULL for failure
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */

char *
#if defined(__STDC__) || defined(__cplusplus)
getRouterMail(ParsedURL *purl)
#else
getRouterMail(purl)
     ParsedURL *purl;    
#endif
{
  char buffer[K_MAX_MESSAGE_SIZE];
  char* retstr;
  int msglen;

  if (getFromTransport(purl, buffer)) {
    msglen = strlen(buffer) + 1;
    retstr = (char*)calloc(sizeof(char),msglen);
    strcpy(retstr,buffer);
    return retstr;
  }

  return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * getRouterMail --
 *
 *	Retrieves a parsed KQML message
 *
 * Results:
 *	received parsed message if successful
 *      NULL for failure
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */
ParsedKQMLMess *
#if defined(__STDC__) || defined(__cplusplus)
getParsedRouterMail(char *mail)
#else
getParsedRouterMail(mail)
  char *mail;
#endif
{
  char *address = NULL;
  char *sender = NULL;
  ParsedKQMLMess *pmessage = 
    (ParsedKQMLMess *) calloc(1, sizeof(ParsedKQMLMess));
  char buffer[K_MAX_MESSAGE_SIZE];

  /* The following approach is used to keep non-ANSI compilers happy.
     gcc allows direct initializtion of the KTable as in
   KeydPair KTable[] = {
    {":sender", &sender},
    {":receiver", &(pmessage->receiver)},
    {(char *) NULL, (char **) NULL}
  }; */
  KeydPair KTable[3];
  char *key1 = ":sender";
  char *key2 = ":receiver";
  KTable[0].keyword = key1;
  KTable[0].value_address = &sender;
  KTable[1].keyword = key2;
  KTable[1].value_address = &(pmessage->receiver);
  KTable[2].keyword = (char *) NULL;
  KTable[2].value_address = (char **) NULL;

  /* Eliminate read flag and time stamp */
  if ((KParse(mail + 15, 
	      &(pmessage->performative), 
	      KTable, 
	      &(pmessage->other_pairs)) != K_OK) || 
      pmessage->performative == NULL) {
    return(NULL);
  }

  if (sender == NULL) {
    sender = (char *) malloc(strlen("unknown") + 1);
    strcpy(sender,"unknown");
  }

  pmessage->sender = sender;

  return(pmessage);
}

/*
 *----------------------------------------------------------------------
 *
 * getRouterString_or_Wait
 *
 *	Receive a message from the router
 *
 * Results:
 *	received message if successful
 *      NULL for failure
 *
 * Side effects:
 *
 *
 *----------------------------------------------------------------------
 */
int
#if defined(__STDC__) || defined(__cplusplus)
KGetRouterString_or_Wait(char* str,char *routername)
#else
KGetRouterString_or_Wait(str,routername)
  char *str;
  char  *routername;    /* a buffer for the string received */
#endif
{
    int result = K_OK;
    ParsedURL *purl = NULL;
    ParsedKQMLMess *pmessage = NULL;
    
    char* rawmail;

    /* Wait for a message  */
    if ((purl = (ParsedURL*)getReceiverURL(routername)) == NULL) {
	return K_NO_ACTIVE_URL;
    }

    if ((rawmail = getRouterMail(purl)) == NULL) {
	return K_NO_MSG;
    }

    /* parse raw message to kqml, extracting read tag and time stamp */
    pmessage = (ParsedKQMLMess*)getParsedRouterMail(rawmail);
    if(pmessage == NULL) {
      free(rawmail);
      return K_BAD_KQML;
    }

    if(rawmail != NULL) {
      free(rawmail);
    }
    
    if (ParsedMessToString(pmessage, str) == NULL) {
      return (K_BAD_KQML);
    }

    DeletePMessage(pmessage);
    return(result);    
}

/*
 *----------------------------------------------------------------------
 *
 * KDisconnectRouter : Disconnect from the router
 *
 *----------------------------------------------------------------------
 */
int
#if defined(__STDC__) || defined(__cplusplus)
KDisconnectRouter(char* myname,char* routername)
#else
KDisconnectRouter(myname,routername)
  char *myname;
  char  *routername;    /* a buffer for the string received */
#endif
{
    int result = K_OK;

    char buffer[K_MAX_MESSAGE_SIZE];
    if((myname == NULL) || (routername == NULL)) {
      return K_BAD_URL;
    }
    
    strcpy(buffer,"(disconnect-agent :sender ");
    strcat(buffer,myname);
    strcat(buffer," :receiver ");
    strcat(buffer,routername);
    strcat(buffer,")");

    KSendStringToAMR(buffer,routername);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * KUnregisterRouterWithString : Unregister from the router by passing
 * address string
 *
 *----------------------------------------------------------------------
 */
int
#if defined(__STDC__) || defined(__cplusplus)
KUnregisterRouterWithString(char* addrstr,char* routername)
#else
KUnregisterRouterWithString(addrstr,routername)
  char *addrstr;
  char  *routername;    /* a buffer for the string received */
#endif
{

  int retval;
  Address* myaddress = (Address *)ConvertKQMLToAddress(addrstr);
  if(myaddress == NULL) {
    return (K_BAD_KQML);
  }

  retval = KUnregisterRouter(myaddress,routername);
  free(myaddress);
  return retval;  
}

/*
 *----------------------------------------------------------------------
 *
 * KUnregisterRouter : Unregister from the router by passing
 * address object
 *
 *----------------------------------------------------------------------
 */
int
#if defined(__STDC__) || defined(__cplusplus)
KUnregisterRouter(Address* addr,char* routername)
#else
KUnregisterRouter(addr,routername)
  Address *addr;
  char  *routername;    /* a buffer for the string received */
#endif
{
    char buffer[K_MAX_MESSAGE_SIZE];
    if((addr == NULL) || (routername == NULL)) {
      return K_BAD_URL;
    }
    
    strcpy(buffer,"(unregister-agent :sender ");
    strcat(buffer,addr->name);
    strcat(buffer," :password ");
    strcat(buffer,addr->password);
    strcat(buffer," :receiver ");
    strcat(buffer,routername);
    strcat(buffer,")");

    KSendStringToAMR(buffer,routername);
    return K_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * KSendDeleteMessageToAMR : Send delete mail request to the router
 *
 *----------------------------------------------------------------------
 */

int
#if defined(__STDC__) || defined(__cplusplus)
KSendDeleteMessageToAMR(char* myname,char* routername)
#else
KSendDeleteMessageToAMR(myname,routername)
  char* myname;
  char* routername;
#endif
{
    char buffer[K_MAX_MESSAGE_SIZE];
    if((myname == NULL) || (routername == NULL)) {
      return K_BAD_URL;
    }
    
    strcpy(buffer,"(delete-message :sender ");
    strcat(buffer,myname);
    strcat(buffer," :receiver ");
    strcat(buffer,routername);
    /* currently always 0. JATLite is more intelligent, though */
    strcat(buffer," :content 0");
    strcat(buffer,")");

    return (KSendStringToAMR(buffer,routername));
}












